#!/usr/bin/python3

#
# Script to automatically generate documentation for Helm Chart
# -------------------------------------------------------------
#
# FIXME FIXME:
#  just like for SyshealthKpiGenerator, move this logic into config_file_generator.py and
#  make documentation just a new export type --export
#

import xlsxwriter
import xml.etree.ElementTree as ET
import os
import argparse
import yaml
import datetime
from common_helpers import xmlstr_to_pytype
from difflib import SequenceMatcher


SIMILAR_RATIO = 0.8


def are_chart_names_similar(a, b):
    return SequenceMatcher(None, a.lower(), b.lower()).ratio() >= SIMILAR_RATIO


def select_default_type(attrib):
    if "default_type" in attrib:
        return attrib["default_type"]
    else:
        return attrib["type"].split()[0]


XML_INFO_TABLE = [
    {"param_name": "name", "name": "Name", "column_size": 30},
    {"param_name": "default_value", "name": "Default Value", "column_size": 12},
    {"param_name": "type", "name": "Type List", "column_size": 30},
    {"param_name": "default_type", "name": "Default Type", "column_size": 30},
    {"param_name": "available_values", "name": "AvailVal", "column_size": 11},
    {"param_name": "hot_reloadable", "name": "HotRel", "column_size": 8},
    {"param_name": "min_value", "name": "Min", "column_size": 8},
    {"param_name": "max_value", "name": "Max", "column_size": 8},
    {"param_name": "short_desc", "name": "Short Desc", "column_size": 30},
    {"param_name": "long_desc", "name": "Long Desc", "column_size": 45},
    {"param_name": "tag", "name": "Tag", "column_size": 10},
]


#
# FUNCTIONS
#


def parse_xml(xmlfile, app):
    """
    Parse an XML file into a 2D array: [sections; options].
    Each section has option childs.
    Each option contains a dictionary of attributes. For example: name default_value type
    If a option has "documentation" attribute set to "HIDDEN", the option will not be added to result.

    Params: xmlfile: path to the xml file to parse
    Params: app: target application
    Returns: a list of sections, each section contains a list of options.
    """
    root = ET.parse(xmlfile).getroot()

    # if XML file does not have helm_chart attribute, the file will be skipped silently
    if not "helm_chart" in root.attrib:
        return []

    # if XML file's helm_chart attribute does not contain target application name or "all", the file will be skipped
    helm_chart = root.attrib["helm_chart"]
    apps = [x.strip() for x in helm_chart.split(",")]
    if not app in apps and helm_chart.lower() != "all":
        # let us inform the user about the skip
        print(f'INFO: not "{app}" nor "all" was found in helm_chart, skipping...')
        # check if into helm_chart there are application name similar to the target application
        # just in case the user made a typo
        similar_apps = [a for a in apps if are_chart_names_similar(app, a)]
        for similar in similar_apps:
            print(f"WARN: found an app name similar to the target one")
            print(f'WARN: "{app}" vs "{similar}"')
            print(f"WARN: maybe you made a typo?")
        return []

    sections = []
    for section in root:
        tag = section.tag.split("}")[-1]
        if tag == "description":
            sections.append(
                {
                    "description": section.text,
                }
            )
        elif tag == "section":
            # if the attribute has documentation attribute equal to "HIDDEN", such attribute will be skipped silently
            is_hidden = (
                lambda prop: prop.attrib.get("documentation", "").upper() == "HIDDEN"
            )
            if is_hidden(section) or all(is_hidden(opt) for opt in section):
                continue

            _name = (
                section.attrib["name"]
                if "name" in section.attrib
                else section.attrib["pattern"]
            )
            sections.append(
                {
                    "name": _name,
                    "tag": section.attrib.get("tag", ""),
                    "options": [opt for opt in section if not is_hidden(opt)],
                }
            )
    return sections


def filename_to_displayname(full_path):
    # remove extension and preceding path
    filename_without_ext = os.path.splitext(os.path.basename(full_path))[0]
    base_filename_split = filename_without_ext.split("-")
    # concatenation of all words as lowercase in the filename except the last one
    displayname = (
        "".join(base_filename_split[:-1]) + base_filename_split[-1].capitalize()
    )
    return displayname


def export_xlsx(
    xml_filename_list, visible_parsed_infos, export_file, app, version, unit_test=False
):
    """Create the Excel spreadsheeet from the provided information.

    Params: configuration_xmls: list of XML configuration files
    Params: visible_parsed_infos: should be a 4D array: [XML files; sections; options; attributes]
    Params: export_file: full path of output file
    Params: app: Helm Chart
    """
    with xlsxwriter.Workbook(export_file) as workbook:
        if unit_test:
            # Only for unit test, using fixed creation date as we don't want md5 sum to change
            workbook.set_properties({"created": datetime.date(2022, 8, 1)})

        bold = workbook.add_format({"bold": True})
        large_bold = workbook.add_format({"bold": True, "font_size": 30})
        top_wrap = workbook.add_format({"valign": "top", "text_wrap": True})
        top_wrap_consolas = workbook.add_format(
            {"valign": "top", "text_wrap": True, "font_name": "Consolas"}
        )
        top_wrap_and_indent = workbook.add_format(
            {"valign": "top", "text_wrap": True, "indent": 2}
        )

        for i, xml_filename in enumerate(xml_filename_list):
            # Don't include worksheets empty or which have all content in sections or options as HIDDEN
            if visible_parsed_infos[i] == [] or not [
                option
                for section in visible_parsed_infos[i]
                for option in section.get("options", [])
            ]:
                continue

            worksheet_name = filename_to_displayname(xml_filename)
            if len(worksheet_name) > 31:
                worksheet_name = worksheet_name[:28] + "..."
            worksheet = workbook.add_worksheet(worksheet_name)
            worksheet.outline_settings(True, False, True, False)

            # Set columns format
            # set_column() takes [first_col, last_col, width, cell_format, options]
            for col, info in enumerate(XML_INFO_TABLE):
                worksheet.set_column(col, col, info["column_size"], top_wrap)

            # Write worksheet header [HelmChart, ConfigMap, Description]
            row = 0
            col = 0
            description_range = f"B{row + 1}:J{col + 1}"
            worksheet.write(row, col, "HelmChart", large_bold)
            worksheet.merge_range(description_range, app, large_bold)  # type: ignore
            row += 1
            description_range = f"B{row + 1}:J{col + 2}"
            if version:
                worksheet.write(row, col, "Version:", bold)
                worksheet.merge_range(description_range, version, bold)  # type: ignore
                row += 1
                description_range = f"B{row + 1}:J{col + 3}"
            worksheet.write(row, col, "configMap:", bold)
            worksheet.merge_range(description_range, worksheet_name, bold)  # type: ignore
            row += 1

            first_child = visible_parsed_infos[i][0]
            if "description" in first_child:
                description = first_child["description"].strip()
                description_lines = len(description.split("\n"))
                if description_lines > 1:
                    worksheet.merge_range(f"A{row + 1}:A{row + description_lines}", "Description:", top_wrap)  # type: ignore
                    worksheet.merge_range(f"B{row + 1}:J{row + description_lines}", description, top_wrap)  # type: ignore
                else:
                    worksheet.write(row, col, "Description:", top_wrap)
                    worksheet.write(row, col + 1, description, top_wrap)
                row += description_lines

            # Write example by first option of first section
            example_section = next(
                section
                for section in visible_parsed_infos[i]
                if "name" in section and section["options"]
            )
            first_option = example_section["options"][0]
            attrib_name = first_option.attrib["name"]
            default_type = select_default_type(first_option.attrib)
            default_value = xmlstr_to_pytype(
                first_option.attrib["default_value"], default_type
            )
            example = yaml.dump(
                {
                    app: {
                        "answers": {
                            worksheet_name: {
                                example_section["name"]: {attrib_name: default_value}
                            }
                        }
                    }
                }
            )

            # example = yaml.safe_dump(example_obj).strip()
            example_lines = len(example.split("\n"))
            example_title_range = f"A{row + 1}:A{row + example_lines}"
            worksheet.merge_range(example_title_range, "Example:", top_wrap)  # type: ignore
            example_value_range = f"B{row + 1}:J{row + example_lines}"
            worksheet.merge_range(example_value_range, example, top_wrap_consolas)  # type: ignore
            row += example_lines

            # Write headers [Name, Default Value, Type, ...]
            for column, info in enumerate(XML_INFO_TABLE):
                worksheet.write(row, col + column, info["name"], bold)

            for section in visible_parsed_infos[i]:
                if "name" in section:
                    row += 1

                    section_title = section["name"] + ":"
                    worksheet.write(row, col, section_title, bold)

                    for option in section["options"]:
                        row += 1

                        # Groups rows of the same section
                        worksheet.set_row(row, None, None, {"level": 1})

                        for column, info in enumerate(XML_INFO_TABLE):
                            pname = info["param_name"]
                            to_print = option.attrib.get(pname, "")
                            if pname == "name":
                                worksheet.write(
                                    row,
                                    col + column,
                                    to_print + ":",
                                    top_wrap_and_indent,
                                )
                            elif pname == "default_type" and to_print == "":
                                worksheet.write(
                                    row,
                                    col + column,
                                    option.attrib["type"].split()[0],
                                    top_wrap,
                                )
                            else:
                                worksheet.write(row, col + column, to_print, top_wrap)
    print(f"Successfully generate the documentation in Excel format as {export_file}")


#
# MAIN
#
if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Script to automatically generate a XLSX file with configuration parameters descriptions"
    )
    parser.add_argument(
        "--app", type=str, help="Target Application (Example: eva-correlator)"
    )
    parser.add_argument(
        "--xml-list",
        nargs="+",
        type=str,
        help="XML list of Helm Chart configuration files",
        required=True,
    )
    parser.add_argument(
        "--output-dir", type=str, default="./", help="Output directory (default: ./)"
    )
    parser.add_argument(
        "--version", type=str, help="Helm Chart version of the Target Application"
    )
    args = parser.parse_args()

    xml_list = sorted(args.xml_list)
    visible_parsed_infos = [
        parse_xml(xml_filename, args.app) for xml_filename in xml_list
    ]

    # generate xlsx
    output_filename = os.path.join(
        os.path.abspath(args.output_dir),
        "-".join(
            filter(None, (args.app, args.version, "configparams-documentation.xlsx"))
        ),
    )
    export_xlsx(xml_list, visible_parsed_infos, output_filename, args.app, args.version)
