import os, json
import ruamel.yaml as yaml
from config_file_generator_constants import COMMENT_OFFSET
from option_properties import OptionProperties
import common_helpers


class YamlConfigFileGenerator(object):
    def __init__(self, logger):
        self.logger = logger

    def process_yaml_property(
        self, yaml_cm, property, parent_json_schema, skip_yaml_write=False
    ):
        option_properties = OptionProperties(property)

        if option_properties.required:
            if not "required" in parent_json_schema:
                parent_json_schema["required"] = list()
            parent_json_schema["required"].append(option_properties.name)
        parent_json_schema["properties"][
            option_properties.name
        ] = option_properties.get_json_schema()

        if option_properties.name and option_properties.type_list:
            if not skip_yaml_write:
                if isinstance(option_properties.default_value, list):
                    yaml_cm[option_properties.name] = yaml.comments.CommentedSeq(option_properties.default_value)  # type: ignore
                    yaml_cm[option_properties.name].fa.set_flow_style()
                else:
                    yaml_cm[option_properties.name] = option_properties.default_value
                yaml_cm.yaml_add_eol_comment(
                    option_properties.get_comment(),
                    option_properties.name,
                    column=COMMENT_OFFSET,
                )

            if option_properties.children is not None:
                parent_json_schema["properties"][option_properties.name]["items"][
                    "properties"
                ] = dict()
                for option_property in option_properties.children:
                    self.process_yaml_property(
                        yaml_cm[option_properties.name],
                        option_property,
                        parent_json_schema["properties"][option_properties.name][
                            "items"
                        ],
                        skip_yaml_write=(
                            option_properties.default_type == "object_list"
                        ),
                    )
        else:
            common_helpers.throw_malformed_error(option_properties)

    def process_yaml_section(self, yaml_cm, json_schema, section):
        skip_yaml_write = False
        section_name = None
        block_name = None

        if "@name" in section:
            section_name = section["@name"]
            block_name = "properties"

        if "@pattern" in section:
            section_name = section["@pattern"]
            block_name = "patternProperties"
            skip_yaml_write = True

        if not section_name:
            # Either 'name' or 'pattern' attribute should be present in section
            common_helpers.throw_error(
                f"Either 'name' or 'pattern' attribute should be present in 'section'"
            )

        yaml_cm[section_name] = yaml.comments.CommentedMap()  # type: ignore
        json_schema[block_name] |= {
            section_name: {
                "type": "object",
                "properties": dict(),
                "additionalProperties": False,
            }
        }
        section_schema = json_schema[block_name][section_name]

        option_list = common_helpers.listify(section["option"])
        for option in option_list:
            # Check if there are options sharing the same name
            if len([opt for opt in option_list if opt["@name"] == option["@name"]]) > 1:
                common_helpers.throw_error(
                    f"Duplicate option {option['@name']} into {section_name} section"
                )

            self.process_yaml_property(
                yaml_cm[section_name], option, section_schema, skip_yaml_write
            )
        if skip_yaml_write:
            yaml_cm.pop(section_name)

    def print_yaml_config_file(
        self, xml_data_dict, first_xml_comm, out_file, out_json_schema
    ):
        base_filename_split = os.path.basename(out_file).split(".yaml")[0].split("-")
        capitalized_filename = base_filename_split[0] + "".join(
            x.capitalize() for x in base_filename_split[1:]
        )

        yaml_cm = yaml.comments.CommentedMap()  # type: ignore
        if first_xml_comm:
            yaml_cm.yaml_set_start_comment(first_xml_comm)

        json_schema = {
            "$schema": "http://json-schema.org/schema#",
            "type": "object",
            # "required": [capitalized_filename],
            "properties": {
                capitalized_filename: {
                    "type": "object",
                    "properties": dict(),
                    "patternProperties": dict(),
                    "additionalProperties": False,
                }
            },
        }

        section_list = common_helpers.listify(xml_data_dict["config-file"]["section"])
        for section in section_list:
            self.process_yaml_section(
                yaml_cm, json_schema["properties"][capitalized_filename], section
            )

        with open(out_json_schema, "w") as json_schema_file:
            json.dump(json_schema, json_schema_file, indent=2)

        with open(out_file, "w") as config_file:
            y = yaml.YAML()
            y.indent(mapping=2, sequence=2, offset=2)
            y.dump(yaml_cm, config_file)
