#
# Makefile TARGETS code to deploy artifacts to Empirix repositories
# Make sure you have included "Deploy-defines.mk" before including this one.
#
# Targets defined by this makefile:
# - deploy_[rhel7_rpm|rhel8_rpm|maven|python|conan|docker|helm]
#

ifeq ($(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR),)
$(error Please define RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR before including this snippet)
endif
ifeq ($(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR),)
$(error Please define EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR before including this snippet)
endif

ifndef SUPPORTS_HELP
# default value
SUPPORTS_HELP=1
endif

# below we use Bash-specific features in this makefile:
SHELL := /bin/bash


# ---------------------------------------------
# Main scripts used to smartly deploy artifacts
# ---------------------------------------------

RPM_DEPLOY_WRAPPER_SCRIPT=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/rpm-deploy-wrapper.sh
RPM_LIST_DEPLOY_WRAPPER_SCRIPT=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/rpm-list-deploy-wrapper.sh
DOCKER_DEPLOY_WRAPPER_SCRIPT=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/docker-deploy-wrapper.sh
CONAN_DEPLOY_WRAPPER_SCRIPT=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/conan-deploy-wrapper.sh
HELM_DEPLOY_WRAPPER=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/helm-deploy-wrapper.sh
PYTHON_DEPLOY_WRAPPER=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/python-deploy-wrapper.sh
REDEPLOY_WRAPPER_SCRIPT=${EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR}/redeploy-artifacts.sh


# ---------------------------------------------
# Credential check targets
# ---------------------------------------------

artifactory_credentials_check:
ifeq ($(ARTIFACTORY_USER),)
	@echo "Please call this makefile target providing ARTIFACTORY_USER env variable"
	@exit 42
endif
ifeq ($(ARTIFACTORY_PW),)
	@echo " Please call this makefile target providing ARTIFACTORY_PW env variable"
	@exit 42
endif

nexus_credentials_check:
ifeq ($(NEXUS_USER),)
	@echo "Please call this makefile target providing NEXUS_USER env variable"
	@exit 42
endif
ifeq ($(NEXUS_PW),)
	@echo "Please call this makefile target providing NEXUS_PW env variable"
	@exit 42
endif

# ---------------------------------------------
# DEPLOY targets
# ---------------------------------------------

# Before invoking one of the "deploy_*" targets just invoke this target to make sure
# the logfile of deployed artifacts is cleaned up from previous runs:
deploy_clean_logfile::
	@mkdir -p $(shell dirname $(DEPLOY_FILELIST_LOG))
	@cat /dev/null >$(DEPLOY_FILELIST_LOG)

deploy_show_destination::
	@echo "=================================================================================="
	@echo " Destination for deploy targets"
	@echo "=================================================================================="
	@echo "Destination for Docker (REPO_TYPE_FOR_DOCKER_DEPLOY) is: $(REPO_TYPE_FOR_DOCKER_DEPLOY)"
	@echo "Destination for Helm (REPO_TYPE_FOR_HELM_DEPLOY) is: $(REPO_TYPE_FOR_HELM_DEPLOY)"
	@echo "Destination for RPM (REPO_TYPE_FOR_RPM_DEPLOY) is: $(REPO_TYPE_FOR_RPM_DEPLOY)"
	@echo "Destination for Pypi (REPO_TYPE_FOR_PYTHON_DEPLOY) is: $(REPO_TYPE_FOR_PYTHON_DEPLOY)"
	@echo "Destination for Conan (REPO_TYPE_FOR_CONAN_DEPLOY) is: $(REPO_TYPE_FOR_CONAN_DEPLOY)"
	@echo "Destination for Maven (REPO_TYPE_FOR_MAVEN_DEPLOY) is: $(REPO_TYPE_FOR_MAVEN_DEPLOY)"
	@echo "Destination decided based on:" 
	@echo "  GIT_BRANCH_TYPE=$(GIT_BRANCH_TYPE)"
	@echo "  GIT_BRANCH_NAME=$(GIT_BRANCH_NAME)"
	@echo "  NUM_COMMITS_SINCE_LAST_TAG=$(NUM_COMMITS_SINCE_LAST_TAG)"
	@echo "  BUILD_AGENT=$(BUILD_AGENT)"
	@echo "  FULL_VERSION=$(FULL_VERSION)"

# RPM deploy through curl: to use just define 
#       RHEL7_RPMS: list of RPMs to deploy to RHEL7-specific repository
deploy_rhel7_rpm::
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),nexus)
	@echo "##teamcity[blockOpened name='Deploying RHEL7 RPMs on Nexus']"
	@echo "=================================================================================="
	@echo " Deploying RHEL7 RPM artifacts on Empirix Nexus servers"
	@echo "=================================================================================="
	@$(MAKE) nexus_credentials_check
	$(RPM_LIST_DEPLOY_WRAPPER_SCRIPT) "$(NEXUS_USER)" "$(NEXUS_PW)" \
		"$(REMOTE_REPO_NEXUS_RPM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" $(RHEL7_RPMS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying RHEL7 RPMs on Nexus']"
else
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),artifactory)
	@echo "##teamcity[blockOpened name='Deploying RHEL7 RPMs on Artifactory']"
	@echo "=================================================================================="
	@echo " Deploying RHEL7 RPM artifacts on Empirix Artifactory server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	$(RPM_LIST_DEPLOY_WRAPPER_SCRIPT) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" \
		"$(REMOTE_REPO_ARTIFACTORY_RPM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" $(RHEL7_RPMS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying RHEL7 RPMs on Artifactory']"
else
$(error Unsupported value for REPO_TYPE_FOR_RPM_DEPLOY=$(REPO_TYPE_FOR_RPM_DEPLOY))
endif
endif

# RPM deploy through curl: to use just define 
#       RHEL8_RPMS: list of RPMs to deploy to RHEL8-specific repository
deploy_rhel8_rpm::
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),nexus)
	@echo "##teamcity[blockOpened name='Deploying RHEL8 RPMs on Nexus']"
	@echo "=================================================================================="
	@echo " Deploying RHEL8 RPM artifacts on Empirix Nexus servers"
	@echo "=================================================================================="
	@$(MAKE) nexus_credentials_check
	$(RPM_LIST_DEPLOY_WRAPPER_SCRIPT) "$(NEXUS_USER)" "$(NEXUS_PW)" \
		"$(REMOTE_REPO_NEXUS_RHEL8_RPM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" $(RHEL8_RPMS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying RHEL8 RPMs on Nexus']"
else
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),artifactory)
	@echo "##teamcity[blockOpened name='Deploying RHEL8 RPMs on Artifactory']"
	@echo "=================================================================================="
	@echo " Deploying RHEL8 RPM artifacts on Empirix Artifactory server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	$(RPM_LIST_DEPLOY_WRAPPER_SCRIPT) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" \
		"$(REMOTE_REPO_ARTIFACTORY_RHEL8_RPM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" $(RHEL8_RPMS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying RHEL8 RPMs on Artifactory']"
else
$(error Unsupported value for REPO_TYPE_FOR_RPM_DEPLOY=$(REPO_TYPE_FOR_RPM_DEPLOY))
endif
endif

# RPM deploy through curl: to use just define 
#       OL9_RPMS: list of RPMs to deploy to OL9-specific repository
deploy_ol9_rpm::
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),nexus)
	@echo "##teamcity[blockOpened name='Deploying OL9 RPMs on Nexus']"
	@echo "=================================================================================="
	@echo " Deploying OL9 RPM artifacts on Empirix Nexus servers"
	@echo "=================================================================================="
	@$(MAKE) nexus_credentials_check
	$(RPM_LIST_DEPLOY_WRAPPER_SCRIPT) "$(NEXUS_USER)" "$(NEXUS_PW)" \
		"$(REMOTE_REPO_NEXUS_OL9_RPM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" $(OL9_RPMS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying OL9 RPMs on Nexus']"
else
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),artifactory)
	@echo "##teamcity[blockOpened name='Deploying OL9 RPMs on Artifactory']"
	@echo "=================================================================================="
	@echo " Deploying OL9 RPM artifacts on Empirix Artifactory server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	$(RPM_LIST_DEPLOY_WRAPPER_SCRIPT) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" \
		"$(REMOTE_REPO_ARTIFACTORY_OL9_RPM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" $(OL9_RPMS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying OL9 RPMs on Artifactory']"
else
$(error Unsupported value for REPO_TYPE_FOR_RPM_DEPLOY=$(REPO_TYPE_FOR_RPM_DEPLOY))
endif
endif

# JAR deploy through Maven: to use just define 
#       JAVA_DIRS: list of Java folders having a pom.xml that need to have their "deploy" target executed
#
# NOTE1: the mvn deploy target can fail if the build number already exist in nexus, hence the || true
# NOTE2: we want to make sure the version:revert is always executed to leave the workspace in clean state
# NOTE3: the targets of the deploy are defined in the <distributionManagement> tag of the pom.xml;
#        the credentials for accessing these repos are defined instead in your $HOME/.m2/settings.xml
# NOTE4: we MUST deploy all child POMs but also all the parent POMs otherwise when trying to use a child POM
#        you will get an error like:
#             [ERROR] ... Failed to read artifact descriptor for com.empirix:events-gpb-lib:jar:1.0.13-develop-SNAPSHOT: 
#             Could not find artifact com.empirix:events-networking-parent:pom:1.0.13-develop-SNAPSHOT in empirix
#        Thus you typically want to have JAVA_DIRS include the '.' directory to deploy the root pom XML.
# NOTE5: the logic here assumes that the <distributionTag> inside the pom.xml is using ${env.REMOTE_REPO_JAVA_RELEASE_FOR_PUSH}
#        and ${env.REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH} as remote repo URLs
#
deploy_maven:
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='Deploying Java artifacts']" || true
	@echo "=================================================================================="
	@echo " Deploying Java artifacts with version=[$(JAVA_FULL_VERSION)] to $(REMOTE_REPO_JAVA_FOR_PUSH)"
	@echo " Remote repository for Maven RELEASES=[$(REMOTE_REPO_JAVA_RELEASE_FOR_PUSH)]"
	@echo " Remote repository for Maven SNAPSHOTS=[$(REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH)]"
	@echo " Selected remote repository for Maven=[$(REMOTE_REPO_JAVA_FOR_PUSH)]"
	@echo "=================================================================================="
	@echo "Using Java version:"
	@java --version
	@echo "Setting version with Maven on all packages:"
	$(MAVEN) versions:set -DnewVersion=$(JAVA_FULL_VERSION)
	@echo "Start the clean+deploy in each Java directory:"
	@for dir in $(JAVA_DIRS); do \
		REMOTE_REPO_JAVA_RELEASE_FOR_PUSH=$(REMOTE_REPO_JAVA_RELEASE_FOR_PUSH) \
		REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH=$(REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH) \
			$(MAVEN) --batch-mode --non-recursive --file $$dir \
				clean deploy 2>&1 || true ; \
		POM_ARTIFACT_ID=$$(xmlstarlet sel -N x='http://maven.apache.org/POM/4.0.0' -t -v '/x:project/x:artifactId' $$dir/pom.xml) ; \
		echo "jar,${REMOTE_REPO_JAVA_FOR_PUSH},$${POM_ARTIFACT_ID}/$(JAVA_FULL_VERSION),deployed" | tee -a $(DEPLOY_FILELIST_LOG) ; \
	done
	@echo "Rolling back version with Maven on all packages:"
	$(MAVEN) versions:revert
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='Deploying Java artifacts']" || true


#
# NOTE: to use deploy_python you need to provide 1 make variable:
#               PYTHON_DIRS: list of python folders having a python package to deploy
deploy_python::
ifeq ($(REPO_TYPE_FOR_PYTHON_DEPLOY),nexus)
	@echo "##teamcity[blockOpened name='Deploying Python Packages on Nexus']"
	@echo "=================================================================================="
	@echo " Deploying Pypi packages on Empirix Nexus server"
	@echo "=================================================================================="
	@$(MAKE) nexus_credentials_check
	$(PYTHON_DEPLOY_WRAPPER) "$(NEXUS_USER)" "$(NEXUS_PW)" \
		"$(REMOTE_REPO_NEXUS_PYTHON_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" \
		$(PYTHON_DIRS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Python Packages on Nexus']"
else
ifeq ($(REPO_TYPE_FOR_PYTHON_DEPLOY),artifactory)
	@echo "##teamcity[blockOpened name='Deploying Python Packages on Artifactory']"
	@echo "=================================================================================="
	@echo " Deploying Pypi packages on Empirix Artifactory server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	$(PYTHON_DEPLOY_WRAPPER) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" \
		"$(REMOTE_REPO_ARTIFACTORY_PYTHON_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" \
		$(PYTHON_DIRS) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Python Packages on Artifactory']"
else
$(error Unsupported value for REPO_TYPE_FOR_PYTHON_DEPLOY=$(REPO_TYPE_FOR_PYTHON_DEPLOY))
endif
endif

#
# NOTE1: to use deploy_conan you need to provide 1 make variable:
#               CONAN_PACKAGES: list of conan packages to deploy to the repo
# NOTE2: since Nexus Conan hosted repo is very buggy, we always deploy to some Artifactory repository, never Nexus.
# NOTE3: since July 2021, we can choose between the Artifactory cloud instance on JFrog or Empirix own ArtifactoryCE 
#        instance on repo-billerica server
deploy_conan::
ifeq ($(REPO_TYPE_FOR_CONAN_DEPLOY),artifactory_billerica)
	@echo "##teamcity[blockOpened name='Deploying Conan Packages on Artifactory Billerica']"
	@echo "=================================================================================="
	@echo " Deploying Conan packages on Empirix Artifactory server (REPO_TYPE_FOR_CONAN_DEPLOY=$(REPO_TYPE_FOR_CONAN_DEPLOY))"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	TOPDIR=$(REALTOP) $(CONAN_DEPLOY_WRAPPER_SCRIPT) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" "$(REMOTE_REPO_ARTIFACTORY_BILLERICA_CONAN_NICKNAME)" \
		"$(REMOTE_REPO_ARTIFACTORY_BILLERICA_CONAN_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" \
		$(CONAN_PACKAGES) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Conan Packages on Artifactory Billerica']"
else
ifeq ($(REPO_TYPE_FOR_CONAN_DEPLOY),artifactory_cloud)
	@echo "##teamcity[blockOpened name='Deploying Conan Packages on Artifactory Cloud']"
	@echo "=================================================================================="
	@echo " Deploying Conan packages on Artifactory cloud server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	TOPDIR=$(REALTOP) $(CONAN_DEPLOY_WRAPPER_SCRIPT) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" "$(REMOTE_REPO_ARTIFACTORY_CLOUD_CONAN_NICKNAME)" \
		"$(REMOTE_REPO_ARTIFACTORY_CLOUD_CONAN_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" \
		$(CONAN_PACKAGES) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Conan Packages on Artifactory Cloud']"
else
$(error Unsupported value for REPO_TYPE_FOR_CONAN_DEPLOY=$(REPO_TYPE_FOR_CONAN_DEPLOY))
endif
endif

#
# NOTE1: to use deploy_docker you need to provide 2 make variables:
#               DOCKER_GROUP: used to define the final location of the docker image e.g. "capture"
#               DOCKER_IMAGES: list of images, space separated, to push in that group e.g. "capture:1.2.0 capture-drivers:1.2.0"
# NOTE2: the choice between Nexus / Artifactory / ECR is driven by the ECR_REGISTRY env variable presence / absence
# NOTE3: since March 2023, the default is to deploy docker images on Amazon ECR always
deploy_docker::
ifeq ($(REPO_TYPE_FOR_DOCKER_DEPLOY),ecr)
	@echo "##teamcity[blockOpened name='Deploying Docker images on Amazon ECR']"
	@echo "=================================================================================="
	@echo " Deploying docker images on Amazon ECR"
	@echo "=================================================================================="
	$(DOCKER_DEPLOY_WRAPPER_SCRIPT) "$(ECR_ACCESS_KEY)" "$(ECR_SECRET_KEY)" \
		"$(REMOTE_REPO_ECR_DOCKER_FOR_PUSH)" "$(REMOTE_REPO_ECR_DOCKER_FOR_PUSH)/$(DOCKER_GROUP)" \
		"$(DEPLOY_FILELIST_LOG)" "$(DOCKER_GROUP)" \
		$(DOCKER_IMAGES) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Docker images on Amazon ECR']"
else
ifeq ($(REPO_TYPE_FOR_DOCKER_DEPLOY),artifactory)
	@echo "##teamcity[blockOpened name='Deploying Docker images on Artifactory']"
	@echo "=================================================================================="
	@echo " Deploying docker images on Empirix Artifactory server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	$(DOCKER_DEPLOY_WRAPPER_SCRIPT) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" \
		"$(REMOTE_REPO_ARTIFACTORY_DOCKER_FOR_PUSH)" "$(REMOTE_REPO_ARTIFACTORY_DOCKER_FOR_PUSH)/$(DOCKER_GROUP)" \
		"$(DEPLOY_FILELIST_LOG)" "$(DOCKER_GROUP)" \
		$(DOCKER_IMAGES) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Docker images on Artifactory']"
endif
endif

#
# NOTE: to use deploy_helm you need to provide 1 make variable:
#               HELM_PACKAGES_LIST: list of helm packages to deploy to the repo
deploy_helm::
ifeq ($(REPO_TYPE_FOR_HELM_DEPLOY),nexus)
	@echo "##teamcity[blockOpened name='Deploying Helm package on Nexus']"
	@echo "=================================================================================="
	@echo " Deploying helm charts on Empirix Nexus servers"
	@echo "=================================================================================="
	@$(MAKE) nexus_credentials_check
	$(HELM_DEPLOY_WRAPPER) "$(NEXUS_USER)" "$(NEXUS_PW)" \
		"$(REMOTE_REPO_NEXUS_HELM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" \
		$(HELM_PACKAGES_LIST) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Helm package on Nexus']"
else
ifeq ($(REPO_TYPE_FOR_HELM_DEPLOY),artifactory)
	@echo "##teamcity[blockOpened name='Deploying Helm package on Artifactory']"
	@echo "=================================================================================="
	@echo " Deploying helm charts on Empirix Artifactory server"
	@echo "=================================================================================="
	@$(MAKE) artifactory_credentials_check
	$(HELM_DEPLOY_WRAPPER) "$(ARTIFACTORY_USER)" "$(ARTIFACTORY_PW)" \
		"$(REMOTE_REPO_ARTIFACTORY_HELM_FOR_PUSH)" "$(DEPLOY_FILELIST_LOG)" \
		$(HELM_PACKAGES_LIST) 2>&1
	@echo "##teamcity[blockClosed name='Deploying Helm package on Artifactory']"
else
$(error Unsupported value for REPO_TYPE_FOR_HELM_DEPLOY=$(REPO_TYPE_FOR_HELM_DEPLOY))
endif
endif


# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Deploy targets (to run INSIDE the builder docker):"
	@echo "    deploy_show_destination:              show what's the destination repo for each type of artifact"
	@echo "    deploy_ol9_rpm:                       deploy all RPMs related to OL9 operating systems"
	@echo "    deploy_rhel7_rpm:                     deploy all RPMs related to RHEL/Centos7 operating systems"
	@echo "    deploy_rhel8_rpm:                     deploy all RPMs related to RHEL/Centos8 operating systems"
	@echo "    deploy_maven:                         deploy all Java/Maven-based projects JARs"
	@echo "    deploy_python:                        deploy all Python-based projects pypis"
	@echo "    deploy_conan:                         deploy all Conan packages"
	@echo "    deploy_docker:                        deploy all Docker images"
	@echo "    deploy_helm:                          deploy all Helm charts"
endif
