#
# This makefile contains the logic to define the version numbers used for packaging
# DiagnostixCloud EvA/ECC software releases (see main Makefile PACKAGING targets)
#
# The calling Makefile must define:
#     BUILD_AGENT
#         If 1 indicates that we are running inside the CI/CD of this project (like Teamcity or Gitlab).
#         If 0 indicates that instead we are running on a developer machine
# The calling Makefile can optionally define:
#     GIT_DESCRIBE
#         This should be the contents of the 'git describe --long' operation. Prepopulating this variable
#         is useful in cases where the Makefile runs, for whatever reason, without the .git folder
#         (e.g. during Conan packaging)
#     GIT_BRANCH_NAME
#         This should be the name of the git branch, verbatim, e.g. "feature/IN-XYZ-myfeature-branch"
#     GIT_STABLE_BRANCH
#         This should be the name of the branch which is meant to produce stable, persistent artifacts.
#         It defaults to "master" but caller makefile may want to change it to "develop".
#         Artifacts coming out of non-stable branches are typically marked as "snapshots" or "unstable"
#         and will be cleaned up from their repositories at some time.
#     GIT_BRANCH_NAME_REGEX
#         This is the regex that must match GIT_BRANCH_NAME.
#         The default regex is (^(feature|bugfix|hotfix|support)\/[-a-zA-Z0-9.]{1,32}$$)|(^(master|HEAD|develop))
# Version symbols exposed to the calling Makefile:
#     GIT_BRANCH_TYPE
#     [FULL|DOCKER|RPM|JAVA|CONAN|HELM|SOLIB]_VERSION
#     CONAN_CHANNEL


# ------------------------------------------------------------------------------------------
# Variables from caller makefile
# ------------------------------------------------------------------------------------------

ifeq ($(BUILD_AGENT),)
$(error Please define BUILD_AGENT before including this snippet)
endif

ifndef GIT_STABLE_BRANCH
# default value
GIT_STABLE_BRANCH=develop
endif


# ------------------------------------------------------------------------------------------
# Extraction of version digits from git tags
# ------------------------------------------------------------------------------------------

# get the git-defined smart versioning using "git describe": 
#    LAST_TAG-NUM_COMMITS_SINCE_LAST_TAG-gLAST_COMMIT_HASH
# IMPORTANT:
#   a) --long is required because in case we are exactly on a tag commit, without --long the format of git describe would be just the tagname
#   b) --tags is required to consider not only annotated tags. E.g. KRAMER creates non-annotated tags. 
#   c) when parsing the "LAST_TAG" part we assume that the tags created in git have just two or three digits: MAJOR.MINOR.PATCH following 
#      https://semver.org/ guidelines: MAJOR version when you make incompatible API changes, MINOR version when you add functionality in 
#      a backwards compatible manner, PATCH version when you make backwards compatible bug fixes.
ifeq ($(GIT_DESCRIBE),)
TAGVERSION:=$(shell git describe --long --tags)
else
TAGVERSION:=$(GIT_DESCRIBE)
endif

# sanitize the git describe output: we use grep to exclude the last part "gLAST_COMMIT_HASH" which is useless to us,
# then we remove dots and dashes and plus signs:
GIT_DESCRIBE_SANITIZED:=$(subst +, ,$(subst -, ,$(subst ., ,$(shell echo $(TAGVERSION) | grep -o '.*-'))))
#$(info GIT_DESCRIBE_SANITIZED=$(GIT_DESCRIBE_SANITIZED))

# NOTE: we assume that the NUM_COMMITS_SINCE_LAST_TAG part is always the last word of the "git describe"
#       when the gLAST_COMMIT_HASH part has been removed. In this way we are more reliable against git tags
#       containing other words after the MAJOR.MINOR part...
override NUM_COMMITS_SINCE_LAST_TAG=$(lastword $(GIT_DESCRIBE_SANITIZED))

ifeq ($(GIT_BRANCH_NAME),)
    ifneq ($(CI_COMMIT_REF_NAME),)
$(info Running build on Gitlab CI/CD [$(CI_COMMIT_REF_NAME)] )
        GIT_BRANCH_NAME:=$(CI_COMMIT_REF_NAME)
    else
        GIT_BRANCH_NAME:=$(shell git rev-parse --abbrev-ref HEAD 2>/dev/null)
        ifneq ($(TC_PULLREQUEST_SOURCE_BRANCH),)
            ifneq ($(TC_PULLREQUEST_SOURCE_BRANCH),%teamcity.pullRequest.source.branch%)
                GIT_BRANCH_NAME:=$(TC_PULLREQUEST_SOURCE_BRANCH)
            endif
        endif
    endif
endif

ifeq ($(GIT_BRANCH_NAME_REGEX),)
GIT_BRANCH_NAME_REGEX := (^(feature|bugfix|hotfix|support)\/[-a-zA-Z0-9.]{1,32}$$)|(^(master|HEAD|develop|dependabot-.+||dependabot\/.+))
endif

override GIT_BRANCH_NAME_NOSLASH:=$(firstword $(subst /,-,$(GIT_BRANCH_NAME)))

# Here we prepare a variable to later be able to detect support branches
# Examples:
#    GIT_BRANCH_NAME         GIT_BRANCH_TYPE
#     feature/IN-xyz         feature
#     hotfix/IN-xyz          hotfix
#     support/1.2.3          support
#     develop                develop
#     master                 master
#     local                  local
#
GIT_BRANCH_TYPE:=$(firstword $(subst -, ,$(GIT_BRANCH_NAME_NOSLASH)))
#$(info GIT_BRANCH_TYPE is [$(GIT_BRANCH_TYPE)])

ifeq ($(GIT_BRANCH_TYPE),support)
# as exception: for SUPPORT branches only simplify the version naming by removing whatever comes after the SLASH;
# so if the current branch is named e.g. 'support/2.1.3' intead of producing as Conan package name
#       8.5.1+support-2.1.3-1
# this makefile will produce:
#       8.5.1+support-1
# which is way cleaner to read
override GIT_BRANCH_NAME_NOSLASH:=$(firstword $(subst /, ,$(GIT_BRANCH_NAME)))
endif

# ------------------------------------------------------------------------------------------
# definition of version numbers
# We follow https://semver.org/ and we craete a version string with this format:
#     <MAJOR>.<MINOR>.<PATCH>+$(GIT_BRANCH_NAME_NOSLASH)<PRERELEASE>
#
# The following logic of this makefile defines the values of numbers inside angular brackets <>
# ------------------------------------------------------------------------------------------

# Always take the major version from the LAST_TAG of the "git describe" command.
# The reason is that regardless of the fact we are building in a CI/CD like Teamcity or locally on a dev machine, 
# we must satisfy Conan requirement of having a coherent major version across the entire dependency tree.
# Using always the git-defined MAJOR allows to use locally-packaged repos in other repos, because their MAJOR version
# remains the same.
override MAJORVERSION=$(word 1,$(GIT_DESCRIBE_SANITIZED))

ifeq ($(BUILD_AGENT),1)
override MINORVERSION=$(word 2,$(GIT_DESCRIBE_SANITIZED))
override PATCHVERSION=$(word 3,$(GIT_DESCRIBE_SANITIZED))
override PRERELEASEVERSION=$(NUM_COMMITS_SINCE_LAST_TAG)
else
# this is a developer machine
override MINORVERSION=0
override PATCHVERSION=0
override PRERELEASEVERSION=0
override NUM_COMMITS_SINCE_LAST_TAG=1
endif


# ------------------------------------------------------------------------------------------
# Definition of FULL_VERSION variable
# FULL_VERSION follows strictly semantic versioning rules, see https://semver.org/
# ------------------------------------------------------------------------------------------

ifeq ($(NUM_COMMITS_SINCE_LAST_TAG),0)
# this build is the build of a GIT TAG... version it accordingly to semver logic without any prerelease/build-metadata part:
FULL_VERSION:=$(MAJORVERSION).$(MINORVERSION).$(PATCHVERSION)
BUILDING_TAG:=1
else
ifeq ($(BUILD_AGENT),1)
# not a git tag build: add semver build metadata information
# For these builds we need to introduce the so-called BUILD METADATA string to avoid conflicts with the
# artifacts generated out of a git tag build. E.g. consider:
#
#              tag X.Y.Z         (new commit)                   (new commit)
#  develop   ------+---------  X.Y.Z+develop-1 build ------- X.Y.Z+develop-2 build  ...
#                   \
#                    \           (new commit)                   (git tag)                       (new commit)
#  support            ----------X.Y.Z+support-1 build ------- X.Y.(Z+1) build ----------  X.Y.(Z+1)+support-1 build 
#
# In the above diagram if the build metadata information was not present in the develop and support branches, then the builds 
# having +1 checkin on top of the X.Y.Z tag would overlap (more precisely: their artifacts would overlap/conflict).
# Note: to reserve "versioning space" for support branches the maintainer must release tags in develop branch bumping
# the MINOR number at least.
FULL_VERSION:=$(MAJORVERSION).$(MINORVERSION).$(PATCHVERSION)+$(GIT_BRANCH_NAME_NOSLASH)-$(NUM_COMMITS_SINCE_LAST_TAG)
BUILDING_TAG:=0
else
# on developer machines, it's not useful to append the number of commits, so the semver build metadata does NOT contain $(NUM_COMMITS_SINCE_LAST_TAG)
# moreover to make it clear this is built on a dev machine, we use -local at the end of the branch name:
FULL_VERSION:=$(MAJORVERSION).$(MINORVERSION).$(PATCHVERSION)+$(GIT_BRANCH_NAME_NOSLASH)-local
BUILDING_TAG:=0
endif
endif


# ------------------------------------------------------------------------------------------
# Definition of RPM/JAVA/PYTHON/SO versions
# ------------------------------------------------------------------------------------------

# RPM versions cannot contains hyphens '-', thus change them into underscores '_'
# see https://twiki.cern.ch/twiki/bin/view/Main/RPMAndDebVersioning
RPM_VERSION:=$(subst -,_,$(FULL_VERSION))

# Nexus repository uses the RPM "Release" field to understand if an RPM is a stable version or not;
# if we write "snapshot" inside that, it is considered to be a non-stable release (and will be cleaned up shortly afterwards)
ifeq ($(GIT_BRANCH_TYPE),$(GIT_STABLE_BRANCH))
RPM_RELEASE:=1
else
ifeq ($(GIT_BRANCH_TYPE),support)
# consider support branches to be stable branches just like the master
RPM_RELEASE:=1
else
RPM_RELEASE:=snapshot
endif
endif

# Python version should follow PEP440:
#    https://www.python.org/dev/peps/pep-0440
# which clearly states
#  'Semantic versions containing a hyphen (pre-releases - clause 10) or a plus sign (builds - clause 11) 
#   are not compatible with this PEP and are not permitted in the public version field.
#   One possible mechanism to translate such semantic versioning based source labels to compatible public
#   versions is to use the .devN suffix to specify the appropriate version order.'
# PEP440 poses 2 serious troubles:
# 1) apparently the semver plus sign for build METADATA is supported by PEP440... but in practice it creates 
#    the so-called "local version identifier" whose format is:
#        <public version identifier>[+<local version label>]
#    The problem however is that just the <public version identifier> part is considered by pip during install;
#    so that if the Pypi repo contains e.g.
#      pypi-example-8.3.0+develop-1
#      pypi-example-8.3.0+feature-in-123
#      pypi-example-8.3.0
#    when running "pip3 install pypi-example==8.3.0" the actual version installed will be the
#    'pypi-example-8.3.0+develop-1' or 'pypi-example-8.3.0+feature-in-123' whatever is found first, since the
#    "local version label" is completely meaningless for dependency resolution.
#    To avoid this problem we must AVOID using the PLUS sign in PYTHON_FULL_VERSION.
#
# 2) it does not allow to use arbitrary strings in versions, like the branch name encoded in the build metadata.
#    We're not the only ones trying to generate unique artifacts depending on the branch name, see
#     https://stackoverflow.com/questions/74535980/install-package-with-pip-that-has-additional-qualifiers
#    but just like in that post we need to resort to use the branch name inside the pypi package NAME itself
#    and encode in the VERSION part "only" the 4 numbers (major, minor, patch, prerelease) that we have 
#    extracted before. That means that 
#       FULL_VERSION=1.2.3+develop-100
#    will be translated to:
#       pipy-package-name-develop   with version 1.2rc3.dev100
#    which is a PEP440-compliant version, see https://peps.python.org/pep-0440/#summary-of-permitted-suffixes-and-relative-ordering
PYTHON_PACKAGE_POSTFIX:=$(GIT_BRANCH_NAME_NOSLASH)
PYTHON_FULL_VERSION:=$(MAJORVERSION).$(MINORVERSION)rc$(PATCHVERSION).dev$(NUM_COMMITS_SINCE_LAST_TAG)

# define JAVA version
ifeq ($(GIT_BRANCH_TYPE),$(GIT_STABLE_BRANCH))
JAVA_FULL_VERSION:=$(FULL_VERSION)
JAVA_IS_SNAPSHOT_VERSION:=0
else
ifeq ($(GIT_BRANCH_TYPE),support)
# consider support branches to be stable branches just like the master
JAVA_FULL_VERSION:=$(FULL_VERSION)
JAVA_IS_SNAPSHOT_VERSION:=0
else
JAVA_FULL_VERSION:=$(FULL_VERSION)-SNAPSHOT
JAVA_IS_SNAPSHOT_VERSION:=1
endif
endif

# Shared object Library (SOLIB) versioning.
# Linux/ELF standard is to use 3 numbers, without any fancy branch name attached:
SOLIB_FULL_VERSION:=$(MAJORVERSION).$(MINORVERSION).$(PATCHVERSION)
SOLIB_SONAME_VERSION:=$(MAJORVERSION).$(MINORVERSION)

# Version for Conan packages
# NOTE: Conan imposes a maximum length of the PACKAGE VERSION of 51 characters.
#       Since in the version we include also the branch name, we might hit this limit with git branches
#       having very long names, so we truncate the version in such cases:
CONAN_VERSION:=$(shell echo "$(FULL_VERSION)" | cut -b 1-50)

ifeq ($(GIT_BRANCH_TYPE),$(GIT_STABLE_BRANCH))
CONAN_CHANNEL:=stable
else
ifeq ($(GIT_BRANCH_TYPE),support)
# consider support branches to be stable branches just like the master
CONAN_CHANNEL:=stable
else
CONAN_CHANNEL:=unstable
endif
endif

# Version for Docker images
# Docker "tags" can contain any string we like EXCEPT for the plus sign used by semver for BUILD METADATA, 
# We also remove any ending period sign.
# see https://github.com/distribution/distribution/issues/1201
DOCKER_VERSION_TMP:=$(subst +,-,$(FULL_VERSION))
DOCKER_VERSION:=$(shell echo "$(DOCKER_VERSION_TMP)" | cut -b 1-63 | sed -r 's/\.$$//')
DOCKER_VERSION_LATEST:=latest-$(GIT_BRANCH_NAME_NOSLASH)

# Version for Helm charts
# NOTE1: in the past the Nexus helm *proxy* plugin present in Nexus>=3.29.2 and Nexus <= 3.34.0, was failing in proxying
#        helm charts with a version like "1.2.3-develop" and only worked with versions like "1.2.3".
#        This is not the case anymore.
# NOTE2: using plus sign + in the helm chart version is producing weird unexpected results; in particular if:
#          * helm repo contains 2 or more versions of a chart like "1.2.3" and "1.2.3+feature-xyz"
#          * "helmfile apply" is then used to fresh-install the version "1.2.3+feature-xyz" 
#        then we noticed that Helm/helmfile will actually choose to install the "1.2.3" version instead of
#        the requested "1.2.3+feature-xyz". This should not happen in theory since build metadata should be
#        ignored only during upgrades when trying to compare 2 different semvers. 
# Given the 2 notes above (in particular NOTE2) we currently deploy helm charts with + replaced by the hyphen:
HELM_VERSION_TMP:=$(subst _,-,$(DOCKER_VERSION))
HELM_VERSION:=$(shell echo "$(HELM_VERSION_TMP)" | cut -b 1-63)

# ------------------------------------------------------------------------------------------
# Helper macro to create GNU make prerequisites from version variables
# ------------------------------------------------------------------------------------------

#
# Use this macro providing as argument a filename only (NOT fullpath)
# e.g.
#
#  somephonytarget:
#        @$(call update_topdir_version_file,fullversion.mypackaging_step)
#        $(MAKE) myfile.ext
#
#  myfile.ext: $(TOPDIR)/.conan_dependencies/fullversion.mypackaging_step
#        ...a slow cmd to regen myfile.ext... this will be skipped if myfile.ext already exists UNLESS the
#        version constants DOCKER_VERSION,HELM_VERSION,etc have changed somehow since last run
#
# The logic is:
#   - in the PHONY target "somephonytarget" we generate from scratch the version file (if missing)
#     or we OVERWRITE it in case of version change; it is left UNTOUCHED if it already exists and
#     there is no change in the value of FULL_VERSION variable
#   - the (slow) target/recipe that creates "myfile.ext" is re-run only when the FULL_VERSION variable changes
# 
define update_topdir_version_file
    @echo "$(FULL_VERSION)$(DEBUG_TAG)" >/tmp/$(1)
    @cmp -s /tmp/$(1) $(TOPDIR)/.conan_dependencies/$(1) || { \
        echo "Detected DOCKER/HELM/RPM/CONAN/JAVA version change... updating file $(TOPDIR)/.conan_dependencies/$(1)" ; \
        mv /tmp/$(1) $(TOPDIR)/.conan_dependencies/$(1) ; \
    }
endef
