#!/usr/bin/python3

import json
import subprocess
from configparser import ConfigParser
import os, glob, argparse
import logging


def conan_command(args):
    output = subprocess.run(
        ["conan"] + args, stdout=subprocess.PIPE, stderr=subprocess.PIPE
    )
    return (
        output.stdout.decode("utf-8"),
        output.stderr.decode("utf-8").strip() if output.stderr else None,
        output.returncode,
    )


def conan_download(args):
    return conan_command(["download"] + args)


def conan_get_storage_path():
    return conan_command(["config", "get", "storage.path"])[0].strip()


def parse_conan_info(text):
    config = ConfigParser(allow_no_value=True)
    config.read_string(text)
    return config


def conan_get(id):
    get_command = conan_command(["get", "-raw", id])
    return parse_conan_info(get_command[0]), get_command[1], get_command[2]


def abi_check(libname, abi_path1, abi_path2, extra_args=[]):
    output = subprocess.run(
        ["abi-compliance-checker", "-l", libname, "-old", abi_path1, "-new", abi_path2]
        + extra_args,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )
    return (
        output.returncode,
        output.stdout.decode("utf-8"),
    )


if __name__ == "__main__":
    # parse the arguments
    parser = argparse.ArgumentParser(
        description="Check ABI compliance of a given library"
    )
    parser.add_argument(
        "--build_type",
        help="Set build type",
        type=str,
        default="Debug",
    )
    parser.add_argument(
        "--runtime",
        help="Set runtime",
        type=str,
        default="OL9",
    )
    parser.add_argument(
        "--install-output",
        help="Path to conan-install-output.json",
        type=str,
        default="conan-install-output.json",
    )
    args = parser.parse_args()
    build_type = args.build_type
    runtime = args.runtime
    install_output = args.install_output
    logging.basicConfig(
        level=logging.DEBUG if os.environ.get("V", "0") == "1" else logging.INFO
    )
    logging.debug(
        f"Running ABI compliance check with build type {build_type} and runtime {runtime}"
    )
    logging.debug(f"conan install info are taken from {install_output}")
    with open(install_output, "r") as f:
        conan_install_output = json.load(f)
        assert not conan_install_output["error"]
        installed_refs = conan_install_output["installed"]

        # A list of the library names for which might make sense to check ABI compliance
        abi_check_valid_libs = [
            reference["recipe"]["name"]
            for reference in installed_refs
            # The library has at least an installed package and...
            if len(reference["packages"]) > 0
            # ...looking at the installed refs again, the library package download has been
            # skipped at least once (and implicitly, it shall appear at least twice, once
            # with a downloaded package and once without it)...
            and any(
                len(skipped_reference["packages"]) == 0
                for skipped_reference in installed_refs
                if skipped_reference["recipe"]["name"] == reference["recipe"]["name"]
            )
            # ...and we have got the related conan info as a ConfigFile
            and (
                conaninfo := conan_get(
                    f"{reference['recipe']['id']}:{reference['packages'][0]['id']}"
                )[0]
            )
            # ...and build_type and runtime are there and they match the requested once
            and "build_type" in conaninfo["full_settings"]
            and "runtime" in conaninfo["full_options"]
            and conaninfo["full_settings"]["build_type"] == build_type
            and conaninfo["full_options"]["runtime"] == runtime
        ]
        logging.debug(f"Libs for which to check ABI compliance: {abi_check_valid_libs}")

        # The list of the libraries info to check in the form:
        # [
        #   {
        #     "name": lib_name,
        #     "recipe_ids": [recipe_id1, recipe_id2, ...],
        #     "package_id": package_id,
        #   },
        #   ...
        # ]
        libs_info = [
            {
                "name": lib_name,
                "recipe_ids": [
                    ref["recipe"]["id"]
                    for ref in installed_refs
                    if ref["recipe"]["name"] == lib_name
                ],
                "package_id": next(
                    ref["packages"][0]["id"]
                    for ref in installed_refs
                    if ref["recipe"]["name"] == lib_name and len(ref["packages"]) > 0
                ),
            }
            for lib_name in abi_check_valid_libs
        ]
        logging.debug(f"Libs info: {libs_info}")
        for ref_info in libs_info:
            for recipe_id in ref_info["recipe_ids"]:
                full_ref = f"{recipe_id}:{ref_info['package_id']}"
                _, stderr, retcode = conan_get(full_ref)
                if retcode != 0:
                    logging.info(
                        f"Got error message from conan get {full_ref}: {stderr}"
                    )
                    if stderr == "ERROR: The specified path doesn't exist":
                        logging.info(
                            f"This error was expected. Going to download missing reference {full_ref}"
                        )
                        cdownload = conan_download([full_ref])
                        assert cdownload[2] == 0
                    else:
                        raise Exception(stderr)

        base_path = conan_get_storage_path()
        for ref_info in libs_info:
            package_id = ref_info["package_id"]
            lib_paths = [
                f"{base_path}/{recipe_id.replace('@', '/')}/package/{package_id}/lib/docker-{runtime}-{build_type.lower()}"
                for recipe_id in ref_info["recipe_ids"]
            ]
            logging.debug(f"Lib paths: {lib_paths}")
            assert all(
                os.path.exists(p) for p in lib_paths
            ), f"Some library paths have been found.\nlib_paths: {lib_paths}"
            abis_lists = [
                sorted(abis_list)  # sorting should be already done by glob actually...
                for abis_list in [glob.glob(f"{p}/*.abi.dump") for p in lib_paths]
                if sum(1 for abis in abis_list if len(abis) > 0)
            ]
            logging.debug(f"List of ABIs: {abis_lists}")
            for lib_abis in zip(*abis_lists):
                for lib_abi in lib_abis[1:]:
                    assert (
                        abi_check(ref_info["name"], lib_abis[0], lib_abi)[0] == 0
                    ), f"ABIs for library {ref_info['name']} are not all equal.\nFailed comparing ABIs: {lib_abis[0]} and {lib_abi}.\nCompat reports generated in compat_reports folder."
