#!/bin/bash

#
# Script that cleans up the local cache of Conan packages
# The logic is to cleanup packages to make sure the Conan cache does not take more than
# a fixed amount of disk space.
#
# IMPORTANT: this script will not ask any confirmation as it is meant to be part of a CI process.
#
# Author: fmontorsi
# Created: Nov 2020
#

# Constants

CONAN_PACKAGES_CREATED_BY_LAST_BUILD="$1"
MAX_AGE_THRESHOLD_SEC=1209600   # 2weeks
MIN_AGE_THRESHOLD_SEC=432000    # 5days
MAX_CONAN_CACHE_SIZE_MB=5000
NUMBER_REGEXP='^[\-]?[[0-9]+$'
CONAN_STORAGE_DIR="/opt/empirix/conan"

# currently commented out: there is no valid usecase today to avoid cleaning up some Conan packages by their name
#CONAN_PACKAGE_NAME_TO_REMOVE_REGEX="[0-9]+.[0-9]+.[0-9]+\+(feature|bugfix|hotfix|HEAD|develop|master|local)"

function assert_is_number()
{
    local NUMBER_STRING="$1"
    if ! [[ $NUMBER_STRING =~ $NUMBER_REGEXP ]]; then
        echo "It was expected to have a number instead [$NUMBER_STRING] was found"
        exit 2
    fi
}

function compute_conan_cache_size()
{
    CURRENT_SIZE_MB="$(du --summarize --block-size=1M $CONAN_STORAGE_DIR | cut -f1)"
    assert_is_number "$CURRENT_SIZE_MB"
}

function remove_conan_packages_older_than()
{
    local AGE_THRESHOLD_SEC="$1"

    local AGE_THRESHOLD_HOURS=$(( AGE_THRESHOLD_SEC/3600 ))
    local NSKIPPED=0
    for conan_pkg_dir in $CONAN_STORAGE_DIR/*/* ; do
        if [ -d "$conan_pkg_dir" ]; then
            local CONAN_PACKAGE_NAME="$(basename $conan_pkg_dir)"
            #if [[ $CONAN_PACKAGE_NAME =~ $CONAN_PACKAGE_NAME_TO_REMOVE_REGEX ]]; then
                TS_CONAN_PKG=`stat --format=%Y $conan_pkg_dir`
                if (( (TS_NOW - TS_CONAN_PKG) > AGE_THRESHOLD_SEC )); then
                    echo "  Removing Conan package $conan_pkg_dir which is more than ${AGE_THRESHOLD_HOURS}hours old."
                    rm -rf $conan_pkg_dir
                    (( NREMOVED++ ))
                else
                    (( NSKIPPED++ ))
                    # this log line is very verbose... disable it to help readability
                    #echo "  Skipping removal of Conan package [$conan_pkg_dir] which was last modified `stat --format=%y $conan_pkg_dir`. It was modified less than ${AGE_THRESHOLD_HOURS}hours ago."
                fi
            #else
            #    echo "  Skipping removal of Conan package [$conan_pkg_dir] since its name does not match the regex of packages to delete"
            #fi
        fi
    done

    echo "Skipped $NSKIPPED packages (below age threshold)."
}


TS_NOW=$(date +%s)
NREMOVED=0
NUM_IT=0
CURR_AGE_THRESHOLD_SEC=$MAX_AGE_THRESHOLD_SEC
AGE_DECREASE_STEP_SEC=86400  # 1day

echo "------------------------------------------------------------------------------------"
echo "Cleaning temporary build files generated by Conan on this machine"
echo "------------------------------------------------------------------------------------"

for CONAN_PACKAGE in ${CONAN_PACKAGES_CREATED_BY_LAST_BUILD}; do
    echo "Cleaning Conan package $CONAN_PACKAGE build folder"
    conan remove --force ${CONAN_PACKAGE} --builds
done

echo "------------------------------------------------------------------------------------"
echo "Cleaning local Conan cache on this machine"
echo "------------------------------------------------------------------------------------"

compute_conan_cache_size
echo "Before cleanup the $CONAN_STORAGE_DIR takes ${CURRENT_SIZE_MB}MB"
while (( CURRENT_SIZE_MB >= MAX_CONAN_CACHE_SIZE_MB )); do
    CURR_AGE_THRESHOLD_HOURS=$(( CURR_AGE_THRESHOLD_SEC/3600 ))
    echo "Current size=${CURRENT_SIZE_MB}MB, target size=${MAX_CONAN_CACHE_SIZE_MB}MB. Starting removal of packages older than ${CURR_AGE_THRESHOLD_HOURS}hours. Iteration ${NUM_IT}"
    remove_conan_packages_older_than "$CURR_AGE_THRESHOLD_SEC"
    compute_conan_cache_size  # for next iteration

    if (( CURR_AGE_THRESHOLD_SEC == MIN_AGE_THRESHOLD_SEC )); then
        # if we reached the min threshold, it's time to stop looping:
        echo "Bailing out after reaching the minimal age threshold of ${CURR_AGE_THRESHOLD_HOURS}hours."
        break
    fi

    # decrease threshold for next iteration:
    CURR_AGE_THRESHOLD_SEC=$(( CURR_AGE_THRESHOLD_SEC - AGE_DECREASE_STEP_SEC ))
    if (( CURR_AGE_THRESHOLD_SEC < MIN_AGE_THRESHOLD_SEC )); then
        # do a last pass with minimal threshold:
        CURR_AGE_THRESHOLD_SEC=${MIN_AGE_THRESHOLD_SEC}
    fi

    (( NUM_IT++ ))
done

echo "Removed a total of $NREMOVED packages. After cleanup the $CONAN_STORAGE_DIR takes ${CURRENT_SIZE_MB}MB"



