#!/usr/bin/python3

"""
This script has been written to download the source code of upstream projects from the JFrog artifactory
directly to it's respective versioned folders in /opt/empirix/conan/.
Since the Conan don't have a direct comand to just download the source, this script will work as an alt-
ernative solution.
Direct source code feature will be available in the upcoming version of Conan 2.0 
(Github issue - https://github.com/conan-io/conan/issues/8121). This script can be replaced with the new 
conan command. 
"""

import yaml, os, requests, tarfile, getpass, argparse, sys
import concurrent.futures

global conandata, conan_dir, conan_items, url_prefix, url_postfix, recipe_user, username, password
conan_items = ["cpp_version_checked_requirements"]
url_postfix = "/0/export/conan_sources.tgz"
repo = "artifactory"
source_tar = "conan_sources.tgz"

# ASCII color for print statements
class bcolors:
    HEADER = "\033[95m"
    OKBLUE = "\033[94m"
    OKCYAN = "\033[96m"
    OKGREEN = "\033[92m"
    WARNING = "\033[93m"
    FAIL = "\033[91m"
    ENDC = "\033[0m"
    BOLD = "\033[1m"
    UNDERLINE = "\033[4m"
    WHITE = "\033[97m"
    BOLD = "\033[1m"

    # print header - takes one line
    def printh(inp_str: str) -> None:
        print(bcolors.BOLD + bcolors.OKCYAN + f"{inp_str}" + bcolors.ENDC)

    # prints message - takes recipe name and log message
    def printm(str_first: str, str_second: str) -> None:
        print(
            bcolors.BOLD
            + bcolors.OKGREEN
            + f"{str_first}: "
            + bcolors.WHITE
            + f"{str_second}"
            + bcolors.ENDC
        )


def parse_yaml_for_recipe(inp_yaml_file: str, yaml_items: list) -> list:
    out_value_list = []
    with open(inp_yaml_file, "r") as file:
        inp_yaml_file = yaml.load(file, Loader=yaml.FullLoader)
        for item in yaml_items:
            for value in inp_yaml_file[item]:
                out_value_list.append(value)
    return out_value_list


# Get source directory using recipe
def get_source_path(recipe: str) -> str:
    return os.path.join(conan_dir, recipe.replace("@", "/"), "source")


# This function takes a list of conan recipe and build a dictionary of recipe with respective artifactory url.
# e.g. lib_file_dict = {"x@u/z":"https://abc/xyx.tgz}
# It returns both the list of directories and the dictionary.
def get_dir_dict(recipe_list: list):
    lib_recipe_list = []
    lib_file_dict = {}

    for l in recipe_list:
        lib_recipe_list.append(l)
        # parse the recipe name to get the user name
        # e.g. recipe :- "networking-libs/1.4.3-develop@empirix/stable"; recipe user will be 'empirix'
        recipe_user = l.split("@")[1].split("/")[0]
        # concatenate all parts of url to form full url to artifact source package
        # lib_file_dict[get_dir_paths] = (
        lib_file_dict[l] = (
            url_prefix
            + "/"
            + recipe_user
            + "/"
            + l.replace(f"@{recipe_user}", "")
            + url_postfix
        )
    # print(lib_file_dict)
    return lib_recipe_list, lib_file_dict


# This function takes the list of recipes and dictionary of recipes and urls.
# It iterates over the dictionary using recipe list to get the url and download
# source package from the artifactory to their respective directories and un-pack it.
def download_file_in_dir(in_list: list, in_dict: dict) -> bool:
    success = True
    retrieve_list = []

    bcolors.printh("Installing (downloading) source code...")
    # download file in folder given in list
    for l in in_list:
        dir = get_source_path(l)
        try:
            # Make new directory
            os.makedirs(dir)
        except FileExistsError:
            if os.path.exists(dir) and not os.path.isfile(dir):
                # condition will be executed if source folder is present and non-empty
                if os.listdir(dir):
                    bcolors.printm(f"{l}", "Already Installed!")
                    continue
        # make a list of recipes whose source is not downloaded
        retrieve_list.append(l)

    with concurrent.futures.ThreadPoolExecutor() as executor:
        # Start the load operations of URLs and mark each future with its recipe
        future_to_url = {
            executor.submit(
                requests.get, in_dict[recipe], auth=(username, password), timeout=10
            ): recipe
            for recipe in retrieve_list
        }
        for future in concurrent.futures.as_completed(future_to_url):
            recipe = future_to_url[future]
            dir = get_source_path(recipe)
            tar_file_name = f"{dir}/{source_tar}"
            try:
                bcolors.printm(recipe, f"Retrieving source code from '{repo}'")
                r = future.result()
            except (requests.ConnectionError, ConnectionResetError) as e:
                bcolors.printm(recipe, f"Connection error: {str(e)}. \n")
            except Exception as e:
                bcolors.printm(recipe, f"Exception - {str(e)} \n")
            else:
                if r.status_code == 200:
                    with open(tar_file_name, "wb") as out:
                        for bits in r.iter_content():
                            out.write(bits)
                        bcolors.printm(recipe, f"Downloading {source_tar} completed")
                    # Open the *.tgz file and extract it's contents
                    tarp = tarfile.open(tar_file_name, "r")
                    tarp.extractall(dir)
                    bcolors.printm(recipe, f"Decompressed {source_tar} inside {dir}")
                    # after files are extracted remove the *.tgz file
                    os.remove(tar_file_name)
                else:
                    bcolors.printm(recipe, f"status_code: {r.status_code} - {r.reason}")

    return success


# Validation of inputs and function calls start from here
"""
Remember to provide at least 3 inputs to this script with following desirable values to it:-
    1. Path to conandata.yml
    2. Pull URL of the artifactory
    3. Conan data directory path (e.g. /opt/empirix/conan)
"""
# parse the arguments to this script
parser = argparse.ArgumentParser(description="Get source code from Artifactory")
parser.add_argument("--conandata_path", help="Conandata.yml file location", type=str)
parser.add_argument(
    "--url_prefix",
    help="JFrog artifacory initial URL to pull the source code package",
    type=str,
)
parser.add_argument(
    "--conan_dir",
    help="Conan location where all source code are downloaded",
    type=str,
)
args = parser.parse_args()
# get path to conandata.yml
conandata = args.conandata_path
# get value of REMOTE_REPO_ARTIFACTORY_CONAN_FOR_PULL
url_prefix = args.url_prefix
# get conan data folder
conan_dir = args.conan_dir

# if username and password are not set then ask the user to provide credentials of artifactory
# NOTE: having these env vars defined but EMPTY is fine and means: no source code download should be attempted
try:
    if len(os.environ["ARTIFACTORY_USERNAME"].strip()) == 0:
        pass
except KeyError:
    os.environ["ARTIFACTORY_USERNAME"] = input(
        "No env var ARTIFACTORY_USERNAME defined. Please enter your username for JFrog Artifactory to download upstream source code (empty to skip): "
    )
try:
    if len(os.environ["ARTIFACTORY_PASSWORD"].strip()) == 0:
        pass
except KeyError:
    os.environ["ARTIFACTORY_PASSWORD"] = getpass.getpass(
        "No env var ARTIFACTORY_PASSWORD defined. Please enter your password for JFrog Artifactory to download upstream source code (empty to skip): "
    )

username = os.environ["ARTIFACTORY_USERNAME"]
password = os.environ["ARTIFACTORY_PASSWORD"]
if len(username) == 0 or len(password) == 0:
    bcolors.printh(
        "Either ARTIFACTORY_USERNAME or ARTIFACTORY_PASSWORD is empty. This means source code donwload will be skipped."
    )
    sys.exit(0)

# get recipe names from conandata.yml file
recipe_list = parse_yaml_for_recipe(conandata, conan_items)
# get list of upstream projects
dir_list, dir_dict = get_dir_dict(recipe_list)
# download upstream source codes
download_file_in_dir(dir_list, dir_dict)
