#!/bin/bash

#
# Smart docker re-deploy from one remote repository to another one
# Created: Aug 2020
# Author: scbradley
#
# Supports the following features:
# - Pull image from one repository to local machine
# - Registry login/authentication
# - Push to target Docker repository
# - Logging of pushed docker images
#
# Usage:
#   ./docker-redeploy-wrapper.sh [target_repo_user] [target_repo_password] [target_repo_url] [source_repo_url] [output_logfile] [docker_group] [image1_name:tag] [image2_name:tag] ...
# E.g.:
#   ./docker-redeploy-wrapper.sh "runner" "artifactory_pw"
#        "empirixartifactory-docker.jfrog.io" "repo-billerica.perform.empirix.com:6000" "logfile.csv" "empirix-ecc-eva"
#        ecc-eva-smm-config:2.7.23_develop
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
#
# Exit code:
#  0: success
#  100: login failure
#  200: tag failure
#  300: push failure
#

set -u

USER="$1"; shift
PW="$1"; shift
REMOTE_REPO_PUSH_REGISTRY_URL="$1"; shift
REMOTE_REPO_PULL_REGISTRY_URL="$1"; shift
DEPLOY_FILELIST_LOG="$1"; shift
DOCKER_GROUP_AND_IMAGE="$1";

docker login -u $USER -p $PW $REMOTE_REPO_PUSH_REGISTRY_URL
if [[ $? -ne 0 ]]; then echo "Failed authentication against $REMOTE_REPO_PUSH_REGISTRY_URL" ; exit 100 ; fi

echo "** Login succeeded... proceeding with tag&push of deployed docker images from [$DEPLOY_FILELIST_LOG]"

docker pull $REMOTE_REPO_PULL_REGISTRY_URL/$DOCKER_GROUP_AND_IMAGE

dockerTag="${REMOTE_REPO_PUSH_REGISTRY_URL}/${DOCKER_GROUP_AND_IMAGE}"

echo "** Tagging source docker image [$REMOTE_REPO_PULL_REGISTRY_URL/DOCKER_GROUP_AND_IMAGE] as [$dockerTag]"
docker tag $REMOTE_REPO_PULL_REGISTRY_URL/$DOCKER_GROUP_AND_IMAGE $dockerTag
if [[ $? -ne 0 ]]; then echo "Failed tagging!" ; exit 200 ; fi

echo "** Pushing docker image [$dockerTag]"
docker push $dockerTag
if [[ $? -ne 0 ]]; then echo "Failed pushing!" ; exit 300 ; fi

DEPLOY_RESULT="deployed"
echo "docker,${REMOTE_REPO_PUSH_REGISTRY_URL},${DOCKER_GROUP_AND_IMAGE},${DEPLOY_RESULT}" | tee -a $DEPLOY_FILELIST_LOG

echo "** Re-deployed Docker image ${DOCKER_GROUP_AND_IMAGE} successfully."
