#!/bin/bash

#
# Smart helm deploy on remote repositories
#
# Supports the following features:
# - HTTP authentication
# Usage:
#   ./helm-deploy-wrapper.sh [repo_user] [repo_password] [repo_url] [output_logfile] [helm_packages list]
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
#
# Exit code:
#  0: success
#  100: deploy failure
#  150: version failure
#  200: redeploy check failure

set -u



USER="$1"; shift
PW="$1"; shift
REMOTE_REPO_URL="$1"; shift
DEPLOY_FILELIST_LOG="$1"; shift
HELM_PACKAGES_LIST="$*"
LOCAL_TAG="0.0-local"

function error_local_tag()
{
  PKG=$1
  echo "-----------------------------------------------------------------------------------------------------"
  echo "-----------------------------------------------------------------------------------------------------"
  echo "ERROR *** Package $PKG can't be pushed to the repo because version $LOCAL_TAG is just for local usage ***"
  echo "-----------------------------------------------------------------------------------------------------"
  echo "-----------------------------------------------------------------------------------------------------"
  exit 150
}

# ensure folder for log file exists
mkdir -p $(dirname $DEPLOY_FILELIST_LOG)

CURL_CMD="curl -v --fail --user $USER:$PW"
CURL_CMD_PRESENCE_CHECK="$CURL_CMD --output /dev/null --silent --head"

NCORRECTLY_DEPLOYED=0
NSKIPPED=0
for helmpkg in $HELM_PACKAGES_LIST; do
    PKG_FILENAME_ONLY="$(basename $helmpkg)"
    if [[ "$PKG_FILENAME_ONLY" == *"$LOCAL_TAG" ]]; then
      error_local_tag $PKG_FILENAME_ONLY
    fi
    if $CURL_CMD_PRESENCE_CHECK $REMOTE_REPO_URL/$PKG_FILENAME_ONLY; then
        echo "WARNING: Apparently the Helm package [$PKG_FILENAME_ONLY] was already deployed: check [$REMOTE_REPO_URL/$PKG_FILENAME_ONLY]. Skipping this deploy."
        DEPLOY_RESULT="skipped"
        (( NSKIPPED++ ))
    else
        echo "Looks like the Helm package file [$PKG_FILENAME_ONLY] has never been deployed before. Proceeding."

        # exit immediately in case of network failure:
        $CURL_CMD -XPUT $REMOTE_REPO_URL \
            -i -T $helmpkg || exit 100

        DEPLOY_RESULT="deployed"
        (( NCORRECTLY_DEPLOYED++ ))
    fi
    echo "helm,${REMOTE_REPO_URL}${PKG_FILENAME_ONLY},${DEPLOY_RESULT}" | tee -a $DEPLOY_FILELIST_LOG
done

echo "Count of helm packages deployed: $NCORRECTLY_DEPLOYED. Skipped Packages: $NSKIPPED"
