#!/bin/bash

#
# Script to re-deploy RPMs and Dockers from one build to another repository, leveraging the existing deploy scripts.
# Most commonly to re-deploy from Nexus to Artifactory.
# Created: Sep 2020
# Author: scbradley
#
# Supports the following features:
# - Registry login/authentication
# - Logging of pushed docker images
#
# Usage:
#   ./redeploy-artifacts.sh [target_repo_user] [target_repo_password] [target_repo_url] [source_repo_url] [output_logfile] [docker_group] [image1_name:tag] [image2_name:tag] ...
# E.g.:
#   ./redeploy-artifacts.sh "runner" "artifactory_pw"
#        "empirixartifactory-docker.jfrog.io" "repo-billerica.perform.empirix.com:6000" "deployed-artifacts.txt"
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
#
# Exit code:
#  0: success
#  100: login failure
#  200: tag failure
#  300: push failure
#

set -u

DEPLOYED_FILE="$1"
SKIP_FILES_REGEX="$2"
ARTIFACTORY_USER="$3"
ARTIFACTORY_PW="$4"
REMOTE_REPO_RPM_URL_PUSH="$5"
REMOTE_REPO_DOCKER_URL_PUSH="$6"
REMOTE_REPO_HELM_URL_PUSH="$7"
DEPLOY_FILELIST_LOG="$8"

# take credentials for Teamcity from env variables
TEAMCITY_USER="${TEAMCITY_USER}"
TEAMCITY_ACCESS_TOKEN="${TEAMCITY_ACCESS_TOKEN}"



function redeploy_rpms()
{
    echo
    echo "Starting RPM redeploy"
    echo

    # For each RPM deployed, download it locally
    cat deployed-artifacts.txt | grep rpm | grep -v ${SKIP_FILES_REGEX} | awk -F ',' '{print $2$3}' | xargs -n1 -i curl -k -O {}
    mkdir -p x86_64
    mv *.rpm ./x86_64

    # Run RPM deploy pointed at Artifactory on the current directory
    ${SCRIPTS_DIR}/rpm-deploy-wrapper.sh ${WORK_DIR} ${ARTIFACTORY_USER} ${ARTIFACTORY_PW} ${REMOTE_REPO_RPM_URL_PUSH} ${DEPLOY_FILELIST_LOG}
    if [ $? -ne 0 ]; then
        echo "Failed RPM redeploy from Nexus to Artifactory. Please check above logs."
        exit 2
    fi

    rm -rf ./x86_64
}

function redeploy_dockers()
{
    echo
    echo "Starting Dockers redeploy"
    echo

    # Run Docker redeploy
    MY_PULL_SERVER=$(cat deployed-artifacts.txt  | grep docker | head -1 | awk -F ',' '{print $2}')
    cat deployed-artifacts.txt | grep docker | grep -v ${SKIP_FILES_REGEX} | awk -F ',' '{print $3}' | xargs -n1 -i ${SCRIPTS_DIR}/docker-redeploy-wrapper.sh ${ARTIFACTORY_USER} ${ARTIFACTORY_PW} ${REMOTE_REPO_DOCKER_URL_PUSH} ${MY_PULL_SERVER} ${DEPLOY_FILELIST_LOG} {}
    if [ $? -ne 0 ]; then
        echo "Failed Docker image redeploy from Nexus to Artifactory. Please check above logs."
        exit 2
    fi
}

function redeploy_helm_charts()
{
    echo
    echo "Starting Helm Chart redeploy"
    echo

    # Use a similar approach for Helm packages as we do for RPMs
    cat deployed-artifacts.txt | grep helm | grep -v ${SKIP_FILES_REGEX} | awk -F ',' '{print $2}' | xargs -n1 -i curl -k -O {}
    for helmfile in $WORK_DIR/*.tgz; do
        ${SCRIPTS_DIR}/helm-deploy-wrapper.sh ${ARTIFACTORY_USER} ${ARTIFACTORY_PW} ${REMOTE_REPO_HELM_URL_PUSH} ${DEPLOY_FILELIST_LOG} ${helmfile}
        if [ $? -ne 0 ]; then
            echo "Failed Helm chart redeploy from Nexus to Artifactory. Please check above logs."
            exit 2
        fi
    done
}

# corner case where the user does not want to blacklist any specific artifact:
if [[ -z "$SKIP_FILES_REGEX" ]]; then
    SKIP_FILES_REGEX="dummy-string-that-will-never-match-any-line"
fi

# ensure folder for log file exists
mkdir -p "$(dirname ${DEPLOY_FILELIST_LOG})"

WORK_DIR=$(pwd)
if [ -f "${WORK_DIR}/helm-deploy-wrapper.sh" ]; then
    # Running directly from the scripts directory use case
    SCRIPTS_DIR="${WORK_DIR}"
else
    # TeamCity use case
    SCRIPTS_DIR="${WORK_DIR}/Scripts"
fi

# Get the deployed artifacts file
rm -rf ./deployed-artifacts.txt
curl --fail --user "${TEAMCITY_USER}:${TEAMCITY_ACCESS_TOKEN}" -k -O ${DEPLOYED_FILE}
if [ $? -ne 0 ]; then
    echo "Failed download of CSV report of deployed artifacts from Teamcity. Please check URL and credentials."
    exit 2
fi

echo "Successfully downloaded the list of artifacts to redeploy from Teamcity:"
cat ./deployed-artifacts.txt

redeploy_rpms
#redeploy_dockers # no need to redeploy docker images -- since March 2023 all docker images are always pushed to Amazon ECR
redeploy_helm_charts
