#!/usr/bin/env python3

import os, sys
import argparse
import datetime

from hashlib import md5
import xlsxwriter
from typing import Tuple


class SystemhealthDocGenerator:
    def __init__(self, verbose: bool):
        """constructor"""
        self.verbose = verbose

    def generate_documentation(
        self,
        export_file_path: str,
        full_kpi_dict: dict,
        kpi_prefix: str,
        fix_creation_date: bool = False,
    ) -> None:
        """Create the Excel spreadsheeet from the provided information.

        Args:
            output_fname: output filename
            full_kpi_dict: each item must be a dictionary describing the syshealth KPI

        Returns:
            Nothing

        Raises:
            Nothing
        """

        assert len(full_kpi_dict) > 0

        with xlsxwriter.Workbook(export_file_path) as workbook:
            if fix_creation_date:
                # Only for unit test, using fixed creation date as we don't want md5 sum to change
                workbook.set_properties({"created": datetime.date(2022, 8, 1)})

            # generate N sheets, one for each source XML file
            bold = workbook.add_format({"bold": True})
            for source_xml in full_kpi_dict:
                kpi_list = full_kpi_dict[source_xml]

                _, sheet_name = os.path.split(source_xml)
                sheet_name = sheet_name.replace(".xml", "")

                if self.verbose:
                    print(f"Generating docs for {len(kpi_list)} KPIs in Excel sheet [{sheet_name}]")

                rows_with_header = self.__create_excel_rows(kpi_list, kpi_prefix)
                worksheet = workbook.add_worksheet(sheet_name[:31])

                # set_column() takes [first_col, last_col, width, cell_format, options]
                worksheet.set_column(0, 0, 50)  # Name
                worksheet.set_column(1, 1, 20)  # Data Type
                worksheet.set_column(2, 2, 20)  # KPI Type
                worksheet.set_column(3, 3, 70)  # Description
                worksheet.set_column(4, 4, 20)  # Service Impact

                # Write list data
                for row_index, row_data in enumerate(rows_with_header):
                    for col_index, data in enumerate(row_data):
                        if row_index == 0:  # write header
                            worksheet.write(row_index, col_index, data, bold)
                        else:
                            worksheet.write(row_index, col_index, data)

        print(f"Successfully generated the excel document in {export_file_path}")

    def __create_excel_rows(self, full_kpi_list: list, kpi_prefix: str) -> list:
        # Prepare row data to be written in excel against columns
        rows_with_header = []
        column_names = [
            "Prometheus Metric Name",
            "Data Type",
            "KPI Type",
            "Description",
            "Service Impact",
        ]
        rows_with_header.append(column_names)
        for field in full_kpi_list:
            # NOTE: the C++ code in common-libs/SystemHealthCollector is doing the same operation to generate
            #       the actual Prometheus metric:
            name = kpi_prefix + field["name"].lower()

            # copy all other attributes
            data_type = field["type"] if "type" in field else ""
            kpi_type = field["kpi_type"] if "kpi_type" in field else ""
            description = field["description"] if "description" in field else ""
            service_impact = (
                field["service_impact"] if "service_impact" in field else ""
            )
            row_data = [name, data_type, kpi_type, description, service_impact]
            rows_with_header.append(row_data)

        return rows_with_header
