#!/bin/bash

source `dirname $(readlink -f $0)`/common.sh

start_tests() {
    start_section tests "Unit test execution"
}

TEST_OUTPUT_DIR=${PYTHON_TEST_OUTPUT_DIR:-"test-results"}
TESTDIR=${PYTHON_TESTS_DIR:-tests}

echo "Running tests in ${TESTDIR} and outputting to ${TEST_OUTPUT_DIR}"

start_section setup "Test environment setup"

if detect_poetry; then
    poetry update
    # detect pytest
    if poetry show | grep pytest 2>/dev/null; then
        ## pytest detected
        # make sure to install all pytest plugin
        PYTEST_DETECTED=1
        poetry add -vv -D coverage[toml] unittest_xml_reporting pytest-cov pytest-mock
        PYTEST_CMD="poetry run python -m pytest"
    else
        PYTEST_DETECTED=0
        poetry add -vv -D coverage[toml] unittest_xml_reporting
        UNITTEST_CMD="poetry run coverage"
    fi
else
    # WIP support for more than one requirements.txt in different pyenv
    uv venv
    source .venv/bin/activate
    find . -type f -name requirements-dev.txt  ! -path "./integration-tests/*" | xargs -r -n 1 -I {} uv pip install -U -r {}
    find . -type f -name requirements.txt  ! -path "./integration-tests/*" | xargs -r -n 1 -I {} uv pip install -U -r {}
    # detect pytest
    if uv pip list | grep pytest 2>/dev/null; then
        PYTEST_DETECTED=1
        uv pip install coverage[toml] unittest_xml_reporting pytest-cov pytest-mock
        PYTEST_CMD="python -m pytest"
    else
        PYTEST_DETECTED=0
        uv pip install coverage[toml] unittest_xml_reporting
        UNITTEST_CMD="coverage"
    fi

fi

mkdir -p ${TEST_OUTPUT_DIR}

end_section setup
start_tests

exit_code=0

if [ "$PYTEST_DETECTED" == "1" ]; then
    # generate coverage targets from comma-separated envvar
    COV_TARGETS_CLI=""
    if [ ! -z ${PYTEST_COVERAGE_TARGETS+x} ]; then
        alltargets=$(awk -F, '{
            split($0, array, ",")
            for (i in array) { print array[i] }
        }' <<< "$PYTEST_COVERAGE_TARGETS")
        for i in `echo $alltargets`; do
            COV_TARGETS_CLI+=" --cov=$i"
            echo "[DEBUG] enabling pytest coverage for directory: $i"
        done
    fi
    # run pytest
    ${PYTEST_CMD} -v --cov-report=term --cov-report=xml --cov-report=html ${COV_TARGETS_CLI} --junit-xml=${TEST_OUTPUT_DIR}/xunit.xml \
      --cov-report=html:${TEST_OUTPUT_DIR}/htmlcov --cov-report=xml:${TEST_OUTPUT_DIR}/coverage.xml \
      ${TESTDIR}/ || exit_code=1
else
    # run unittest
    ${UNITTEST_CMD} run -m xmlrunner --output-file ${TEST_OUTPUT_DIR}/xunit.xml discover -v $TESTDIR/ || exit_code=1
    ${UNITTEST_CMD} xml -o ${TEST_OUTPUT_DIR}/coverage.xml
    ${UNITTEST_CMD} report
fi

# cleanup coverage file
if [ ! -z "${CI}" ]; then
    sed -i 's:'${GITHUB_WORKSPACE}'/::' ${TEST_OUTPUT_DIR}/coverage.xml
fi

end_section tests

exit $exit_code
