#
# Empirix Conan file to package C++ artifacts generated by this repository
# as a Conan package
#
# A few important points on how Conan is used here:
# 1) Conan is used only for the PACKAGING and does NOT drive the C++ build.
#    GNU make drives the C++ compilation, linking and installation steps.
# 2) Conan versioning is obtained from GNU make, through the use of the
#    "make get_conan_version" command... in this way the version declared by
#    GNU make will always be perfectly aligned to the one declared inside the
#    Conan package.
# 3) The installation of the artifacts into the Conan package is done through
#    "make install" command, instead of hardcoding here a long list of "copy"
#    commands.

from conans import ConanFile, tools
from conans.errors import ConanInvalidConfiguration

import os
import subprocess
import json
import string
import glob
import sys

class EmpirixConanBase(object):

    # Defined the runtime option and its possible values
    options = {"runtime": ["None", "FC38", "CENTOS7", "OL9"]}

    # Default value for header-only project
    default_options = {"runtime": "OL9"}

    #
    # Empirix utility functions that we want to share across all Conan packages
    #

    def empirix_get_buildid(self, arch, runtime, cfg):
        return arch + "-" + runtime + "-" + cfg

    def empirix_get_runtime(self):
        self.output.info("Executing empirix_get_runtime() method")
        return str(self.options.runtime)
    
    def empirix_inject_runtime_default_options(self, data, options):
        self.output.info("Executing empirix_inject_runtime_default_options() method")
        runtimeE = self.options.runtime
        runtimeEnv = str(runtimeE)
        self.output.info("Injecting runtime " + runtimeEnv)
        # For each dependency, set a 'runtime' option
        # For example, base-libs/1.0.0+local@empirix/unstable will set an option "base-libs:runtime=<RUNTIME>"
        for dep in data:
            # Extract the name of the dependency
            index = res = dep.find('/')
            depName = dep[0:index]
            self.output.info("Injecting option for dependency " + depName)
            self.options[depName].runtime = runtimeEnv

    def empirix_get_cfg(self):
        # XXX IN-46372 the following regex can be used
        #     for direct transformation PascalCase => hyphen-case
        # pattern = re.compile(r'(?<!^)(?=[A-Z])')
        # return pattern.sub('-', str(self.settings.build_type)).lower()
        return {
            "Release": "release",
            "Debug": "debug",
            "DebugGcov": "debug-gcov",
            "DebugAsan": "debug-asan",
            "DebugTsan": "debug-tsan",
        }[str(self.settings.build_type)]

    def empirix_ask_version_to_gnu_make(self, git_branch_name: str, git_describe_output: str, folder_with_makefile: str):
        my_env = os.environ.copy()  # pass all the env vars that Conan client receives down to GNU make
        my_env["GIT_DESCRIBE"] = git_describe_output
        my_env["GIT_BRANCH_NAME"] = git_branch_name
        my_env["LAB"] = "billerica"
        make_proc = subprocess.run(
            [
                "/usr/bin/make",
                "--quiet",
                "-C",
                folder_with_makefile,
                "get_conan_version",
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            env=my_env,
        )
        if make_proc.returncode != 0:
            # this typically happens when the use is running "make check_deps" and the "devel" package
            # is thus not yet availale to GNU make: the root Makefile will not contain the definition of
            # the "get_conan_version" target and thus will fail
            self.output.info(
                "Failed running the [get_conan_version] target from GNU make. Setting version [VerNotAvail]. This is normal in the [check_deps] stage."
            )
            return "VerNotAvail"
        return make_proc.stdout.decode("utf-8")

    def empirix_write_env_vars(self, git_branch_name: str, git_describe_output: str):
        env_vars = {}
        env_vars["GIT_DESCRIBE"] = git_describe_output
        env_vars["GIT_BRANCH_NAME"] = git_branch_name
        self.output.info("Dumping env vars as a JSON inside the Conan package")
        json.dump(env_vars, open(".conan_env_vars.json", "w"))

    def empirix_read_env_vars(self):
        env_vars = json.load(open(".conan_env_vars.json"))
        self.output.info("Loaded env var GIT_DESCRIBE=%s" % env_vars["GIT_DESCRIBE"])
        self.output.info("Loaded env var GIT_BRANCH_NAME=%s" % env_vars["GIT_BRANCH_NAME"])
        return env_vars

    def empirix_verify_option(self, opt: str):
        bool = hasattr(self.options, opt)
        if not bool:
            self.output.info("No [{opt}] Conan package OPTION found... did you forget it?".format(opt=opt))
        return bool

    def empirix_replace_strings_in_file(self, filepath: str, replacements: str):
        self.output.info("Replacing into the file {f} the following map of keywords/values:\n{m}".format(f=filepath, m=replacements))
        try:
            with open(filepath, "r") as fd:
                # string.Template utility of Python is a small templating engi1 able to replace variables
                # encoded in text as $var or ${var}
                template = string.Template(fd.read())
                tempstr = template.safe_substitute(replacements)
            with open(filepath, "w") as fd:
                fd.write(tempstr)
        except IOError as e:
            self.output.error("Failed to replace strings into [{s}]".format(s=filepath))

    def empirix_check_env_vars(self, required_env_vars: list):
        for env_var in required_env_vars:
            if not os.environ.get(env_var):
                self.output.error(
                    f"Environment variable {env_var} is not defined. "
                    f"This typically happens when running 'make check_deps' and for some reason Conan decides to try rebuilding a required Conan package on the fly. "
                    f"Rebuilding a package during 'make check_deps' is not supported and does not work. "
                    f"If you get this error please verify that the whole tree of Conan packages required is using exactly the same version of the conan-base package."
                )
                sys.exit(199)

    def empirix_validate_copy(self, pattern_str: str, dst: str = "", exact_num_files: int = None, min_num_files: int = None):
        list_of_copied = self.copy(pattern_str, dst)
        if exact_num_files != None and len(list_of_copied) != exact_num_files:
            raise ConanInvalidConfiguration(
                f"When copying pattern [{pattern_str}] a total of {exact_num_files} files were expected to be copied but instead {len(list_of_copied)} files were copied: {list_of_copied}"
            )
        if min_num_files != None and len(list_of_copied) < min_num_files:
            raise ConanInvalidConfiguration(
                f"When copying pattern [{pattern_str}] a min of {min_num_files} files were expected to be copied but only {len(list_of_copied)} files were copied: {list_of_copied}"
            )

    def empirix_render_and_package_docker_compose_yaml(self):
        if not self.empirix_verify_option("docker_image_variant"):
            return

        self.empirix_check_env_vars(["DOCKER_VERSION_FOR_TEST_BUNDLES", "DOCKER_GROUP"])
        replacements = {
            # CONTAINER_IMAGE_VERSION is DEPRECATED:
            # reason is that it contains DOCKER informations... we keep them for backward compatibility, but if possible, use the DOCKER_ var
            "CONTAINER_IMAGE_VERSION": str(os.environ.get("DOCKER_VERSION_FOR_TEST_BUNDLES")),
            "DOCKER_VERSION": str(os.environ.get("DOCKER_VERSION_FOR_TEST_BUNDLES")),
            "DOCKER_GROUP": str(os.environ.get("DOCKER_GROUP")),
        }
        for yaml_file in glob.glob("docker-compose.*.yml"):
            self.copy(yaml_file)
            self.empirix_replace_strings_in_file(os.path.join(self.package_folder, yaml_file), replacements)

    def empirix_render_and_package_charts_yaml(self):
        if not self.empirix_verify_option("docker_image_variant"):
            return

        self.empirix_check_env_vars(
            [
                "DOCKER_REGISTRY_FOR_TEST_BUNDLES",
                "DOCKER_VERSION_FOR_TEST_BUNDLES",
                "DOCKER_GROUP",
                "REMOTE_REPO_NEXUS_DOCKER_FOR_PULL",
                "HELM_REPO_FOR_TEST_BUNDLES",
                "HELM_VERSION_FOR_TEST_BUNDLES",
            ]
        )
        replacements = {
            # CONTAINER_IMAGE_VERSION is DEPRECATED:
            # CONTAINER_IMAGE contains DOCKER informations... we keep them for backward compatibility, but if possible, use the DOCKER_ vars
            "CONTAINER_IMAGE_VERSION": str(os.environ.get("DOCKER_VERSION_FOR_TEST_BUNDLES")),
            "DOCKER_VERSION": str(os.environ.get("DOCKER_VERSION_FOR_TEST_BUNDLES")),
            "DOCKER_GROUP": str(os.environ.get("DOCKER_GROUP")),
            "HELM_VERSION": os.environ.get("HELM_VERSION_FOR_TEST_BUNDLES"),
        }
        # recursively copy & render all YAML files under the pre-defined "helm" folder; this typically has a structure like:
        #    helm/chart.<component>.yaml
        #    helm/docker_images.yaml
        #    helm/chart_values/<additional-value-file1>.yaml
        #    helm/chart_values/<additional-value-file2>.yaml
        for yaml_file in glob.glob("helm/**/*.yaml", recursive=True):
            self.copy(yaml_file)
            self.empirix_replace_strings_in_file(os.path.join(self.package_folder, yaml_file), replacements)

    #
    # Conan standard functions, invoked by Conan at runtime
    #

    def set_version(self):
        """
        This function is called by Conan in a very early stage (possibly before any other of the standard functions)
        to set dynamically the version of this Conan package. By the time this function is called, the versioning informations
        is still available because Conan has not yet moved the source code into the "build folder" or the "package folder".
        Thus we use a combination of tools.Git() and "make get_conan_version" to find out the version from GNU make.
        """

        self.output.info("Executing set_version() method")

        # collect all info from GitHub
        github_ref_type = os.getenv("GITHUB_REF_TYPE")
        github_head_ref = os.getenv("GITHUB_HEAD_REF")
        github_ref_name = os.getenv("GITHUB_REF_NAME")
        github_sha = os.getenv("GITHUB_SHA")

        # collect all info we need from GIT:
        git = tools.Git()
        git_describe_output = git.run("describe --long --tags")

        git_branch_name_raw = git.get_branch()

        # get the name of the branch we are in:
        if (github_sha != None) and (github_sha != ""):
            github_branch_sha = git.run("branch -r --contains " + github_sha)

            # remove the "origin/" prefix
            branches = github_branch_sha.split()

            if len(branches) > 0:
                git_branch_name_raw = branches[0].replace("origin/", "")

        if github_ref_type and github_ref_type != "":
            # we are in a gh/gitlab ci pipeline
            if github_ref_type == "branch":
                if github_head_ref and github_head_ref != "":
                    git_branch_name_raw = github_head_ref
                else:
                    git_branch_name_raw = github_ref_name
            
        git_branch_name = git_branch_name_raw.replace("#", "")

        folder_with_makefile = git.get_repo_root()

        # set the Conan package version:
        self.version = self.empirix_ask_version_to_gnu_make(git_branch_name, git_describe_output, folder_with_makefile)

        # set also some metadata about the source branch for this Conan package:
        self.branch = git_branch_name

        # save the git-obtained versioning informations inside the Conan package itself
        self.empirix_write_env_vars(git_branch_name, git_describe_output)

    def config_options(self):
        self.output.info("Executing config_options() method")

        install_pipeline = os.getenv("INSTALL_PIPELINE_ONLY") == "True"
        if install_pipeline:
            return
        if self.conan_data is None:
            return
        if "cpp_free_version_requirements" in self.conan_data:
            self.empirix_inject_runtime_default_options(self.conan_data["cpp_free_version_requirements"], self.options)
        if "cpp_version_checked_requirements" in self.conan_data:
            self.empirix_inject_runtime_default_options(self.conan_data["cpp_version_checked_requirements"], self.options)

    def requirements(self):
        """
        This function will read the requirements for this Conan package from the file "conandata.yml" which is exposed
        by Conan into the self.conan_data member variable.
        We differentiate between
        - pipeline_requirements
        - noarch_requirements
        - cpp_free_version_requirements requirements
        - cpp_version_checked_requirements requirements
        """
        self.output.info("Executing requirements() method")
        install_pipeline = os.getenv("INSTALL_PIPELINE_ONLY") == "True"
        if self.conan_data is None:
            return
        # private=True is used to tell Conan that whatever ABI is inside the binaries of these packages, they
        # completely hide them. This is to instruct Conan to tolerate different major versions of these packages
        # across a dependency tree
        if install_pipeline:
            if "pipeline_requirements" in self.conan_data:
                for req in self.conan_data["pipeline_requirements"]:
                    self.requires(req, private=True)
        else:
            if "cpp_free_version_requirements" in self.conan_data:
                for req in self.conan_data["cpp_free_version_requirements"]:
                    self.requires(req, private=True)
            if "cpp_version_checked_requirements" in self.conan_data:
                for req in self.conan_data["cpp_version_checked_requirements"]:
                    self.requires(req, private=False)
            if "noarch_requirements" in self.conan_data:
                for req in self.conan_data["noarch_requirements"]:
                    self.requires(req, private=True)

    def package_id(self):
        # apply major_mode for all the dependencies of the package
        # This is necessary when using version numbers lower than 1.0.0
        self.info.requires.major_mode()


class ConanBase(ConanFile, EmpirixConanBase):

    # metadata specific to this repo:
    name = "conan-base"
    description = "Package to used to deploy the EmpirixConanBase class used by the projects in eva-ecc"
    # metadata common to all Empirix C++ repos:
    generators = "make"
    author = "Empirix EVA/ECC team"
    settings = "os", "compiler", "build_type", "arch"
    license = "Empirix Commercial License"
    topics = None

    # NOTE:
    # build_policy=never would be the most accurate build_policy option: with that Conan won't attempt to rebuild the
    # package during "make check_deps". HOWEVER build_policy=never can be used only for packages created using the
    # 'conan export-pkg' method. Currently our pipeline-framework code instead is using 'conan create'.
    # That's why we're forced in using build_policy=missing.
    # However keep in mind that our Conan packages CANNOT be rebuilt inplace during "make check_deps" because the Conan
    # package does not include all source code and tools to support the build; and even if the build was possible...
    # it would be very SLOW!
    build_policy = "missing"

    def package_id(self):
        """
        This function is called by Conan to understand the type of package to create. In case there are no precompiled binaries,
        we must inform Conan that this is a so-called header-only package.
        Note that to correctly install header-only packages apparently on the download side you will need to always give the --build
        flag to the 'conan install' command.
        """
        self.info.header_only()
