#
# This makefile contains several targets that allow to start building an EVA/ECC project
# using the BUILDER docker produced by the 'docker-base' repository.
# In practice all targets defined here are just a 'docker run' command to start a GNU make
# instance inside the builder docker to actually execute some other target.
# The convention is to have 
#     make docker_XYZ
# being equivalent to:
#     docker run --name builder_docker --entrypoint=/usr/bin/make ...   XYZ
#


# ------------------------------------------------------------------------------------------
# Caller makefile parameters
# ------------------------------------------------------------------------------------------

ifeq ($(HAS_FULLDIST_IN_BUILDER_DOCKER),)
# default value
HAS_FULLDIST_IN_BUILDER_DOCKER=1
endif
ifeq ($(HAS_FULLDIST_IN_BUILDER_DOCKER_CENTOS7),)
# default value
HAS_FULLDIST_IN_BUILDER_DOCKER_CENTOS7=0
endif
ifeq ($(HAS_TARGET_ALL_IN_BUILDER_DOCKER),)
# default value
HAS_TARGET_ALL_IN_BUILDER_DOCKER=1
endif

ifeq ($(HAS_CLEAN_IN_BUILDER_DOCKER),)
# default value
HAS_CLEAN_IN_BUILDER_DOCKER=1
endif

ifeq ($(HAS_TEST_IN_BUILDER_DOCKER),)
# default value
HAS_TEST_IN_BUILDER_DOCKER=1
endif

ifeq ($(HAS_TEST_IN_BUILDER_DOCKER_OL9),)
# default value
HAS_TEST_IN_BUILDER_DOCKER_OL9=0
endif
ifeq ($(HAS_UNIT_TESTS_IN_BUILDER_DOCKER),)
# default value
HAS_UNIT_TESTS_IN_BUILDER_DOCKER=1
endif

ifeq ($(HAS_PACKAGING_IN_BUILDER_DOCKER),)
# default value
HAS_PACKAGING_IN_BUILDER_DOCKER=1
endif
ifeq ($(HAS_FULLPACKAGING_IN_BUILDER_DOCKER),)
# default value
HAS_FULLPACKAGING_IN_BUILDER_DOCKER=1
endif

ifeq ($(HAS_PACKAGING_IN_BUILDER_DOCKER_CENTOS7),)
# default value
HAS_PACKAGING_IN_BUILDER_DOCKER_CENTOS7=0
endif
ifeq ($(HAS_FULLPACKAGING_IN_BUILDER_DOCKER_CENTOS7),)
# default value
HAS_FULLPACKAGING_IN_BUILDER_DOCKER_CENTOS7=0
endif


ifeq ($(HAS_DOCUMENTATION_IN_BUILDER_DOCKER),)
# default value
HAS_DOCUMENTATION_IN_BUILDER_DOCKER=1
endif
ifeq ($(HAS_DEPLOY_IN_BUILDER_DOCKER),)
# default value
HAS_DEPLOY_IN_BUILDER_DOCKER=1
endif

# ------------------------------------------------------------------------------------------
# Builder docker image auxiliary variables
# ------------------------------------------------------------------------------------------

#
# If BUILDER_DOCKER_IMAGE_FOR_RUN is not defined, we create it here, taking the contents of some variables;
# note that these variables are typically defined by the support makefile generated by the docker-base repo
# and can be defined in the caller makefile with:
#
#   -include $(EMPIRIX_DOCKER_BASE_MAKEFILE_SUPPORT_DIR)/DockerBase-support.mk
#

ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
# calculate a default value for the builder docker image to pass to the "docker run"

ifneq ($(BUILDER_DOCKER_OL9_IMAGE_NAME),)
ifneq ($(BUILDER_DOCKER_OL9_GROUP),)
ifneq ($(BUILDER_DOCKER_OL9_REMOTE_NEXUS_DOCKER_REGISTRY),)

# let's build here the name of the builder docker to pass to "docker run"
BUILDER_DOCKER_IMAGE_FOR_RUN=$(BUILDER_DOCKER_OL9_REMOTE_NEXUS_DOCKER_REGISTRY)/$(BUILDER_DOCKER_OL9_GROUP)/$(BUILDER_DOCKER_OL9_IMAGE_NAME)

endif
endif
endif

endif

ifeq ($(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN),)
# calculate a default value for the builder docker image to pass to the "docker run"

ifneq ($(BUILDER_DOCKER_CENTOS7_IMAGE_NAME),)
ifneq ($(BUILDER_DOCKER_CENTOS7_GROUP),)
ifneq ($(BUILDER_DOCKER_CENTOS7_REMOTE_NEXUS_DOCKER_REGISTRY),)

# let's build here the name of the builder docker to pass to "docker run"
BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN=$(BUILDER_DOCKER_CENTOS7_REMOTE_NEXUS_DOCKER_REGISTRY)/$(BUILDER_DOCKER_CENTOS7_GROUP)/$(BUILDER_DOCKER_CENTOS7_IMAGE_NAME)

endif
endif
endif

endif

ifeq ($(BUILDER_DOCKER_MAKE_OUTPUT_MODE),)
# default value "target" provides an immediate output on developer machines, very useful to understand what's going on.
# But Teamcity need a perfectly-ordered output to get nice, readable sections -- in that case we will use instead "recurse"
ifeq ($(BUILD_AGENT),1)
BUILDER_DOCKER_MAKE_OUTPUT_MODE=recurse
else
BUILDER_DOCKER_MAKE_OUTPUT_MODE=target
endif
endif

ifeq ($(REALTOP),)
# in case the caller makefile has defined only TOPDIR but not REALTOP
REALTOP=$(shell readlink -f $(TOPDIR))
endif

ifneq ($(LOAD_AVG),)
# NOTE: it's important NOT to use -j alone: -j alone means "infinite jobs" and we've seen a lot
#       of cases where using -j alone triggers a lot of OOM for GCC
LOAD_AVG_OPT:=--max-load=$(LOAD_AVG)
endif

# --cap-add=SYS_PTRACE unblocks the basic ptrace() syscall required by GDB
# --security-opt seccomp=unconfined avoids the "warning: error disabling address space randomization: operation not permitted"
BUILDER_DOCKER_ALLOW_GDB_PARAMS := --cap-add=SYS_PTRACE --security-opt seccomp=unconfined

# from docker docs:
#  --sig-proxy=true|false
#    Proxy all received signals to the process (non-TTY mode only). SIGCHLD, SIGKILL, and
#    SIGSTOP are not proxied. The default is true.
# We set it to false because we don't want to Docker to mess up with CTRL+C (SIGINT) signal
#  --stop-timeout 
# is used to change the default 10sec wait to 0 (instant exit when using CTRL+C)
BUILDER_DOCKER_ALLOW_CTRLC_PARAMS := --stop-timeout=0 
 ###--sig-proxy=false


# FIXME IN-41192
# About CONAN_DATA_DIR and CONAN_LOCAL_PROFILE_DIR: this is to allow for non-root builds:
# the data dir will be /opt/empirix/conan while the profile dir might be e.g. /home/xyz/.conan
# but will be mapped inside the docker as the /home/builder/.conan folder.
# In other words, inside the builder docker, we try to do a simple mapping of whatever user "xyz" is 
# in use on the host with the "builder" user.
# In theory we would need docker userns-remap: https://docs.docker.com/engine/security/userns-remap/
#
# About tmpfs mount on /home/builder/rpmbuild: this is to allow rpmbuild temporary file creation
# 
# About /var/run/docker.sock and /etc/docker/daemon.json: this is to allow Docker-in-Docker
# since we need to invoke "docker build" from within the docker container
BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS := \
	--volume $(REALTOP):/source \
	--volume $(CONAN_DATA_DIR):$(CONAN_DATA_DIR) \
	--volume $(CONAN_LOCAL_PROFILE_DIR):$(CONAN_DOCKER_PROFILE_DIR) \
	--volume $(CONAN_REMOTES_CONFIG_FILE):$(CONAN_REMOTES_CONFIG_FILE) \
	--mount type=tmpfs,destination=/home/builder/rpmbuild \
	--volume /var/run/docker.sock:/var/run/docker.sock \
	--volume /etc/docker/daemon.json:/etc/docker/daemon.json \
	--volume ~/.config/helm/repositories.yaml:/root/.config/helm/repositories.yaml

# The config.json file contains the login tokens for docker-registries.
# Using this volume bind, any docker pull launched inside the builder docker context (docker-in-docker)
# will use the credentials of the host machine
BUILDER_DOCKER_AUTHENTICATION_PARAMS := \
	--volume ~/.docker/config.json:/root/.docker/config.json

# This variable defined some container mapped files to use local proxies (jfrog & nexus)
BUILDER_DOCKER_PROXIES_PARAMS := \
	--volume ~/.m2/settings.xml:/root/.m2/settings.xml \
	--volume /etc/yum/vars/artifactory-yum-repository:/etc/yum/vars/artifactory-yum-repository

# This variable contains some major/critical env variables used by the targets defined in the "devel-environment"
# repository that alter the behavior of the GNU make build system
# More details:
#  - TEAMCITY_VERSION and BUILD_AGENT variables: they are used to detect/force the versioning scheme:
#    if the former is non-empty OR the second has value "1" then the production versioning scheme is used with
#    artifacts named based on the output of "git describe" (see Version-defines.mk).
#    These variables need to be provided to the builder docker to ensure also the builder docker can select the 
#    "official build" versioning scheme (teamcity) or "local" versioning scheme (dev machine)
#  - TC_PULLREQUEST_SOURCE_BRANCH: used by Version-defines.mk to set correct git branch name for repositories
#    checked out in "HEAD detached" mode (like for merge request builds in Teamcity)
#  - CFG: if non-empty defines which type of build to run inside the builder docker
#  - NUM_CPUS: allow customization of the number of CPUs passed to "-j" flag of GNU make; e.g. on some build agents
#    we prefer not to impose too much load and we purposely decrease this number from the maximum
#  - V: verbose or quiet build
#  - APT_BUILD: if 1 indicates a performance build (APT) is running; this allows the Makefile logic to skip debug & gcov builds
BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS := \
	--env TEAMCITY_VERSION="$(TEAMCITY_VERSION)" \
	--env TC_PULLREQUEST_SOURCE_BRANCH="$(TC_PULLREQUEST_SOURCE_BRANCH)" \
	--env BUILD_AGENT="$(BUILD_AGENT)" \
	--env CFG="$(CFG)" \
	--env NUM_CPUS="$(NUM_CPUS)" \
	--env V="$(V)" \
	--env LAB="$(LAB)" \
	--env APT_BUILD="$(APT_BUILD)" \
	--env DOCKER_USER_PARAMS="$(DOCKER_USER_PARAMS)"

# The following env variables are only used by the "docker_deploy" target:
BUILDER_DOCKER_DEPLOY_SCRIPTS_ENV_VARS_PARAMS := \
	--env REPO_TYPE_FOR_CONAN_DEPLOY="$(REPO_TYPE_FOR_CONAN_DEPLOY)" \
	--env ARTIFACTORY_USER="$(ARTIFACTORY_USER)" \
	--env ARTIFACTORY_PW="$(ARTIFACTORY_PW)" \
	--env NEXUS_USER="$(NEXUS_USER)" \
	--env NEXUS_PW="$(NEXUS_PW)" \
	--env ECR_REGISTRY="$(ECR_REGISTRY)" \
	--env ECR_ACCESS_KEY="$(ECR_ACCESS_KEY)" \
	--env ECR_SECRET_KEY="$(ECR_SECRET_KEY)"


# FIXME IN-41192 -- HOME env var
# BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS += \
#	--env HOME="/home/builder"
# This would be a workaround to allow non-root builds without using the docker userns remap feature
#

# About --network=host: 
#   on Centos7 it's not actually needed: the builder docker will be able to ping and use DNS
#   resolution even without it. However on OracleLinux8 it's needed otherwise the builder docker is unable to resolve
#   e.g. repo-billerica.perform.empirix.com FQDN.
# About --privileged:
#   this is useful in case we need to use 'perf' inside the builder docker bash
BUILDER_DOCKER_PERMISSIONS:=\
	--network=host \
	--privileged

# FIXME IN-41192 -- BUILDER_DOCKER_USER_PARAMS
#
## This variable allows to start the processes inside the builder docker as a non-root user
## Note that as documented at 
##   https://docs.docker.com/engine/reference/run/
## "When passing a numeric ID, the user does not have to exist in the container."
## This allows us to use whatever Linux user is present on the developer machine without the need
## to have such user present in the builder docker!
#DOCKER_GID=$(shell getent group docker | cut -d: -f3)
#ifeq ($(BUILDER_DOCKER_USER),)
## simply use as user for the builder docker whatever user launched the GNU make docker_* target
## NOTE: to allow Docker-in-Docker the GID must always be the GID of the "docker" group, assuming that
##       a) the current user is inside that group
##       b) the group of /var/run/docker.sock is "docker"
#BUILDER_DOCKER_USER_PARAMS := --user $(shell id -u):$(DOCKER_GID)
#else
#BUILDER_DOCKER_USER_PARAMS := --user $(shell id -u $(BUILDER_DOCKER_USER)):$(DOCKER_GID)
#endif

# This variable is used to set --workdir in a smart way to build only the current subtree
BUILDER_DOCKER_RELATIVE_WORKDIR := /source/$(shell realpath --relative-to=$(TOPDIR) .)


# ------------------------------------------------------------------------------------------
# Builder docker image auxiliary targets
# ------------------------------------------------------------------------------------------

# This target allows you to get access to a Bash inside the builder docker image:
docker_builder_bash::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.docker_builder_bash_history
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm -it  \
		--volume $(HOME)/.docker_builder_bash_history:/root/.bash_history:rw \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--env TEAMCITY_VERSION="$(TEAMCITY_VERSION)" \
		--env BUILD_AGENT="$(BUILD_AGENT)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) /bin/bash


# a synonym
bash_in_builder_docker: docker_builder_bash

# This target allows you to get access to a Bash inside the builder docker image:
docker_builder_bash_ol9::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.docker_builder_bash_history
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm -it  \
		--volume $(HOME)/.docker_builder_bash_history:/root/.bash_history:rw \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--env TEAMCITY_VERSION="$(TEAMCITY_VERSION)" \
		--env BUILD_AGENT="$(BUILD_AGENT)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) /bin/bash

ifeq ($(HAS_TARGET_ALL_IN_BUILDER_DOCKER),1)

docker_all::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_all target for CFG=$(CFG))
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			all \
			-j$(NUM_CPUS) $(LOAD_AVG_OPT) \
			--no-print-directory \
			--output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1

endif # HAS_TARGET_ALL_IN_BUILDER_DOCKER


ifeq ($(HAS_CLEAN_IN_BUILDER_DOCKER),1)

# in the docker_clean target we provide also the CFG=release/debug variable to the builder docker,
# plus we set the --workdir in a smart way to build only the current subtree
docker_clean::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			clean \
			-j$(NUM_CPUS) --no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1


# by its nature the "fullclean" does not need CFG propagated:
docker_fullclean::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			fullclean \
			-j$(NUM_CPUS) --no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1

endif # HAS_CLEAN_IN_BUILDER_DOCKER


ifeq ($(HAS_TEST_IN_BUILDER_DOCKER),1)

docker_test::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			test \
			--output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1


docker_test_valgrind::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			test_valgrind \
			--output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1

endif # HAS_TEST_IN_BUILDER_DOCKER


# NOTE: when running fulldist in this way you cannot CTRL+C it... so we rather give it a defined name
#       to make it easier to run "docker stop builder_docker" from another shell just in case
ifeq ($(HAS_FULLDIST_IN_BUILDER_DOCKER),1)
docker_fulldist::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_fulldist target for CFG=$(CFG) for Oracle Linux 9)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	-docker stop builder_docker   # in case it was hanging since a previous build perhaps
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=/source \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			fulldist \
			-j$(NUM_CPUS) \
			--no-print-directory

endif # HAS_FULLDIST_IN_BUILDER_DOCKER

ifeq ($(HAS_FULLDIST_IN_BUILDER_DOCKER_CENTOS7),1)
docker_fulldist::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_fulldist target for CFG=$(CFG) for CentOS 7)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	-docker stop builder_docker   # in case it was hanging since a previous build perhaps
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=/source \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN) \
			fulldist \
			-j$(NUM_CPUS) \
			--no-print-directory

endif # HAS_FULLDIST_IN_BUILDER_DOCKER_CENTOS7

ifeq ($(HAS_UNIT_TESTS_IN_BUILDER_DOCKER),1)
docker_unit_tests::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_unit_tests target for CFG=$(CFG) for Oracle Linux 9)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=/source \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
		unit_tests --no-print-directory 2>&1

endif # HAS_UNIT_TESTS_IN_BUILDER_DOCKER

ifeq ($(HAS_PACKAGING_IN_BUILDER_DOCKER),1)
docker_packaging::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_packaging target for CFG=$(CFG) for Oracle Linux 9)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		--env GCOV="$(GCOV)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			packaging \
			--no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1
endif # HAS_PACKAGING_IN_BUILDER_DOCKER

ifeq ($(HAS_FULLPACKAGING_IN_BUILDER_DOCKER),1)
docker_fullpackaging::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_fullpackaging target for CFG=$(CFG) for Oracle Linux 9)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		--env GCOV="$(GCOV)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			fullpackaging \
			--no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1
endif # HAS_FULLPACKAGING_IN_BUILDER_DOCKER

ifeq ($(HAS_PACKAGING_IN_BUILDER_DOCKER_CENTOS7),1)
docker_packaging::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_packaging target for CFG=$(CFG) for CentOS 7)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		--env GCOV="$(GCOV)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN) \
			packaging \
			--no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1
endif # HAS_PACKAGING_IN_BUILDER_DOCKER_CENTOS7

ifeq ($(HAS_FULLPACKAGING_IN_BUILDER_DOCKER_CENTOS7),1)
docker_fullpackaging::
	@echo "======================================================================================================================"
	@$(call print_green,Running docker_fullpackaging target for CFG=$(CFG) for CentOS 7)
	@echo "======================================================================================================================"
ifeq ($(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		--env GCOV="$(GCOV)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN) \
			fullpackaging \
			--no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1
endif # HAS_FULLPACKAGING_IN_BUILDER_DOCKER_CENTOS7

ifeq ($(HAS_DOCUMENTATION_IN_BUILDER_DOCKER),1)
docker_documentation::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		--env GCOV="$(GCOV)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			documentation \
			--no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1
endif # HAS_DOCUMENTATION_IN_BUILDER_DOCKER


ifeq ($(HAS_DEPLOY_IN_BUILDER_DOCKER),1)
docker_deploy::
ifeq ($(BUILDER_DOCKER_IMAGE_FOR_RUN),)
	@echo "Please provide a valid BUILDER_DOCKER_IMAGE_FOR_RUN... was DockerBase-support.mk included in the Makefile of this directory?"
	@exit 100
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		$(BUILDER_DOCKER_DEPLOY_SCRIPTS_ENV_VARS_PARAMS) \
		--workdir=$(BUILDER_DOCKER_RELATIVE_WORKDIR) \
		--entrypoint=/usr/bin/make \
		--env GCOV="$(GCOV)" \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			deploy \
			--no-print-directory --output-sync=$(BUILDER_DOCKER_MAKE_OUTPUT_MODE) 2>&1
endif # HAS_DEPLOY_IN_BUILDER_DOCKER





docker_make_any_target:
ifeq ($(DOCKER_MAKE_TARGET),)
	@echo "Please invoke this target providing the option DOCKER_MAKE_TARGET=<some-target>"
	@exit 99
endif
	touch $(HOME)/.config/helm/repositories.yaml
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=/source \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
		$(DOCKER_MAKE_TARGET) --no-print-directory 2>&1


# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Builder docker targets (to run OUTSIDE the builder docker itself!):"
	@echo "    docker_builder_bash:                  start a bash inside the builder docker"
	@echo "    docker_all:                           run the [all] target inside the builder docker"
ifeq ($(HAS_FULLDIST_IN_BUILDER_DOCKER),1)    
	@echo "    docker_fulldist:                      run the [fulldist] target inside the builder docker"
endif    
	@echo "    docker_clean:                         run the [clean] target inside the builder docker"
	@echo "    docker_fullclean:                     run the [fullclean] target inside the builder docker"
	@echo "    docker_test:                          run the [test] target inside the builder docker"
	@echo "    docker_test_valgrind:                 run the [test_valgrind] target inside the builder docker"
ifeq ($(HAS_UNIT_TESTS_IN_BUILDER_DOCKER),1)    
	@echo "    docker_unit_tests:                    run the [unit_tests] target inside the builder docker"
endif
	@echo "    docker_packaging:                     run the [packaging] target inside the builder docker"
	@echo "    docker_fullpackaging:                 run the [fullpackaging] target inside the builder docker"
	@echo "    docker_make_any_target:               run with DOCKER_MAKE_TARGET=sometarget to run it inside the builder docker"
endif # SUPPORTS_HELP
