#
# This is the file which is typically included by top-level-folder makefiles of a git repository
# that needs to collect C/C++ code coverage results from CFG=debug-gcov builds
#

# ------------------------------------------------------------------------------------------
# Calling makefile parameters
# ------------------------------------------------------------------------------------------

ifndef TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT
$(error Please define TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT before including this snippet)
endif

ifeq ($(HAS_DIFF_COVER_SUPPORT),)
# disabled for now since it is way too much verbose inside MRs...
HAS_DIFF_COVER_SUPPORT=0
endif

ifeq ($(DIFF_COVER_MAIN_BRANCH),)
DIFF_COVER_MAIN_BRANCH=origin/develop
endif

ifeq ($(FUNCTION_COVERAGE_THRESHOLD_PERC),)
FUNCTION_COVERAGE_THRESHOLD_PERC=50
endif

ifndef SUPPORTS_HELP
# default value
SUPPORTS_HELP=1
endif

# ------------------------------------------------------------------------------------------
# constants
# ------------------------------------------------------------------------------------------

FASTCOV_OPTS:=--validate-sources --branch-coverage --gcov gcov --exclude /usr/include /usr/lib /opt/empirix/conan TeamcityArtifacts/ oss/
FASTCOV_FINAL_EXCLUSIONS:='*/UnitTests*' '*/oss/*'
FASTCOV_FINAL_OUTPUT_FILE:=$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/totals.fastcov.json
LCOV_FINAL_OUTPUT_FILE:=$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/totals.lcov
SONARQUBE_FINAL_OUTPUT_FILE:=$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/totals.sonarqube.xml

# IMPORTANT: the list of environment variables provided to the builder docker are different to the list
#            of env vars provided to the same builder docker at e.g. fulldist or fullpackaging steps!!
BUILDER_DOCKER_CODECOVERAGE_ENV_VARS_PARAMS := \
		--env TC_PULLREQUEST_NUMBER="$(TC_PULLREQUEST_NUMBER)" \
		--env TC_PULLREQUEST_SOURCE_BRANCH="$(TC_PULLREQUEST_SOURCE_BRANCH)" \
		--env TC_PULLREQUEST_TARGET_BRANCH="$(TC_PULLREQUEST_TARGET_BRANCH)" \
		--env SONAR_HOST="$(SONAR_HOST)" \
		--env SONAR_TOKEN="$(SONAR_TOKEN)" \
		--env GITLAB_USER="$(GITLAB_USER)" \
		--env GITLAB_ACCESS_TOKEN="$(GITLAB_ACCESS_TOKEN)" \
		--env ATF_ENABLE="$(ATF_ENABLE)" \
		--env HAS_SONAR_SCANNER_SUPPORT="$(HAS_SONAR_SCANNER_SUPPORT)" \
		--env TEAMCITY_VERSION="$(TEAMCITY_VERSION)" \
		--env BUILD_AGENT="$(BUILD_AGENT)" \
		--env CFG="$(CFG)" \
		--env V="$(V)"

# ------------------------------------------------------------------------------------------
# Sonar(Qube) parameters
# ------------------------------------------------------------------------------------------

SONAR_LINK_BRANCH_PR_PARAM:="branch=$(GIT_BRANCH_NAME)"
IS_MERGE_REQUEST_BUILD=0

# IMPORTANT: in some builds (non merge-request builds) teamcity will not expand at all the variables related to
#            merge request stuff so they will still contain %key% text! Consider in this case we have no MR data
ifneq ($(TC_PULLREQUEST_NUMBER),)
ifneq ($(TC_PULLREQUEST_NUMBER),%teamcity.pullRequest.number%)
SONAR_BRANCH_NAME_PARAMS+=-Dsonar.pullrequest.key=$(TC_PULLREQUEST_NUMBER)
SONAR_LINK_BRANCH_PR_PARAM:="pullRequest=$(TC_PULLREQUEST_NUMBER)"
IS_MERGE_REQUEST_BUILD=1
endif
endif

ifneq ($(TC_PULLREQUEST_SOURCE_BRANCH),)
ifneq ($(TC_PULLREQUEST_SOURCE_BRANCH),%teamcity.pullRequest.source.branch%)
SONAR_BRANCH_NAME_PARAMS+=-Dsonar.pullrequest.branch=$(TC_PULLREQUEST_SOURCE_BRANCH)
IS_MERGE_REQUEST_BUILD=1
endif
endif

ifneq ($(TC_PULLREQUEST_TARGET_BRANCH),)
ifneq ($(TC_PULLREQUEST_TARGET_BRANCH),%teamcity.pullRequest.target.branch%)
SONAR_BRANCH_NAME_PARAMS+=-Dsonar.pullrequest.base=$(TC_PULLREQUEST_TARGET_BRANCH)
IS_MERGE_REQUEST_BUILD=1
endif
endif

# ------------------------------------------------------------------------------------------
# Enable sonar only if supported and we're building: tags, merge requests and develop
# ------------------------------------------------------------------------------------------

SONAR_SCAN_DISABLE_REASON:=Sonarqube scan disabled because: building a branch which is not develop and not building a MergeRequest

ifeq ($(IS_MERGE_REQUEST_BUILD),1)
# pull/merge request build, SONAR_BRANCH_NAME_PARAMS var already set
SONAR_SCAN_ENABLE_REASON:=Sonarqube scan enabled because: building a MergeRequest
IS_SONAR_SCAN_ENABLED:=1
endif

ifeq ($(GIT_BRANCH_NAME),develop)
SONAR_SCAN_ENABLE_REASON:=Sonarqube scan enabled because: building DEVELOP branch
IS_SONAR_SCAN_ENABLED:=1

ifeq ($(SONAR_BRANCH_NAME_PARAMS),)
ifeq ($(NUM_COMMITS_SINCE_LAST_TAG),0)

# running a build (probably of develop branch) immediately after a tag was released
# give as branch name the FULL_VERSION that will be used to qualify artifacts of this build.
# in this way we get in sonarqube dashboard a new entry named with FULL_VERSION that allows
# us to track the quality of different released versions 
SONAR_BRANCH_NAME_PARAMS:=-Dsonar.branch.name=$(FULL_VERSION)

else  # NUM_COMMITS_SINCE_LAST_TAG!=0

# if SONAR_BRANCH_NAME_PARAMS is still empty it means this is not a PR/MR nor a build for a tag;
# this means this is just a normal build of a develop branch:
SONAR_BRANCH_NAME_PARAMS:=-Dsonar.branch.name=develop

endif # NUM_COMMITS_SINCE_LAST_TAG
endif # SONAR_BRANCH_NAME_PARAMS empty
endif # GIT_BRANCH_NAME=develop

# ------------------------------------------------------------------------------------------
# Disable sonarqube based on some conditions
# ------------------------------------------------------------------------------------------

ifeq ($(HAS_SONAR_SCANNER_SUPPORT),0)
SONAR_SCAN_DISABLE_REASON:=Sonarqube scan disabled because: the project root makefile or an environment variable is setting HAS_SONAR_SCANNER_SUPPORT to zero
IS_SONAR_SCAN_ENABLED:=0
endif

ifeq ($(ATF_ENABLE),0)
# if build was launched with automated tests disabled, skipping Sonarqube check since most likely 
# it will not pass (due to low coverage) and the test-less build would fail.
# If the build was launched with ATF_ENABLE=0 it's probably because you just need artifacts quickly in the repo.
SONAR_SCAN_DISABLE_REASON:=Sonarqube scan disabled because: the project root makefile or an environment variable is disabling all automated tests
IS_SONAR_SCAN_ENABLED:=0
endif
ifeq ($(ATF_ENABLE),false)
SONAR_SCAN_DISABLE_REASON:=Sonarqube scan disabled because: the project root makefile or an environment variable is disabling all automated tests
IS_SONAR_SCAN_ENABLED:=0
endif

ifeq ($(SONAR_HOST),)
SONAR_SCAN_DISABLE_REASON:=Sonarqube scan disabled because: Sonarqube server not configured (SONAR_HOST is empty)
IS_SONAR_SCAN_ENABLED:=0
endif

ifeq ($(SONAR_TOKEN),)
SONAR_SCAN_DISABLE_REASON:=Sonarqube scan disabled because: Sonarqube token not configured (SONAR_TOKEN is empty)
IS_SONAR_SCAN_ENABLED:=0
endif

# ------------------------------------------------------------------------------------------
# diff-cover parameters
# ------------------------------------------------------------------------------------------

ifeq ($(GITLAB_USER),)
HAS_DIFF_COVER_SUPPORT=0
endif
ifeq ($(GITLAB_ACCESS_TOKEN),)
HAS_DIFF_COVER_SUPPORT=0
endif


# ------------------------------------------------------------------------------------------
# code coverage targets (to run inside the builder docker)
# ------------------------------------------------------------------------------------------

gcov_clean:
	# Recursively delete all gcda files
	fastcov --zerocounters
	rm -f $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/*.fastcov.json $(LCOV_FINAL_OUTPUT_FILE) $(FASTCOV_FINAL_OUTPUT_FILE)

# the following target is typically linked to the "unit_tests" target which is run into the "docker_unit_tests" target and
# thus runs inside the builder docker:
gcov_process_intree_gcda_files:
ifeq ($(GCOV_TEST_NAME),)
	@echo "Please provide a compact string for the GCOV_TEST_NAME variable. Aborting."
	@exit 200
endif
	@echo "##teamcity[blockOpened name='Processing GCDA data generated by $(GCOV_TEST_NAME)']"
	mkdir -p $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)
	# find all gcda files and assemble them into a "coverage-report" both in FASTCOV JSON format
	fastcov $(FASTCOV_OPTS) --output $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/coverage-$(GCOV_TEST_NAME).fastcov.json || true
	# make sure we don't leave .gcda files around if they were already processed!
	fastcov --zerocounters
	@echo "##teamcity[blockClosed name='Processing GCDA data generated by $(GCOV_TEST_NAME)']"

# the following target is typically linked to the "docker_gcov_finalize" target and rusn inside the builder docker:
gcov_process_automated_tests_gcda_tarballs:
	@echo "##teamcity[blockOpened name='Processing automated tests GCDA data']"
	@echo "Processing automated tests GCDA tarballs from $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)"
	@if [ -n "$$(ls -A $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/*.tar.gz 2>/dev/null)" ]; then \
		for testsuiteGcdaTarball in $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/*.tar.gz; do \
			outFile=$$(echo $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/$$(basename $$testsuiteGcdaTarball) | cut -f 1 -d '.'); \
			echo "--------------------------------------------------------------------------------------------------------" ; \
			echo "Processing automated tests GCOV data from $$testsuiteGcdaTarball --> $$outFile.fastcov.json" ; \
			echo "--------------------------------------------------------------------------------------------------------" ; \
			fastcov --zerocounters || exit 100 ; \
			tar -xvzf $$testsuiteGcdaTarball -C $(TOPDIR) || exit 200; \
			fastcov $(FASTCOV_OPTS) --output $$outFile.fastcov.json || true; \
		done ; \
	fi
	#@echo "Validating list of gcda files"
	#cd Internal-Utilities/scripts && \
	#	./validate_gcda_files.sh $(REALTOP) $(TEAMCITY_ARTIFACTS_DIR_MAIN) /tmp/gcda/gcda_list    # script will cleanup complete directory containing gcda list
	# make sure we don't leave .gcda files around if they were already processed!
	fastcov --zerocounters
	@echo "##teamcity[blockClosed name='Processing automated tests GCDA data']"

# the following target is typically linked to the "docker_gcov_finalize" target and runs inside the builder docker and it does:
#   for each automatedtest tarball:
#        .tar.gz -> .lcov 
#   join of all .lcov files (with filter)
gcov_merge_and_finalize:
	@echo "##teamcity[blockOpened name='Processing all FASTCOV JSON']"
	@echo "Combining all FASTCOV json together"
	-rm -f $(FASTCOV_FINAL_OUTPUT_FILE) # remove leftover from other runs (if any)
	fastcov \
		-C $(wildcard $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/*.fastcov.json) \
		--exclude $(FASTCOV_FINAL_EXCLUSIONS) \
		--output $(FASTCOV_FINAL_OUTPUT_FILE)
	fastcov -C $(FASTCOV_FINAL_OUTPUT_FILE) --lcov --output $(LCOV_FINAL_OUTPUT_FILE)
	fastcov_to_sonarqube -o $(SONARQUBE_FINAL_OUTPUT_FILE) $(FASTCOV_FINAL_OUTPUT_FILE)
	genhtml \
		--quiet --legend --function-coverage --show-details \
		--rc "genhtml_hi_limit = 90" \
		--rc "genhtml_med_limit = 60" \
		--output-directory $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT) $(LCOV_FINAL_OUTPUT_FILE)
	lcov_cobertura $(LCOV_FINAL_OUTPUT_FILE) -o $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/cobertura.xml
	@echo "Done generating the HTML report"
	@echo "##teamcity[blockClosed name='Processing all FASTCOV JSON']"

gcov_coverage_check:
	@echo "##teamcity[blockOpened name='Code coverage checks']"
ifneq ($(ATF_ENABLE), 0)
	@echo "Performing a check on the ABSOLUTE/TOTAL VALUE OF FUNCTION COVERAGE level measured combining unit tests and automated tests."
	@echo "The MIN threshold for this project is $(FUNCTION_COVERAGE_THRESHOLD_PERC)%. If the coverage falls below, the build fails."
	fastcov_summary --function-coverage-threshold $(FUNCTION_COVERAGE_THRESHOLD_PERC) $(FASTCOV_FINAL_OUTPUT_FILE)
else
	@echo "Automated tests were disabled, so skipping this check since most likely it will not pass and the test-less build would fail."
	@echo "If the build was launched with ATF_ENABLE=0 it's probably because you just need artifacts quickly in the repo."
endif
	@echo "##teamcity[blockClosed name='Code coverage checks']"

diff_cover:
	@echo "##teamcity[blockOpened name='DiffCover tool']"
ifeq ($(HAS_DIFF_COVER_SUPPORT),1)
	# diff-cover utility works by running a "git diff origin/develop..HEAD" command and then cross-checking the 
	# data in the Cobertura XML file with the data out of the "git diff" operation.
	# The problem here is that Teamcity and several other CI servers will not checkout all the branches of the project to build,
	# only a strictly required subset of "refs". This makes the git diff step fail with an error like:
	#    diff_cover.command_runner.CommandError: fatal: ambiguous argument 'origin/develop...HEAD': unknown revision or path not in the working tree.
	# To workaround the problem here we force the build agent to fetch all references in the remote repository, including "origin/develop".
	# To do that we need however to have gitlab credentials available and provide them in a non-interactive way to git fetch command,
	# see https://stackoverflow.com/questions/8536732/can-i-hold-git-credentials-in-environment-variables/43022442#43022442
	@echo '#!/bin/bash' > /tmp/diff-cover-git-credential-helper.sh
	@echo "sleep 1" >> /tmp/diff-cover-git-credential-helper.sh
	@echo "echo username=$$GITLAB_USER" >> /tmp/diff-cover-git-credential-helper.sh
	@echo "echo password=$$GITLAB_ACCESS_TOKEN" >> /tmp/diff-cover-git-credential-helper.sh
	@echo "Now fetching all git references from remote Gitlab repository"
	@git -c credential.helper="/bin/bash /tmp/diff-cover-git-credential-helper.sh" fetch
	@rm -f /tmp/diff-cover-git-credential-helper.sh
	@mkdir -p $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT)
	@echo "Now running diff-cover tool to create a report about coverage of files modified in this branch compared to $(DIFF_COVER_MAIN_BRANCH)"
	@echo "# Automatic differential coverage analysis" >$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT)/diff-coverage.txt
	diff-cover \
		--compare-branch=$(DIFF_COVER_MAIN_BRANCH) \
		--markdown-report $(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT)/diff-coverage.md \
		$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_GCOV_REPORT)/cobertura.xml
ifneq ($(GITLAB_PROJECT_ID),)
ifneq ($(TC_PULLREQUEST_NUMBER),)
ifneq ($(TC_PULLREQUEST_NUMBER),%teamcity.pullRequest.number%)
	@echo "Creating message to post in Gitlab as temporary file /tmp/diff-cover-comment.json"
	@jq -Rs <$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT)/diff-coverage.md >/tmp/diff-cover-tmp.json
	@echo '{ "body": "" }' | jq --slurpfile texts /tmp/diff-cover-tmp.json '.body=$$texts[0]' >/tmp/diff-cover-comment.json
	@echo "Now adding a comment to the associated MR number $(TC_PULLREQUEST_NUMBER) in gitlab with the results of diff-cover."
	curl --include --show-error --request POST \
		--header "Private-Token: $(GITLAB_ACCESS_TOKEN)" \
		--header "Content-Type: application/json" \
		--header "Accept: application/json" \
		--data "@/tmp/diff-cover-comment.json" \
		https://gitlab.com/api/v4/projects/$(GITLAB_PROJECT_ID)/merge_requests/$(TC_PULLREQUEST_NUMBER)/notes || true
	@echo
	@echo
	@echo "Please check text above to see if the Gitlab REST API was successful or not. Diff-cover target completed."
endif
endif
endif
else
	@echo "Diff-cover disabled for this project."
endif
	@echo "##teamcity[blockClosed name='DiffCover tool']"

sonar_cli_scan:
	@echo "##teamcity[blockOpened name='Sonarqube Scanner tool']"
ifeq ($(IS_SONAR_SCAN_ENABLED),1)
	@echo $(SONAR_SCAN_ENABLE_REASON)
	@echo "<a href='$(SONAR_HOST)dashboard?id=$(shell crudini --get sonar-project.properties '' sonar.projectKey)&$(SONAR_LINK_BRANCH_PR_PARAM)'>Link to Sonarqube dashboard</a>." >$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT)/sonarqube.html
	@echo "Please right click and choose to open this link in a new tab." >>$(TEAMCITY_ARTIFACTS_DIR_COVERAGE_HTML_REPORT)/sonarqube.html
	$(SONARQUBE_CLI_SCAN_BIN) \
		-Dsonar.host.url=$(SONAR_HOST) \
		-Dsonar.login=$(SONAR_TOKEN) \
		-Dsonar.qualitygate.wait=true \
		-Dsonar.cfamily.build-wrapper-output=$(SONARQUBE_WRAPPER_OUTDIR) \
		-Dsonar.cfamily.cache.enabled=false \
		-Dsonar.projectVersion=$(FULL_VERSION) \
		-Dsonar.coverageReportPaths=$(SONARQUBE_FINAL_OUTPUT_FILE) \
		-Dsonar.cfamily.threads=$(NUM_CPUS) \
		$(SONAR_BRANCH_NAME_PARAMS)
else
	@echo "Sonar scan disabled for this build. Sonar scans will be run only on: develop builds and MR builds triggered by Gitlab (they appear in Teamcity as merge_request/XX)."
	@echo "$(SONAR_SCAN_DISABLE_REASON)"
endif
	@echo "##teamcity[blockClosed name='Sonarqube Scanner tool']"

# ------------------------------------------------------------------------------------------
# builder docker targets
# ------------------------------------------------------------------------------------------

docker_gcov_finalize:
	docker run --rm --name builder_docker \
		--cap-add=SYS_PTRACE --security-opt seccomp=unconfined \
		--volume $(REALTOP):/source  \
		--volume $(CONAN_DATA_DIR):$(CONAN_DATA_DIR) \
		--volume /var/run/docker.sock:/var/run/docker.sock \
		--volume /etc/docker/daemon.json:/etc/docker/daemon.json \
		--workdir=/source \
		--entrypoint=/usr/bin/make \
		$(BUILDER_DOCKER_CODECOVERAGE_ENV_VARS_PARAMS) \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			--no-print-directory \
			gcov_process_automated_tests_gcda_tarballs \
			gcov_merge_and_finalize \
			diff_cover \
			sonar_cli_scan \
			gcov_coverage_check 2>&1


# ------------------------------------------------------------------------------------------
# code metrics targets
# ------------------------------------------------------------------------------------------

docker_get_code_metrics:
	# NOTE: I didn't find a way to ask "cloc" to only count C/C++ source code files... 
	#       it's always counting all recognized programming languages (a lot)
	docker run --rm \
		--volume $(REALTOP):/source \
		aldanial/cloc \
		/source --csv --report-file=/source/source_code_analysis.csv



# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Code coverage targets (to run OUTSIDE the builder docker):"
	@echo "    docker_gcov_finalize:                 run GCOV postprocessing, SonarQube and other code coverage checks"
endif # SUPPORTS_HELP
