#
# Makefile logic to decide which repository to host each artifact type
# Make sure the following variables are defined
#   REALTOP
#   LAB                              -> modena/billerica/massy
#   GIT_BRANCH_NAME                  -> defined by Version-defines.mk: name of current GIT branch, used to decide REPO_TYPE_FOR_DEPLOY value
#   GIT_BRANCH_TYPE                  -> defined by Version-defines.mk: type of current GIT branch, used to decide REPO_TYPE_FOR_DEPLOY value (e.g. support, master, develop)
#   NUM_COMMITS_SINCE_LAST_TAG       -> defined by Version-defines.mk
#
# Optionally-defined variables:
#   REPO_TYPE_FOR_DEPLOY             -> deprecated now
#   REPO_TYPE_FOR_DOCKER_DEPLOY      -> defaults to "ecr", override to decide the destination repository
#   REPO_TYPE_FOR_HELM_DEPLOY        -> defaults to "artifactory", override to decide the destination repository
#   REPO_TYPE_FOR_RPM_DEPLOY         -> defaults to "artifactory", override to decide the destination repository
#   REPO_TYPE_FOR_PYTHON_DEPLOY      -> defaults to "artifactory", override to decide the destination repository
#   REPO_TYPE_FOR_CONAN_DEPLOY       -> defaults to "artifactory_billerica", override to decide the destination repository
#   DEPLOY_FILELIST_LOG              -> logfile of deployed file; by default goes under TeamcityArtifacts
# 
# Very optional variables (mostly used only inside pipelines, not useful for developers):
#   ARTIFACTORY_USER                 -> user to use to upload to Artifactory
#   ARTIFACTORY_PW                   -> pw to use to upload to Artifactory
#   NEXUS_USER                       -> user to use to upload to Nexus
#   NEXUS_PW                         -> pw to use to upload to Nexus
#


# ---------------------------------------------
# required parameters
# ---------------------------------------------

ifeq ($(REALTOP),)
$(error Please define REALTOP before including this snippet)
endif
ifeq ($(LAB),)
$(error Please define LAB before including this snippet)
endif
ifeq ($(GIT_BRANCH_NAME),)
$(error Please define GIT_BRANCH_NAME before including this snippet)
endif
ifeq ($(GIT_BRANCH_TYPE),)
$(error Please define GIT_BRANCH_TYPE before including this snippet)
endif
ifeq ($(NUM_COMMITS_SINCE_LAST_TAG),)
$(error Please define NUM_COMMITS_SINCE_LAST_TAG before including this snippet)
endif


# ---------------------------------------------
# includes
# ---------------------------------------------

# include the constants of the various remotes
include ${EMPIRIX_PIPELINE_FRAMEWORK_MAKEFILE_SUPPORT_DIR}/Internal-repoconstants.mk


# ---------------------------------------------
# Repository selection logic
# ---------------------------------------------

# REPO_TYPE_FOR_DOCKER_DEPLOY accepts only 3 values:
#  - ecr:            the Amazon ECR cloud instance [DEFAULT]
#  - artifactory:    the Infovista cloud instance of Artifactory -- pushing dockers here is now deprecated
#  - nexus:          the hosted docker registry instance in the Infovista on-prem repo
ifeq ($(REPO_TYPE_FOR_DOCKER_DEPLOY),)
	REPO_TYPE_FOR_DOCKER_DEPLOY=ecr
else
	ifneq ($(REPO_TYPE_FOR_DOCKER_DEPLOY), ecr)
		ifneq ($(REPO_TYPE_FOR_DOCKER_DEPLOY), artifactory)
$(error Unsupported value for REPO_TYPE_FOR_DOCKER_DEPLOY variable: $(REPO_TYPE_FOR_DOCKER_DEPLOY))
		endif
	endif
endif

# REPO_TYPE_FOR_CONAN_DEPLOY accepts only 2 values:
#  - artifactory_billerica:  the Infovista-owned instance of Artifactory CE  [DEFAULT]
#  - artifactory_cloud:      the Infovista cloud instance of Artifactory
# NOTE: since Conan packages are meant to transport intermediate C/C++ artifacts (e.g. static&shared libraries or testbundles)
#       which do not constitute a runnable product by themselves, by default we do not deploy them on the remote cloud instance 
#       of Artifactory (artifactory_cloud); rather we deploy them in Infovista-owned private instance of Artifactory CE (artifactory_billerica)
ifeq ($(REPO_TYPE_FOR_CONAN_DEPLOY),)
	REPO_TYPE_FOR_CONAN_DEPLOY=artifactory_billerica
else
	ifneq ($(REPO_TYPE_FOR_CONAN_DEPLOY), artifactory_cloud)
		ifneq ($(REPO_TYPE_FOR_CONAN_DEPLOY), artifactory_billerica)
$(error Unsupported value for REPO_TYPE_FOR_CONAN_DEPLOY variable: $(REPO_TYPE_FOR_CONAN_DEPLOY))
		endif
	endif
endif

# REPO_TYPE_FOR_RPM_DEPLOY accepts only 2 values:
#  - artifactory:            the Infovista cloud instance of Artifactory
#  - nexus:                  some hosted repository on Infovista Nexus instances local to each LAB
# See DE-1564: since Sept 2023, by default all RPMs (even if produced by feature branches) will be deployed on Artifactory cloud instance.
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),)
	REPO_TYPE_FOR_RPM_DEPLOY=artifactory
else
	ifneq ($(REPO_TYPE_FOR_RPM_DEPLOY), nexus)
		ifneq ($(REPO_TYPE_FOR_RPM_DEPLOY), artifactory)
$(error Unsupported value for REPO_TYPE_FOR_RPM_DEPLOY variable: $(REPO_TYPE_FOR_RPM_DEPLOY))
		endif
	endif
endif

# REPO_TYPE_FOR_HELM_DEPLOY accepts only 2 values:
#  - artifactory:            the Infovista cloud instance of Artifactory
#  - nexus:                  some hosted repository on Infovista Nexus instances local to each LAB
# See DE-1564: since Sept 2023, by default all HELM charts (even if produced by feature branches) will be deployed on Artifactory cloud instance.
ifeq ($(REPO_TYPE_FOR_HELM_DEPLOY),)
	REPO_TYPE_FOR_HELM_DEPLOY=artifactory
else
	ifneq ($(REPO_TYPE_FOR_HELM_DEPLOY), nexus)
		ifneq ($(REPO_TYPE_FOR_RPM_DEPLOY), artifactory)
$(error Unsupported value for REPO_TYPE_FOR_HELM_DEPLOY variable: $(REPO_TYPE_FOR_HELM_DEPLOY))
		endif
	endif
endif

# REPO_TYPE_FOR_PYTHON_DEPLOY accepts only 2 values:
#  - artifactory:            the Infovista cloud instance of Artifactory
#  - nexus:                  some hosted repository on Infovista Nexus instances local to each LAB
# See DE-1564: since Sept 2023, by default all PYPI packages (even if produced by feature branches) will be deployed on Artifactory cloud instance.
ifeq ($(REPO_TYPE_FOR_PYTHON_DEPLOY),)
	REPO_TYPE_FOR_PYTHON_DEPLOY=artifactory
else
	ifneq ($(REPO_TYPE_FOR_PYTHON_DEPLOY), nexus)
		ifneq ($(REPO_TYPE_FOR_RPM_DEPLOY), artifactory)
$(error Unsupported value for REPO_TYPE_FOR_PYTHON_DEPLOY variable: $(REPO_TYPE_FOR_PYTHON_DEPLOY))
		endif
	endif
endif

# REPO_TYPE_FOR_MAVEN_DEPLOY accepts only 2 values:
#  - artifactory:            the Infovista cloud instance of Artifactory
#  - nexus:                  some hosted repository on Infovista Nexus instances local to each LAB
# See DE-1564: since Sept 2023, by default all MAVEN packages (even if produced by feature branches) will be deployed on Artifactory cloud instance.
ifeq ($(REPO_TYPE_FOR_MAVEN_DEPLOY),)
	REPO_TYPE_FOR_MAVEN_DEPLOY=artifactory
else
	ifneq ($(REPO_TYPE_FOR_MAVEN_DEPLOY), nexus)
		ifneq ($(REPO_TYPE_FOR_RPM_DEPLOY), artifactory)
$(error Unsupported value for REPO_TYPE_FOR_MAVEN_DEPLOY variable: $(REPO_TYPE_FOR_MAVEN_DEPLOY))
		endif
	endif
endif



ifeq ($(DEPLOY_FILELIST_LOG),)
# by default list all deployed files in the folder that Teamcity will pick up:
DEPLOY_FILELIST_LOG=$(REALTOP)/TeamcityArtifacts/deployed-artifacts.txt
endif

ifneq ($(LAB), massy)
	ifneq ($(LAB), modena)
		ifneq ($(LAB), billerica)
$(error Unsupported value for LAB variable: $(LAB))
	endif
		endif
endif


# ------------------------------------------------------------------------------------------
# RPM output directories
#
# NOTE: we used to have these directories to be placed inside 'TeamcityArtifacts'
#       so that we could easily archive the RPMs both in Teamcity and in the chosen
#       repo (nexus/artifactory); however it's confusing to have the RPMs
#       available in two places and moreover some projects may generate a lot of MBs
#       of RPMs so that it's best not to move these to Teamcity. 
# ------------------------------------------------------------------------------------------

RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR=$(REALTOP)/RpmArtifacts-$(RUNTIME)/deployments
RPM_OUTDIR_FOR_DEVEL_OUTDIR:=$(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR)



# ------------------------------------------------------------------------------------------
# repository constants abstracted from REPO instance
#
# General naming scheme is:   REMOTE_REPO_<artifactType>_FOR_<operation>
#  where
#     artifactType can be DOCKER,HELM,RPM,CONAN,PYTHON,JAVA
#     operation can be PUSH,PULL
#
# ------------------------------------------------------------------------------------------

## docker images

ifeq ($(REPO_TYPE_FOR_DOCKER_DEPLOY),ecr)
REMOTE_REPO_DOCKER_FOR_PUSH:=$(REMOTE_REPO_ECR_DOCKER_FOR_PUSH)
endif
ifeq ($(REPO_TYPE_FOR_DOCKER_DEPLOY),artifactory)
# NOTE: pushing dockers here is now deprecated
REMOTE_REPO_DOCKER_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_DOCKER_FOR_PUSH)
endif

# always pull dockers through the local cache/proxy for speed:
REMOTE_REPO_DOCKER_FOR_PULL:=$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)


## helm charts

ifeq ($(REPO_TYPE_FOR_HELM_DEPLOY),nexus)
REMOTE_REPO_HELM_FOR_PUSH:=$(REMOTE_REPO_NEXUS_HELM_FOR_PUSH)
endif
ifeq ($(REPO_TYPE_FOR_HELM_DEPLOY),artifactory)
REMOTE_REPO_HELM_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_HELM_FOR_PUSH)
endif

# always pull helm charts through the local cache/proxy for speed:
REMOTE_REPO_HELM_FOR_PULL:=$(REMOTE_REPO_NEXUS_HELM_ROOT_FOR_PULL)/artifactory-helm/


## pypi

ifeq ($(REPO_TYPE_FOR_PYTHON_DEPLOY),nexus)
REMOTE_REPO_PYTHON_FOR_PUSH:=$(REMOTE_REPO_NEXUS_PYTHON_FOR_PUSH)
endif
ifeq ($(REPO_TYPE_FOR_PYTHON_DEPLOY),artifactory)
REMOTE_REPO_PYTHON_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_PYTHON_FOR_PUSH)
endif


## Maven

ifeq ($(REPO_TYPE_FOR_MAVEN_DEPLOY),nexus)
REMOTE_REPO_JAVA_RELEASE_FOR_PUSH:=$(REMOTE_REPO_NEXUS_JAVA_RELEASE_FOR_PUSH)
REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH:=$(REMOTE_REPO_NEXUS_JAVA_SNAPSHOT_FOR_PUSH)
endif
ifeq ($(REPO_TYPE_FOR_MAVEN_DEPLOY),artifactory)
REMOTE_REPO_JAVA_RELEASE_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_JAVA_RELEASE_FOR_PUSH)
REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_JAVA_SNAPSHOT_FOR_PUSH)
endif
ifeq ($(JAVA_IS_SNAPSHOT_VERSION),1)
REMOTE_REPO_JAVA_FOR_PUSH:=$(REMOTE_REPO_JAVA_SNAPSHOT_FOR_PUSH)
else
REMOTE_REPO_JAVA_FOR_PUSH:=$(REMOTE_REPO_JAVA_RELEASE_FOR_PUSH)
endif


## RPMs

ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),nexus)
REMOTE_REPO_RPM_FOR_PUSH:=$(REMOTE_REPO_NEXUS_RPM_FOR_PUSH)
REMOTE_REPO_RHEL8_RPM_FOR_PUSH:=$(REMOTE_REPO_NEXUS_RHEL8_RPM_FOR_PUSH)
REMOTE_REPO_OL9_RPM_FOR_PUSH:=$(REMOTE_REPO_NEXUS_OL9_RPM_FOR_PUSH)
endif
ifeq ($(REPO_TYPE_FOR_RPM_DEPLOY),artifactory)
REMOTE_REPO_RPM_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_RPM_FOR_PUSH)
REMOTE_REPO_RHEL8_RPM_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_RHEL8_RPM_FOR_PUSH)
REMOTE_REPO_OL9_RPM_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_OL9_RPM_FOR_PUSH)
endif


## Conan packages

ifeq ($(REPO_TYPE_FOR_CONAN_DEPLOY),artifactory_billerica)
REMOTE_REPO_CONAN_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_BILLERICA_CONAN_FOR_PUSH)
REMOTE_REPO_CONAN_FOR_PULL:=$(REMOTE_REPO_ARTIFACTORY_BILLERICA_CONAN_FOR_PULL)
REMOTE_REPO_CONAN_NICKNAME:=$(REMOTE_REPO_ARTIFACTORY_BILLERICA_CONAN_NICKNAME)
endif
ifeq ($(REPO_TYPE_FOR_CONAN_DEPLOY),artifactory_cloud)
REMOTE_REPO_CONAN_FOR_PUSH:=$(REMOTE_REPO_ARTIFACTORY_CLOUD_CONAN_FOR_PUSH)
REMOTE_REPO_CONAN_FOR_PULL:=$(REMOTE_REPO_ARTIFACTORY_CLOUD_CONAN_FOR_PULL)
REMOTE_REPO_CONAN_NICKNAME:=$(REMOTE_REPO_ARTIFACTORY_CLOUD_CONAN_NICKNAME)
endif


