#
# This is the file which is included by any Makefiles that wants to
# build a RPM. Be sure to include this file at the end
# of the makefile.
#
# The calling Makefile must define:
#
#     PACKAGE_DOCKER_IMAGE=0/1
#         To enable/disable creation of a docker image. If 1, then DOCKER_IMAGE_NAME, DOCKER_IMAGE_CONTENT_FILES and
#         PRODBASE_OL9_DOCKER_IMAGE_NAME variables should be defined.
#         The latter is provided by the "docker-base" support makefile.
#     PACKAGE_HELM_CHART=0/1
#         To enable/disable creation of a Helm chart.
#         If 1 the caller makefile must define HELM_CHART_DIR and HELM_CHART_NAME variables and also the 'chart_config_files' target
#     PACKAGE_TEST_BUNDLE=0/1
#         To enable/disable creation of a test bundle.
#         If 1, the caller makefile must provide the "test-bundle-fill" GNU make target to fill the TEST_BUNDLE_DIR directory.
#     PACKAGE_RPMS=0/1
#         To enable/disable creation of RPMs.
#         If 1 the caller makefile must define RPM_SPEC_FILES
#
# DOCKERs:
#     DOCKER_IMAGE_NAME
#         Name of the docker image to generate
#     DOCKER_IMAGE_CONTENT_FILES
#         List of files that need to be ready for "docker build" to be able to generate the Docker image;
#         for each file a GNU make recipe must be present to generate it
#     DOCKER_BUILDER_EXTRA_FILES
#         A list of files that will be copied in the location where they are accessible by "docker build"
#         (and after docker image creation will be cleaned up); differently from DOCKER_IMAGE_CONTENT_FILES
#         GNU make does not expect to have a rule to build/create/update these files 
#     DOCKER_BUILDER_EXTRA_ARGS
#         Additional arguments to be provided to the "docker build" stage
#
# HELMs:
#     HELM_CHART_DIR
#         Fullpath of the folder containing the Chart.yaml* and values.yaml* files, e.g. /source/Capture/HelmChart/eva-capture/
#     HELM_CONFIG_DIR
#         Fullpath of the folder containing the configuration files used during Helm packaging, e.g. /source/Capture/HelmChart/eva-capture/configs/opt
#     HELM_VALUES_FILES
#         List of files that contain values and for which a JSON schema must be created
#     HELM_ADDITIONAL_CONFIG_FILES
#         List of files that will be used as configmap and for which a JSON schema using the filename as root must be created
#
# CONAN TESTBUNDLEs:
#     CONAN_CREATE_OPTIONS_EXTRA
#         Optional extra CLI options to provide to 'conan create' command to support testbundle creation
#
# RPMs:
#     RPM_SPEC_FILES
#         List of .spec files under $(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR) that the makefile builds as RPMs.
#         If this variable is empty, then no RPM will be produced.
#     RPM_GROUP_NAME
#         Name of the group of RPMs -- this is used when creating tarballs
#     RPM_CONTENT_FILES
#         List of files that need to be ready for "rpmbuild" to be able to generate the RPM;
#         for each file a GNU make recipe must be present to generate it
#

# ---------------------------------------------
# check calling Makefile parameters:
# ---------------------------------------------

ifndef RPM_VERSION
$(error RPM_VERSION must be defined. Did you include Version-defines.mk?)
endif
ifndef DOCKER_VERSION
$(error DOCKER_VERSION must be defined Did you include Version-defines.mk?)
endif
ifndef HELM_VERSION
$(error HELM_VERSION must be defined Did you include Version-defines.mk?)
endif
ifndef CONAN_VERSION
$(error CONAN_VERSION must be defined Did you include Version-defines.mk?)
endif
ifndef REMOTE_REPO_NEXUS_DOCKER_FOR_PULL
$(error REMOTE_REPO_NEXUS_DOCKER_FOR_PULL must be defined Did you include Deploy-defines.mk?)
endif
ifndef DOCKER_GROUP
$(error DOCKER_GROUP must be defined)
endif

ifndef PACKAGE_DOCKER_IMAGE
PACKAGE_DOCKER_IMAGE=0
endif

ifndef PACKAGE_TEST_BUNDLE
PACKAGE_TEST_BUNDLE=0
endif

ifndef PACKAGE_HELM_CHART
PACKAGE_HELM_CHART=0
endif

ifndef PACKAGE_RPMS
PACKAGE_RPMS=0
endif

ifndef DOCKER_BUILDER_EXTRA_FILES
# default value
DOCKER_BUILDER_EXTRA_FILES=
endif

ifndef DOCKER_BUILDER_EXTRA_ARGS
# default value
DOCKER_BUILDER_EXTRA_ARGS=
endif

ifndef CLEAN_TEMPORARY_FILES
# default value
CLEAN_TEMPORARY_FILES=1
endif

ifndef SUPPORTS_HELP
# default value
SUPPORTS_HELP=1
endif

ifndef HELM_CONFIG_DIR
# default value
HELM_CONFIG_DIR=$(HELM_CHART_DIR)/configs/opt
endif

ifndef HELM_VALUES_FILES
# empty list of values is the default
HELM_VALUES_FILES=
endif
ifndef HELM_ADDITIONAL_CONFIG_FILES
# empty list of values is the default
HELM_ADDITIONAL_CONFIG_FILES=
endif

# old variable names (version 4.x)
ifneq ($(UPSTREAM_TARBALLS),)
$(error Please replace UPSTREAM_TARBALLS with DOCKER_IMAGE_CONTENT_FILES)
endif
ifneq ($(CHART_DIR),)
$(error Please replace CHART_DIR with HELM_CHART_DIR)
endif
ifneq ($(CONFIG_DIR),)
$(error Please replace CONFIG_DIR with HELM_CONFIG_DIR)
endif
ifneq ($(CHART_NAME),)
$(error Please replace CHART_NAME with HELM_CHART_NAME)
endif


# ------------------------------------------------------------------------------------------
# inclusion checks
# ------------------------------------------------------------------------------------------

INCLUDED_PACKAGING_TARGETS:=1

# blacklist other makefiles that for some reason are incompatible 
# (reason could be as simple as 'the combination has never been tested' or 'does not make much sense')
ifeq ($(INCLUDED_RECURSIVE_TARGETS),1) 
$(error Packaging-targets.mk is incompatible with Recursive-targets.mk. Don't include both.)
endif



# ---------------------------------------------
# Constants
# ---------------------------------------------

REALTOP=$(shell readlink -f $(TOPDIR))

RPM_BUILD_ROOT=$(REALTOP)/RpmArtifacts-$(RUNTIME)/tmp-rpm-build

RPM_FULL_LIST := \
	$(foreach spec,$(RPM_SPEC_FILES),$(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR)/x86_64-$(RUNTIME)/$(spec)-$(RPM_VERSION)-$(RPM_RELEASE).$(ARCH).rpm)

# The following list of RPMs is without version and uses wildcards to match any version of the RPMs:
RPM_FULL_LIST_NOVERSION := \
	$(foreach spec,$(RPM_SPEC_FILES),$(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR)/x86_64-$(RUNTIME)/$(spec)-$(MAJORVERSION).*.rpm)

DOCKER_TMPINSTALL_DIR:=/tmp/docker-install-dir
ifeq ($(DOCKER_TARBALL_DIR),)
DOCKER_TARBALL_DIR:=$(TOPDIR)/DockerArtifacts-$(RUNTIME)/$(DOCKER_IMAGE_NAME)
endif

GIT_REPO_URL:=$(shell git config --get remote.origin.url)

# all the constants related to the BUILDER DOCKER typically are transported by the support Makefile of "eva-ecc-cpp-docker-base"
BUILDER_DOCKER_IMAGE_FOR_RUN=$(BUILDER_DOCKER_OL9_REMOTE_NEXUS_DOCKER_REGISTRY)/$(BUILDER_DOCKER_OL9_GROUP)/$(BUILDER_DOCKER_OL9_IMAGE_NAME)
BUILDER_DOCKER_CENTOS7_IMAGE_FOR_RUN=$(BUILDER_DOCKER_CENTOS7_REMOTE_NEXUS_DOCKER_REGISTRY)/$(BUILDER_DOCKER_CENTOS7_GROUP)/$(BUILDER_DOCKER_CENTOS7_IMAGE_NAME)

ifneq ($(DOCKER_VERSION), 1.0.0-local)
FULL_DOCKER_IMAGE=$(REMOTE_REPO_REGISTRY_NAME)/$(DOCKER_GROUP)/$(DOCKER_IMAGE_NAME):$(DOCKER_VERSION)$(DEBUG_TAG)
else
FULL_DOCKER_IMAGE=$(DOCKER_IMAGE_NAME):$(DOCKER_VERSION)$(DEBUG_TAG)
endif

TEST_BUNDLE_DIR = $(shell readlink -f .)




# ---------------------------------------------
# Target "all" and "packaging"
# ---------------------------------------------

all:: packaging

ifeq ($(INSIDE_BUILDER_IMAGE),)

packaging::
	@echo "The 'packaging' target can be invoked only from inside the builder docker."
	@echo "Please use the 'docker_packaging' or 'docker_fullpackaging' target instead."
	@exit 99

else

# IMPORTANT: please note that very often TestBundles contain Helm charts, so it's important
#            to package HelmChart folder BEFORE the TestBundle folder

packaging::
	$(call update_topdir_version_file,fullversion.dockerimage)
	$(call update_topdir_version_file,fullversion.helmvalues)
ifeq ($(PACKAGE_DOCKER_IMAGE),1)
	$(call open_section,DOCKER IMAGE $(DOCKER_IMAGE_NAME) with CFG=$(CFG))
	@$(MAKE) --no-print-directory .docker_image
	$(call print_green, READY DOCKER IMAGE [$(DOCKER_IMAGE_REF_MAIN)])
	$(call close_section,DOCKER IMAGE $(DOCKER_IMAGE_NAME) with CFG=$(CFG))
endif
ifeq ($(PACKAGE_HELM_CHART),1)
	$(call open_section,HELM CHART $(HELM_CHART_NAME))
	@$(MAKE) --no-print-directory helm_chart
	$(call print_green, READY HELM CHART [$(CHART_TARBALL)])
	$(call close_section,HELM CHART $(HELM_CHART_NAME))
endif
ifeq ($(PACKAGE_TEST_BUNDLE),1)
	$(call open_section,TESTBUNDLE $(DOCKER_IMAGE_NAME) with CFG=$(CFG))
	@$(MAKE) --no-print-directory test_bundle MAKEFLAGS=
	$(call print_green, READY CONAN TESTBUNDLE [$(CONAN_PACKAGE_REF)])
	$(call close_section,TESTBUNDLE $(DOCKER_IMAGE_NAME) with CFG=$(CFG))
endif
ifeq ($(PACKAGE_RPMS),1)
	$(call open_section,RPMs $(RPM_SPEC_FILES))
	@$(MAKE) --no-print-directory rpm
	$(call print_green, READY RPMs [$(RPM_SPEC_FILES)])
	$(call close_section,RPMs $(RPM_SPEC_FILES))
endif

endif # INSIDE_BUILDER_IMAGE

# ------------------------------------------------------------------------------------------
# Docker image helper targets
#
#	In order to produce a working docker image a temporary installation directory is created and shipped
#	as a set of tarballs during image building.
# 	The tarballs should contain:
#		- a valid installation of all upstream libraries (events-networking, third-party-cpp-libs, etc)
#		- a valid installation of the common libraries
#		- a valid installation of the binary
#
# 	Note that it's assumed that DOCKER_PROD_IMAGE_NAME and DOCKER_BUILDER_IMAGE_NAME have the same version
# ------------------------------------------------------------------------------------------

ifeq ($(DOCKER_PROD_IMAGE_VERSION),)
# if not provided, DOCKER_PROD_IMAGE_VERSION matches the builder image version!
DOCKER_PROD_IMAGE_VERSION:=$(DOCKER_BUILDER_IMAGE_VERSION)
endif

# Base images computation (depending on the runtime)
ifeq ($(RUNTIME),OL9)
PRODBASE_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY_FOR_CURRENT_RUNTIME=$(PRODBASE_DOCKER_OL9_REMOTE_NEXUS_DOCKER_REGISTRY)
PRODBASE_DOCKER_IMAGE_NAME_FOR_CURRENT_RUNTIME=$(PRODBASE_DOCKER_OL9_IMAGE_NAME)
endif

ifeq ($(BASE_IMAGE),)
ifeq ($(CFG),release)
# NOTE: for this CFG build the docker image for RELEASE, based on the lightweight PRODUCTION BASE
BASE_IMAGE:=$(PRODBASE_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY_FOR_CURRENT_RUNTIME)/$(PRODBASE_DOCKER_OL9_GROUP)/$(PRODBASE_DOCKER_IMAGE_NAME_FOR_CURRENT_RUNTIME)
endif
ifeq ($(CFG),debug)
# NOTE: for this CFG build the docker image using as BASE image the BUILDER DOCKER -- this simplifies debugging since the BUILDER DOCKER comes
#       with a GDB having all features (e.g. Python pretty printers) enabled, not just a _minimal_ GDB
BASE_IMAGE:=$(PRODBASE_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY_FOR_CURRENT_RUNTIME)/$(PRODBASE_DOCKER_OL9_GROUP)/$(PRODBASE_DOCKER_IMAGE_NAME_FOR_CURRENT_RUNTIME)
# Pass the build-arg which will be used to set ASAN_OPTIONS env variable in Dockerfile.prod for debug builds
DOCKER_BUILDER_EXTRA_ARGS += --build-arg IS_DEBUG_BUILD=1
endif
ifeq ($(CFG),debug-gcov)
# NOTE: for this CFG build the docker image using as BASE image the BUILDER DOCKER -- this simplifies debugging since the BUILDER DOCKER comes
#       with a GDB having all features (e.g. Python pretty printers) enabled, not just a _minimal_ GDB
BASE_IMAGE:=$(PRODBASE_DOCKER_REMOTE_NEXUS_DOCKER_REGISTRY_FOR_CURRENT_RUNTIME)/$(PRODBASE_DOCKER_OL9_GROUP)/$(PRODBASE_DOCKER_IMAGE_NAME_FOR_CURRENT_RUNTIME)
# Pass the build-arg which will be used to set ASAN_OPTIONS env variable in Dockerfile.prod for debug builds
DOCKER_BUILDER_EXTRA_ARGS += --build-arg IS_DEBUG_BUILD=1
endif
ifeq ($(CFG),debug-tsan)
# NOTE: for this CFG build the docker image using as BASE image the BUILDER DOCKER -- this simplifies debugging since the BUILDER DOCKER comes
#       with a GDB having all features (e.g. Python pretty printers) enabled, not just a _minimal_ GDB
BASE_IMAGE:=$(BUILDER_DOCKER_IMAGE_FOR_RUN)
# Pass the build-arg which will be used to set ASAN_OPTIONS env variable in Dockerfile.prod for debug builds
DOCKER_BUILDER_EXTRA_ARGS += --build-arg IS_DEBUG_BUILD=1
endif
endif

DOCKER_IMAGE_REF_MAIN := $(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)/$(DOCKER_GROUP)/$(DOCKER_IMAGE_NAME):$(DOCKER_VERSION)$(DEBUG_TAG)
DOCKER_IMAGE_REF_LATEST := $(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)/$(DOCKER_GROUP)/$(DOCKER_IMAGE_NAME):$(DOCKER_VERSION_LATEST)$(DEBUG_TAG)

DOCKER_BUILDER_OPTS:= \
	--tag $(DOCKER_IMAGE_REF_MAIN) \
	--tag $(DOCKER_IMAGE_REF_LATEST) \
	--build-arg BUILDKIT_INLINE_CACHE=1 \
	--build-arg DEBUG_TAG=$(DEBUG_TAG) \
	--build-arg GIT_REPO_URL=$(GIT_REPO_URL) \
	--build-arg DOCKER_VERSION=$(DOCKER_VERSION) \
	--build-arg DOCKER_PROD_FULLNAME_IMAGE_NAME=$(BASE_IMAGE) \
	--build-arg BUILDER_DOCKER_FULLNAME_IMAGE_NAME=$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
	--cache-from $(DOCKER_IMAGE_REF_LATEST) \

$(TOPDIR)/.conan_dependencies/fullversion.dockerimage:
	$(call update_topdir_version_file,fullversion.dockerimage)

# IMPORTANT #1:
# Why do we use REMOTE_REPO_NEXUS_DOCKER_FOR_PULL instead of REMOTE_REPO_NEXUS_DOCKER_FOR_PUSH in the "docker build" step below?
#   The latter would be more correct since we create an image that needs to be pushed later on the remote registry; however during automated
#   tests we will need to use a mix of docker images from remote registries and from the local registry
#   (depending on which images of other repositories we use).
#   To keep ATF simple, we do not provide it any knowledge of which docker images have been generated locally and which are not.
#   We rely on the fact that the docker-compose/docker daemon will just receive a list of docker images to use, through the YAMLs inside the
#   test bundles and will automatically detect what's already available locally (because just-built and not yet pushed) and what needs to
#   be fetched from the remote instead.
#   For this to work, we need that the YAMLs inside the test bundles built out of this repo reference exactly the same registry/docker-group
#   used during this "docker build" step. Since the test bundles must contain the REMOTE_REPO_NEXUS_DOCKER_FOR_PULL registry, we must use here
#   the REMOTE_REPO_NEXUS_DOCKER_FOR_PULL registry as well. Of course in the "deploy_docker" step later we will replace the registry reference
#   with the good one for PUSHING
#
# IMPORTANT #2:
# The "docker pull" and "--cache-from" steps below are used to try to recycle docker image layers as much as possible: if a previous
# build was tagged with $(DOCKER_VERSION_LATEST)$(DEBUG_TAG), it will be used in future builds as cache of layers with no changes.
# This allow us to recycle up to 90% of the layers of a typical docker image.
#
# IMPORTANT #3:
# The 'buildkit'-related variables are used to share docker layers as much as possible, see https://testdriven.io/blog/faster-ci-builds-with-docker-cache/
#
# NOTE ABOUT BUILD ARGS:
# The "docker build" stage is provided with many --build-args, which can be optionally used by the downstream project Dockerfile.
# The BUILDER_DOCKER_FULLNAME_IMAGE_NAME is one of these build args and it's provided to allow the Dockerfile to contain e.g. "COPY --from=builder" if needed.
#
# NOTE ABOUT PREREQUISITES:
#   - DOCKER_IMAGE_CONTENT_FILES: these are obviously prerequisites since they are expected to be used with COPY or ADD inside the Dockerfile
#   - $(TOPDIR)/conandata.yml: any change to the 'docker-base' or 'pipeline-framework' upstreams must trigger docker recreation
#   - $(TOPDIR)/.fullversion.dockerimage: any change to the DOCKER_VERSION variable must trigger docker recreation
.docker_image: \
	$(DOCKER_IMAGE_CONTENT_FILES) \
	$(TOPDIR)/conandata.yml \
	$(TOPDIR)/.conan_dependencies/fullversion.dockerimage

ifeq ($(PRODBASE_DOCKER_OL9_IMAGE_NAME),)
	@echo "Please provide a valid PRODBASE_DOCKER_OL9_IMAGE_NAME environment variable or GNU make option."
	@exit 200
endif
ifneq ($(DOCKER_IMAGE_NAME),)
	@mkdir -p $(DOCKER_TMPINSTALL_DIR)
	@cp $(DOCKER_IMAGE_CONTENT_FILES) $(DOCKER_BUILDER_EXTRA_FILES) ./        # copy over the local folder where packaging is happening (docker build limitation)
ifeq ($(V),1)
	@echo "DOCKER BUILD: Using as base image $(BASE_IMAGE)"
	DOCKER_BUILDKIT=1 \
		docker build \
			$(DOCKER_BUILDER_OPTS) \
			$(DOCKER_BUILDER_EXTRA_ARGS) \
			. -f Dockerfile.prod && \
			touch .docker_image
else
	$(call print_green, DOCKER BUILD [$(DOCKER_IMAGE_REF_MAIN)])
	@DOCKER_BUILDKIT=1 \
		docker build --quiet \
			$(DOCKER_BUILDER_OPTS) \
			$(DOCKER_BUILDER_EXTRA_ARGS) \
			. -f Dockerfile.prod && \
			touch .docker_image
endif
	@# remove all temporary files
ifeq ($(CLEAN_TEMPORARY_FILES),1)
	@rm -r $(RM_OPTS) $(DOCKER_TMPINSTALL_DIR) docker-*.tar.gz $(notdir $(DOCKER_BUILDER_EXTRA_FILES))
else
	@echo "Skipping cleaning of docker temporary files (CLEAN_TEMPORARY_FILES=0)"
endif
else
	@echo "No docker image to create"
endif


# the clean target is meant to remove anything that can be safely regenerated:
docker_image_clean:
ifneq ($(DOCKER_IMAGE_CONTENT_FILES),)
	@echo "CLEAN DOCKER IMAGE CONTENT FILES"
	@rm $(RM_OPTS) $(DOCKER_IMAGE_CONTENT_FILES) $(TOPDIR)/.conan_dependencies/fullversion.dockerimage
endif

# ------------------------------------------------------------------------------------------
# Helm chart helper targets
# ------------------------------------------------------------------------------------------

HELM_CHART_PARENT_DIR:=$(realpath $(HELM_CHART_DIR)/../)
CHART_TARBALL:=$(HELM_CHART_PARENT_DIR)/$(HELM_CHART_NAME)-$(HELM_VERSION).tgz

$(TOPDIR)/.conan_dependencies/fullversion.helmvalues:
	$(call update_topdir_version_file,fullversion.helmvalues)

# this target runs Jinja2 on template files to produce final Chart.yaml and values.yaml files:
$(HELM_CHART_DIR)/values.yaml $(HELM_CHART_DIR)/Chart.yaml: \
		$(TOPDIR)/conandata.yml \
		$(HELM_CHART_DIR)/values.yaml.j2 \
		$(HELM_CHART_DIR)/Chart.yaml.j2 \
		$(TOPDIR)/.conan_dependencies/fullversion.helmvalues

	@# create the GNU make variables HELM_VALUES_SUBSTITUTIONS / HELM_CHART_SUBSTITUTIONS
	@# note that both the docker registry and the helm chart repository are expanded at
	@# runtime, based on the LAB env variable value (see Internal-repoconstants.mk)
	$(eval HELM_VALUES_SUBSTITUTIONS+= \
		CHART_NAME=$(HELM_CHART_NAME) \
		REMOTE_REPO_NEXUS_DOCKER_FOR_PULL="$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)/" \
		DOCKER_GROUP="$(DOCKER_GROUP)/" \
		DOCKER_VERSION=$(DOCKER_VERSION) \
		SHELLMGR_DOCKER_VERSION=$(SHELLMGR_DOCKER_VERSION) \
		EVA_ECC_EXPORTER_TAG=$(EVA_ECC_EXPORTER_TAG) \
	)
	$(eval HELM_CHART_SUBSTITUTIONS+= \
		CHART_NAME=$(HELM_CHART_NAME) \
		DOCKER_VERSION=$(DOCKER_VERSION) \
		HELM_VERSION=$(HELM_VERSION) \
		HELM_BASE_HELM_CHART_NAME=$(HELM_BASE_HELM_CHART_NAME) \
		HELM_BASE_HELM_REPO=$(REMOTE_REPO_NEXUS_HELM_ROOT_FOR_PULL)/artifactory-helm/ \
		HELM_BASE_HELM_VERSION=$(HELM_BASE_HELM_VERSION) \
	)
	@# run Jinja2 to obtain the actual chart YAML file inside the component's Helm chart directory
	@echo "J2 $(HELM_CHART_DIR)/*.yaml.j2"
ifeq ($(V),1)
	$(HELM_VALUES_SUBSTITUTIONS) j2 $(HELM_CHART_DIR)/values.yaml.j2 -o $(HELM_CHART_DIR)/values.yaml
	$(HELM_CHART_SUBSTITUTIONS) j2 $(HELM_CHART_DIR)/Chart.yaml.j2 -o $(HELM_CHART_DIR)/Chart.yaml
else
	@$(HELM_VALUES_SUBSTITUTIONS) j2 $(HELM_CHART_DIR)/values.yaml.j2 -o $(HELM_CHART_DIR)/values.yaml
	@$(HELM_CHART_SUBSTITUTIONS) j2 $(HELM_CHART_DIR)/Chart.yaml.j2 -o $(HELM_CHART_DIR)/Chart.yaml
endif

# this target creates the HELM_CONFIG_DIR foler and fills it with the component config files:
# NOTE1: the target 'chart_config_files' must be defined in caller Makefile and it should install all
#        .yaml config files and their .schema.json associated files into the HELM_CONFIG_DIR
# NOTE2: since it's not possible to detect when source YAML config files are changed, we simply run
#        this target all the times (phony target) to make sure the HELM_CONFIG_DIR is up to date
helm_chart_install_configs:
	@echo "INSTALL CONFIG FILES + JSON SCHEMAS -> $(HELM_CONFIG_DIR)"
	@mkdir -p $(HELM_CONFIG_DIR)
	@$(MAKE) --no-print-directory chart_config_files

# this target generates the JSON SCHEMA for the entire Helm chart;
# its prerequisites are all the individual .schema.json files
$(HELM_CHART_DIR)/values.schema.json: \
		$(TOPDIR)/conandata.yml \
		$(HELM_CONFIG_DIR) \
		$(HELM_CONFIG_DIR)/etc/*.schema.json \
		$(HELM_CHART_DIR)/configs/*.schema.json \
		$(HELM_VALUES_FILES) \
		$(HELM_ADDITIONAL_CONFIG_FILES) \
		$(HELM_CHART_DIR)/Chart.yaml

	@# to make the JSON schema validation more powerful, we try to produce the .schema.json files also from 
	@# YAML files that do NOT come from any master XML file:	
	@for values_file in $(HELM_VALUES_FILES); do \
		echo "CFGFILE $${values_file} -> $${values_file}.partial.schema.json" ; \
		python3 $(YAML_TO_JSON_SCHEMA_SCRIPT) -i $${values_file} -o $${values_file}.partial.schema.json; \
		done

	@for config_file in $(HELM_ADDITIONAL_CONFIG_FILES); do \
		echo "CFGFILE $${config_file} -> $${config_file}.partial.schema.json" ; \
		python3 $(YAML_TO_JSON_SCHEMA_SCRIPT) --add_root_filename -i $${config_file} -o $${config_file}.partial.schema.json; \
		done

	@# Reduce all the json schema to a single one. -s (slurp) option treats all the schemas as if they were a single big array
	@echo "MERGING ALL JSON SCHEMAS TOGETHER -> $(HELM_CHART_DIR)/values.schema.json"
	@jq -s 'reduce .[] as $$item ({}; . * $$item)' \
		$(HELM_CONFIG_DIR)/etc/*.schema.json \
		$(HELM_CHART_DIR)/configs/*.schema.json \
		$(HELM_CHART_DIR)/*.partial.schema.json \
		$(CONFIG_FILE_GENERATOR_DIR)/additional_root_properties.schema.json \
			> $(HELM_CHART_DIR)/values.schema.json
	@DEP_LIST=$$(yq e '.dependencies[].name' $(HELM_CHART_DIR)/Chart.yaml); \
		for DEP in $$DEP_LIST; do \
			jq '.properties."'$$DEP'" += {"type": "object"}' $(HELM_CHART_DIR)/values.schema.json > $(HELM_CHART_DIR)/values_tmp.schema.json; \
			mv $(HELM_CHART_DIR)/values_tmp.schema.json $(HELM_CHART_DIR)/values.schema.json; \
		done

# this target downloads the dependencies (subcharts), e.g. helm-base, which are encoded inside Chart.yaml:
# NOTE: to get 'helm dependency build' to work without Helm repositories being registered in Helm, we need to remove Chart.lock:
$(HELM_CHART_DIR)/.helm_dependencies: \
		$(TOPDIR)/conandata.yml \
		$(HELM_CHART_DIR)/Chart.yaml

	@echo "HELM DEPENDENCIES DOWNLOAD"
	@rm -f $(HELM_CHART_DIR)/Chart.lock
	@helm dependency build $(HELM_CHART_DIR)
	@touch $(HELM_CHART_DIR)/.helm_dependencies

helm_chart:
ifeq ($(HELM_CHART_DIR),)
	@echo "Please define HELM_CHART_DIR in the caller makefile"
	@exit 34
endif
ifeq ($(HELM_CHART_NAME),)
	@echo "Please define HELM_CHART_NAME in the caller makefile"
	@exit 34
endif
	@$(MAKE) --no-print-directory $(HELM_CHART_DIR)/values.yaml $(HELM_CHART_DIR)/Chart.yaml
	@$(MAKE) --no-print-directory helm_chart_install_configs
	@$(MAKE) --no-print-directory $(HELM_CHART_DIR)/values.schema.json
	@$(MAKE) --no-print-directory $(HELM_CHART_DIR)/.helm_dependencies
ifeq ($(V),1)
	helm-docs --chart-search-root "$(HELM_CHART_DIR)"
	helm lint $(HELM_CHART_DIR)
	helm package $(HELM_CHART_DIR)
	if [ -d $(HELM_CHART_DIR)/../tests ]; then \
		cd $(HELM_CHART_DIR)/.. ; \
		pytest -vvv --path $(HELM_CHART_NAME) tests/ --junitxml=$(TOPDIR)/TeamcityArtifacts/unit-tests-junit/helm_chart_unit_tests.xml ; \
	fi
else
	$(call print_green, HELM PACKAGE [$(CHART_TARBALL)])
	@helm-docs --chart-search-root "$(HELM_CHART_DIR)" >/dev/null 2>&1
	@helm lint $(HELM_CHART_DIR)  >/dev/null
	@helm package $(HELM_CHART_DIR)  >/dev/null
	@if [ -d $(HELM_CHART_DIR)/../tests ]; then \
		echo "UNIT TESTING HELM PACKAGE [$(CHART_TARBALL)]" ; \
		cd $(HELM_CHART_DIR)/.. ; \
		pytest --quiet --path $(HELM_CHART_NAME) tests/ ; \
	fi
	$(call print_green, HELM UNIT TESTS PASSED [$(CHART_TARBALL)])
endif

# the clean target is meant to remove anything that can be safely regenerated:
helm_chart_clean:
ifneq ($(HELM_CONFIG_DIR),)
	@echo "CLEAN HELM CONFIGS DIR $(HELM_CONFIG_DIR)"
	@rm -r $(RM_OPTS) $(HELM_CONFIG_DIR)
endif
ifneq ($(HELM_CHART_DIR),)
	@echo "CLEAN HELM DIR $(HELM_CHART_DIR)"
	@rm $(RM_OPTS) \
		$(HELM_CHART_DIR)/values.yaml $(HELM_CHART_DIR)/Chart.yaml \
		$(HELM_CHART_DIR)/README.md \
		$(HELM_CHART_DIR)-*.tgz $(HELM_CHART_DIR)/Chart.lock \
		$(HELM_CHART_DIR)/values.schema.json \
		$(HELM_CHART_DIR)/.helm_dependencies \
		$(TOPDIR)/.conan_dependencies/fullversion.helmvalues
endif

# ------------------------------------------------------------------------------------------
# Testbundle generation targets
# ------------------------------------------------------------------------------------------

# NOTE: typically testbundles use helpers defined in the 'conan-base' repository to perform substitutions/templating
#       of files like: helm charts, docker compose files, etc
#       To support templating at Conan packaging time, we provide a lot of env variables to Conan:
CONAN_CREATE_OPTIONS:= \
	-e DOCKER_REGISTRY_FOR_TEST_BUNDLES="$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)/$(DOCKER_GROUP)" \
	-e REMOTE_REPO_NEXUS_DOCKER_FOR_PULL="$(REMOTE_REPO_NEXUS_DOCKER_FOR_PULL)" \
	-e DOCKER_GROUP="$(DOCKER_GROUP)" \
	-e DOCKER_VERSION_FOR_TEST_BUNDLES="$(DOCKER_VERSION)$(DEBUG_TAG)" \
	-e REMOTE_REPO_HELM_FOR_PULL="$(REMOTE_REPO_HELM_FOR_PULL)" \
	-e HELM_REPO_FOR_TEST_BUNDLES="$(REMOTE_REPO_HELM_FOR_PULL)" \
	-e HELM_VERSION_FOR_TEST_BUNDLES="$(HELM_VERSION)" \
	-o docker_image_variant=$(DEBUG_TAG) \
	$(CONAN_CREATE_OPTIONS_EXTRA)

ifeq ($(PACKAGE_TEST_BUNDLE),1)
# conan inspect is kind of slow, so set these variables only when PACKAGE_TEST_BUNDLE=1
CONAN_PACKAGE_NAME:=$(shell TOPDIR="$(REALTOP)" conan inspect . --json /tmp/conaninfo.json >/dev/null && jq .name /tmp/conaninfo.json)
CONAN_PACKAGE_REF:=$(CONAN_PACKAGE_NAME)/$(CONAN_VERSION)@empirix/$(CONAN_CHANNEL)
endif

# FIXME: this is currently a PHONY target always rebuilt... we should find a clever way to rebuild only when needed...
#        e.g. we can assume that testbundles only depend from a set of variables...
test_bundle:
	@mkdir -p settings && rm -rf settings/*
	$(MAKE) --no-print-directory test-bundle-fill  # provided by the caller makefile
ifeq ($(V),1)
	@echo "Now packaging the testbundle $(CONAN_PACKAGE_NAME) for docker_image_variant=$(DEBUG_TAG)"
	TOPDIR="$(REALTOP)" \
		conan create \
			$(CONAN_CREATE_OPTIONS) \
			. empirix/$(CONAN_CHANNEL) --options runtime=$(RUNTIME)
else
	$(call print_green, CONAN CREATE [$(CONAN_PACKAGE_REF)])
	@TOPDIR="$(REALTOP)" \
		conan create \
			$(CONAN_CREATE_OPTIONS) \
			. empirix/$(CONAN_CHANNEL) --options runtime=$(RUNTIME) >/dev/null 2>&1
endif
ifeq ($(CLEAN_TEMPORARY_FILES),1)
	@rm -rf settings
else
	@echo "Skipping cleaning of test-bundle temporary files (CLEAN_TEMPORARY_FILES=0)"
endif

test_bundle_clean:
	@# nothing to cleanup ?



# ---------------------------------------------
# RPM generation targets
# NOTE: these support parallel mode -jN
# ---------------------------------------------

RPM_DEFINES := \
	--define "abs_lib_hmcommon         /home/hammer/hmcommon/lib"                             \
	--define "abs_debuginfo_dir        /usr/lib/debug"                                        \
	--define "abs_empirix         	   /opt/empirix"                                          \
	--define "abs_eva_ecc         	   %{abs_empirix}/eva-ecc"                                \
	--define "topdir                   $(REALTOP)"                                            \
	--define "_rpmdir                  $(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR)"                  \
	--define "spec_utils               %{topdir}/Common/Packaging"                            \
	--define "build_root               %{buildroot}"                                          \
	--define "debuginfo_hammer_root    %{buildroot}/%{abs_debuginfo_dir}/%{abs_hammer_root}"  \
	--define "eva_ecc_root             %{buildroot}/%{abs_eva_ecc}"                           \
	--define "debuginfo_eva_ecc_root   %{buildroot}/%{abs_debuginfo_dir}/%{eva_ecc_root}"     \
	--define "bash_required            which, sed, coreutils, net-tools, patch, gawk, crudini, lsof, psmisc" \
	--define "_buildshell              /bin/bash"                                             \
	--define "__jar_repack             %{nil}"

# empty targets are special to GNU make: they will provoke any target that depends on them to be always rebuilt
EMPTY_TARGET:

# About the QA_RPATHS: this is a new feature of rpmbuild starting from Fedora35; if not defined, rpmbuild will likely fail;
# we define QA_RPATHS to allow to pass through:
#   0x0002 invalid RPATHs; rpmbuild considers e.g. /opt/empirix/third-party-cpp-libs/lib/docker-release/ to be invalid
#   0x0010 empty RPATHs
$(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR)/x86_64-$(RUNTIME)/%-$(RPM_VERSION)-$(RPM_RELEASE).$(ARCH).rpm: %.spec $(RPM_CONTENT_FILES) EMPTY_TARGET
	$(call open_section,Building RPM from spec file $<)
	@mkdir -p $(RPM_OUTDIR_FOR_DEPLOYMENTS_OUTDIR)
	@# start from a clean folder when executing the %install section of the spec file
	@echo "CLEANING RPM BUILD ROOT $(RPM_BUILD_ROOT)-$(subst .spec,,$<)"
	@rm -r $(RM_OPTS) $(RPM_BUILD_ROOT)-$(subst .spec,,$<)/* && mkdir -p $(RPM_BUILD_ROOT)-$(subst .spec,,$<)
	@echo "RPMBUILD $<"
ifeq ($(V),1)
	export QA_RPATHS=$$(( 0x0002|0x0010 )) ; \
		rpmbuild -bb $< --buildroot $(RPM_BUILD_ROOT)-$(subst .spec,,$<) \
			$(RPM_DEFINES) \
			$(RPM_ADDITIONAL_DEFINES) 2>&1
else
	@export QA_RPATHS=$$(( 0x0002|0x0010 )) ; \
		rpmbuild -bb $< --buildroot $(RPM_BUILD_ROOT)-$(subst .spec,,$<) \
			$(RPM_DEFINES) \
			$(RPM_ADDITIONAL_DEFINES) >/dev/null 2>&1
endif
	$(call close_section,Building RPM from spec file $<)

post_rpm_generation::

# target to re-package only RPMs:
rpm:: $(RPM_FULL_LIST) post_rpm_generation

# the clean target is meant to remove anything that can be safely regenerated:
rpm_clean:
ifneq ($(RPM_FULL_LIST_NOVERSION),)
	@echo "CLEAN RPMS $(RPM_FULL_LIST_NOVERSION)"
	@rm -rf $(RPM_BUILD_ROOT)-*
	@rm $(RM_OPTS) $(RPM_FULL_LIST_NOVERSION)
endif


# ---------------------------------------------
# Clean targets
# ---------------------------------------------

clean::
	@$(MAKE) --no-print-directory docker_image_clean
	@$(MAKE) --no-print-directory helm_chart_clean
	@$(MAKE) --no-print-directory test_bundle_clean
	@$(MAKE) --no-print-directory rpm_clean

distclean:: clean

packaging_clean:: clean

.PHONY: all clean distclean


# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Packaging targets (to run INSIDE the builder docker):"
	@echo "    packaging:                            run all possible packagings for this project"
	@echo "    docker_image:                         run just the Docker image creation"
	@echo "    test_bundle:                          run just the Conan TestBundle creation"
	@echo "    helm_chart:                           run just the Helm Chart creation"
	@echo "    rpm:                                  run just the RPM packaging"
endif

# ---------------------------------------------
# include shared targets:
# ---------------------------------------------

# provides the 'generate_json_schema_from_yaml' target:
include ${EMPIRIX_PIPELINE_FRAMEWORK_MAKEFILE_SUPPORT_DIR}/ConfigFiles-targets.mk
