#
# This is the file which is included by the ROOT Makefiles of downstream projects.
# It contains makefile targets that make sense only in the ROOT folder of a project,
# not inside any subdirectory.
#
# The calling Makefile must define:
#     EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR
#         Absolute path of the folder containing the "Scripts" subdir of the pipeline-framework repository source code
#

ifndef SUPPORTS_HELP
# default value
SUPPORTS_HELP=1
endif

# ------------------------------------------------------------------------------------------
# check_deps_upstream target
# ------------------------------------------------------------------------------------------
#
# Once pipeline framework has been retrieved, this target becomes available to everyone.
# Therefore, it is necessary to install via conan pipeline-framework, putting it into
# the pipeline_dependency category into conandata.yml
# This is usually achieved by the check_deps_pipeline target.
check_deps_upstream: 
ifeq (,$(CONAN_DEPENDENCIES_DIR))
	@echo "ERROR: must define CONAN_DEPENDENCIES_DIR (usually in your local Makefiles/Repo-defines.mk). Aborting."
	@exit 123
endif
	@mkdir -p $(CONAN_DEPENDENCIES_DIR) && rm -rf $(CONAN_DEPENDENCIES_DIR)/*
	@if ! conan remote list | grep $(REMOTE_CONAN_NAME_FOR_PULL) 2>/dev/null 1>&2; then \
		echo "ERROR: there is no Conan remote repository named [$(REMOTE_CONAN_NAME_FOR_PULL)]. Please run setup-devel-[YOUR-OS].sh again." ; \
		exit 123 ; \
	fi
	@echo "======================================================================================================================"
	@$(call print_green,Checking and downloading UPSTREAM dependencies for CFG=$(CFG) RUNTIME=$(RUNTIME))
	@echo "======================================================================================================================"
	@mkdir -p $(CONAN_DEPENDENCIES_DIR) && \
		cd $(CONAN_DEPENDENCIES_DIR) && \
		conan install ../.. --settings build_type=$(CONAN_INSTALL_BUILD_TYPE) --options runtime=$(RUNTIME) --json conan-install-output.json $(if $(filter $(V),1),,>/dev/null) && \
		jq -r '.installed[].recipe.id' conan-install-output.json | sort
ifeq ($(CONAN_INSTALL_BUILD_TYPE),Debug)
ifneq ($(SKIP_CHECK_ABI_COMPLIANCE),1)
	@echo "======================================================================================================================"
	@$(call print_green,Checking ABIs of the dependencies)
	@echo "======================================================================================================================"
ifeq ($(ARCH), docker)
	@python3 $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/check_abi_compliance.py \
			--build_type $(CONAN_INSTALL_BUILD_TYPE) \
			--runtime $(RUNTIME) \
			--install-output $(CONAN_DEPENDENCIES_DIR)/conan-install-output.json \
			2>&1
else
	docker run --rm --name builder_docker \
		$(BUILDER_DOCKER_PERMISSIONS) \
		$(BUILDER_DOCKER_ALLOW_CTRLC_PARAMS) \
		$(BUILDER_DOCKER_ALLOW_GDB_PARAMS) \
		$(BUILDER_DOCKER_USER_PARAMS) \
		$(BUILDER_DOCKER_STANDARD_VOLUMES_PARAMS) \
		$(BUILDER_DOCKER_AUTHENTICATION_PARAMS) \
		$(BUILDER_DOCKER_PROXIES_PARAMS) \
		$(BUILDER_DOCKER_MAKEFILES_ENV_VARS_PARAMS) \
		--workdir=/source \
		$(BUILDER_DOCKER_ADDITIONAL_PARAMS) \
		$(BUILDER_DOCKER_IMAGE_FOR_RUN) \
			python3 $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/check_abi_compliance.py \
			--build_type $(CONAN_INSTALL_BUILD_TYPE) \
			--runtime $(RUNTIME) \
			--install-output $(CONAN_DEPENDENCIES_DIR)/conan-install-output.json \
			2>&1
endif
	@echo "ABI compliance check done!"
endif
endif
	@# as last step, ensure that the (proxy) repositories on this machine (for Maven/pypi/etc) are correctly setup for the current value of "LAB"
	@# this has nothing to do with Conan packages strictly but we need to carry out this step before the actual build starts, so hooking this step
	@# under "check_deps_upstream" is convenient
	@$(MAKE) configure_repositories

# ------------------------------------------------------------------------------------------
# verify_branch_name target
# ------------------------------------------------------------------------------------------

verify_branch_name:
ifneq ($(shell [[ $(GIT_BRANCH_NAME) =~ $(GIT_BRANCH_NAME_REGEX) ]] && echo matched), matched)
	$(error "ERROR on branching name. Please rename your branch '$(GIT_BRANCH_NAME)' using the following syntax: '$(GIT_BRANCH_NAME_REGEX)'")
else
	@echo "Branch name '$(GIT_BRANCH_NAME)' verified"
endif

# ------------------------------------------------------------------------------------------
# info target
# ------------------------------------------------------------------------------------------

info::
ifneq ($(FULL_VERSION),)
	@# only from this top-level makefile: communicate the git-decided version number to Teamcity:
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[buildNumber '$(CONAN_VERSION)']" || true
	@# requestPinning is a nice feature of the https://github.com/ocroquette/teamcity-autopin plugin
	@[ ! -z "$(TEAMCITY_VERSION)" -a "$(BUILDING_TAG)" = "1" ] && echo "##teamcity[requestPinning]" || true
	@[ ! -z "$(TEAMCITY_VERSION)" -a "$(GIT_BRANCH_TYPE)" = "support" ] && echo "##teamcity[requestPinning]" || true
	@echo "======================================================================================================================"
	@echo "GIT branch name: $(GIT_BRANCH_NAME)"
	@echo "Selected configuration is: $(CFG)"
	@echo "Target architecture is: $(ARCH)"
	@echo "Runtime is: $(RUNTIME)"
	@echo "Verbosity: $(V)"
	@echo "This machine is a build agent: $(BUILD_AGENT)"
	@echo "Number of commits since last tag in git: $(NUM_COMMITS_SINCE_LAST_TAG)"
	@echo "Building a git tag: $(BUILDING_TAG)"
	@echo "Shared libraries version: $(SOLIB_FULL_VERSION)"
	@echo "Full version: $(FULL_VERSION)"
	@echo "RPM version: $(RPM_VERSION)"
	@echo "Conan version@user/channel: $(CONAN_VERSION)@empirix/$(CONAN_CHANNEL)"
	@echo "Python version: $(PYTHON_FULL_VERSION)"
	@echo "Docker version: $(DOCKER_VERSION)"
	@echo "Helm version: $(HELM_VERSION)"
	@echo "======================================================================================================================"
ifneq ($(GIT_BRANCH_NAME), local)
	$(MAKE) verify_branch_name
endif

else
	@echo "Version strings not available. Please check if Version-defines.mk has been included"
endif

# ------------------------------------------------------------------------------------------
# configure_repositories target
# ------------------------------------------------------------------------------------------

# This target is useful to dynamically configure which Docker/Helm/Maven/Pypi/RPM repositories 
# will be used inside docker "build" containers to fetch packages/libraries to build
# the project.
# A few considerations:
# * as of today this target is actually re-configuring only the MAVEN+YUM repositories; 
#   other artifact types (DOCKER,HELM,MAVEN,PYPI,etc) are fetched using the right client
#   (dockerd,helm,maven,pip,etc) on the fly and the URL is expanded dynamically based 
#   on the LAB env var (see e.g. ATF logic to expand docker registries)
# * the reason to set as YUM repository the OracleLinux9 is because the EVA/ECC builder docker is
#   based on a recent Fedora release and OL9 is the "closest" match to Fedora>=34
#   (note that as of Oct2023 there is no pipeline that fetches RPMs anyway, so this is just
#   theoretical work)
configure_repositories:
	@echo "======================================================================================================================"
	@$(call print_green,Configuring Nexus repository proxies for LAB=$(LAB))
	@echo "======================================================================================================================"
	@$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/configure-proxy-repositories.sh \
		$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR) \
		$(REMOTE_REPO_NEXUS_OL9_RPM_FOR_PULL)

# ------------------------------------------------------------------------------------------
# get_conan_version target
# ------------------------------------------------------------------------------------------

# This target is useful to integrate our GNU makefiles with other tools (like Conan) and thus
# will provide a machine-friendly, compact output
# See conan-base python code
get_conan_version:
ifneq ($(CONAN_VERSION),)
	@echo "$(CONAN_VERSION)"
else
	@echo "Conan version not available. Please check if Version-defines.mk has been included"
endif

# ------------------------------------------------------------------------------------------
# Formatting targets
# 1. Check YAML/YML and JSON config file format. Stop if format not correct or warning.
# 2. Correct the YAML/YML and JSON config file format. Stop if any syntactical error .
# ------------------------------------------------------------------------------------------

format_check_json_yaml:
ifeq ($(ARCH), docker)
	@if [ ! -f /opt/empirix/nodejs_tools/bin/prettier ]; then \
		echo "Cannot find the /opt/empirix/nodejs_tools/bin/prettier binary. You are probably trying to use this target from the BUILDER docker." ; \
		echo "Please invoke this target from inside the ATF docker instead, using either:" ; \
		echo " make docker_format_check_json_yaml        OR:" ; \
		echo " make bash_in_atf_docker ; make format_check_json_yaml " ; \
		exit 2 ; \
	fi
	@echo "======================================================================================================================"
	@echo "== Checking JSON/YAML formatting =="
	@echo "======================================================================================================================"
	PATH=$(PATH):/opt/empirix/nodejs_tools/bin/ && \
		/opt/empirix/nodejs_tools/bin/prettier --check .		# this command will check the JSON/YAML format and linting using Prettier tool
else  # !ARCH=docker
	@echo
	@echo "The config format check using Prettier tool is disabled when ARCH!=docker."
	@echo
endif

format_inplace_json_yaml:
ifeq ($(ARCH), docker)
	@if [ ! -f /opt/empirix/nodejs_tools/bin/prettier ]; then \
		echo "Cannot find the /opt/empirix/nodejs_tools/bin/prettier binary. You are probably trying to use this target from the BUILDER docker." ; \
		echo "Please invoke this target from inside the ATF docker instead, using either:" ; \
		echo " make docker_format_check_json_yaml        OR:" ; \
		echo " make bash_in_atf_docker ; make format_check_json_yaml " ; \
		exit 2 ; \
	fi
	@echo "======================================================================================================================"
	@echo "== Reformatting JSON/YAML =="
	@echo "======================================================================================================================"
	PATH=$(PATH):/opt/empirix/nodejs_tools/bin/ && \
		/opt/empirix/nodejs_tools/bin/prettier --write .		# this command will correct the JSON/YAML format and linting using Prettier tool
else  # !ARCH=docker
	@echo
	@echo "The config format check using Prettier tool is disabled when ARCH!=docker."
	@echo
endif

# ------------------------------------------------------------------------------------------
# Cache-Cleanup targets
# ------------------------------------------------------------------------------------------

# IMPORTANT:
#  In most repositories the cleanup of local Conan cache has to happen AFTER the very last step of "deploy";
#  reason is that the cleanup is not so smart to avoid cleaning up the same Conan packages required by currently-running build, so
#  it's best to do the cleanup only as very last step.
#  Moreover the cleanup cannot be the very first step (immediately before "check_deps" step) because in some pipelines we check out the whole
#  source tree as brand new on build agents and as such the "EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR" variable becomes available only
#  in the GNU make steps that run after the "check_deps" one.
#  Conclusion: the "conan_cleanup_local_cache" has to be placed EXACTLY as the last step of each pipeline, that currently means
#  the "check_teamcity_artifacts" step.
conan_cleanup_local_cache::
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='Cleaning Conan local cache']" || true
	@-[ -f $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/conan-cleanup-local-cache.sh ] && $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/conan-cleanup-local-cache.sh $(CONAN_PACKAGES)
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='Cleaning Conan local cache']" || true

docker_cleanup_local_cache::
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockOpened name='Cleaning Docker local cache']" || true
	@echo "Cleaning local docker cache"
	@-[ -f $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/docker-cleanup-local-cache.sh ] && sh $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/docker-cleanup-local-cache.sh
	@[ ! -z "$(TEAMCITY_VERSION)" ] && echo "##teamcity[blockClosed name='Cleaning Docker local cache']" || true


# ------------------------------------------------------------------------------------------
# Teamcity-specific targets
# ------------------------------------------------------------------------------------------

#
# This target is the cleanup phase executed in all pipelines, to ensure the build agent is left in a good shape for next builds.
# This step is executed even if:
#  - some other earlier build step FAILED
#  - if the user has clicked "Stop" in the Teamcity UI for this build
#
check_teamcity_artifacts:
	# if ATF created the file 'restart-docker-daemon' it means that we need to restart the daemon due to a network bug inside docker
	@if [ -f $(TEAMCITY_ARTIFACTS_DIR_MAIN)/restart-docker-daemon ]; then \
	     echo "Detected docker daemon bug with docker networks; restarting now docker daemon to recover from the bug." ; \
         systemctl restart docker ; \
         rm -f $(TEAMCITY_ARTIFACTS_DIR_MAIN)/restart-docker-daemon ; \
    fi
	# NOTE: OPTIONAL_ARGS allows to pass extra arguments to the Python script
	python3 $(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/check_teamcity_artifacts.py $(OPTIONAL_ARGS)
	# Invoke CONAN Cleanup
	$(MAKE) conan_cleanup_local_cache
	# Invoke DOCKER Cleanup
	$(MAKE) docker_cleanup_local_cache

# ------------------------------------------------------------------------------------------
# Credentials-cache targets
# ------------------------------------------------------------------------------------------


ifeq ($(TIMEOUT),)
# this GNU make var. sets the cache time after which the username and password will expire (default is 1 hour)
# NOTE: use TIMEOUT=0 to setup an 'infinite' timeout for credentials
TIMEOUT:=3600
endif

# NOTE: NEXUS_PCAPBOT_USER/NEXUS_PCAPBOT_PW are typically environment variables available only on build agents.
#       On developer machines, they will expand to nothing and the script will ask for credentials.
cache_credentials:
ifeq ($(V),0)
	@$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/cache-nexus-credentials.sh \
						write_ciphered_file \
						$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR) \
						$(REMOTE_REPO_NEXUS_PCAP_FOR_PULL) $(TIMEOUT) \
						$(NEXUS_PCAPBOT_USER) $(NEXUS_PCAPBOT_PW)
else
	$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/cache-nexus-credentials.sh \
						write_ciphered_file \
						$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR) \
						$(REMOTE_REPO_NEXUS_PCAP_FOR_PULL) $(TIMEOUT) \
						$(NEXUS_PCAPBOT_USER) $(NEXUS_PCAPBOT_PW)
endif

cache_credentials_clean:
	@$(EMPIRIX_PIPELINE_FRAMEWORK_SCRIPT_SUPPORT_DIR)/cache-nexus-credentials.sh \
						cache_credentials_clean


# ------------------------------------------------------------------------------------------
# Help target
# NOTE1: descriptions must be all aligned at column 54 as standard (including the initial TAB)
# NOTE2: only most useful targets are documented; internal/obscure targets must NOT be listed
# ------------------------------------------------------------------------------------------

ifeq ($(SUPPORTS_HELP),1)
help::
	@echo
	@echo "Misc targets (to run OUTSIDE the builder docker):"
	@echo "    info:                                 print information about the configuration, version and automatically-detected build params"
	@echo "    configure_repositories:               switch repositories from which dependencies are pulled based on LAB value"
	@echo
	@echo "Misc targets (to run INSIDE the ATF docker):"
	@echo "    format_check_json_yaml:               run Prettier formatting tool to validate formatting of all JSON/YAML files"
	@echo "    format_inplace_json_yaml:             run Prettier formatting tool to reformat all JSON/YAML files"
	@echo
	@echo "Local cache cleanups (to run OUTSIDE the builder docker):"
	@echo "    conan_cleanup_local_cache:            cleanup all old Conan packages"
	@echo "    docker_cleanup_local_cache:           cleanup all old Docker images, containers, networks"
	@echo
	@echo "Nexus PCAPs credentials (to run OUTSIDE the builder docker):"
	@echo "    cache_credentials:                    cache Nexus PCAPs credentials; optionally provide TIMEOUT=<num-secs> to customize timeout"
	@echo "    cache_credentials_clean:              immediately remove cached Nexus PCAPs credentials"
	@echo
endif


.PHONY: conan_cleanup_local_cache docker_cleanup_local_cache check_teamcity_artifacts cache_credentials cache_credentials_clean help
