#!/usr/bin/env python3

"""
This module allows to create a file to cache credentials for Nexus PCAPs repository in an encoded string.
The credentials file must not be shared with other users since it's actually easy to decode and then
find out the clear-text password
"""

import getpass, os, sys, base64, json, requests, readline

def input_with_prefill(prompt, text):
    def hook():
        readline.insert_text(text)
        readline.redisplay()
    readline.set_pre_input_hook(hook)
    result = input(prompt)
    readline.set_pre_input_hook()
    return result

def ask_and_validate_credentials(pcap_repo, nexus_pcapbot_user, nexus_pcapbot_pw):
    if not nexus_pcapbot_user or nexus_pcapbot_user == "null":
        username_guess = os.environ["EMAILTO"] if "EMAILTO" in os.environ else ""
        nexus_pcapbot_user = input_with_prefill("Enter your username for PCAP repository: ", username_guess)

    if not nexus_pcapbot_pw or nexus_pcapbot_pw == "null":
        nexus_pcapbot_pw = getpass.getpass(
            "Enter your password for PCAP repository: "
        )

    # check if credentials are correct by launching an HTTP GET on a folder like "/repository/pcap/automated-tests/"
    # From empirical testing we know that Nexus will answer with 401 or 403 if credentials are wrong and with 404 if credentials are OK
    # (404 is returned because Nexus does not allow to download a directory)
    print(f"Checking if credentials are correct for {pcap_repo}...")
    r = requests.get(
        url=pcap_repo,
        auth=(nexus_pcapbot_user, nexus_pcapbot_pw),
        allow_redirects=True,
    )

    if r.status_code == 401 or r.status_code == 403:
        print(f"Username and/or password not correct! HTTP status code is {r.status_code}. Please try again")
        sys.exit(2)
    elif r.status_code != 404:
        print(f"Failed a simple HTTP GET against {pcap_repo} with the provided username/password combination: HTTP status code is {r.status_code}")
        sys.exit(2)

    print(f"Credentials are valid!")
    return (nexus_pcapbot_user, nexus_pcapbot_pw)

def write_ciphered_file(credentials_file, actual_user, actual_pw):
    credentials_dict = {
        "NEXUS_PCAPBOT_USER": actual_user,
        "NEXUS_PCAPBOT_PW": actual_pw,
    }
    json_string = json.dumps(credentials_dict)
    encoded_str = base64.encodebytes(json_string.encode())

    os.makedirs(os.path.dirname(credentials_file), exist_ok=True)
    with open(credentials_file, "w+") as f:
        f.write(encoded_str.decode())

    print("Credentials file saved correctly")


if __name__ == "__main__":
    credentials_file = sys.argv[1]
    pcap_repo = sys.argv[2] # this would look something like e.g. "https://nexus-modena.dev.infovista.com/repository/pcap/automated-tests/"
    nexus_pcapbot_user = sys.argv[3]
    nexus_pcapbot_pw = sys.argv[4]

    if os.path.exists(credentials_file):
        print(
            f"File {credentials_file} already exists. If you want to remove it run 'make cache_credentials_clean'"
        )
        sys.exit(99)

    print(
        f"\nSaving a file with ciphered credentials for PCAP repository {pcap_repo}. Please remember to run [make cache_credentials_clean] after running automated tests!\n"
    )
    actual_user, actual_pw = ask_and_validate_credentials(pcap_repo, nexus_pcapbot_user, nexus_pcapbot_pw)
    write_ciphered_file(credentials_file, actual_user, actual_pw)
