#!/usr/bin/env python3

"""
This Python script will be useful in removing the artifact packages from the Klerity JFrog Artifacts.
User has to pass a list of repo project names and version numbers in a pipe separated list with various options.
It has a check if the artifact is unused for a very long time then remove it. Number of last download and
modification days option can be provided in command.
Currently, this script is configured to remove helm-local and docker-local artifacts.
"""

from pyartifactory import Artifactory
from artifactory import ArtifactoryPath
from datetime import datetime, timezone
import os, sys, argparse, re

global ARTIFACTORY_USER, ARTIFACTORY_API_KEY, BASE_ARTIFACTORY_REST_API_URL, OUTPUT_DIRECTORY
BASE_ARTIFACTORY_REST_API_URL = "https://empirixartifactory.jfrog.io/artifactory"
OUTPUT_DIRECTORY = "TeamcityArtifacts"

def get_artifacts_info_and_stats(artifact_file_name):
    py_art = Artifactory(
        url=BASE_ARTIFACTORY_REST_API_URL, auth=(ARTIFACTORY_USER, ARTIFACTORY_API_KEY)
    )
    info = py_art.artifacts.info(artifact_file_name)
    stats = py_art.artifacts.stats(artifact_file_name)
    return info, stats


def calculate_number_of_days_from_unix_timestamp(timestamp):
    converted_timestamp = datetime.fromtimestamp(round(timestamp / 1000))
    current_time_utc = datetime.utcnow()
    days = current_time_utc - converted_timestamp
    return days.days


def final_run(artifact_last_downloaded, artifact_last_modified, artifacts: list):
    with open(
        os.path.join(OUTPUT_DIRECTORY, "not_downloaded_in_long_time.txt"), "w+"
    ) as not_downloaded_in_long_time_file, open(
        os.path.join(OUTPUT_DIRECTORY, "never_downloaded.txt"), "w+"
    ) as never_downloaded_file:

        never_downloaded = list()
        not_downloaded_in_long_time = list()
        number_of_days_last_downloaded_check = artifact_last_downloaded
        number_of_days_last_modified_check = artifact_last_modified

        for artifact in artifacts:
            artifact_info, artifact_stats = get_artifacts_info_and_stats(
                artifact.replace(BASE_ARTIFACTORY_REST_API_URL, "")
            )
            number_of_days_last_downloaded = calculate_number_of_days_from_unix_timestamp(
                artifact_stats.lastDownloaded
            )
            number_of_days_last_modified = calculate_number_of_days_from_unix_timestamp(
                int(artifact_info.lastModified.timestamp() * 1000)
            )
            if (
                artifact_stats.lastDownloaded == 0
                and number_of_days_last_modified > number_of_days_last_modified_check
            ):
                never_downloaded.append(artifact)
                never_downloaded_file.write(artifact)
                never_downloaded_file.write("\n")
            elif number_of_days_last_downloaded > number_of_days_last_downloaded_check:
                not_downloaded_in_long_time.append(artifact)
                not_downloaded_in_long_time_file.write(artifact)
                not_downloaded_in_long_time_file.write("\n")

    print(
        f"Total {len(not_downloaded_in_long_time)} artifact files not downloaded since "
        f"{artifact_last_downloaded} days. Please check not_downloaded_in_long_time.txt file."
    )
    print(
        f"Total {len(never_downloaded)} artifact files never downloaded but were last modified before "
        f"{artifact_last_modified} days. Please check never_downloaded.txt file."
    )

    return never_downloaded, not_downloaded_in_long_time


def delete_artifacts(artifact_path_list):
    delete_artifacts = open(
        os.path.join(OUTPUT_DIRECTORY, "deleted_artifacts.txt"), "w+"
    )
    for artifact in artifact_path_list:
        if artifact.endswith("manifest.json"):
            artifact = artifact.replace("manifest.json", "")
        dohq_art = ArtifactoryPath(
            artifact, auth=(ARTIFACTORY_USER, ARTIFACTORY_API_KEY)
        )
        if dohq_art.exists():
            print(f"Start deleting '{artifact}'")
            # Call to unlik function
            dohq_art.unlink()
            print(f"Finished deleting '{artifact}'")
            delete_artifacts.write(artifact)
            delete_artifacts.write("\n")
    delete_artifacts.close()


def get_artifacts_eligible_to_be_removed_list(
    artifacts_list: list,
    project_name: str,
    version_list: list,
    url_suffix: str,
    file_pattern: str,
) -> None:
    dohq_art = ArtifactoryPath(
        f"{BASE_ARTIFACTORY_REST_API_URL}/{url_suffix}/",
        auth=(ARTIFACTORY_USER, ARTIFACTORY_API_KEY),
    )
    for path in dohq_art.glob(f"{project_name}{file_pattern}"):
        # If it is a bugfix or feature branch or a 0.0.x version, skip it
        # NOTE: we are being careful not to include versions like 10.0.4
        #       by using \D0\.0\.0\d+
        #       crf AP-3290
        if (
            "bugfix" in path.__str__()
            or "feature" in path.__str__()
            or re.search(r"\D0\.0\.\d+", path.__str__())
        ):
            if len(version_list) > 0:
                for version in version_list:
                    if f"{version}" in path.__str__():
                        artifacts_list.append(path.__str__())
            else:
                artifacts_list.append(path.__str__())


def get_docker_helm_local_file_list(
    artifacts_list: list,
    repository: str,
    projects: str,
    versions: str = "",
    docker_group: str = "",
) -> None:
    project_list = projects.split("|") if projects else []
    version_list = versions.split("|") if versions else []

    for project_name in project_list:
        # Set the flag when the given master folder found in the list
        dir_list_flag = False
        # Keep adding any master folder in the condition and define parts of url pattern
        if repository == "helm-local":
            url_suffix = repository
            file_pattern = "*.tgz"
            dir_list_flag = True
        elif repository == "docker-local":
            url_suffix = f"{repository}/{docker_group}"
            file_pattern = "/**/manifest.json"
            dir_list_flag = True
        else:
            print(f"WARN: Unknown master folder {repository}")
        if dir_list_flag:
            get_artifacts_eligible_to_be_removed_list(
                artifacts_list, project_name, version_list, url_suffix, file_pattern
            )


# Operation of script starts from here
if __name__ == "__main__":
    # parse the arguments to this script
    parser = argparse.ArgumentParser(
        description="Remove artifacts from the Artifactory. Authentication should be provided by environment variables ARTIFACTORY_USER, ARTIFACTORY_PW",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument(
        "--helm_projects",  # repositories
        help="Pipe-separated helm project patterns to match in Artifactory",
        type=str,
        default="eva-|eva-ecc-|ecc-|kubernetes-post-deploy-util-|redis-cluster-operator-|watch-util-",
    )
    parser.add_argument(
        "--docker_projects",  # repositories
        help="Pipe-separated docker project patterns to match in Artifactory",
        type=str,
        default="*",
    )
    parser.add_argument(
        "--docker_groups",
        help="Pipe-separated docker groups names under 'docker-local' repository in Artifactory",
        type=str,
        default="empirix-eva-ecc|empirix-eem-epm",
    )
    parser.add_argument(
        "--specific_versions",  # specific_versions
        help="Pipe-separated version list of projects",
        type=str,
        default="",
    )
    parser.add_argument(
        "--exclude_downloaded_in_last_days",  # last_download_days
        help="Skip cleaning of artifacts that were downloaded in the last N days",
        type=int,
        default=0,
    )
    parser.add_argument(
        "--exclude_modified_in_last_days",  # last_modify_days
        help="Skip cleaning of artifacts that were modified/uploaded in the last N days",
        type=int,
        default=0,
    )
    args = parser.parse_args()

    try:
        if len(os.environ["ARTIFACTORY_USER"].strip()) == 0:
            pass
    except KeyError:
        print(
            "No env var ARTIFACTORY_USER defined. Please set ARTIFACTORY_USER env variable to connect JFrog Artifactory."
        )
        sys.exit(0)
    try:
        if len(os.environ["ARTIFACTORY_PW"].strip()) == 0:
            pass
    except KeyError:
        print(
            "No env var ARTIFACTORY_PW defined. Please set ARTIFACTORY_PW env variable to connect JFrog Artifactory."
        )
        sys.exit(0)

    ARTIFACTORY_USER = os.environ["ARTIFACTORY_USER"]
    ARTIFACTORY_API_KEY = os.environ["ARTIFACTORY_PW"]

    print(
        f"Searching in {BASE_ARTIFACTORY_REST_API_URL} for artifacts to be deleted..."
    )

    try:
        os.mkdir(OUTPUT_DIRECTORY)
    except:
        # folder already exists
        pass

    docker_helm_local_file_list = list()
    if args.helm_projects and len(args.helm_projects) > 0:
        get_docker_helm_local_file_list(
            docker_helm_local_file_list,
            repository="helm-local",
            projects=args.helm_projects,
            versions=args.specific_versions,
        )
    if args.docker_projects and len(args.docker_projects) > 0:
        if args.docker_groups and len(args.docker_groups) > 0:
            docker_group_list = args.docker_groups.split("|")
            for docker_group_name in docker_group_list:
                get_docker_helm_local_file_list(
                    docker_helm_local_file_list,
                    repository="docker-local",
                    projects=args.docker_projects,
                    versions=args.specific_versions,
                    docker_group=docker_group_name,
                )
        else:
            print("WARN: The --docker_groups cannot remain empty")

    _, not_downloaded_in_long_time_list = final_run(
        artifact_last_downloaded=args.exclude_downloaded_in_last_days,
        artifact_last_modified=args.exclude_modified_in_last_days,
        artifacts=docker_helm_local_file_list,
    )
    print(datetime.now())
    delete_artifacts(not_downloaded_in_long_time_list)
    print(datetime.now())
