#!/bin/bash

#
# Smart docker deploy on remote repositories
# Created: Aug 2020 @fmontorsi
# Updated: Mar 2023 @danielefranceschi ECR management
#
# Supports the following features:
# - Registry login/authentication
# - Logging of pushed docker images
#
# Usage:
#   ./docker-deploy-wrapper.sh [repo_user] [repo_password] [repo_registry_name] [repo_url] [output_logfile] [docker_group] [image1_name:tag] [image2_name:tag] ...
# E.g.:
#   ./docker-deploy-wrapper.sh "developer" "hammer"
#        "repo-billerica.perform.empirix.com:6000" "repo-billerica.perform.empirix.com:6000/empirix-ecc-eva" "logfile.csv" "empirix-ecc-eva"
#        "my-project-group"
#         myprj:1.1 myprj-sidecar:1.1
#
# or when using ECR it could be e.g.:
#  ./docker-deploy-wrapper.sh "theECRAccessKey" "theECRSecretKey"
#        "255142282392.dkr.ecr.us-east-1.amazonaws.com" "255142282392.dkr.ecr.us-east-1.amazonaws.com/empirix-ecc-eva" "logfile.csv" "empirix-ecc-eva"
#        "my-project-group"
#         myprj:1.1 myprj-sidecar:1.1
#
# To ease aggregation of multiple runs of the script, the logfile will NOT be truncated, just used in APPEND mode.
# NOTE: this script does not require any env variable to be set, its behavior is entirely defined by CLI arguments
#
# Exit code:
#  0: success
#  100: login failure
#  200: tag failure
#  300: push failure
#

set -u

USER="$1"; shift
PW="$1"; shift
REMOTE_REPO_REGISTRY_NAME="$1"; shift
REMOTE_REPO_URL_PUSH="$1"; shift
DEPLOY_FILELIST_LOG="$1"; shift
DOCKER_GROUP="$1"; shift
LOCAL_DOCKER_IMAGES_AND_TAGS="$*"
LOCAL_TAG="0.0-local"
NCORRECTLY_DEPLOYED=0
NSKIPPED=0

function error_local_tag()
{
  IMG=$1
  echo "-----------------------------------------------------------------------------------------------------"
  echo "-----------------------------------------------------------------------------------------------------"
  echo "ERROR *** Image $IMG can't be pushed to the repo because version $LOCAL_TAG is just for local usage ***"
  echo "-----------------------------------------------------------------------------------------------------"
  echo "-----------------------------------------------------------------------------------------------------"
  exit 200
}

function ecr_preauthenticate()
{
  # for readability, rename the USER/PW arguments to something that makes sense in ECR terminology:
  ECR_ACCESS_KEY="$USER"
  ECR_SECRET_KEY="$PW"

  ECR_REGION=`echo "${REMOTE_REPO_REGISTRY_NAME}" | cut -d. -f4`
  ECR_REGISTRY_ID=`echo "${REMOTE_REPO_REGISTRY_NAME}" | cut -d. -f1`
  echo "** pre-authenticating to ECR/${ECR_REGION}"

  aws_configure_failed=0
  aws configure set aws_access_key_id "${ECR_ACCESS_KEY}" || aws_configure_failed=1
  aws configure set aws_secret_access_key "${ECR_SECRET_KEY}" || aws_configure_failed=1
  aws configure set region "${ECR_REGION}" || aws_configure_failed=1
  aws configure set output "text" || aws_configure_failed=1
  if [ $aws_configure_failed -ne 0 ]; then echo "FATAL awscli setup failed" ; exit 100 ; fi
}

function ecr_create_repository()
{
  local dockerImage="$1"

  # ECR needs the repository to be created before pushing the image.
  # first of all, remove the tag for repo creation
  imageNameWithoutTag=`echo $dockerImage | cut -d: -f1`
  echo "** creating repository ${DOCKER_GROUP}/${imageNameWithoutTag} on ECR/${ECR_REGION}"

  # then use awscli to create the repo on ECR
  aws_cli_exit_code=0
  aws ecr create-repository --region "${ECR_REGION}" --repository-name "${DOCKER_GROUP}/${imageNameWithoutTag}" --image-tag-mutability MUTABLE > /tmp/awscli.log 2>&1 || aws_cli_exit_code=$?
  if [ "$aws_cli_exit_code" != "0" ]; then
    # the log may contain an exception reporting that the repo already exists
    if ! grep -q RepositoryAlreadyExistsException /tmp/awscli.log; then
      # unexpected error, report and exit
      echo "FATAL error creating repository ${imageNameWithoutTag} on ECR/${ECR_REGION}"
      cat /tmp/awscli.log
      exit 150
    fi
  fi
  rm -f /tmp/awscli.log
}

# ensure folder for log file exists
mkdir -p $(dirname "$DEPLOY_FILELIST_LOG")

if [[  "$REMOTE_REPO_REGISTRY_NAME" == *".ecr."*  ]]; then
  USING_ECR=true

  # let's preauthenticate... this is a step required for ECR only
  ecr_preauthenticate

  DOCKERBUILD_PASSWORD=`aws ecr get-login-password --region ${ECR_REGION}`
  if [ $? -ne 0 ]; then echo "FATAL ECR pre-authentication failed" ; exit 100 ; fi
  DOCKERBUILD_USER=AWS
  DOCKERBUILD_REGISTRY="$REMOTE_REPO_REGISTRY_NAME"
else
  USING_ECR=false
  DOCKERBUILD_USER="$USER"
  DOCKERBUILD_PASSWORD="$PW"
  DOCKERBUILD_REGISTRY="$REMOTE_REPO_REGISTRY_NAME"
fi

docker login -u "$DOCKERBUILD_USER" -p "$DOCKERBUILD_PASSWORD" "$DOCKERBUILD_REGISTRY"
if [ $? -ne 0 ]; then echo "FATAL Failed authentication against $DOCKERBUILD_REGISTRY" ; exit 100 ; fi

echo "** Login succeeded... proceeding with tag&push of following docker images [$LOCAL_DOCKER_IMAGES_AND_TAGS]"
for srcDockerImageAndTag in $LOCAL_DOCKER_IMAGES_AND_TAGS; do

  if [[ "$srcDockerImageAndTag" == *"$LOCAL_TAG" ]]; then
    error_local_tag "$srcDockerImageAndTag"
  fi
  # in some cases the images listed via CLI already contain the registry and docker group...
  # in such case just remove whatever registry they contain and replace with the one we got via CLI.
  # The following syntax keeps the portion of string coming after LAST slash:
  dockerImage="${srcDockerImageAndTag##*/}"
  targetDockerImageAndTag="${DOCKERBUILD_REGISTRY}/${DOCKER_GROUP}/${dockerImage}" 

  if [[ "$srcDockerImageAndTag" != "$targetDockerImageAndTag" ]]; then
    echo "** Re-tagging source docker image [$srcDockerImageAndTag] as [$targetDockerImageAndTag]"
    docker tag "$srcDockerImageAndTag" "$targetDockerImageAndTag"
    if [ $? -ne 0 ]; then echo "Failed tagging!" ; exit 200 ; fi
  else
    echo "** The docker image [$srcDockerImageAndTag] has already correct docker registry, group and tag. Skipping re-tagging"
  fi

  if $USING_ECR; then
    ecr_create_repository "$dockerImage"
  fi

  echo "** Pushing docker image [$targetDockerImageAndTag]"  
  docker push "$targetDockerImageAndTag"
  if [ $? -ne 0 ]; then echo "Failed pushing $targetDockerImageAndTag !" ; exit 300 ; fi

  DEPLOY_RESULT="deployed"

  # save this docker image artifact in the output CSV file
  echo "docker,${DOCKERBUILD_REGISTRY},${DOCKER_GROUP}/${dockerImage},${DEPLOY_RESULT}" | tee -a $DEPLOY_FILELIST_LOG
  (( NCORRECTLY_DEPLOYED++ ))
done

echo "** Count of docker images deployed: $NCORRECTLY_DEPLOYED."
