#!/usr/bin/env python3

#
# Utility to automatically generate headers and documentation for System Health KPIs.
# Note that this utility is built around the differentiation:
# 1) common KPIs
# 2) non-common KPIs
#

import os, sys
import argparse
import datetime

from hashlib import md5
import xlsxwriter
from systemhealth_xml_loader import SystemhealthXMLLoader
from systemhealth_cpp_code_generator import SystemhealthCppCodeGenerator
from systemhealth_doc_generator import SystemhealthDocGenerator
from typing import Tuple

 
class SystemhealthKpiGenerator:
    def __init__(self, args):
        """constructor"""
        self.system_health_xmls = args.system_health_xmls
        self.system_health_prefix = args.system_health_prefix

        if args.output_dir:
            self.output_dir = os.path.abspath(args.output_dir)
        else:
            self.output_dir = os.path.abspath(
                self.__get_parent_of_master_format_directory(args.system_health_xmls)
            )

        self.export_type = args.export_type
        self.code_formatter = args.code_formatter
        self.verbose = args.verbose

        # FIXME: this is really hackish:
        if hasattr(args, "unit_test_mode"):
            self.unit_test = args.unit_test_mode
        else:
            self.unit_test = False

    def __get_parent_of_master_format_directory(self, system_health_xmls: list):
        master_format = "/master-format/"
        for xml in system_health_xmls:
            if master_format in xml:
                return xml.split(master_format)[0]
        print(f"None of the xml from {system_health_xmls} is stored under the expected folder 'master-format'... cannot compute the output directory automatically. Please user --output-dir option explicitly.")
        sys.exit(3)

    def __generate_header(self) -> None:
        """
        Generate C++ .h header file from systemhealth XML files provided at ctor time
        """

        # The End KPI id for application (e.g. capture, statemachine, etc.) is defined by "APPLICATION_KPI_END_ID"
        # and its value would be defined in the last processed xml file using the last assigned KPI enum id.

        loader = SystemhealthXMLLoader(self.verbose)
        full_xml_list = loader.expand_xml_list(self.system_health_xmls)

        generator = SystemhealthCppCodeGenerator(self.verbose, self.code_formatter)

        for index, system_health_xml in enumerate(full_xml_list):
            xml_filepath, xml_filename = os.path.split(system_health_xml)

            # IMPORTANT: the processname="common" is handled in a specific, very special way by C++ code generator
            processname = xml_filename[0 : xml_filename.rfind("_")]

            kpi_list = loader.parse_systemhealth_xml(
                os.path.join(xml_filepath, xml_filename)
            )
            first_xmlfile_being_parsed = index == 0
            last_xmlfile_being_parsed = index == len(self.system_health_xmls) - 1

            generator.generate_cpp_header_file(
                self.output_dir,
                xml_filename,
                processname,
                kpi_list,
                first_xmlfile_being_parsed,
                last_xmlfile_being_parsed,
            )

    def __generate_documentation(self) -> None:
        """
        Generate docs from systemhealth XML files provided at ctor time
        """

        loader = SystemhealthXMLLoader(self.verbose)
        full_kpi_dict = loader.get_all_kpis_recursively(self.system_health_xmls)

        # FIXME: currently we hardcode as Excel output the syshealth prefix minus last char, since that's usually a "_"
        display_name = self.system_health_prefix[:-1]

        export_file_name = f"{display_name}_prometheus_metrics_docs.xlsx"
        export_file_path = os.path.join(self.output_dir, export_file_name)

        generator = SystemhealthDocGenerator(self.verbose)
        generator.generate_documentation(
            export_file_path, full_kpi_dict, self.system_health_prefix, self.unit_test
        )

    def generate(self) -> None:
        """Generate header source code or documentation"""
        if self.export_type == "header":
            self.__generate_header()
        elif self.export_type == "documentation":
            self.__generate_documentation()
        else:
            print(f"Invalid export type {self.export_type} provided. Aborting.")
            sys.exit(2)


def parse_cmdline() -> dict:
    """Parse commandline options"""
    parser = argparse.ArgumentParser(
        description="Script to translate Systemhealth XML files into C++ code or Excel documentation"
    )
    parser.add_argument(
        "--system-health-xmls",
        nargs="*",
        type=str,
        help="List of XML syshealth descriptor files",
    )
    # FIXME: add handling of the syshealth prefix
    parser.add_argument(
        "--system-health-prefix",
        type=str,
        help="Prefix for all the component syshealth KPIs",
        default="",
    )
    parser.add_argument(
        "--output-dir",
        type=str,
        help="Output directory where the .h or .xlsx is produced",
    )
    parser.add_argument(
        "--export-type",
        type=str,
        help="'header' : generates source code, 'documentation' : generates excel document",
        default="header"
    )
    parser.add_argument(
        "--code-formatter",
        type=str,
        help="Absolute full path of the formatter utility to re-format .h files",
    )
    parser.add_argument(
        "--verbose",
        help="Be verbose",
        action='store_true'
    )

    # do the actual parsing
    retval = parser.parse_args()
    if not retval.system_health_xmls:
        print(f"At least one syshealth XML descriptor file must be specified through --system-health-xmls. Use --help for usage help.")
        sys.exit(2)

    return retval


#
# MAIN
#
if __name__ == "__main__":
    args = parse_cmdline()
    sysheathkpi = SystemhealthKpiGenerator(args)
    sysheathkpi.generate()
