/* 
 * File:         highsee_adaptor.h
 * Description:  HighSee Adaptor External interface
 *
 * (C) COPYRIGHT nTh s.r.l. 2015-2019. ALL RIGHTS RESERVED.
 * NO PART OF THIS PROGRAM MAY BE PHOTOCOPIED REPRODUCED OR
 * TRANSLATED TO ANOTHER PROGRAM LANGUAGE WITHOUT THE PRIOR
 * WRITTEN CONSENT OF nTh s.r.l.
 */

#ifndef HIGHSEE_ADAPTOR_H
#define	HIGHSEE_ADAPTOR_H

#if __GNUC__ >= 4
    #define HIGHSEE_DLL_PUBLIC __attribute__ ((visibility ("default")))
    #define HIGHSEE_DLL_LOCAL  __attribute__ ((visibility ("hidden")))
#else
    #define HIGHSEE_DLL_PUBLIC
    #define HIGHSEE_DLL_LOCAL
#endif

typedef void (*OutputMessageCallback_t)(unsigned int level, const char* msg);

#include <stdint.h>

#define TYPE_PRODUCER                 0
#define SM_INTERFACE_MASK             0x00040000
#define TCP_INTERFACE_MASK            0x00080000
#define FUNCT_INTERFACE_MASK          0x00200000
#define HTTP_PAYLOAD_MASK             0x10000000
#define TIMESTAMP_TYPE                5
#define UNKN_PACKET_TYPE              0
#define COMMAND_TYPE                 14
#define INNERIP_PACKET_TYPE          34
#define INNERIP_TRUNC_PACKET_TYPE    37

#define ENABLE_DUMP_INPUT_COMMAND     3
#define DISABLE_DUMP_INPUT_COMMAND    4
#define SET_LOG_LEVEL_COMMAND         5
#define	ENABLE_DUMP_TRACE_COMMAND     13
#define	DISABLE_DUMP_TRACE_COMMAND    14
#define READ_LICENSE_COMMAND          15

#define COMMAND_VALUE_SIZE          100

#define TCP_PAYLOAD_TYPE            6
#define UDP_PAYLOAD_TYPE            17

#define CELL_ID_TYPE_4G 0
#define CELL_ID_TYPE_3G 1

#define MAX_HS_ADDRESS_LENGTH      16
#define MAX_HS_SUMMARY_LENGTH   64000
#define MAX_HS_STATS_LENGTH     100000

#define INPUT_OK                   0
#define INPUT_ERROR_NULL_POINTER   1
#define INPUT_ERROR_INT_NOT_INIT   2
#define INPUT_ERROR_PROT_UNKNOWN   3
#define INPUT_ERROR_PORT_UNKNOWN   4
#define INPUT_ERROR_SERVER_UNKNOWN 5
#define INPUT_ERROR_TCP_PARSING    6
#define INPUT_ERROR_UDP_PARSING    7
#define INPUT_ERROR_SSL_PARSING    8
#define INPUT_ERROR_QUIC_PARSING   9
#define INPUT_ERROR_HTTP_PARSING   10
#define INPUT_ERROR_PACKET_SEND    11
#define INPUT_ERROR_WRONG_FORMAT   12
#define INPUT_ERROR_UNORDERED_TIME 13
#define INPUT_ERROR_WRONG_PAYLOAD_LEN 14

#define SESSION_OUT_TYPE_NONE      0
#define SESSION_OUT_TYPE_STOP      1
#define SESSION_OUT_TYPE_START     2
#define SESSION_OUT_TYPE_SR_CG     3
#define SESSION_OUT_TYPE_PR_CG     4
#define SESSION_OUT_TYPE_PR_HS     5
#define SESSION_OUT_TYPE_SR_CT     6
#define SESSION_OUT_TYPE_PR_CT     7

#define HIGHSEE_OK                 0
#define HIGHSEE_FAIL              -1

#define UP_QFI_NOT_VALID     0xFF

#define HS_SET_INDEX(a,b)   (a << 16 | b)
#define HS_GET_PRINDEX(a)   (a >> 16)
#define HS_GET_HSINDEX(a)   (a & 0xFFFF)

typedef struct
{
    uint32_t    sec;
    uint32_t    nsec;
}   highsee_timestamp_t;

typedef struct
{
    const uint8_t  *p_data;
    uint16_t        len;
}   highsee_ip_address_t;

typedef struct
{
    /*  Source IP Address */
    highsee_ip_address_t    src_ip;
    /*  Destination IP Address */
    highsee_ip_address_t    dst_ip;
    /*  Source and Destination UDP/TCP ports */
    uint32_t    src_port;
    uint32_t    dst_port;
}   highsee_ip_packet_t;

typedef struct
{
    uint16_t       imsi_len;
    const uint8_t* imsi;
    uint16_t       imei_len;
    const uint8_t* imei;
}   highsee_user_id_t;

typedef struct
{
    uint16_t       mcc;
    uint16_t       mnc; 
    uint8_t        type;   // 0: 4G cellid; 1: 3G cellid;
    union 
    {
        uint32_t   cellid4g;
        struct 
        {
            uint16_t  lac;
            uint8_t   rac;
            uint16_t  sac;
        } cellid3g; 
    } cellid;
}   highsee_cell_id_t;

typedef struct
{
    highsee_ip_address_t     src_ip;
    highsee_ip_address_t     dst_ip;
    uint32_t                 teid;
}   highsee_gtp_packet_t;

typedef struct
{
    highsee_ip_address_t     ip_addr;
    uint32_t                 teid;
}   highsee_gtp_endpoint_t;

typedef struct
{
    highsee_timestamp_t     timestamp;
}   highsee_syncro_packet_t;

typedef struct
{
    uint32_t                command_type;
    uint16_t                messase_len; 
    char                    message_value[COMMAND_VALUE_SIZE];
}   highsee_command_packet_t;

typedef struct
{
    uint32_t   error_code;
    void*      out_object;
} highsee_output_packet_t;

typedef struct
{
    uint32_t               session_type;
    uint64_t               session_id;
    highsee_timestamp_t    timestamp;
    highsee_ip_address_t   client_ip;
    uint8_t                client_ip_bytes[MAX_HS_ADDRESS_LENGTH];
    uint8_t                up_qfi;
    highsee_gtp_endpoint_t ul_gtp_endpoint;
    uint8_t                ul_gtp_ip_bytes[MAX_HS_ADDRESS_LENGTH];
    highsee_gtp_endpoint_t dl_gtp_endpoint;
    uint8_t                dl_gtp_ip_bytes[MAX_HS_ADDRESS_LENGTH];
    uint32_t               summary_info_len;
    char	           summary_info[MAX_HS_SUMMARY_LENGTH];      
} highsee_session_packet_t;

typedef struct
{
    uint32_t    stats_len;
    char	stats[MAX_HS_STATS_LENGTH];      
} highsee_stats_packet_t;

typedef struct
{
    highsee_timestamp_t     timestamp;
    highsee_ip_address_t    src_ip;
    highsee_ip_address_t    dst_ip;
    uint32_t                num_frame_bytes;
    uint32_t                num_payload_bytes;
    const uint8_t*          payload_bytes;
    uint8_t                 payload_type;     
    uint8_t                 ttl_hoplimit;
    // Optional parameters
    uint8_t                 is_gtp_tunnel;
    highsee_gtp_packet_t    gtp_tunnel;
    uint8_t                 is_cell_id;
    highsee_cell_id_t       cell_id; 
    uint8_t                 is_user_id;
    highsee_user_id_t       user_id;
    uint8_t                 is_upQfi;
    uint8_t                 upQfi;        
} highsee_innerip_packet_t;

typedef struct
{
    highsee_timestamp_t     timestamp;
    highsee_ip_address_t    src_ip;
    highsee_ip_address_t    dst_ip;
    uint32_t                num_frame_bytes;
    uint32_t                num_payload_bytes;
    uint32_t                num_capt_payload_bytes;
    const uint8_t*          payload_bytes;
    uint8_t                 payload_type;     
    uint8_t                 ttl_hoplimit;
    // Optional parameters
    uint8_t                 is_gtp_tunnel;
    highsee_gtp_packet_t    gtp_tunnel;
    uint8_t                 is_cell_id;
    highsee_cell_id_t       cell_id; 
    uint8_t                 is_user_id;
    highsee_user_id_t       user_id;
    uint8_t                 is_upQfi;
    uint8_t                 upQfi;        
} highsee_innerip_trunc_packet_t;

#ifdef	__cplusplus
extern "C" {
#endif
    
HIGHSEE_DLL_PUBLIC void* highsee_initialize(int int_type, int param, char* p_arg, int pr_index, int hs_index);
HIGHSEE_DLL_PUBLIC int highsee_process(void* p_packet, int type, highsee_output_packet_t* p_packout, void* p_inst);
HIGHSEE_DLL_PUBLIC int highsee_receive(highsee_output_packet_t* p_packout, void* p_inst);
HIGHSEE_DLL_PUBLIC int highsee_getstats(highsee_stats_packet_t* p_stats, void* p_inst);
HIGHSEE_DLL_PUBLIC int highsee_release(void* p_inst);

#ifdef	__cplusplus
}
#endif

#endif	/* HIGHSEE_ADAPTOR_H */

