#!/bin/bash
#
# Check the GCC version and the RPATHs stored inside shared libraries
#
# Usage:
#   ./check-shared-libraries.sh file1.so file2.so ...
#
#
# F. Montorsi
# Nov 2016, Nov 2018
#
set -u

#############################################################################################################
# Constants
#############################################################################################################

declare -r RESTORE=$(echo -en '\033[0m')
declare -r RED=$(echo -en '\033[00;31m')
declare -r GREEN=$(echo -en '\033[00;32m')
declare -r YELLOW=$(echo -en '\033[00;33m')
declare -r WHITE=$(echo -en '\033[01;37m')


#############################################################################################################
# Functions:
#############################################################################################################

function check_elf_file()
{
    local file="$1"
    local nwarnings=0

    # head -n1 is because in some cases (e.g. libUtil.so, several symbols may be present in the .so)
    local GCC_VERSION=$(strings -a $file | grep "GCC: ("| sed -e "s@: (GNU) @@g" | sed -e 's/@//g' | sed -e 's/fCGCC/GCC/g' | head -n1)
    if [[ "$GCC_VERSION" != "$EXPECTED_GCC" ]]; then
        echo "    ${RED}ERROR: File $file does is compiled using $GCC_VERSION (expecting $EXPECTED_GCC). Wrong toolchain?${RESTORE}"
        RETURN_VALUE=2
    fi

    local RPATH=$(patchelf --print-rpath $file)
    if [[ -z "${RPATH}" ]]; then
        echo "    ${YELLOW}WARNING: File $file does NOT have RPATH/RUNPATH tag inside. This may be ok if the ELF linking it has its own RPATH/RUNPATH tags.${RESTORE}"
        (( nwarnings++ ))
    else
        local FOUND_REQUIRED=false
        IFS=':' read -ra RPATHS <<< "$RPATH"
        for path in ${RPATHS[@]}; do
            if [ "$path" = "${RUNTIME_DIR}" ]; then
                FOUND_REQUIRED=true
            elif [ "$path" = "${TOOLLIB_DIR}" ]; then
                # IMPORTANT:
                # This directory is totally unnecessary for production installations. However if it's present, it does not hurt.
                # Since it's difficult to remove it, just ignore it here and don't report a warning for that.
                true
            else
                echo "    ${YELLOW}WARNING: File $file contains unnecessary directory in RPATH: ${WHITE}${path}${RESTORE}"
                (( nwarnings++ ))
            fi
        done

        if ! $FOUND_REQUIRED; then
            echo "    ${RED}ERROR: File $file does NOT have ${RUNTIME_DIR} inside its RPATH/RUNPATH!"
            echo "           Actual RPATH: ${WHITE}${RPATH}${RESTORE}"
            RETURN_VALUE=2
        fi
    fi

    if (( RETURN_VALUE == 0 && nwarnings == 0 )); then
        echo "    ${GREEN}OK: File $file has expected GCC version and all required RPATHs inside. Checks passed.${RESTORE}"
    fi
}

function check_files()
{
    local FILES="$*"

    for file in $FILES; do
        # skip symlinks:
        if [ ! -L "$file" ]; then

            # skip non-ELF:
            file "$file" | grep -q "ELF"
            if [ $? = 0 ]; then
                check_elf_file "$file"
            fi
        fi
    done
}



#############################################################################################################
# Main code
#############################################################################################################

declare DO_CHECK_LIBEMPIRIX=1
declare RETURN_VALUE=0
declare FILES_TO_CHECK="$*"

if [ -z "$FILES_TO_CHECK" ]; then
    FILES_TO_CHECK="*"
fi

# need to know selected toolchain:
if [[  -z "${CROSSTOOLDIR:-}" || -z "${RUNTIME_DIR:-}" || -z "${TOOLCHAIN_GCC_VERSION:-}" ]]; then
    echo "Please export as environment variables: CROSSTOOLDIR, RUNTIME_DIR, TOOLCHAIN_GCC_VERSION"
    exit 1
else
    declare -r TOOLLIB_DIR="$CROSSTOOLDIR/tools/lib"
    declare -r EXPECTED_GCC="GCC${TOOLCHAIN_GCC_VERSION}"
fi

# check only precompiled binaries in specified directories:
check_files "$FILES_TO_CHECK"

# check also /lib/empirix:
if ((DO_CHECK_LIBEMPIRIX)); then
    echo ""
    echo "Now checking ${RUNTIME_DIR} shared libraries as well:"
    echo ""
    check_files "${RUNTIME_DIR}/*so*"
fi

exit ${RETURN_VALUE}

