/*
 * This file is a part of the Qosmos ixEngine.
 *
 * Copyright ENEA Qosmos Division 2000-2022 - All rights reserved
 *
 * This computer program and all its components are protected by
 * authors' rights and copyright law and by international treaties.
 * Any representation, reproduction, distribution or modification
 * of this program or any portion of it is forbidden without
 * Qosmos explicit and written agreement and may result in severe
 * civil and criminal penalties, and will be prosecuted
 * to the maximum extent possible under the law.
*/

#ifndef __QMDPI_H__
#define __QMDPI_H__

#include <stdint.h>

#ifndef _MSC_VER
#include <sys/time.h>
#else
#include <windows.h>
#endif

/* Constants */
#include "qmdpi_const.h"
/* Structures */
#include "qmdpi_struct.h"
/* Engine build informations */
#ifndef QMPROTOCOLS
#include "qmdpi_engine_info.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------------
 * Engine API
 *----------------------------------------------------------------------------*/
/**
 * @file qmdpi.h
 * @brief Main ixEngine API file containing headers of ixEngine functions, along with typedefs, enumerations, and macros.
 *
 * @defgroup engine Engine API
 * @defgroup engine_mgmt Engine Management
 * @ingroup engine
 * @{
 */

/*--------------------------------------
 * Engine Management
 *------------------------------------*/

/**
 * Create an engine instance: initialize and allocate resources for this newly created engine instance using the specified input configuration.
 *
 * @param       config configuration string; if set to NULL, then default values are used.
 * @note The format of the configuration string is as follows:
 * @code
config_param1=config_value1;config_param2=config_value2; @endcode
 *
 * @warning To override ixEngine functions, call ::qmdpi_interface_set before calling ::qmdpi_engine_create.
 * @return      Either:
 * - pointer to the engine instance created by the function
 * - NULL if error (with errno set)
 */
struct qmdpi_engine *qmdpi_engine_create(char const *config);

/**
 * Destroy the specified engine instance.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      An integer:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_engine_destroy(struct qmdpi_engine *engine);

/**
 * Get the version string of the specified engine instance.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - version string if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_engine_version_get_string(struct qmdpi_engine *engine);

/**
 * Get the identifier of the specified engine instance.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - engine instance identifier if success
 * - error code otherwise
 */
int qmdpi_engine_id_get(struct qmdpi_engine *engine);

/**
 * Get the amount of allocated memory **effectively used** within the specified engine instance.
 *
 * @attention The returned allocated size **does not** take into account the **unused** pre-allocated memory.
 *
 * @note For example, if the specified engine instance pre-allocated one 512 MB block and uses half of it then the function will return 256 000 000;
 * whereas if engine pre-allocates one 512 MB block and uses 600 MB then the function will return 600 000 000.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - allocated size if success
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_engine_malloc_get_size(struct qmdpi_engine *engine);

/**
 * Get the total amount of allocated memory within an engine instance
 * independently from being used or not
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - allocated size if success
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_engine_malloc_get_total_size(struct qmdpi_engine *engine);

/**
 * Get the number of flow contexts (qmdpi_flow) referenced by all the bundle instances
 * attached to the specified engine instance.
 *
 * @note useful for getting the number of flows in STREAM MODE.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - flow count if success
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_engine_flow_get_count(struct qmdpi_engine *engine);

/**
 * Get the number of flow contexts (qmdpi_flow) present in the flow tables of
 * the specified engine instance (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - flow count if success
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_engine_flow_table_get_count(struct qmdpi_engine *engine);

/**
 * Get the number of allocated flow contexts for the specified engine instance (PACKET MODE ONLY).
 * @warning     Packet-mode only.
 *
 * @note this is only useful when flow contexts are preallocated at init
 * (see fm_flow_table_alloc_mode config parameter for more details),
 * in this case flows may be allocated but not used (i.e. pre-allocated).
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - flow count if success
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_engine_flow_table_allocated_get_count(struct qmdpi_engine
        *engine);

/**
 * Get the number of pre-allocated elements in the specified pool type in the specified engine instance.
 *
 * @param       engine pointer to a valid engine instance.
 * @param       pool_type type of pool.
 * @return      One of the following:
 * - number of pre-allocated elements if success
 * - 0 if error (with errno set)
 */
uint32_t qmdpi_engine_pool_get_count(struct qmdpi_engine *engine,
                                     int pool_type);

/**
 * Get the size of the element for the specified pool type in the specified engine instance.
 *
 * @param       engine pointer to a valid engine instance.
 * @param       pool_type type of pool.
 * @return      One of the following:
 * - element size if success
 * - 0 if error (with errno set)
 */
uint32_t qmdpi_engine_pool_get_size(struct qmdpi_engine *engine, int pool_type);

/**
 * Get the number of elements effectively used in the specified pool type in the specified engine instance.
 *
 * @param       engine pointer to a valid engine instance.
 * @param       pool_type type of pool.
 * @return      One of the following:
 * - number of currently used elements in the pool allocated for the object
 * engine if success
 * - 0 if error (with errno set)
 */
uint32_t qmdpi_engine_pool_get_used(struct qmdpi_engine *engine, int pool_type);

/** @} */

/*--------------------------------------
 * Engine configuration
 *------------------------------------*/

/**
 * @defgroup engine_config Engine Configuration
 * @ingroup engine
 * @{
 */

/**
 * Set engine configuration.
 *
 * @param       engine pointer to a valid engine instance.
 * @param       param configuration parameter.
 * @param       val parameter value.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_engine_config_set(struct qmdpi_engine *engine, const char *param,
                            uint64_t val);

/**
 * Get the value of the specified engine configuration parameter.
 *
 * @param       engine pointer to a valid engine instance.
 * @param       param configuration parameter.
 * @return      One of the following:
 * - parameter value if success
 * - error code otherwise
 */
int64_t qmdpi_engine_config_get(struct qmdpi_engine *engine,
                                const char *param);

/**
 * Output function invoked by the dump functions.
 *
 * @param       output Pointer to output stream.
 * @param       format Format of output string (Printf style).
 * @return      Return value defined by function.
 */
typedef int (*qmdpi_output_fn_t)(void *output, const char *format, ...);

/**
 * Dump the specified engine's configuration to the specified output stream.
 *
 * @param       engine Pointer to a valid engine instance.
 * @param       output Pointer to output stream.
 * @param       output_fn Output function.
 * @return      One of the following:
 * - return value of the output function
 * - error code otherwise
 */
int qmdpi_engine_config_dump(struct qmdpi_engine *engine, void *output,
                             qmdpi_output_fn_t output_fn);

/**
 * Set a default port for a given protocol in flow_manager classification
 * **for any engine instance** (global configuration).
 * @param   proto_id Protocol whose port will be set, valid values are:
 * - 0 (to reset a protocol for a given port)
 * - Q_PROTO_CAPWAP
 * - Q_PROTO_DNS
 * - Q_PROTO_GTP
 * - Q_PROTO_L2TP
 * - Q_PROTO_MOBILE_IP
 * - Q_PROTO_NSH
 * - Q_PROTO_VXLAN
 * - Q_PROTO_TEREDO
 * @param   l4_proto    Transport layer (only Q_PROTO_UDP is valid for now).
 * @param   port        Port (in host endianness).
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_flow_table_set_port(int proto_id, int l4_proto, uint16_t port);


/**
 * For the specified virtual network identifier,
 * set the default port for a given protocol in flow_manager classification
 * **for any engine instance** (global configuration).
 * @param   engine pointer to a valid engine instance
 * @param   proto_id    Protocol whose port will be set. Valid values are:
 * - 0 (to reset a protocol for a given port)
 * - Q_PROTO_CAPWAP
 * - Q_PROTO_DNS
 * - Q_PROTO_GTP
 * - Q_PROTO_L2TP
 * - Q_PROTO_MOBILE_IP
 * - Q_PROTO_NSH
 * - Q_PROTO_VXLAN
 * - Q_PROTO_TEREDO
 * @param   l4_proto    Transport layer (only Q_PROTO_UDP is valid for now)
 * @param   port        Port (in host endianness)
 * @param   virtual_id virtual network identifier
 * @return  0 if success,\n
 *          error code otherwise
 */
int32_t qmdpi_flow_table_set_port_byvid(struct qmdpi_engine *engine,
                                        int32_t proto_id,
                                        int32_t l4_proto, uint16_t port, int32_t virtual_id);

/** @} */

/*--------------------------------------
 * Statistics monitoring
 *------------------------------------*/

/**
 * @defgroup engine_stats Engine Statistics Monitoring
 * @ingroup engine
 * @{
 */

/**
 * To be called before stats initialization starts.
 * @param       engine pointer to a valid engine instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_stats_init_start(struct qmdpi_engine *engine);

/**
 * To be called after stats initialization is done.
 * @param       engine pointer to a valid engine instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_stats_init_end(struct qmdpi_engine *engine);

/**
 * To be called before the exit of the application
 * @param       engine pointer to a valid engine instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_stats_exit(struct qmdpi_engine *engine);

/**
 * Dump @a engine statistics to @a output and invoke the @a output_fn function.
 * @param       engine pointer to a valid engine instance
 * @param       output pointer to output stream
 * @param       output_fn output function
 * @return      One of the following:
 * - return value of the output function
 * - error code otherwise
 */
int qmdpi_stats_dump(struct qmdpi_engine *engine, void *output,
                     qmdpi_output_fn_t output_fn);

/** @} */

/*------------------------------------------------------------------------------
 * Layer store management API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup engine_lstore Layer Store Management
 * @ingroup engine
 * @{
 */

/**
 * Get maximum size for the specified layer store.
 * @param       engine pointer to a valid engine instance
 * @param       store_name name of the layer store
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_layer_store_get_maxsize(struct qmdpi_engine *engine,
                                  char const *store_name);

/**
 * Set maximum size for the specified layer store.
 * @param       engine pointer to a valid engine instance
 * @param       store_name name of the layer store
 * @param       store_size maximum size of the layer store
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_layer_store_set_maxsize(struct qmdpi_engine *engine,
                                  char const *store_name, uint32_t store_size);

/**
 * Destroy (free) all entries or all expired entries of the specified layer store.
 * @param       worker Pointer to worker instance.
 * @param       store_name Name of the layer store.
 * @param       expired Allowed values:
 * - 0: Free all entries.
 * - 1: Free expired entries only.
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_layer_store_flush(struct qmdpi_worker *worker,
                            char const *store_name, int expired);

/**
 * Print info on the specified layer store.
 * @param       engine pointer to a valid engine instance
 * @param       store_name name of the layer store
 * @param       f user-specific data passed to print function
 * @param       printfn print function
 * @param       prefix prefix to prepend to each output line
 * @param       level level of information requested:
 *                      - 0: Global statistics
 *                      - 1: Hash table line statistics
 *                      - 2: Hash table node information
 *                      - 3: Hash table node content
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_layer_store_print_info(struct qmdpi_engine *engine,
                                 char const *store_name,
                                 void *f, int (*printfn)(void *f, char const *fmt, ...),
                                 char const *prefix, unsigned int level);

/**
 * Build a list of @a layer @a store @a IDs.
 * The user should call ::qmdpi_layer_store_destroy_list to free all memory allocated by this function.
 * @param engine pointer to engine instance
 * @param l is set to the layer store list.
 * @return      One of the following:
 * - Number of elements in the list
 * - error code otherwise
 */
int qmdpi_layer_store_build_list(struct qmdpi_engine *engine, char ***l);

/**
 * Destroy a layer store ID list.
 *
 * @param engine pointer to engine instance
 * @param l the NULL-terminated @a layer @a store @a ID list to destroy
 */
void qmdpi_layer_store_destroy_list(struct qmdpi_engine *engine, char **l);

/**
 * Callback prototype to be called when a layer store entry is created/updated.
 * @param store_name Name of layer store.
 * @param entry Serialized layer store entry.
 * @param entry_len Serialized layer store entry's length.
 * @param arg Callback argument.
 */
typedef void (*qmdpi_layer_store_on_update_entry_callback_t)(
    const char *store_name, char *entry, int entry_len, void *arg);

/**
 * Register an action to be performed when a layer store entry is created/updated.
 * @note Supported store_name : "imap_ip4_uid_store", "imap_ip6_uid_store", "uclassification_cache_ip4_store", "uclassification_cache_ip4_store".
 *
 * @param engine Pointer to engine instance.
 * @param store_name Name of the layer store.
 * @param callback Callback to be executed on new entry.
 * @param arg Callback argument.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_layer_store_on_update_entry_callback_set(struct qmdpi_engine *engine,
        const char *store_name,
        qmdpi_layer_store_on_update_entry_callback_t callback, void *arg);

/**
 * Insert/update an entry into layer store using serialized entry
 * (layer store name is stored in the entry).
 * @param worker Pointer to worker instance.
 * @param serialized_entry Serialized layer store entry.
 * @param entry_len Serialized layer store entry's length.
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_layer_store_insert_entry(struct qmdpi_worker *worker,
                                   void *serialized_entry,
                                   int entry_len);

/**
 * Dump layer store entries for a given layer store.
 *
 * @param       worker Pointer to a valid worker instance.
 * @param       store_name Name of the layer store.
 * @param       output Pointer to output buffer.
 * @param       output_fn Output function.
 * @param       header If set to 1, adds a header line at the top of the dump.
 * @return      One of the following:
 * - return value of the output function,
 * - error code otherwise.
 */
int qmdpi_layer_store_dump(struct qmdpi_worker *worker,
                           char const *store_name,
                           void *output,
                           qmdpi_output_fn_t output_fn,
                           uint8_t header);

/**
 * Set the lifetime of a layer store entry.
 *
 * @param       engine Pointer to a valid engine instance.
 * @param       store_name Name of the layer store.
 * @param       lifetime Lifetime of a ulayer store entry.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_layer_store_lifetime_set(struct qmdpi_engine *engine,
                                   char const *store_name,
                                   uint32_t lifetime);

/**
 * Get the lifetime of a layer store entry.
 * @param       engine Pointer to a valid engine instance.
 * @param       store_name Name of the layer store.
 * @param       lifetime Lifetime of a ulayer store entry.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_layer_store_lifetime_get(struct qmdpi_engine *engine,
                                   char const *store_name, uint32_t *lifetime);

/** @} */

/*------------------------------------------------------------------------------
 * Worker API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup worker Worker API
 */

/**
 * @defgroup worker_mgmt Worker Management
 * @ingroup worker
 * @{
 */

/*--------------------------------------
 * Worker management
 *------------------------------------*/

/**
 * Create worker instance for the specified engine instance:
 * initialize and allocate resources for the worker instance,
 * according to engine configuration.
 *
 * @param       engine pointer to a valid engine instance.
 * @return      One of the following:
 * - a pointer to the created worker instance
 * - NULL if error (with errno set)
 */
struct qmdpi_worker *qmdpi_worker_create(struct qmdpi_engine *engine);

/**
 * Destroy the specified worker instance:
 * free resources for an engine instance, according to input configuration.
 *
 * @param       worker pointer to a valid worker instance.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_destroy(struct qmdpi_worker *worker);

/**
 * Get the identifier of the specified worker.
 *
 * @param       worker pointer to a valid worker instance.
 * @return      One of the following:
 * - worker instance id if success
 * - error code otherwise
 */
int qmdpi_worker_id_get(struct qmdpi_worker *worker);

/**
 * Get the engine instance bound to the specified worker.
 *
 * @param       worker pointer to a valid worker instance.
 * @return      One of the following:
 * - pointer to active worker instance
 * - NULL if error (with errno set)
 */
struct qmdpi_engine *qmdpi_worker_engine_get(struct qmdpi_worker *worker);

/**
 * Callback: invoked by qmdpi_worker_flow_foreach.
 *
 * @param       worker pointer to a valid worker instance.
 * @param       flow pointer to a flow context
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_worker_flow_callback_t)(struct qmdpi_worker *worker,
        struct qmdpi_flow *flow,
        void *arg);

/**
 * Call the specified callback for each flow context in the specified worker (PACKET MODE ONLY).
 * @warning     Packet-mode only.
 * @param       worker pointer to a valid worker instance.
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_flow_foreach(struct qmdpi_worker *worker,
                              qmdpi_worker_flow_callback_t callback,
                              void *arg);

/**
 * Dump the flow table (PACKET MODE ONLY).
 * @warning     Packet-mode only.
 * @param       worker pointer to a valid worker instance.
 * @param       output pointer to output stream
 * @param       output_fn output function
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_flow_table_dump(struct qmdpi_worker *worker, void *output,
                                 qmdpi_output_fn_t output_fn);

/** @} */

/*--------------------------------------
 * Worker stats
 *------------------------------------*/

/**
 * Get the current number of allocated flow contexts in the flow table
 * of the specified worker instance (PACKET MODE ONLY).
 *
 * @note This is only useful when flow contexts are preallocated at init
 * (see fm_flow_table_alloc_mode config parameter for more details). In
 * this case, flows may be allocated but not used (i.e. pre-allocated).
 *
 * @warning     Packet-mode only.
 *
 * @param       worker pointer to a valid worker instance
 * @return      One of the following:
 * - flow count if success
 * - errno set if error
 */
uint64_t qmdpi_worker_flow_table_allocated_get_count(struct qmdpi_worker
        *worker);

/**
 * Get the current number of used flow contexts in the flow table
 * of the specified worker instance (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 *
 * @param       worker pointer to a valid worker instance.
 * @return      One of the following:
 * - flow count if success
 * - errno set if error
 */
uint64_t qmdpi_worker_flow_table_get_count(struct qmdpi_worker *worker);

/** @} */

/*--------------------------------------
 * PDU setup / processing
 *------------------------------------*/

/**
 * @defgroup worker_pdu PDU Setup and Processing
 * @ingroup worker
 * @{
 */

/**
 * Execute DPI processing of the specified PDU.
 *
 * @param       worker pointer to a valid worker instance.
 * @param       flow pointer to the flow to which this PDU belongs; In @a PACKET @a MODE, this parameter is ignored.
 * @param       result pointer to DPI result.
 * @return      One of the following:
 * - 0 if success
 * - an integer > 0 if something happenned during Flow Manager stage (see qmdpi_const.h for QMDPI_PROCESS_* values)
 * - error code otherwise
 */
int qmdpi_worker_process(struct qmdpi_worker *worker,
                         struct qmdpi_flow *flow,
                         struct qmdpi_result **result);

/**
 * Set PDU instance input parameters; set information about PDU for DPI processing.
 *
 * - PACKET MODE: @a dir represents the "way" in which the traffic is flowing (incoming or outgoing).
 * - STREAM MODE: @a dir is the direction of the packet relative to the 2 network peers involved in the communication, can be client-to-server (QMDPI_DIR_CTS) or server-to-client (QMDPI_DIR_STC).
 *
 * @note To classify IPSEC over UDP, or Skype over TCP, some information must be obtained from L3-L4 headers, so users who wish to classify these must call ::qmdpi_worker_pdu_header_set right after calling this function.
 *
 * @param       worker pointer to worker.
 * @param       data PDU data.
 * @param       data_len data length.
 * @param       ts packet timestamp.
 * @param       first_header protocol of first header (STREAM MODE: this parameter is ignored).
 * @param       dir direction of the packet (Note: meaning differs depending on injection mode).
 * @param       virtual_id formerly virtual network identifier (optional), used to avoid overlapping of flows (i.e. flows having the same N-tuples but coming from different Networks).
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_pdu_set(struct qmdpi_worker *worker, void const *data,
                         int data_len,
                         struct timeval const *ts, int first_header,
                         int dir, int virtual_id);

/**
 * Set timestamp for current worker.
 *
 * @param       worker Pointer to a valid worker instance.
 * @param       tv Timestamp value.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_timestamp_set(struct qmdpi_worker *worker,
                               const struct timeval *tv);

/**
 * Return the pointer and the length of the payload of the PDU.
 *
 * @param       worker pointer to worker instance
 * @param       data a pointer to the pointer to the payload
 * @param       data_len a pointer to the payload length
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_stream_get(struct qmdpi_worker *worker, void const **data,
                            int *data_len);

/**
 * Dump PDU n-tuple to @a output and invoke the @a output_fn function (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to a valid worker instance
 * @param       output pointer to output stream
 * @param       output_fn output function
 * @return      One of the following:
 * - return value of the output function
 * - error code otherwise
 */
int qmdpi_worker_pdu_ntuple_dump(struct qmdpi_worker *worker, void *output,
                                 qmdpi_output_fn_t output_fn);

/**
 * Get n-th protocol of n-tuple (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to a worker instance
 * @param       index index of requested protocol
 * @return      One of the following:
 * - protocol ID if success
 * - error code otherwise
 */
int qmdpi_worker_pdu_ntuple_get_proto(struct qmdpi_worker *worker, int index);

/**
 * Get n-th header of n-tuple (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to a worker instance
 * @param       index index of requested header
 * @return      One of the following:
 * - pointer to header if success
 * - NULL if error (with errno set)
 */
void *qmdpi_worker_pdu_ntuple_get_header(struct qmdpi_worker *worker,
        int index);

/**
 * Get number of elements in PDU n-tuple (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to a worker instance
 * @return      One of the following:
 * - number of elements in n-tuple
 * - error code otherwise
 */
int qmdpi_worker_pdu_ntuple_get_size(struct qmdpi_worker *worker);


/**
 * Set l3/l4 headers for PDU (STREAM MODE ONLY).
 * The headers are those of inner most headers.
 *
 * @param       worker pointer to a worker instance
 * @param       l3_header pointer to l3 header (e.g. IP or IP6)
 * @param       l4_header pointer to l4 header (e.g. TCP, UDP or SCTP)
 *
 * @note See also ::qmdpi_worker_pdu_set
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_pdu_header_set(struct qmdpi_worker *worker,
                                void *l3_header, void *l4_header);

/**
 * Set the classification of a flow context associated with previously set PDU.
 *
 * @param       worker pointer to a valid worker instance.
 * @param       flow pointer to the flow to which the PDU belongs;
                In @a PACKET @a MODE, this parameter is ignored and classification is set on the flow currently processed by the worker.
 * @param       path_index path_index where classification update should begin.
 * @param       path_array array of proto_id to add.
 * @param       path_array_len the number of proto_id's to add.
 * @param       classified set the classified status of the flow. 1 to mark the flow as fully classified, 0 otherwise.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_classification_set(struct qmdpi_worker *worker,
                                    struct qmdpi_flow *flow,
                                    int path_index,
                                    int path_array[],
                                    int path_array_len,
                                    int classified);

/**
 * Get the qmdpi_flow associated with the provided PDU (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to worker.
 * @param       data PDU data.
 * @param       data_len data length.
 * @param       first_header protocol of first header.
 * @param       virtual_id formerly virtual network identifier (optional), used to avoid overlapping of flows (i.e. flows having the same N-tuples but coming from different Networks).
 *
 * @return      One of the following:
 * - pointer to the flow,
 * - NULL if error (with errno set)
 */
struct qmdpi_flow *qmdpi_flow_get_bypdu(struct qmdpi_worker *worker,
                                        void const *data,
                                        int data_len,
                                        int first_header,
                                        int virtual_id);

/** @} */

/*------------------------------------------------------------------------------
 * Bundle API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup bundle Bundle API
 */

/**
 * @defgroup bundle_mgmt Bundle Management
 * @ingroup bundle
 * @{
 */

/*--------------------------------------
 * Bundle Management
 *------------------------------------*/

/**
 * Create bundle instance from file:
 * load shared library object, initialize bundle and attach it to the specified engine.
 *
 * @param       engine pointer to a valid engine instance
 * @param       filename filename of shared library object to load;
 * if set to NULL, the bundle instance gets created from the library linked with
 * the user application.
 * @return      One of the following:
 * - a pointer to created bundle instance
 * - NULL if error (with errno set)
 */
struct qmdpi_bundle *qmdpi_bundle_create_from_file(struct qmdpi_engine *engine,
        const char *filename);

/**
 * Callback: called when a bundle is destroyed.
 *
 * @param       engine pointer to a valid engine instance
 * @param       arg callback argument
 */
typedef void (*qmdpi_bundle_destroy_callback_t)(struct qmdpi_engine *engine,
        void *arg);

/**
 * Set the callback that will be invoked when the specified bundle is destroyed.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       destroy_callback the callback called when bundle is destroyed
 * @param       arg argument to be called in the callback (e.g. for tracking bundle destruction)
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_destroy_callback_set(struct qmdpi_bundle *bundle,
                                      qmdpi_bundle_destroy_callback_t destroy_callback,
                                      void *arg);

/**
 * Mark the specified bundle as "to be destroyed".
 * Bundle will be destroyed when all of its flow contexts have expired.
 * Existing flow contexts used by a bundle marked as "to be destroyed" are expired
 * by calling ::qmdpi_flow_expire_next.
 *
 * @note It is recommended to unregister all the attributes that have been
 * registered for extraction, via ::qmdpi_bundle_attr_unregister,
 * before destroying the Protocol Bundle.
 *
 * @warning If automatic bundle removal is disabled (bundle_automgmt_enable=0),
 * then ::qmdpi_bundle_destroy should not be called if there are any flows
 * still referenced by the specified bundle instance.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_destroy(struct qmdpi_bundle *bundle);

/**
 * Activate bundle instance.
 * This will mark this bundle as active in the attached engine instance;
 * all new flows will use this bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_activate(struct qmdpi_bundle *bundle);

/**
 * Whether or not the specified bundle is active.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - 0 if inactive
 * - 1 if active
 * - error code otherwise
 */
int qmdpi_bundle_is_active(struct qmdpi_bundle *bundle);

/**
 * Get the active bundle instance for the specified engine instance.

 * @param       engine pointer to a valid engine instance
 * @return      One of the following:
 * - a pointer to the active bundle instance
 * - NULL if error (with errno set)
 */
struct qmdpi_bundle *qmdpi_bundle_get_active(struct qmdpi_engine *engine);

/**
 * Get the version string of the specified bundle instance.
 *
 * @param       bundle pointer to a valid engine instance
 * @return      One of the following:
 * - version string if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_bundle_version_get_string(struct qmdpi_bundle *bundle);

/**
 * Get the identifier of the specified bundle instance.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - bundle instance ID if success
 * - error code otherwise
 */
int qmdpi_bundle_id_get(struct qmdpi_bundle *bundle);

/**
 * Get count of flows referenced by the specified bundle instance.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - number of active flow contexts for that bundle
 * - error code otherwise
 */
int64_t qmdpi_bundle_flow_get_count(struct qmdpi_bundle *bundle);

/**
 * Add an entry to the DNS Cache of the specified bundle instance
 * if the hostname matches a pdata rule concerning DNS matching.
 *
 * @param       bundle Pointer to a valid bundle instance.
 * @param       clientl3proto Client layer 3 protocol (can be Q_PROTO_IP or Q_PROTO_IP6).
 * @param       clientl3addr IP Address of the client issuing the DNS request in Network Byte Order.
 * @param       hostl3proto Host layer 3 protocol (can be Q_PROTO_IP or Q_PROTO_IP6).
 * @param       hostl3addr IP Address of the resolved hostname in Network Byte Order.
 * @param       hostname NULL-terminated name of host.
 * @param       ttl How long the entry should stay valid, in seconds.
 * @param       tv The current time.
 * @param       vid Virtual Network Identifier (0 if not applicable).
 * @return      One of the following:
 * - 0 if success,
 * - QMDPI_EBUNDLE_RULE_DOES_NOT_EXIST if hostname did not match against any PDATA signature,
 * - another error code otherwise.
 */
int qmdpi_bundle_dns_cache_add(struct qmdpi_bundle *bundle,
                               int clientl3proto,
                               void *clientl3addr,
                               int hostl3proto,
                               void *hostl3addr,
                               char *hostname,
                               int ttl,
                               struct timeval *tv,
                               uint32_t vid);

/**
 * Dump the content of the DNS Cache of the specified bundle instance.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       output pointer to output stream
 * @param       output_fn output function
 * @param       tv is the current time
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_dns_cache_dump(struct qmdpi_bundle *bundle, void *output,
                                qmdpi_output_fn_t output_fn, struct timeval *tv);

/** @} */

/*--------------------------------------
 * Bundle Configuration
 *------------------------------------*/

/**
 * @defgroup bundle_config Bundle Configuration
 * @ingroup bundle
 * @{
 */

/**
 * Register attribute for extraction.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature signature name
 * @param       attr attribute name or the rule for a user-defined attribute
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_attr_register(struct qmdpi_bundle *bundle,
                               char const *signature, char const *attr);

/**
 * Unregister the specified attribute.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature signature name
 * @param       attr attribute name or the rule for a user-defined attribute
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
*/
int qmdpi_bundle_attr_unregister(struct qmdpi_bundle *bundle,
                                 char const *signature, char const *attr);
/**
 * Register attribute for extraction.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       Attr_list string containing the list of attributes to register;
 * @note The format of the attr_list string is as follows:
 * @code protocol_name:attribute_name;protocol_name:attribute_name; @endcode
 * @param       usage type of usage (0 for standard mode, 1 for attributes needed by the IDS Rule Engine)
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_attr_list_register(struct qmdpi_bundle *bundle,
                                    char const *attr_list,
                                    int usage);

/**
 * Unregister attribute for extraction.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       Attr_list string containing the list of attributes to unregister;
 * @note The format of the attr_list string is as follows:
 * @code protocol_name:attribute_name;protocol_name:attribute_name; @endcode
 * @param       usage type of usage (0 for standard mode, 1 for attributes needed by the IDS Rule Engine)
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_attr_list_unregister(struct qmdpi_bundle *bundle,
                                      char const *attr_list,
                                      int usage);

/**
 * Get the IDS metadata buffer to be fed into the IDS Rule Engine.
 *
 * @param       worker pointer to a valid worker instance.
 * @param       ids_metadata_buf pointer to IDS metadata buffer.
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */

int qmdpi_worker_ids_metadata_buf_get(struct qmdpi_worker *worker,
                                      void **ids_metadata_buf);


/* Classification methods */
enum {
    QMDPI_CLASSIF_METHOD_IMPLICIT = 0,    /*!< is_proto, AKA Pattern Matching */
    QMDPI_CLASSIF_METHOD_EXPLICIT,        /*!< get_upper + flow manager, AKA Explicit */
    QMDPI_CLASSIF_METHOD_PDATA,           /*!< PDATA, AKA Protocol Data Signature */
    QMDPI_CLASSIF_METHOD_PDATA_IP,        /*!< PDATA (from ip address) */
    QMDPI_CLASSIF_METHOD_PDATA_DNS,       /*!< PDATA (from dns cache), AKA DNS Caching */
    QMDPI_CLASSIF_METHOD_PREDICTIVE,      /*!< l3l4 cache, dns cache, AKA Session Correlation */
    QMDPI_CLASSIF_METHOD_SPID,            /*!< packet statistics / spid, AKA Statistical Protocol Identification */
    QMDPI_CLASSIF_METHOD_BEHAVIORAL,      /*!< behavioral, AKA Session Behavior */
    QMDPI_CLASSIF_METHOD_PORT_BASED,      /*!< port-based, AKA Port-based classification over SSL */
    QMDPI_CLASSIF_METHOD_DOMAIN_FRONTING, /*!< domain fronting detection techniques */
};

/**
 * Register a generic attribute for all layers.
 * @warning     As of ixEngine 5.3.0, only the attribute "classification_method" is supported.
 * @warning     This function is deprecated; generic attributes can now be registered via the function ::qmdpi_bundle_attr_register.
 *              e.g: qmdpi_bundle_attr_register(bundle, NULL, "classification_method")
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       gen_attr generic attribute name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_gen_attr_register(struct qmdpi_bundle *bundle,
                                   char const *gen_attr);


/**
 * Unregister a generic attribute for all layers.
 * @warning     As of ixEngine 5.3.0, only the attribute "classification_method" is supported.
 * @warning     This function is deprecated; generic attributes can now be unregistered via the function ::qmdpi_bundle_attr_unregister.
 *              e.g: qmdpi_bundle_attr_unregister(bundle, NULL, "classification_method")
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       gen_attr generic attribute name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_gen_attr_unregister(struct qmdpi_bundle *bundle,
                                     char const *gen_attr);

/**
 * Operations on Signatures (AKA Protocols Plug-ins).
 */

/**
 * Enable all signatures for the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_enable_all(struct qmdpi_bundle *bundle);

/**
 * Disable all signatures for the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
*/
int qmdpi_bundle_signature_disable_all(struct qmdpi_bundle *bundle);

/**
 * Enable the specified signature for the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_enable(struct qmdpi_bundle *bundle,
                                  const char *signature_name);

/**
 * Disable the specified signature for the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_disable(struct qmdpi_bundle *bundle,
                                   const char *signature_name);

/**
 * Whether a signature enabled in the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - 0 if not enabled, 1 if enabled
 * - error code otherwise
 */
int qmdpi_bundle_signature_is_enabled(struct qmdpi_bundle *bundle,
                                      const char *signature_name);

/**
 * In the specified bundle, enable all signatures having the specified tag.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       tag_name tag name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_enable_bytag(struct qmdpi_bundle *bundle,
                                        const char *tag_name);

/**
 * In the specified bundle, disable all signatures having the specified tag.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       tag_name tag name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_disable_bytag(struct qmdpi_bundle *bundle,
        const char *tag_name);

/**
 * In the specified bundle, enable all signatures belonging to the specified family.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       family_name family name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_enable_byfamily(struct qmdpi_bundle *bundle,
        const char *family_name);
/**
 * In the specified bundle, disable all signatures belonging to the specified family.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       family_name family name
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_disable_byfamily(struct qmdpi_bundle *bundle,
        const char *family_name);
/**
 * In the specified bundle, set specific parameters by running an ioctl command.
 * @warning This function is deprecated; the new version of this function is ::qmdpi_worker_signature_ctl_set.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature signature name
 * @param       code code command
 * @param       arg pointer to parameter to set
 * @param       arg_len length of parameter
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_ctl_set(struct qmdpi_bundle *bundle,
                                   const char *signature, int code,
                                   void *arg, int arg_len);
/**
 * In the specified bundle, set specific parameters by running an ioctl command.
 *
 * @param       worker pointer to a valid worker
 * @param       bundle pointer to a valid bundle instance
 * @param       signature signature name
 * @param       code code command
 * @param       arg pointer to parameter to set
 * @param       arg_len length of parameter
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_signature_ctl_set(struct qmdpi_worker *worker,
                                   struct qmdpi_bundle *bundle,
                                   const char *signature, int code,
                                   void *arg, int arg_len);

/**
 * In the specified bundle, get specific parameters by running an ioctl command.
 * @warning This function is deprecated; the new version of this function is ::qmdpi_worker_signature_ctl_get.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature signature name
 * @param       code code command
 * @param       arg pointer to parameter to get
 * @param       arg_len of parameter
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_ctl_get(struct qmdpi_bundle *bundle,
                                   const char *signature, int code,
                                   void *arg, int arg_len);

/**
 * In the specified bundle, get the upper layers of a specified protocol.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       proto_id protocol identifier
 * @param       upper_array referece to an array of (uint32_t) protocol identifiers
 * @param       upper_array_len length of the array of protocol identifiers
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_uppers_get(struct qmdpi_bundle *bundle,
                                      uint32_t             proto_id,
                                      const uint32_t     **upper_array,
                                      uint32_t            *upper_array_len);


/**
 * In the specified bundle, get specific parameters by running an ioctl command.
 *
 * @param       worker pointer to a valid worker
 * @param       bundle pointer to a valid bundle instance
 * @param       signature signature name
 * @param       code code command
 * @param       arg pointer to parameter to get
 * @param       arg_len of parameter
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_worker_signature_ctl_get(struct qmdpi_worker *worker,
                                   struct qmdpi_bundle *bundle,
                                   const char *signature, int code,
                                   void *arg, int arg_len);

/** @} */

/*--------------------------------------
 * Upper layer signature
 *------------------------------------*/
/**
 * @defgroup bundle_upper Upper Layer Signature
 * @ingroup bundle_config
 * @{
 */

/**
 * Add a signature for a layer above a specific layer (currently only HTTP or HTTPS)
 * for the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       bottom_signature_name signature name of the bottom layer
 * @param       signature_name name of the signature to be added
 * @param       signature_long_name long name of the signature to be added
 * @param       signature_id id of the signature to be added
 * @param       pattern pattern string to match
 * @return      One of the following:
 * - ID of the created signature
 * - error code otherwise
 */
int qmdpi_bundle_signature_upper_add(struct qmdpi_bundle *bundle,
                                     const char *bottom_signature_name,
                                     const char *signature_name,
                                     const char *signature_long_name,
                                     int         signature_id,
                                     const char *pattern);

/**
 * Remove an upper layer signature for the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name name of the signature to be deleted
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_upper_del(struct qmdpi_bundle *bundle,
                                     const char *signature_name);

/**
 * Dump upper layer signatures for the specified bundle to @a output, and
 * invoke the @a output_fn function.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       bottom_signature_name signature name of the bottom layer
 * @param       output pointer to output stream
 * @param       output_fn output function to be called for each upper layer signatures
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_upper_dump(struct qmdpi_bundle *bundle,
                                      const char *bottom_signature_name,
                                      void *output,
                                      qmdpi_output_fn_t output_fn);

/** @} */

/*--------------------------------------
 * Bundle info
 *------------------------------------*/
/**
 * @defgroup bundle_info Bundle Information
 * @ingroup bundle
 * @{
 */

/*--------------------------------------
 * Signature info
 *------------------------------------*/
/**
 * @defgroup sig_info Signature Information
 * @ingroup bundle_info
 * @{
 */

/**
 * Callback invoked by ::qmdpi_bundle_signature_foreach, ::qmdpi_bundle_signature_foreach_enabled, ::qmdpi_bundle_signature_foreach_permitted.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_signature_callback_t)(struct qmdpi_bundle *bundle,
        struct qmdpi_signature *sig,
        void *arg);

/**
 * Callback invoked by ::qmdpi_bundle_signature_foreach_by_mpa, ::qmdpi_bundle_signature_foreach_enabled_by_mpa, ::qmdpi_bundle_signature_foreach_permitted_by_mpa.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @param       attr pointer to a valid attribute
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_signature_attr_callback_t)(struct qmdpi_bundle
        *bundle,
        struct qmdpi_signature *sig,
        struct qmdpi_attr *attr,
        void *arg);

/**
 * Call the specified callback for each signature in the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_foreach(struct qmdpi_bundle *bundle,
                                   qmdpi_bundle_signature_callback_t callback,
                                   void *arg);

/**
 * Call the specified callback for each enabled signature in the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_foreach_enabled(struct qmdpi_bundle *bundle,
        qmdpi_bundle_signature_callback_t callback,
        void *arg);


/**
 * Call the specified callback for each signature covered by the license in the specified bundle.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_foreach_permitted(struct qmdpi_bundle *bundle,
        qmdpi_bundle_signature_callback_t callback,
        void *arg);


/**
 * Call the specified callback for each signature that has the specified
 * multi-protocol attribute (MPA).
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       attribute Multi-Protocol Attribute MPA
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_foreach_by_mpa(struct qmdpi_bundle *bundle,
        char const *attribute,
        qmdpi_bundle_signature_attr_callback_t callback,
        void *arg);

/**
 * Call the specified callback for each enabled signature that has the specified
 * multi-protocol attribute (MPA).
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       attribute Multi-Protocol Attribute MPA
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_foreach_enabled_by_mpa(struct qmdpi_bundle *bundle,
        char const *attribute,
        qmdpi_bundle_signature_attr_callback_t callback,
        void *arg);


/**
 * Call the specified callback for each permitted signature that has the specified
 * multi-protocol attribute (MPA).
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       attribute Multi-Protocol Attribute MPA
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_foreach_permitted_by_mpa(struct qmdpi_bundle *bundle,
        char const *attribute,
        qmdpi_bundle_signature_attr_callback_t callback,
        void *arg);


/**
 * Get signature name.
 *
 * @param       sig pointer to a valid signature
 * @return      signature name string, errno set if error.
 */
const char *qmdpi_signature_name_get(struct qmdpi_signature *sig);

/**
 * Get signature ID.
 *
 * @param       sig pointer to a valid signature
 * @return      signature ID,\n
 *              errno set if error
 */
int qmdpi_signature_id_get(struct qmdpi_signature *sig);

/**
 * Get signature long name.
 *
 * @param       sig pointer to a valid signature
 * @return      signature long name string, errno set if error.
 */
const char *qmdpi_signature_longname_get(struct qmdpi_signature *sig);

/**
 * Get signature family name.
 *
 * @param       sig pointer to a valid signature
 * @return      signature family string, errno set if error.
 */
const char *qmdpi_signature_family_get(struct qmdpi_signature *sig);

/**
 * Get signature family ID.
 *
 * @param       sig pointer to a valid signature
 * @return      signature family ID, errno set if error.
 */
int qmdpi_signature_family_id_get(struct qmdpi_signature *sig);

/**
 * Get signature tag bitfield.
 * To test if a tag is set, use ::QMDPI_TAG_ISSET(tag_id, bitfield).
 * Tag ID list is available in protodef.h header file.
 *
 * @param       sig pointer to a valid signature
 * @return      signature tags bitfield, errno set if error.
 */
uint64_t qmdpi_signature_tags_get(struct qmdpi_signature *sig);

/**
 * Set custom tags for a signature. Can only be called for signatures which have been enabled.
 *
 *
 * @param      bundle pointer to a valid bundle instance
 * @param      sig pointer to a valid signature
 * @param      custom_tags bitfield indicating which custom tags to set
 * @return     One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_signature_custom_tags_set(struct qmdpi_bundle *bundle,
                                    struct qmdpi_signature *sig, uint64_t custom_tags);

/**
 * Get custom tags for a signature. Can only be called for signatures which have been enabled.
 *
 *
 * @param      sig pointer to a valid signature
 * @param      custom_tags address of custom tags field
 * @return     One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_signature_custom_tags_get(struct qmdpi_signature *sig,
                                    uint64_t *custom_tags);


/**
 * Whether signature attributes are merged.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @return      One of the following:
 * - 1 if signature attributes are merged
 * - 0 if not
 * - on error, -1 is returned and errno set
 */
int8_t qmdpi_signature_has_attr_merged(struct qmdpi_signature *sig);

/**
 * Callback: invoked for each tag set to 1.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       tag_id tag ID
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_signature_tag_callback_t)(struct qmdpi_bundle *bundle,
        int tag_id, void *arg);

/**
 * Iterate over signature tag bitfield.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @param       callback callback to be called for each signature tag
 * @param       arg callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_signature_tag_foreach(struct qmdpi_bundle *bundle,
                                struct qmdpi_signature *sig,
                                qmdpi_signature_tag_callback_t callback, void *arg);

/**
 * Callback: invoked for each lower layer.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig lower layer signature
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_signature_lower_callback_t)(struct qmdpi_bundle
        *bundle,
        struct qmdpi_signature *sig,
        void *arg);

/**
 * Iterate over signature lower layers.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @param       callback callback to be called for each signature lower layer
 * @param       arg callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_lower_foreach(struct qmdpi_bundle *bundle,
        struct qmdpi_signature *sig,
        qmdpi_bundle_signature_lower_callback_t callback,
        void *arg);

/**
 * Callback: invoked for each child layer.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig child layer signature
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_signature_child_callback_t)(struct qmdpi_bundle
        *bundle,
        struct qmdpi_signature *sig,
        void *arg);

/**
 * Iterate over signature child layers (e.g: Q_PROTO_FTP_DATA is a child layer of Q_PROTO_FTP).
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @param       callback callback to be called for each signature child layer
 * @param       arg callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_child_foreach(struct qmdpi_bundle *bundle,
        struct qmdpi_signature *sig,
        qmdpi_bundle_signature_child_callback_t callback,
        void *arg);

/**
 * Get signature features.
 *
 * @param       sig pointer to a valid signature
 * @return      signature features bitfield (values are defined in qmdpi_const.h),\n
 *              errno set if error.
 */
uint8_t qmdpi_signature_features_get(struct qmdpi_signature *sig);

/**
 * Get signature version.
 *
 * This bitfield has the following structure :
 *
 *\verbatim
0         8        16              31
+--------+--------+--------+--------+
| major  | minor  |    revision     |
+--------+--------+--------+--------+
 *\endverbatim
 *
 * @param       sig pointer to a valid signature
 * @return      signature version bitfield,\n
 *              errno set if error.
 */
uint32_t qmdpi_signature_version_get(struct qmdpi_signature *sig);

/**
 * Get signature timeout value previously defined by the user for the given signature.
 *
 * @param       sig pointer to a valid signature
 * @return      signature timeout value defined by the user (0 if never defined)
 *              errno set if error.
 */
int32_t qmdpi_signature_timeout_get(struct qmdpi_signature *sig);

/**
 * Set signature timeout value.
 * @warning: For signatures that have a proto tune that can change the layer timeout (e.g. ssl_store_timeout), use the proto tune instead of ::qmdpi_signature_timeout_set.
 *
 * @param       bundle Pointer to valid bundle instance.
 * @param       signature_name Signature name.
 * @param       timeout Pointer to timeval timeout value (tv_sec max value = 8191, tv_usec is ignored).
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_signature_timeout_set(struct qmdpi_bundle *bundle,
                                const char *signature_name,
                                struct timeval *timeout);

/** @} */

/*--------------------------------------
 * Attribute info
 *------------------------------------*/

/**
 * @defgroup attr_info Attribute Information
 * @ingroup bundle_info
 * @{
 */

/**
 * Callback:
 * invoked for each attribute when calling @a qmdpi_bundle_attr_foreach, @a qmdpi_bundle_attr_foreach_permitted.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       attr pointer to a valid attribute
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_attr_callback_t)(struct qmdpi_bundle *bundle,
        char const *signature_name,
        struct qmdpi_attr const *attr,
        void *arg);

/**
 * Call the specified callback for each attribute of a signature.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_attr_foreach(struct qmdpi_bundle *bundle,
                              const char *signature_name,
                              qmdpi_bundle_attr_callback_t callback,
                              void *arg);


/**
 * Call the specified callback for each attribute of a signature covered by the license.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_attr_foreach_permitted(struct qmdpi_bundle *bundle,
                                        const char *signature_name,
                                        qmdpi_bundle_attr_callback_t callback,
                                        void *arg);

/**
 * Callback: invoked for each transaction supported by signature
 *
 * @param       transaction_id transaction ID
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_signature_transaction_callback_t)(
    struct qmdpi_bundle    *bundle,
    struct qmdpi_signature *sig,
    const unsigned int      transaction_id,
    void                   *arg);

/**
 * Iterate over signature supported transactions.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       sig pointer to a valid signature
 * @param       callback callback to be called for each signature transaction
 * @param       arg callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_signature_transaction_foreach(struct qmdpi_bundle
        *bundle,
        struct qmdpi_signature                 *sig,
        qmdpi_bundle_signature_transaction_callback_t  callback,
        void                                   *arg);

/**
 * Get attribute name.
 *
 * @param       attr pointer to a valid attribute
 * @return      attribute name string, errno set if error.
 */
char const *qmdpi_attr_name_get(struct qmdpi_attr const *attr);

/**
 * Get attribute ID.
 *
 * @param       attr pointer to a valid attribute
 * @return      attribute ID, errno set if error.
 */
int qmdpi_attr_id_get(struct qmdpi_attr const *attr);

/**
 * Get attribute data type.
 *
 * @param       attr pointer to a valid attribute
 * @return      attribute data type, errno set if error.
 */
uint8_t qmdpi_attr_datatype_get(struct qmdpi_attr const *attr);

/**
 * Get attribute direction:
 * the direction of the PDU from which the attribute was extracted.
 *
 * @param       attr pointer to a valid attribute
 * @return      attribute direction,\n
 *              errno set if error
 */
uint8_t qmdpi_attr_dir_get(struct qmdpi_attr const *attr);

/**
 * Get attribute parent ID.
 *
 * @param       attr pointer to a valid attribute
 * @return      attribute parent ID,\n
 *              errno set if error
 */
int32_t qmdpi_attr_parent_id_get(struct qmdpi_attr const *attr);

/**
 * Indicates whether this attribute is a streamed attribute.
 *
 * @param       attr pointer to a valid attribute
 * @return      1 if streamed, 0 if not,\n
 *              errno set if error
 */
uint8_t qmdpi_attr_is_streamed(struct qmdpi_attr const *attr);

/**
 * Indicates whether this attribute is a content attribute.
 *
 * @param       attr pointer to a valid attribute
 * @return      1 if content, 0 if not,\n
 *              errno set if error
 */
uint8_t qmdpi_attr_is_content(struct qmdpi_attr const *attr);

/**
 * Get attribute scope.
 *
 * @param       attr pointer to a valid attribute
 * @return      attribute scope (see qmdpi_const.h for QMDPI_SCOPE_* values),\n
 *              errno set if error
 **/
uint8_t qmdpi_attr_scope_get(struct qmdpi_attr const *attr);

/** @} */

/*--------------------------------------
 * Tune info
 *------------------------------------*/

/**
 * @defgroup tune_info Signature Tune Information
 * @ingroup bundle_info
 * @{
 */


/**
 * Callback: invoked for each signature config param
 * when calling @a qmdpi_bundle_tune_foreach.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       tune pointer to a valid signature config param (AKA tune)
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_bundle_tune_callback_t)(struct qmdpi_bundle *bundle,
        char const *signature_name,
        struct qmdpi_tune *tune,
        void *arg);

/**
 * Call the specified callback for each config param (AKA tune) of a signature.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_tune_foreach(struct qmdpi_bundle *bundle,
                              const char *signature_name,
                              qmdpi_bundle_tune_callback_t callback,
                              void *arg);

/**
 * Get tune name.
 *
 * @param       tune pointer to a valid tune
 * @return      tune name string,\n
 *              errno set if error
 */
const char *qmdpi_tune_name_get(struct qmdpi_tune const *tune);

/**
 * Get tune value.
 *
 * @param       tune pointer to a valid tune
 * @return      tune value,\n
 *              errno set if error
 */
uint32_t qmdpi_tune_value_get(struct qmdpi_tune const *tune);

/**
 * Set signature configuration parameter (AKA tune).
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       tune_name tune name
 * @param       tune_value tune value
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_bundle_tune_set(struct qmdpi_bundle *bundle,
                          const char *signature_name, const char *tune_name, uint32_t tune_value);

/** @} */

/*--------------------------------------
 * Bundle Object Accessors
 *------------------------------------*/
/**
 * @defgroup object_accessors Bundle Object Accessors
 * @ingroup bundle_info
 * @{
 */


/**
 * Get signature object by name.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_signature_get_byname.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - pointer to signature struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_signature *qmdpi_bundle_signature_get_byname(
    struct qmdpi_bundle *bundle,
    const char *signature_name);
/**
 * Get signature family name by signature name.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_signature_family_get_byname
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - family name of the signature if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_signature_family_get_byname(struct qmdpi_bundle *bundle,
        const char *signature_name);

/**
 * Get signature family name by signature ID.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_signature_family_get_byid.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_id signature id
 * @return      One of the following:
 * - family name of the signature if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_signature_family_get_byid(struct qmdpi_bundle *bundle,
        int signature_id);

/**
 * Get attribute object by name.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_attr_get_byname.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       attr_name attribute name
 * @return      One of the following:
 * - pointer to attr struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_attr *qmdpi_bundle_attr_get_byname(struct qmdpi_bundle *bundle,
        const char *signature_name,
        const char *attr_name);

/**
 * Get config param (AKA tune) object by name.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_tune_get_byname.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       tune_name tune name
 * @return      One of the following:
 * - tune pointer to tune struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_tune *qmdpi_bundle_tune_get_byname(struct qmdpi_bundle *bundle,
        const char *signature_name,
        const char *tune_name);

/**
 * Get signature object by ID.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_signature_get_byid.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_id signature identifier
 * @return      One of the following:
 * - pointer to signature struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_signature *qmdpi_bundle_signature_get_byid(
    struct qmdpi_bundle *bundle,
    int signature_id);

/**
 * Get attribute object by ID.
 *
 * @warning This function is not thread-safe; thread-safe version of this function is ::qmdpi_worker_attr_get_byid.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_id signature identifier
 * @param       attr_id attr identifier
 * @return      One of the following:
 * - pointer to attr struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_attr *qmdpi_bundle_attr_get_byid(struct qmdpi_bundle *bundle,
        int signature_id, int attr_id);
/** @} */

/*--------------------------------------
 * Bundle Object Accessors (Thread-Safe)
 *------------------------------------*/
/**
 * @defgroup object_accessors_thread Bundle Object Accessors (thread-safe)
 * @ingroup bundle_info
 * @{
 */

/**
 * Get signature object by name (thread-safe).
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - pointer to signature struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_signature *qmdpi_worker_signature_get_byname(
    struct qmdpi_worker *worker, struct qmdpi_bundle *bundle,
    const char *signature_name);
/**
 * Get signature family name by signature name.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 **
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @return      One of the following:
 * - family name of the signature if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_worker_signature_family_get_byname(struct qmdpi_worker
        *worker, struct qmdpi_bundle *bundle, const char *signature_name);

/**
 * Get signature family name by signature id.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_id signature id
 * @return      One of the following:
 * - family name of the signature if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_worker_signature_family_get_byid(struct qmdpi_worker *worker,
        struct qmdpi_bundle *bundle, int signature_id);

/**
 * Get attribute object by name.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       attr_name attribute name
 * @return      One of the following:
 * - pointer to attr struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_attr *qmdpi_worker_attr_get_byname(struct qmdpi_worker *worker,
        struct qmdpi_bundle *bundle, const char *signature_name, const char *attr_name);

/**
 * Get config param (AKA tune) object by name.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_name signature name
 * @param       tune_name tune name
 * @return      One of the following:
 * - pointer to tune struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_tune *qmdpi_worker_tune_get_byname(struct qmdpi_worker *worker,
        struct qmdpi_bundle *bundle, const char *signature_name, const char *tune_name);

/**
 * Get signature object by identifier.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_id signature identifier
 * @return      One of the following:
 * - pointer to signature struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_signature *qmdpi_worker_signature_get_byid(
    struct qmdpi_worker *worker, struct qmdpi_bundle *bundle, int signature_id);

/**
 * Get attribute object by indentifier.
 *
 * @warning Validity of returned pointer is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       signature_id signature identifier
 * @param       attr_id attr identifier
 * @return      One of the following:
 * - pointer to attr struct if success
 * - NULL if error (with errno set)
 */
struct qmdpi_attr *qmdpi_worker_attr_get_byid(struct qmdpi_worker *worker,
        struct qmdpi_bundle *bundle, int signature_id, int attr_id);

/** @} */
/** @} */

/*------------------------------------------------------------------------------
 * Family / Tags info
 *----------------------------------------------------------------------------*/
/**
 * @defgroup family_tags Family / Tags info
 * @ingroup bundle
 * @{
 */

/**
 * Get signature family ID.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       family_name family name
 * @return      a valid family ID or an error code otherwise
 */
int qmdpi_family_id_get_byname(struct qmdpi_bundle *bundle,
                               const char *family_name);

/**
 * Get signature family name.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       family_id family ID
 * @return      a valid family name, NULL if error (with errno set)
 */
const char *qmdpi_family_name_get_byid(struct qmdpi_bundle *bundle,
                                       int family_id);

/**
 * Callback: invoked by @a qmdpi_family_foreach.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       family_id family ID
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_family_callback_t)(struct qmdpi_bundle *bundle,
                                       int family_id, void *arg);

/**
 * Call the specified callback for each family.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_family_foreach(struct qmdpi_bundle *bundle,
                         qmdpi_family_callback_t callback, void *arg);

/**
 * Get signature tag ID.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       tag_name tag name
 * @return      a valid tag ID, an error code otherwise
 */
int qmdpi_tag_id_get_byname(struct qmdpi_bundle *bundle, const char *tag_name);

/**
 * Get signature tag name.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       tag tag ID
 * @return      a valid tag name, NULL if error (with errno set)
 */
const char *qmdpi_tag_name_get_byid(struct qmdpi_bundle *bundle, int tag);

/**
 * Callback: invoked by @a qmdpi_tag_foreach.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       tag_id tag ID
 * @param       arg callback argument
 * @return      defined by callback
 */
typedef int (*qmdpi_tag_callback_t)(struct qmdpi_bundle *bundle, int tag_id,
                                    void *arg);

/**
 * Call the specified callback for each tag.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       callback function to be called
 * @param       arg optional callback argument
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_tag_foreach(struct qmdpi_bundle *bundle,
                      qmdpi_tag_callback_t callback, void *arg);



/** @} */

/*------------------------------------------------------------------------------
 * Protocol Data Binary (PDB)
 *----------------------------------------------------------------------------*/

/**
 * @defgroup bundle_pdb Protocol Data Binary (PDB) Management
 * @ingroup bundle
 * @{
 */

/**
 * Create PDB instance from an address and provide a PDB handle to user.
 *
 * @note PDB handle may be safely deleted on success (using ::qmdpi_pdb_destroy)
 *
 * @param       bundle pointer to valid bundle instance
 * @param       pdb_addr address of PDB buffer
 * @param       pdb_len length of PDB buffer
 * @return      One of the following:
 * - pointer to handle of created PDB
 * - NULL if error (with errno set)
 */
struct qmdpi_pdb *qmdpi_bundle_pdb_create_from_addr(struct qmdpi_bundle *bundle,
        char *pdb_addr, int pdb_len);

/**
 * Destroy PDB handle.
 *
 * @note This only destroys qmdpi_pdb handle, not the PDB itself.
 *
 * @param       pdb_addr address of PDB buffer
 * @return      0 on sucess, error otherwise.
 */
int qmdpi_bundle_pdb_destroy(struct qmdpi_pdb *pdb_addr);

/**
 * Activate PDB instance.
 *
 * @param       pdb pointer to PDB handle
 * @return      0 on sucess, error otherwise.
 */
int qmdpi_bundle_pdb_activate(struct qmdpi_pdb *pdb);

/**
 * Get active PDB handle.
 *
 * @param       bundle pointer to bundle instance
 * @return      One of the following:
 * - active PDB handle
 * - NULL if no PDB instance is set to active
 * - NULL with errno set if error
 */
struct qmdpi_pdb *qmdpi_bundle_pdb_get_active(struct qmdpi_bundle *bundle);

/**
 * Get number of referenced flows for the specified PDB.
 *
 * @param       pdb pointer to PDB handle
 * @return      One of the following:
 * - number of referenced flows
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_bundle_pdb_get_refcount(struct qmdpi_pdb *pdb);

/**
 * Get default PDB address.
 *
 * @param       bundle pointer to valid bundle instance
 * @param       pdb_addr address of PDB buffer
 * @param       pdb_len length of PDB buffer
 * @return      0 on success, error otherwise.
 */
int qmdpi_bundle_pdb_get_default_addr(struct qmdpi_bundle *bundle,
                                      char **pdb_addr, int *pdb_len);

/**
 * Get default PDB handle.
 *
 * @param       bundle pointer to valid bundle instance
 * @return      One of the following:
 * - pointer to default PDB handle
 * - NULL if error (with errno set)
 */
struct qmdpi_pdb *qmdpi_bundle_pdb_get_default(struct qmdpi_bundle *bundle);

/**
 * Get the version string of the specified PDB.
 *
 * @param       pdb pointer to pdb handle
 * @return      One of the following:
 * - version string if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_pdb_version_get_string(struct qmdpi_pdb *pdb);


/**
 * Get the version string of the active PDB.
 *
 * @param       bundle pointer to valid bundle instance
 * @return      One of the following:
 * - version string if success
 * - NULL if error (with errno set)
 */
const char *qmdpi_bundle_pdb_version_get_string(struct qmdpi_bundle *bundle);

/** @} */

/*------------------------------------------------------------------------------
 * Custom signature
 *----------------------------------------------------------------------------*/

/**
 * @defgroup bundle_cs Custom Signature API (Requires a specific license)
 * @ingroup bundle
 * @{
 */

/**
 * Attempt to add new signatures in the input Protocol Data Binary
 * @a pdb_addr / @a pdb_len, and then load and activate it.
 * If @a pdb_addr is null then the default Protocol Data Binary will be used.
 *
 * @param bundle pointer to bundle instance
 * @param pdd_addr      Pointer to the buffer holding the content of a pdata description file (PDD).
 * @param pdd_len       Length of the PDD buffer.
 * @param pdb_addr      Pointer to the buffer holding the content of a pdata binary file (PDB).
 * @param pdb_len       Length of PDB buffer.
 * @return 0 on sucess, error otherwise.
 */
int qmdpi_bundle_pdb_compile_and_activate(struct qmdpi_bundle *bundle,
        char *pdd_addr,
        int pdd_len,
        char *pdb_addr,
        int pdb_len);

/**
 * Creates a Protocol Data Binary and integrates it into the specified @a bundle,
 * accessible via the pointers @a pdb_out_addr and @a pdb_out_len,
 * constructed from the input Protocol Data Binary @a pdb_in_addr / @a pdb_in_len
 * plus the addition of the new pdata signatures
 * passed-in by @a pdd_addr / @a pdd_len.
 * If the @a pdb_in_addr argument is null, the signatures will be added to the
 * default Protocol Data Binary.
 *
 * @param bundle pointer to bundle instance
 * @param pdd_addr      A buffer pointer holding the content of a pdata description file.
 * @param pdd_len       Length of signature.
 * @param pdb_in_addr   Pointer to the buffer holding the content of the pdata binary file (PDB) to compile.
 * @param pdb_in_len    Length of the PDB to compile.
 * @param pdb_out_addr  Pointer to the buffer holding the newly created PDB.
 * @param pdb_out_len   Length of the buffer holding the newly created PDB.
 * @return 0 on sucess, error otherwise.
 *
 * @warning After calling this function and retrieving the PDB, the caller must call
 * ::qmdpi_bundle_pdb_compile_cleanup to free the @a pdb_out_addr buffer.
 *
 */
int qmdpi_bundle_pdb_compile(struct qmdpi_bundle *bundle, char *pdd_addr,
                             int pdd_len, char *pdb_in_addr, int pdb_in_len, char **pdb_out_addr,
                             int *pdb_out_len);

/**
 * Attempts to free the @a pdb_out_addr buffer.
 * This function must be used after ::qmdpi_bundle_pdb_compile
 * in order to free the new pdata binary file returned.
 *
 * @param bundle pointer to bundle instance
 * @param pdb_addr the buffer pointer returned by the function ::qmdpi_bundle_pdb_compile (AKA @a pdb_out_addr)
 * @return 0 on sucess, error otherwise.
 */
int qmdpi_bundle_pdb_compile_cleanup(struct qmdpi_bundle *bundle,
                                     char **pdb_addr);

/** @} */

/*------------------------------------------------------------------------------
 * Traffic Distribution Module
 *----------------------------------------------------------------------------*/

/**
 * Initialize and allocate resources for the traffic distribution module.
 *
 * @param engine pointer to engine instance.
 * @param config configuration string, with format: "config_param1=config_value1;config_param2=config_value2;".
 *               If set to NULL, default values are used.
 *
 * @return 0 if success, error code otherwise.
 */
int qmdpi_report_init(struct qmdpi_engine *engine, char const *config);

/**
 * Start feeding the report table.
 *
 * @param engine pointer to engine instance.
 *
 * @return 0 if success, error code otherwise.
 */
int qmdpi_report_start(struct qmdpi_engine *engine);

/**
 * Dump the report of traffic distribution module.
 * TDM has to be disabled (::qmdpi_report_stop) to dump the report.
 * All dumped reports are sorted by volume (biggest to smallest).
 *
 * @param engine    Pointer to engine instance.
 * @param table_id  The identifier of the table to dump: ProtocolDistribution, Unknown, New, Combined.
 * @param output    Pointer to output stream
 * @param output_fn Output function
 * @return 0 if success, error code otherwise.
 */
int qmdpi_report_dump(struct qmdpi_engine *engine, uint32_t table_id,
                      void *output, qmdpi_output_fn_t output_fn);

/**
 * Stop feeding the report table.
 *
 * @param engine pointer to engine instance.
 * @return 0 if success, error code otherwise.
 */
int qmdpi_report_stop(struct qmdpi_engine *engine);

/**
 * Free the the resources allocated for the Traffic Distribution Module (TDM).
 *
 * @param engine pointer to engine instance.
 * @return 0 if success, error code otherwise
 */
int qmdpi_report_cleanup(struct qmdpi_engine *engine);

/**
 * Create a monitoring context on this flow (STREAM MODE ONLY).
 *
 * @param f flow to monitor.
 * @return 0 if success, error code otherwise.
 */
int qmdpi_flow_monitor(struct qmdpi_flow *f);

/**
 * Add volume stats to monitoring context (STREAM MODE ONLY).
 * This would typically called before expiration.
 *
 * @param f flow to update
 * @param cts_bytes client-to-server bytes of the flow.
 * @param stc_bytes server-to-client bytes of the flow.
 * @param cts_packets client-to-server number of packets of the flow.
 * @param stc_packets server-to-client number of packets of the flow.
 * @return 0 if success, error code otherwise.
 */
int qmdpi_flow_monitor_stats_update(struct qmdpi_flow *f,
                                    uint64_t cts_bytes, uint64_t stc_bytes,
                                    uint32_t cts_packets, uint32_t stc_packets);

/**
 * Get Classification extension and write it to buffer.
 *
 * @warning Not thread-safe; thread-safe version of this function is ::qmdpi_worker_flow_classif_ext_get
 * @warning Calling this API function requires a specific license
 *
 * @param       flow pointer to a flow context
 * @param       buffer user-allocated buffer
 * @param       buffer_len length of provided buffer
 * @return      One of the following:
 * - number of bytes copied if success, including the terminating null byte '\0'
 * - QMDPI_EOVERFLOW if buffer was too small (string will be truncated)
 * - QMDPI_ENOEXIST if no extension found
 * - another error code otherwise
 */
int qmdpi_flow_classif_ext_get(struct qmdpi_flow *flow, char *buffer,
                               int buffer_len);

/**
 * Enable classification extension feature.
 *
 * @warning Calling this API function requires a specific license
 *
 * @param engine pointer to engine instance.
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_classif_ext_enable(struct qmdpi_engine *engine);

/**
 * Get classification extension and write it to buffer.
 * @warning Calling this API function requires a specific license.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       flow pointer to a flow context
 * @param       buffer user-allocated buffer
 * @param       buffer_len length of provided buffer
 * @return      One of the following:
 * - number of bytes copied if success, including the terminating null byte '\0'
 * - QMDPI_EOVERFLOW if buffer was too small (string will be truncated)
 * - QMDPI_ENOEXIST if no extension found
 * - another error code otherwise
 */
int qmdpi_worker_flow_classif_ext_get(struct qmdpi_worker *worker,
                                      struct qmdpi_flow *flow, char *buffer, int buffer_len);

/** @} */

/*------------------------------------------------------------------------------
 * Data presentation
 *----------------------------------------------------------------------------*/

/**
 * @defgroup bundle_data Data Presentation
 * @ingroup bundle
 * @{
 */

/**
 * Convert attribute value to appropriate human-readable string.
 *
 * @warning Not thread-safe; thread-safe version of this function is ::qmdpi_worker_data_attr_to_buffer
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       buffer pointer to buffer containing converted data
 * @param       buffer_len buffer length
 * @param       proto_id protocol identifier
 * @param       attr_id attribute identifier
 * @param       attr_value attribute value
 * @param       attr_value_len attribute value length
 * @return      One of the following:
 * - written length (or 0 if not applicable) if success
 * - error code otherwise
 */
int qmdpi_data_attr_to_buffer(struct qmdpi_bundle *bundle, char *buffer,
                              int buffer_len,
                              int proto_id, int attr_id, char const *attr_value, int attr_value_len);


/**
 * Convert numerical path value to appropriate human-readable path
 * (for example ip\.tcp\.http).
 *
 * @warning Not thread-safe; thread-safe version of this function is ::qmdpi_worker_data_path_to_buffer
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       buffer pointer to buffer containing converted data
 * @param       buffer_len buffer length
 * @param       path pointer to classification path
 * @return      One of the following:
 * - written length (or 0 if not applicable) if success
 * - error code otherwise
 */
int qmdpi_data_path_to_buffer(struct qmdpi_bundle *bundle, char *buffer,
                              int buffer_len,
                              struct qmdpi_path const *path);

/**
 * Convert attribute value to appropriate human-readable string.
 *
 * @warning Validity of output data is guaranteed:
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       buffer pointer to buffer containing converted data
 * @param       buffer_len buffer length
 * @param       proto_id protocol identifier
 * @param       attr_id attribute identifier
 * @param       attr_type attribute data type (0 for automatic detection)
 * @param       attr_value attribute value
 * @param       attr_value_len attribute value length
 * @return      One of the following:
 * - written length (or 0 if not applicable) if success
 * - error code otherwise
 */
int qmdpi_worker_data_attr_to_buffer(struct qmdpi_worker *worker,
                                     struct qmdpi_bundle *bundle,
                                     char *buffer, int buffer_len, int proto_id, int attr_id, int attr_type,
                                     char const *attr_value, int attr_value_len);

/**
 * Convert numerical path value to appropriate human-readable path
 * (for example ip\.tcp\.http).
 *
 * @warning Validity of output data is guaranteed :
 * - if a function of the current group is not called concurrently (e.g. on another thread) on the same worker
 * - before another function of the current group is called on the same thread.
 *
 * @param       worker pointer to a valid worker (if NULL, function can only be called from one thread, e.g. control thread)
 * @param       bundle pointer to a valid bundle instance
 * @param       buffer pointer to buffer containing converted data
 * @param       buffer_len buffer length
 * @param       path pointer to classification path
 * @return      One of the following:
 * - written length (or 0 if not applicable) if success
 * - error code otherwise
 */
int qmdpi_worker_data_path_to_buffer(struct qmdpi_worker *worker,
                                     struct qmdpi_bundle *bundle, char *buffer,
                                     int buffer_len, struct qmdpi_path const *path);

/**
 * Get transaction name by transaction ID.
 *
 * @param       bundle pointer to bundle instance
 * @param       transaction_id transaction identifier
 * @param       transaction_name transaction name (output)
 * @param       transaction_name_len transaction name length (output)
 * @return      0 if success, error code otherwise.
 */
int qmdpi_data_transaction_get_name(struct qmdpi_bundle *bundle,
                                    uint32_t transaction_id, char const **transaction_name,
                                    int *transaction_name_len);

/** @} */

/*------------------------------------------------------------------------------
 * DPI processing results
 *----------------------------------------------------------------------------*/

/**
 * @defgroup result DPI Processing Result
 * @{
 */

/**
 * Get result flags from a given result object.
 *
 * @param       result pointer to a valid result object
 * @return      One of the following:
 * - pointer to flags if success
 * - NULL if error (with errno set)
 */
struct qmdpi_result_flags const *qmdpi_result_flags_get(
    struct qmdpi_result *result);

/**
 * Whether classified state has changed or not.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow classification status has changed,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_CLASSIFIED_STATE_CHANGED(result_flags)    ((result_flags)->qrf.values.qrf_classified_state_changed)

/**
 * Retrieve classified status.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow is fully classified,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_CLASSIFIED_STATE(result_flags)            ((result_flags)->qrf.values.qrf_classified_state)

/**
 * Retrieve classification extension status.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if classification extension is possible,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_NEED_CLASSIF_EXT(result_flags)                 ((result_flags)->qrf.values.qrf_classif_ext)

/**
 * Retrieve classified final status.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow is fully classified and final,
 * - 0 otherwise.
 *
 */
#define QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE(result_flags)      ((result_flags)->qrf.values.qrf_classified_final_state)

/**
 * Whether classified final state has changed or not.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow final classification status has changed,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_CLASSIFIED_FINAL_STATE_CHANGED(result_flags)      ((result_flags)->qrf.values.qrf_classified_final_state_changed)


/**
 * Whether offloading state has changed or not.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow offloading status changed,
 * - 0 otherwise.
 *
 */
#define QMDPI_RESULT_FLAGS_OFFLOADED_STATE_CHANGED(result_flags)     ((result_flags)->qrf.values.qrf_offloaded_state_changed)

/**
 * Retrieve offloading state.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow is offloaded,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_OFFLOADED_STATE(result_flags)             ((result_flags)->qrf.values.qrf_offloaded_state)

/**
 * Whether classification path has changed or not.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow classification path has changed,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)                ((result_flags)->qrf.values.qrf_path_changed)

/**
 * Whether the flow has expired or not.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow has expired,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags)                ((result_flags)->qrf.values.qrf_flow_expired)

/**
 * Whether the flow has just been created or not.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow has just been created,
 * - 0 otherwise.
 *
 */
#define QMDPI_RESULT_FLAGS_FLOW_CREATED(result_flags)                ((result_flags)->qrf.values.qrf_flow_created)

/**
 * Whether or not the packet has been ignored because of a full TCP reassembly queue.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if the packet has been ignored,
 * - 0 otherwise.
 *
 */
#define QMDPI_RESULT_FLAGS_TCP_REASS_QUEUE_FULL(result_flags)        ((result_flags)->qrf.values.qrf_tcp_reass_queue_full)

/**
 * Whether or not some metadata extraction failed because of a full metadata buffer.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if a metadata extraction failed,
 * - 0 otherwise.
 *
 */
#define QMDPI_RESULT_FLAGS_ATTR_BUF_FULL(result_flags)               ((result_flags)->qrf.values.qrf_attr_buf_full)

/**
 * Retrieve the direction of the packet either Server-To-Client or Client-To-Server.
 *
 * @note The flow direction may be udpated during the flow lifetime, so packets of the same flow and going in the same way could have different values.
 * @param       result_flags pointer
 * @return      packet direction: QMDPI_DIR_CTS or QMDPI_DIR_STC
 *
 */
#define QMDPI_RESULT_FLAGS_PDU_DIR(result_flags)                     ((result_flags)->qrf.values.qrf_pdu_dir)

/**
 * Whether or not the flow direction has changed.
 * In some cases, packet parsing can help reordering client and server ends.
 * This could occur for for example on TCP flows without TCP handshake.
 *
 * @param       result_flags pointer
 * @return      One of the following:
 * - 1 if flow direction changed,
 * - 0 otherwise.
 */
#define QMDPI_RESULT_FLAGS_PDU_DIR_CHANGED(result_flags)             ((result_flags)->qrf.values.qrf_pdu_dir_changed)

/**
 * Retrieve the arbitrary direction of the packet.
 * All packets of the same flow and going in the same way should have the same value.
 *
 * @param       result_flags pointer
 * @return      packet direction index QMDPI_DIR0, QMDPI_DIR1
 *
 */
#define QMDPI_RESULT_FLAGS_PDU_DIR_INDEX(result_flags)               ((result_flags)->qrf.values.qrf_pdu_dir_index)

/**
 * Retrieve the length of the cacheable path.
 *
 * @note The length of the cacheable path of a flow is less of equal to the length of the qmdpi_path.
 * @note This flag is computed if and only if the configuration parameter cacheable_path_len_enable is set to 1.
 * @param       result_flags pointer
 * @return      length of the cacheable path (0 if not cacheable)
 *
 */
#define QMDPI_RESULT_FLAGS_CACHEABLE_PATH_LEN(result_flags)          ((result_flags)->qrf.values.qrf_cacheable_path_len)


/**
 * Get the flow context from a given result object.
 *
 * @param       result pointer to a valid result object
 * @return      One of the following:
 * - pointer to flow context if success
 * - NULL if error (with errno set)
 */
struct qmdpi_flow *qmdpi_result_flow_get(struct qmdpi_result *result);

/**
 * Get classification path from a given result object.
 *
 * @param       result pointer to a valid result object
 * @return      One of the following:
 * - pointer to path if success
 * - NULL if error (with errno set)
 */
struct qmdpi_path *qmdpi_result_path_get(struct qmdpi_result *result);

/**
 * Get cached classification path from a given result object.
 *
 * @param       result pointer to a valid result object
 * @return      One of the following:
 * - pointer to path if success
 * - NULL if error (with errno set)
 */
struct qmdpi_path *qmdpi_result_cached_path_get(struct qmdpi_result *result);

/**
 * Reset the classification cache
 *
 * @param engine pointer to engine instance.
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_classification_cache_reset(struct qmdpi_engine *engine);

/**
 * Lookup for a cached path in the classification cache.
 *
 * @warning: if called with a worker that does not process packets, then the worker timestamp
 *           must be set via ::qmdpi_worker_timestamp_set prior to calling this API function.
 *
 * @param       worker pointer to worker
 * @param       l3proto layer 3 protocol (allowed values: Q_PROTO_IP, Q_PROTO_IP6)
 * @param       l4proto layer 4 protocol (allowed values: Q_PROTO_TCP, Q_PROTO_UDP)
 * @param       l3server layer 3 server addr (Network Byte Order)
 * @param       l4server layer 4 server port (Network Byte Order)
 * @param       virtual_id virtual network identifier
 * @param       cached_path pointer to cached classification path if success, NULL otherwise (output)
 * @param       offloaded set to 1 if cache entry was created out of a previously offloaded flow,
 *              set to 0 otherwise (output)
 * @return      One of the following:
 * - 0 if success.
 * - QMDPI_EINIT if the classification cache was not previously enabled.
 * - QMDPI_ENOEXIST if no cache entry is found.
 * - another error code otherwise.
 */
int qmdpi_classification_cache_lookup(struct qmdpi_worker *worker,
                                      int l3proto,
                                      int l4proto,
                                      void const *l3server,
                                      void const *l4server,
                                      int virtual_id,
                                      struct qmdpi_path *cached_path,
                                      int *offloaded);

/**
 * Get next attribute value from a given result object.
 *
 * @param       result pointer to a valid result object
 * @param       proto_id protocol identifier for that attribute
 * @param       attr_id attribute identifier
 * @param       attr_value attribute value
 * @param       attr_value_len attribute value length
 * @param       attr_flags attribute flags
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_result_attr_getnext(struct qmdpi_result *result,
                              int *proto_id,
                              int *attr_id,
                              char const **attr_value,
                              int *attr_value_len,
                              int *attr_flags);

/**
 * Get next attribute value from a given result object.
 *
 * @param       result pointer to a valid result object
 * @param       proto_id protocol identifier for that attribute
 * @param       path_index path_index for that attribute
 * @param       attr_id attribute identifier
 * @param       attr_value attribute value
 * @param       attr_value_len attribute value length
 * @param       attr_flags attribute flags
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_result_attr_getnext_with_index(struct qmdpi_result *result,
        int *proto_id,
        int *path_index,
        int *attr_id,
        char const **attr_value,
        int *attr_value_len,
        int *attr_flags);

/** @} */

/*------------------------------------------------------------------------------
 * Flow API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup flow Flow API
 */

/**
 * @defgroup flow_mgmt Flow Context Management
 * @ingroup flow
 * @{
 */

/*--------------------------------------
 * Flow context management
 *------------------------------------*/

/**
 * Create a flow context (STREAM MODE ONLY): initialize and allocate resources for a flow context.
 * All mechanisms related to cache or IP classification obtain addresses and ports directly in the flow sig which is built using information provided by the client.
 *
 * @param       worker pointer to worker
 * @param       l3proto layer 3 protocol (allowed values: Q_PROTO_IP, Q_PROTO_IP6)
 * @param       l4proto layer 4 protocol (allowed values: any protocol defined in Protocol field of the IPv4 header or the Next Header field of the IPv6 header)
 * @param       l3client layer 3 client addr (Network Byte Order)
 * @param       l4client layer 4 client port (Network Byte Order)
 *              - NULL when l4proto is Q_PROTO_ICMP or Q_PROTO_ICMP6.
 * @param       l3server layer 3 server addr (Network Byte Order)
 * @param       l4server layer 4 server port (Network Byte Order)
 *              - NULL when l4proto is Q_PROTO_ICMP or Q_PROTO_ICMP6.
 * @return      One of the following:
 * - a pointer to flow context created,
 * - NULL if error (with errno set).
 */
struct qmdpi_flow *qmdpi_flow_create(struct qmdpi_worker *worker,
                                     int l3proto,
                                     int l4proto,
                                     void const *l3client,
                                     void const *l4client,
                                     void const *l3server,
                                     void const *l4server);

/**
 * Destroy a flow context.
 *
 * @param       worker pointer to worker
 * @param       flow pointer to a flow context
 * @param       result pointer to DPI result
 *
 * @warning     In Packet-mode, to allow access to flow information after the call,
 * flow context is not destroyed until next call to ::qmdpi_worker_process / ::qmdpi_flow_expire_next.
 * @warning     In Stream-mode, destruction does not happen until the next call
 * to ::qmdpi_flow_reset() / ::qmdpi_flow_offload() / ::qmdpi_flow_destroy().
 *
 * @return      One of the following:
 * - 0 if success,
 * - QMDPI_PROCESS_MORE if one or more TCP Segments of the flow are still present in the reassembly queue.
 * - error code otherwise.
 */
int qmdpi_flow_destroy(struct qmdpi_worker *worker,
                       struct qmdpi_flow *flow,
                       struct qmdpi_result **result);

/**
 * Force the offloading of the specified flow and free resources that are no longer needed
 * by the flow context associated to the specified flow; this is irreversible.
 *
 * @param       worker pointer to worker
 * @param       flow pointer to a flow context
 * @param       result pointer to DPI result
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_flow_offload(struct qmdpi_worker *worker,
                       struct qmdpi_flow *flow,
                       struct qmdpi_result **result);

/**
 * Reset an already allocated flow context (STREAM MODE ONLY):
 * initialize a flow context.
 *
 * @param       worker pointer to worker
 * @param       flow pointer to a flow context
 * @param       result pointer to DPI result
 * @param       l3proto layer 3 protocol (allowed values: Q_PROTO_IP, Q_PROTO_IP6)
 * @param       l4proto layer 4 protocol (allowed values: Q_PROTO_TCP, Q_PROTO_UDP or Q_PROTO_SCTP)
 * @param       l3client pointer to layer 3 client addr (Network Byte Order)
 * @param       l4client pointer to layer 4 client port (Network Byte Order)
 * @param       l3server pointer to layer 3 server addr (Network Byte Order)
 * @param       l4server pointer to layer 4 server port (Network Byte Order)
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_flow_reset(struct qmdpi_worker *worker,
                     struct qmdpi_flow *flow,
                     struct qmdpi_result **result,
                     int l3proto,
                     int l4proto,
                     void const *l3client,
                     void const *l4client,
                     void const *l3server,
                     void const *l4server);

/**
 * Get the virtual network identifier associated with the flow_context.
 *
 * @param      flow pointer to a valid flow context
 * @return      One of the following:
 * - virtual network identifier if success,
 * - 0 if error (with errno set).
 */
int qmdpi_flow_vid_get(struct qmdpi_flow *flow);

/**
 * Destroy next expired flow from Qosmos Flow Manager (PACKET MODE ONLY).
 *
 * @note If there exist a bundle set to be destroyed (this is done by calling ::qmdpi_bundle_destroy),
 * this will destroy one remaining flow used by this bundle.
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to worker
 * @param       ts expiration timestamp (use NULL to expire flow regardless of timestamp)
 * @param       result pointer to DPI result
 * @return      One of the following:
 * - 0 if success,
 * - QMDPI_PROCESS_MORE if one or more TCP Segments of the flow are still present in the reassembly queue.
 * - error code otherwise.
 */
int qmdpi_flow_expire_next(struct qmdpi_worker *worker,
                           struct timeval const *ts,
                           struct qmdpi_result **result);
/** @} */

/*--------------------------------------
 * Flow context accessors
 *------------------------------------*/

/**
 * @defgroup flow_access Flow Context Accessors
 * @ingroup flow
 * @{
 */

/**
 * Get flow context information.
 *
 * @warning     This function must not be called on a flow that has been destroyed via ::qmdpi_flow_destroy.
 * @param       flow pointer to a valid flow context
 * @return      One of the following:
 * - flow flags
 * - NULL if error (with errno set)
 */
struct qmdpi_flow_info *qmdpi_flow_info_get(struct qmdpi_flow *flow);

/**
 * Retrieve offloading state.
 *
 * @param       flow_info pointer to a valid ::qmdpi_flow_info instance.
 * @return      One of the following:
 * - 1 if flow is offloaded,
 * - 0 otherwise.
 */
#define QMDPI_FLOW_INFO_OFFLOADED(flow_info)          (flow_info->qf_flags.qf_offload)

/**
 * Retrieve classification status.
 *
 * @param       flow_info pointer to a valid ::qmdpi_flow_info instance.
 * @return      One of the following:
 * - 1 if flow is classified,
 * - 0 otherwise.
 */
#define QMDPI_FLOW_INFO_CLASSIFIED(flow_info)         (flow_info->qf_flags.qf_classified)

/**
 * Retrieve classification final status.
 *
 * @param       flow_info pointer to a valid ::qmdpi_flow_info instance.
 * @return      One of the following:
 * - 1 if flow is fully classified and final,
 * - 0 otherwise.
 *
 */
#define QMDPI_FLOW_INFO_CLASSIFIED_FINAL_STATE(flow_info)  (flow_info->qf_flags.qf_classified_final_state)

/**
 * Retrieve classification status through classification cache.
 *
 * @param       flow_info pointer to a valid ::qmdpi_flow_info instance.
 * @return      One of the following:
 * - 1 if flow has classification cached,
 * - 0 otherwise.
 */
#define QMDPI_FLOW_INFO_CLASSIF_CACHED(flow_info)         (flow_info->qf_flags.qf_classif_cached)

/**
 * Whether TCP handshake has been received for this flow.
 *
 * @param       flow_info pointer to a valid ::qmdpi_flow_info instance.
 * @return      One of the following:
 * - 1 if TCP handshake has been seen,
 * - 0 otherwise.
 */
#define QMDPI_FLOW_INFO_TCP_HANDSHAKE_SEEN(flow_info) (flow_info->qf_flags.qf_tcp_handshake_seen)

/**
 * Get flow context identifier.
 *
 * @param       flow pointer to a valid flow context
 * @return      One of the following:
 * - flow context ID if success
 * - 0 if error (with errno set)
 */
uint64_t qmdpi_flow_id_get(struct qmdpi_flow *flow);

/**
 * Get flow path object.
 *
 * @param       flow pointer to a valid flow context
 * @return      One of the following:
 * - pointer to classification path if success
 * - NULL if error (with errno set)
 */
struct qmdpi_path *qmdpi_flow_path_get(struct qmdpi_flow *flow);

/**
 * Get parent flow.
 * In the case of external demultiplexing (SCTP, SPDY, HTTP2),
 * this flow is the root of all multiplexed flows (AKA subflows).
 *
 * @param       flow pointer to a valid flow context
 * @return      One of the following:
 * - pointer to parent flow,
 * - NULL if error (with errno set).
 */
struct qmdpi_flow *qmdpi_flow_parent_get(struct qmdpi_flow *flow);

/**
 * Get the bundle instance of the specified flow.
 *
 * @param       flow pointer to a valid flow context
 * @return      One of the following:
 * - pointer to bundle if success,
 * - NULL if error (with errno set).
 */
struct qmdpi_bundle *qmdpi_flow_bundle_get(struct qmdpi_flow *flow);

/**
 * Set a user handle for a given flow.
 *
 * @param       flow pointer to a valid flow context
 * @param       user_handle user handle (pointer to user data)
 * @return      One of the following:
 * - 0 if success,
 * - error code otherwise.
 */
int qmdpi_flow_user_handle_set(struct qmdpi_flow *flow, void *user_handle);

/**
 * Get a user handle for a given flow.
 *
 * @param       flow pointer to a valid flow context
 * @return      One of the following:
 * - pointer to user handle if success
 * - NULL if error (with errno set)
 */
void *qmdpi_flow_user_handle_get(struct qmdpi_flow *flow);

/**
 * Get a pointer to the first flow context in 'garbage collector' flow list.
 *
 * @param       worker pointer to worker
 * @param       bundle pointer to a valid bundle instance
 * @return      One of the following:
 * - pointer to the head of the list,
 * - NULL if list is empty, with errno set if error.
 */
struct qmdpi_flow *qmdpi_bundle_get_gc_head(struct qmdpi_worker *worker,
        struct qmdpi_bundle *bundle);

/**
 * Get a pointer to the next flow in the 'garbage collector' flow list.
 *
 * @param       flow pointer to a flow context
 * @return      One of the following:
 * - pointer to the next flow context in the flow list,
 * - NULL if next flow does not exist, with errno set if error.
 */
struct qmdpi_flow *qmdpi_bundle_get_gc_next(struct qmdpi_flow *flow);

/**
* Get INNERMOST 5-tuple data for the specified flow.
*
* @param       flow pointer to flow context
* @param       l3proto layer 3 protocol (e.g. Q_PROTO_IP)
* @param       l4proto layer 4 protocol (e.g. Q_PROTO_TCP, Q_PROTO_UDP, Q_PROTO_SCTP),
*                      Note: Only layer 4 protocols being part of the flow sig are returned
* @param       l3client layer 3 client addr (Network Byte Order)
* @param       l4client layer 4 client port (Network Byte Order)
* @param       l3server layer 3 server addr (Network Byte Order)
* @param       l4server layer 4 server port (Network Byte Order)
* @return      One of the following:
* - number of packets (only if the flow metrics computation is enabled),
* - a negative integer if error.
*/
int
qmdpi_flow_5tuple_get(struct qmdpi_flow *flow,
                      int *l3proto,
                      int *l4proto,
                      void **l3client,
                      void **l4client,
                      void **l3server,
                      void **l4server);

/**
 * Get the flow state (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param      flow pointer to a valid flow context
 * @return     One of the following:
 * - flow state identifier if success (always 0 for non-TCP flows),
 * - a negative integer if error.
*/
int qmdpi_flow_state_get(struct qmdpi_flow *flow);

/** @} */

/*--------------------------------------
 * Flow metrics
 *------------------------------------*/

/**
 * @defgroup flow_metrics Flow Metrics
 * @ingroup flow
 * @{
 */

/**
 * Get number of packets for the specified flow.
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of packets
 */
uint64_t qmdpi_flow_packets_get(struct qmdpi_flow *flow);

/**
 * Get number of packets for the specified flow (client-to-server direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of packets
 */
uint64_t qmdpi_flow_packets_get_cts(struct qmdpi_flow *flow);

/**
 * Get number of packets for the specified flow (server-to-client direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of packets
 */
uint64_t qmdpi_flow_packets_get_stc(struct qmdpi_flow *flow);

/**
 * Get number of bytes for the specified flow.
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of bytes
 */
uint64_t qmdpi_flow_bytes_get(struct qmdpi_flow *flow);

/**
 * Get number of bytes for the specified flow (client-to-server direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of bytes
 */
uint64_t qmdpi_flow_bytes_get_cts(struct qmdpi_flow *flow);

/**
 * Get number of bytes for the specified flow (server-to-client direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of bytes
 */
uint64_t qmdpi_flow_bytes_get_stc(struct qmdpi_flow *flow);

/**
 * Get number of packets processed by the DPI engine for the specified flow.
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of packets
 */
uint64_t qmdpi_flow_packets_processed_get(struct qmdpi_flow *flow);

/**
 * Get number of packets processed by the DPI engine for the specified flow
 * (client-to-server direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of packets
 */
uint64_t qmdpi_flow_packets_processed_get_cts(struct qmdpi_flow *flow);

/**
 * Get number of packets processed by the DPI engine for the specified flow
 * (server-to-client direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of packets
 */
uint64_t qmdpi_flow_packets_processed_get_stc(struct qmdpi_flow *flow);

/**
 * Get number of bytes processed by the DPI engine for the specified flow.
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of bytes
 */
uint64_t qmdpi_flow_bytes_processed_get(struct qmdpi_flow *flow);

/**
 * Get number of bytes processed by the DPI engine for the specified flow
 * (client-to-server direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of bytes
 */
uint64_t qmdpi_flow_bytes_processed_get_cts(struct qmdpi_flow *flow);

/**
 * Get number of bytes processed by the DPI engine for the specified flow
 * (server-to-client direction).
 * @note If the Flow Manager defragmentation is disabled
 * (as it is by default), the Volume (Bytes) and Volume (Packets) API functions
 * do not take into account fragmented IP packets. Flow Manager defragmentation
 * can be enabled at Engine initialization (see fm_ip_defrag_enable).
 *
 * @param flow pointer to flow context
 * @return number of bytes
 */
uint64_t qmdpi_flow_bytes_processed_get_stc(struct qmdpi_flow *flow);

/**
 * Get the start time of the specified flow.
 *
 * @param flow pointer to flow context
 * @param tv flow's start time
 */
void qmdpi_flow_start_time_get(struct qmdpi_flow *flow, struct timeval *tv);

/**
 * Get the last packet timestamp of the specified flow.
 *
 * @param flow pointer to flow context
 * @param tv flow's last packet timestamp
 */
void qmdpi_flow_last_packet_time_get(struct qmdpi_flow *flow,
                                     struct timeval *tv);

/**
 * Get flow duration.
 *
 * @param flow pointer to flow context
 * @param tv flow duration.
 */
void qmdpi_flow_duration_get(struct qmdpi_flow *flow, struct timeval *tv);

/**
 * Get flow TCP SYN counter.
 *
 * @param flow pointer to flow context
 * @return      One of the following:
 * - TCP syn state count
 * - 0 otherwise (e.g. not a TCP flow)
 */
uint32_t qmdpi_flow_tcp_syn_get(struct qmdpi_flow *flow);

/**
 * Get flow TCP ACK counter.
 *
 * @param flow pointer to flow context
 * @return      One of the following:
 * - TCP ack state count
 * - 0 otherwise (e.g. not a TCP flow)
 */
uint32_t qmdpi_flow_tcp_ack_get(struct qmdpi_flow *flow);

/**
 * Get flow TCP SYNACK counter.
 * @param flow pointer to flow context
 * @return      One of the following:
 * - TCP syn/ack state count
 * - 0 otherwise (e.g. not a TCP flow)
 */
uint32_t qmdpi_flow_tcp_synack_get(struct qmdpi_flow *flow);

/**
 * Get flow TCP FIN counter.
 *
 * @param flow pointer to flow context
 * @return      One of the following:
 * - TCP fin state count,
 * - 0 otherwise (e.g. not a TCP flow)
 */
uint32_t qmdpi_flow_tcp_fin_get(struct qmdpi_flow *flow);

/**
 * Get flow TCP FINACK counter.
 *
 * @param flow pointer to flow context
 * @return      One of the following:
 * - TCP fin/ack state count
 * - 0 otherwise (e.g. not a TCP flow)
 */
uint32_t qmdpi_flow_tcp_finack_get(struct qmdpi_flow *flow);

/**
 * Get flow TCP RST counter.
 *
 * @param flow pointer to flow context
 * @return      One of the following:
 * - TCP rst state count
 * - 0 otherwise (e.g. not a TCP flow)
 */
uint32_t qmdpi_flow_tcp_rst_get(struct qmdpi_flow *flow);

/**
 * Reset the value of the specified flow metric (Packet Mode Only).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to worker
 * @param       flow pointer to flow
 * @param       flow_metric_bitfield flow metric to be reset
 * @return      0 if success, error code otherwise
 */
int qmdpi_flow_metric_reset(struct qmdpi_worker *worker,
                            struct qmdpi_flow *flow,
                            uint64_t flow_metric_bitfield);

/** @} */

/*--------------------------------------
 * Flow timeout
 *------------------------------------*/

/**
 * @defgroup flow_timeout Flow Timeout
 * @ingroup flow
 * @{
 */

/**
 * Get flow timeout value for the specified flow.
 *
 * @param flow pointer to flow context
 * @param timeout set to requested timeout
 */
void qmdpi_flow_timeout_get(struct qmdpi_flow *flow, struct timeval *timeout);

/**
 * Get flow expiration timestamp for the specified flow.
 *
 * @param flow pointer to flow context
 * @param ts expiration timestamp
 */
void qmdpi_flow_expiration_ts_get(struct qmdpi_flow *flow, struct timeval *ts);

/**
 * Callback: invoked when updating timeout value for the specified flow.
 *
 * @param       flow pointer to a valid flow context
 * @param       timeout new timeout value
 */
typedef void (*qmdpi_flow_timeout_updated_callback_t)(struct qmdpi_flow *flow,
        struct timeval const *timeout);

/**
 * Set function called when flow timeout is updated.
 *
 * @param       callback callback
 * @return      0 if success, error code otherwise
 */
int qmdpi_flow_timeout_updated_callback_set(
    qmdpi_flow_timeout_updated_callback_t callback);

/**
 * Update timeout value (PACKET MODE ONLY).
 *
 * @warning     Packet-mode only.
 * @param       worker pointer to worker
 * @param       flow pointer to flow
 * @param       timeout new timeout value (relative value)
 * @return      0 if success, error code otherwise
 */
int qmdpi_flow_timeout_update(struct qmdpi_worker *worker,
                              struct qmdpi_flow *flow,
                              struct timeval *timeout);

/** @} */

/*--------------------------------------
 * Flow offloading
 *------------------------------------*/

/**
 * @defgroup flow_offloading Flow Offloading
 * @ingroup flow
 * @{
 */

/**
 * Mark the specified attribute as discarded for the specified flow.
 * Calls to this function should be followed by a call to ::qmdpi_flow_update.
 *
 * @param worker pointer to a valid worker instance.
 * @param flow pointer to flow context
 * @param path_index path index
 * @param signature_id identifier of the signature containing the attribute to be discarded
 * @param attr_id identifier of the attribute to be discarded
 * @return      0 on success, error code otherwise
 */
int qmdpi_flow_offloading_attr_discard(struct qmdpi_worker *worker,
                                       struct qmdpi_flow *flow,
                                       int path_index,
                                       int signature_id,
                                       int attr_id);
/**
 * Compute and update the flow's offloading status
 * (based on registered and discarded metadata).
 * Should be called once per PDU after last ::qmdpi_flow_offloading_attr_discard call.
 *
 * @param worker pointer to a valid worker instance.
 * @param flow   pointer to flow context
 * @return      One of the following:
 * - 1 if the flow is now offloaded
 * - 0 if not
 */
int qmdpi_flow_offloaded_status_update(struct qmdpi_worker *worker,
                                       struct qmdpi_flow   *flow);

/**
 * Force the offloading of the specified flow; this is irreversible.
 * @warning This function is deprecated; the new version of this function is ::qmdpi_flow_offload.
 *
 * @param flow pointer to flow context
 * @return 0 if success, error code otherwise
 */
int qmdpi_flow_set_offloaded(struct qmdpi_flow *flow);

/**
 * Get the offloaded state of the specified flow.
 *
 * @param flow pointer to flow context
 * @return 1 if offloaded, 0 if not
 */
int qmdpi_flow_is_offloaded(struct qmdpi_flow *flow);

/** @} */

/*--------------------------------------
 * Flow Table Synchronization
 *------------------------------------*/

/**
 * @defgroup flow_table_synchronization Flow Table Synchronization
 *
 * @note the Flow Table Synchronization API MUST ONLY be used
 * between ixEngine instances having both identical version and flavor.
 *
 * @warning the Flow Table Synchronization API is DEPRECATED.
 *
 * @{
 */

/**
 * (PACKET MODE ONLY) Create a flow context in a given worker from a blob (flow manager)
 *
 * @warning This function is deprecated.
 *
 * @warning Packet-mode only.
 * @param   worker pointer to worker
 * @param   blob flow blob
 * @param   len blob len
 * @return  a pointer to flow context created,\n
 *          NULL if error (with errno set)
 */
struct qmdpi_flow *qmdpi_worker_flow_create_from_blob(struct qmdpi_worker
        *worker,
        void *blob,
        unsigned int len);

/**
 * (PACKET MODE ONLY) Update a flow context in a given worker from a blob (flow manager)
 *
 * @warning  This function is deprecated.
 *
 * @warning  Packet-mode only.
 * @param    worker pointer to worker
 * @param    blob flow blob
 * @param    len blob len
 * @return   a pointer to flow context updated,\n
 *           NULL if error (with errno set)
 */
struct qmdpi_flow *qmdpi_worker_flow_update_from_blob(struct qmdpi_worker
        *worker,
        void *blob,
        unsigned int len);

/**
 * Expire flow by blob (PACKET MODE ONLY)
 *
 * @warning  This function is deprecated.
 *
 * @warning  Packet-mode only.
 * @param    worker pointer to worker
 * @param    blob pointer to flow blob
 * @param    len blob len
 * @return   0 if success,\n
 *           ENOENT if flow was not found, EINVAL otherwise
 */
int qmdpi_worker_flow_expire_by_blob(struct qmdpi_worker *worker,
                                     void *blob,
                                     unsigned int len);

/**
 * Get flow blob length (PACKET MODE ONLY)
 *
 * @warning This function is deprecated.
 *
 * @warning Packet-mode only.
 * @param   flow pointer to flow
 * @param   len pointer to len of blob
 * @param   flags requested blob flags (see below)
 * @return  0 if success, error code otherwise
 */
int qmdpi_flow_blob_len_get(struct qmdpi_flow *flow,
                            unsigned int *len,
                            int flags);

/**
 * Get flow blob (PACKET MODE ONLY)
 *
 * @warning This function is deprecated.
 *
 * @warning Packet-mode only.
 * @param   flow pointer to flow
 * @param   blob pointer to blob (allocated by user)
 * @param   len  len of blob
 * @param   flags requested blob flags (see below)
 * @return  0 if success, error code otherwise
 */
int qmdpi_flow_blob_get(struct qmdpi_flow *flow,
                        void *blob,
                        unsigned int len,
                        int flags);

/**
 * Get flow's inner most 5-tuple values from blob
 *
 * @warning This function is deprecated.
 *
 * @param   blob pointer to flow blob
 * @param   len blob len
 * @param   l3proto layer 3 protocol (e.g. Q_PROTO_IP)
 * @param   l4proto layer 4 protocol (e.g. Q_PROTO_TCP)
 * @param   l3client layer 3 client addr (Network Byte Order)
 * @param   l4client layer 4 client port (Network Byte Order)
 * @param   l3server layer 3 server addr (Network Byte Order)
 * @param   l4server layer 4 server port (Network Byte Order)
 * @return  0 if success, error code otherwise
 */
int qmdpi_flow_blob_5tuple_inner_get(void *blob,
                                     unsigned int len,
                                     int *l3proto,
                                     int *l4proto,
                                     void **l3client,
                                     void **l4client,
                                     void **l3server,
                                     void **l4server);


/**
 * Get blob flags
 *
 * @warning This function is deprecated.
 *
 * @param   blob pointer to flow blob
 * @param   len blob len
 * @param   flags pointer to blob flags
 * @return  0 if success, error code otherwise
 */
int qmdpi_flow_blob_flags_get(void *blob,
                              unsigned int len,
                              int *flags);

/**
 * Calculate the "Community ID" flow hash.
 * This is a standardized string identifier representing a given network flow,
 * used to correlate between data sets.
 *
 * @warning     In Packet-mode, the API function must only be called within the
 *              qmdpi_result processing logic of the related flow.
 *
 * @param       flow pointer to flow context
 * @param       icmp_type value indicating type of icmp or icmp6 message,
 *              e.g. QMDPI_ICMP_ECHO (see qmdpi_const.h for QMDPI_ICMP* values)
 *              In @a PACKET @a MODE, this parameter is ignored.
 *              In @a STREAM @a MODE, this parameter is used only for flows
 *              where the layer 4 protocol is Q_PROTO_ICMP or Q_PROTO_ICMP6
 * @ param      seed seed value used to compute community IDs.
 * @ param      base64_encoding  set to 1 to get base64-encoded community ID;
 *              0 otherwise.
 * @ param      community_id user allocated buffer of QMDPI_CID_MAX_SIZE for
 *              storing the computed community ID value.
 *
 * @return      One of the following:
 * - actual size of community_id string if success
 * - error code otherwise
 */

int qmdpi_flow_community_id_get(struct qmdpi_worker *worker,
                                struct qmdpi_flow *flow,
                                uint16_t icmp_type,
                                uint16_t seed,
                                uint8_t base64_encoding,
                                char *community_id);


/** @} */

/*------------------------------------------------------------------------------
 * License API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup license License API
 * @{
 */

/**
 * Load license from a given license file (.bin).
 *
 * @param       filename filename
 * @return      0 if success, error code otherwise
 */
int qmdpi_license_load_from_file(const char *filename);

/**
 * Load license from a buffer address.
 *
 * @param       license pointer to license buffer
 * @param       len length of buffer
 * @param       serial serial number string
 * @return      0 if success, error code otherwise
 */
int qmdpi_license_load_from_addr(const char *license,
                                 int len,
                                 const char *serial);

/**
 * Tell if license is loaded or not
 *
 * @return      1 if loaded,\n
 *              0 if not loaded
 */
int qmdpi_license_is_loaded(void);

/**
 * Destroy the license
 */
void qmdpi_license_destroy(void);

/**
 * Dump license information.
 *
 * @param       output Pointer to output stream
 * @param       output_fn output function
 * @return      return value of the output function, error code otherwise
 * @note Output format is as follows:<BR>
 *     Serial Number: __serial_number__<BR>
 *     Expiration date (DD-MM-YYYY): __date__
 */
int qmdpi_license_dump(void *output,
                       qmdpi_output_fn_t output_fn);

/** @} */

/*------------------------------------------------------------------------------
 * Exception API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup exception Exception API
 * @{
 */

/**
 * Callback: invoked when an exception is thrown.
 *
 * @param       excetion_type exception type
 * @param       ctx pointer to context of exception (output parameter) (for future use)
 * @param       arg pointer to user defined object
 * @return      defined by callback
 */
typedef int (*qmdpi_exception_callback_t)(uint32_t exception_type, void *ctx,
        void *arg);

/**
 * Set exception callback.
 *
 * @param       callback function to be called on exception
 * @param       arg pointer to user defined object
 * @return      0 if success, error code otherwise
 */
int qmdpi_exception_callback_set(qmdpi_exception_callback_t callback,
                                 void *arg);

/**
 * Get exception string for the specified exception ID.
 *
 * @param       exception Exception ID
 * @return      exception string if success,\n
 *              NULL if error (with errno set)
 */
char const *qmdpi_exception_get_string(int exception);

/** @} */

/*------------------------------------------------------------------------------
 * Error API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup error Error API
 * @{
 */

/**
 * Get errno value.
 *
 * @return      errno value
 */
int qmdpi_error_get(void);

/**
 * Get error string for the specified error ID.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       error error value
 * @return      error string if success,\n
 *              NULL if error (with errno set)
 */
char const *qmdpi_error_get_string(struct qmdpi_bundle *bundle, int error);

/** @} */

/*------------------------------------------------------------------------------
 * Function Override API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup override Function Override API
 */

/**
 * @defgroup override_interface Set Functions
 * @ingroup override
 * @{
 */

/**
 * Set overriden function.
 *
 * @warning Calls to ::qmdpi_interface_set must be executed prior to engine
 * creation, i.e. before calling ::qmdpi_engine_create.
 *
 * @param       func_string the name of the function to override
 * @param       func_ptr pointer to the function to override
 * @return      0 if success, error code otherwise
 */
int qmdpi_interface_set(const char *func_string, void *func_ptr);

/** @} */

/*--------------------------------------
 * Default allocator
 *------------------------------------*/

/**
 * @defgroup override_malloc_default Default Allocator API
 * @ingroup override
 * @{
 */

/**
 * Internal default malloc.
 * Can be used when the default malloc has been overridden
 * to provide internal malloc functionality.
 *
 * @param nr : number of elements to allocate
 * @param size : size of one element to allocate
 * @param ctx : memory context
 * @param sid : structure id
 *
 * @return      One of the following:
 * - allocated pointer
 * - NULL if error
 */
void *qm_malloc_default(uint32_t nr, uint32_t size, unsigned int ctx,
                        unsigned int sid);

/**
 * Internal default free.
 * Can be used when the default free has been overridden
 * to provide internal free functionality.
 *
 * @param ptr : pointer to free
 * @param ctx : memory context
 * @param sid : structure id
 */
void qm_free_default(void *ptr, unsigned int ctx, unsigned int sid);

/**
 * Internal default realloc.
 * Can be used when default realloc has been overridden
 * to provide internal free functionality.
 *
 * @param ptr : pointer to realloc
 * @param nr : number of element to allocate
 * @param size : size of one element to allocate
 * @param ctx : memory context
 * @param sid : structure id
 *
 * @return      One of the following:
 * - reallocated pointer
 * - NULL if error
 */
void *qm_realloc_default(void *ptr,
                         uint32_t nr,
                         uint32_t size,
                         unsigned int ctx,
                         unsigned int sid);

/**
 * Gets the size of the memchunk that was allocated with this pointer.
 *
 * @warning Use only with a pointer allocated with qm_{malloc,realloc}_default
 * @param ptr : pointer that was used to allocate
 *
 * @return the size of the allocated memchunk to which ptr points
 */
uint64_t qm_memchunk_size_get_default(void *ptr);

/**
 * Gets the type of internal pool from which an allocated memory chunk belongs.
 *
 * @warning Use only with a pointer allocated with qm_{malloc,realloc}_default
 * @param ptr : pointer that was used to allocate
 * @return the internal pool type
 */
qmdpi_pool_type_t qm_memchunk_type_get_default(void *ptr);


/**
 * Internal default allocation system initialization (e.g. pool initializations).
 *
 * @return 0 if success, negative value otherwise.
 *
 * @note If the user overloads @a qm_malloc_init and does not call
 * @a qm_malloc_init_default in his implementation, then he will not be able to
 * use the functions @a qm_malloc_default, @a qm_free_default,
 * @a qm_realloc_default.
 */
int qm_malloc_init_default(void);

/**
 * Internal default allocation system cleanup.
 */
void qm_malloc_cleanup_default(void);

/** @} */

/*--------------------------------------
 * Internal alloc stats
 *------------------------------------*/

/**
 * @defgroup override_malloc_stats Internal Alloc Stats
 * @ingroup override
 * @{
 */

/**
 * Get allocation context description by ID.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       ctx_id ID of the allocation context
 * @return      context description,\n
 *              NULL if error (with errno set)
 */
char const *qmdpi_malloc_ctx_desc_get_byid(struct qmdpi_bundle *bundle,
        int ctx_id);

/**
 * Get allocated structure description by ID.
 *
 * @param       bundle pointer to a valid bundle instance
 * @param       struct_id ID of the allocatd structure
 * @return      allocated struct description,\n
 *              NULL if error (with errno set)
 */
char const *qmdpi_malloc_struct_desc_get_byid(struct qmdpi_bundle *bundle,
        int struct_id);

/** @} */

/*------------------------------------------------------------------------------
 * Platform-specific API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup platform Platform-specific API
 * @{
 */
/** @} */

/**
 * @defgroup platform_caviumse Cavium SE API
 * @ingroup platform
 * @{
 */

/*--------------------------------------
 * CAVIUMSE
 *------------------------------------*/

/**
 * Set default memory arena used by internal allocators.
 *
 * @param       arena cvmx_arena_list_t object
 */
void qmdpi_caviumse_arena_set(void *arena);

/** @} */

/*------------------------------------------------------------------------------
 * Misc API
 *----------------------------------------------------------------------------*/

/**
 * @defgroup misc Misc API
 * @{
 */

/**
 * Get packet 2-tuple hashkey.
 *
 * @param       data pointer to packet data
 * @param       datalen packet data len
 * @param       first_header packet's first header type
 * @return      hashkey value if success,\n
 *              0 if error (with errno set)
 */
uint32_t qmdpi_packet_hashkey_get(void const *data, int datalen,
                                  int first_header);

/**
 * Get packet N-tuple hashkey.
 * Note : if a 5-tuple hashkey is requested, then
 *        a 5-tuple hashkey is computed on packets with l4_proto in TCP/UDP/SCTP,
 *        and a 2-tuple hashkey is computed on packets with different l4_proto.
 *
 * @param       data pointer to packet data
 * @param       datalen packet data len
 * @param       first_header packet's first header type
 * @param       ntuple 5-tuple hashkey (QMDPI_5TUPLE_HASHKEY) or 2-tuple hashkey (QMDPI_2TUPLE_HASHKEY)
 * @param       hashkey output hashkey value
 * @return      One of the following:
 * - QMDPI_5TUPLE_HASHKEY if 5-tuple hashkey is computed.
 * - QMDPI_2TUPLE_HASHKEY if 2-tuple hashkey is computed.
 * - QMDPI_EINVAL if error
 */
int qmdpi_packet_ntuple_hashkey_get(void const     *data,
                                    int             datalen,
                                    int             first_header,
                                    int             ntuple,
                                    uint32_t       *hashkey);

/** @} */

/*--------------------------------------
 * Extra Library Management
 *------------------------------------*/

/**
 * @defgroup extra Extra Library Management
 * @{
 */

/**
 * Create instance for the specified extra library from file:
 * load library object, initialize and attach it to the specified engine.
 *
 * @warning Requires:
 * - Qosmos Protocol Bundle version 1.570.0-22 or higher and
 * - Qosmos ixEngine version 5.7.0-36 or higher.
 * @param       engine pointer to a valid engine instance
 * @param       extralib_type type of the extralib. Allowed value: QMDPI_EXTRALIB_LIBQMMLETC
 * @param       filename filename of shared library object to load
 * if set to NULL, the extra library instance gets created from the library linked with
 * the user application.
 * @return      One of the following:
 * - a pointer to created extralib instance
 * - NULL if error (with errno set)
 */
struct qmdpi_extralib *qmdpi_extralib_create_from_file(struct qmdpi_engine
        *engine,
        int                  extralib_type,
        const char          *filename);
/**
 * Destroy the specified extra library.
 *
 * @warning Requires:
 * - Qosmos Protocol Bundle version 1.570.0-22 or higher and
 * - Qosmos ixEngine version 5.7.0-36 or higher.
 * @param       extralib pointer to a valid extralib instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_extralib_destroy(struct qmdpi_extralib *extralib);

/**
 * Activate the specified extralib instance.
 * This will mark this extralib as active in the attached engine instance.
 *
 * @warning Requires:
 * - Qosmos Protocol Bundle version 1.570.0-22 or higher and
 * - Qosmos ixEngine version 5.7.0-36 or higher.
 * @param       extralib pointer to a valid extralib instance
 * @return      One of the following:
 * - 0 if success
 * - error code otherwise
 */
int qmdpi_extralib_activate(struct qmdpi_extralib *extralib);

/**
 * Get partial classification path out of the specified FQDN.
 *
 * @param     worker Pointer to a valid worker
 * @param     bundle Pointer to a valid bundle instance
 * @param     fqdn NULL-terminated FQDN to be processed
 * @param     fqdn_classif_path Pointer to the classification path
 * @return    One of the following:
 * - Length of fqdn_classif_path if success
 * - a negative integer if error
 */

int qmdpi_worker_process_fqdn(struct qmdpi_worker *worker,
                              struct qmdpi_bundle *bundle,
                              char const          *fqdn,
                              struct qmdpi_path   *fqdn_classif_path);
/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __QMDPI_H__ */
