/*
 * This file is a part of the Qosmos ixEngine.
 *
 * Copyright ENEA Qosmos Division 2000-2022 - All rights reserved
 *
 * This computer program and all its components are protected by
 * authors' rights and copyright law and by international treaties.
 * Any representation, reproduction, distribution or modification
 * of this program or any portion of it is forbidden without
 * Qosmos explicit and written agreement and may result in severe
 * civil and criminal penalties, and will be prosecuted
 * to the maximum extent possible under the law.
*/

#ifndef __QMDPI_CONST_H__
#define __QMDPI_CONST_H__

#define QMDPI_INSTANCE_MAX 2048   /*!< max number of engine instances */
#define QMDPI_WORKER_MAX    256   /*!< max number of workers per instance */


/** @brief Attribute flags */
#define QMDPI_ATTR_PARENT_START                (1 << 0) /*!< Attribute flag: parent start */
#define QMDPI_ATTR_PARENT_END                  (1 << 1) /*!< Attribute flag: parent end */
#define QMDPI_ATTR_DIR_CTS                     (1 << 2) /*!< Attribute flag: direction client-to-server */
#define QMDPI_ATTR_METADATA                    (1 << 4) /*!< Attribute flag: std attr needed  */
#define QMDPI_ATTR_IDS                         (1 << 5) /*!< Attribute flag: attr needed by IDS rule engine */

/** @brief Signature features (used for offloading decision) */
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
#define QMDPI_SIG_FEATURE_TUNNELING             (1 << 0) /*!< Obsolete */
#define QMDPI_SIG_FEATURE_MULTIPLEXING          (1 << 1)
#define QMDPI_SIG_FEATURE_SIGNALIZATION         (1 << 2)
#define QMDPI_SIG_FEATURE_INHERITANCE           (1 << 3) /*!< Obsolete */
#define QMDPI_SIG_FEATURE_DECLASSIFICATION      (1 << 4)
#define QMDPI_SIG_FEATURE_NO_CACHING            (1 << 5) /*!< internal use */
#else
#if __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
#define QMDPI_SIG_FEATURE_TUNNELING             (1 << 7) /*!< Obsolete */
#define QMDPI_SIG_FEATURE_MULTIPLEXING          (1 << 6)
#define QMDPI_SIG_FEATURE_SIGNALIZATION         (1 << 5)
#define QMDPI_SIG_FEATURE_INHERITANCE           (1 << 4) /*!< Obsolete */
#define QMDPI_SIG_FEATURE_DECLASSIFICATION      (1 << 3)
#define QMDPI_SIG_FEATURE_NO_CACHING            (1 << 2) /*!< internal use */
#else
#error "Please fix LITTLE/BIG_ENDIAN define"
#endif
#endif

/**
 * @brief Link modes
 */
enum {
    QMDPI_PROTO_ETH    = 1,
    QMDPI_PROTO_IP     = 2,
    QMDPI_PROTO_MPLS   = 3,
    QMDPI_PROTO_IP6    = 6,
    QMDPI_PROTO_PPP    = 7,
    QMDPI_PROTO_8021Q  = 8,
    QMDPI_PROTO_8021AD = 9,
    QMDPI_PROTO_802_11 = 10,
};

/** @brief DPI processing return values. */
enum {
    QMDPI_PROCESS_OK                          = 0, /*!< Packet has been processed successfully */
    QMDPI_PROCESS_MORE                        = 1, /*!< A chunk of the packet has been processed, more chunks are available in the packet */
    QMDPI_PROCESS_IP_DEFRAG_BUF               = 2, /*!< Packet has been buffered by IP defragmentation module, a previous fragment was missing */
    QMDPI_PROCESS_IP_DEFRAG_DROP              = 3, /*!< Packet has been dropped by IP defragmentation module, this fragment overlaps a previous fragment */
    QMDPI_PROCESS_BOUNDS_FAILURE              = 4, /*!< Packet has not been decoded successfully by the Flow Manager */
    QMDPI_PROCESS_BOUNDS_DROP                 = 5, /*!< Packet has been dropped by the Flow Manager (i.e: fragmented IP packets when IP defragmentation module is disabled) */
    QMDPI_PROCESS_TCP_SEGMENT_NOT_PROCESSED   = 6, /*!< Packet has been buffered by TCP reassembly module (i.e. packet was not processed) */
    QMDPI_PROCESS_TCP_SEGMENT_DROPPED         = 7, /*!< Packet has been dropped by TCP reassembly module (i.e. packet was not processed) */
};


/**
 * @brief Direction indices
 */
enum {
    QMDPI_DIR_0 = 0,
    QMDPI_DIR_1,
};

/**
 * @brief Direction (for Packet injection mode), there are 2 different use cases:<BR/>
 * - Packets are captured on a active network interface: Incoming packets from a source A to a destination B should be associated with Outgoing packets from B to A.<BR/>
 * In this case incoming packets should be processed with the direction @a QMDPI_DIR_INCOMING and outgoing packets should be processed with the direction @a QMDPI_DIR_OUTGOING.<BR/>
 * - Packets are received from a tapping interface: Packet direction cannot be determined during the capture. Both packets from A to B and From B to A are seen as incoming packets.<BR/>
 * In this case incoming packets should be processed with the direction QMDPI_DIR_DEFAULT.
 */
enum {
    QMDPI_DIR_INCOMING = 0,
    QMDPI_DIR_OUTGOING,
    QMDPI_DIR_DEFAULT,
};

/**
 * @brief Direction (for Stream injection mode)
 */
enum {
    QMDPI_DIR_CTS = 1,  /*!< Client to Server */
    QMDPI_DIR_STC,      /*!< Server to Client */
    QMDPI_DIR_UNKNOWN,  /*!< Should not be used */
};

/**
 * @brief Data types
 */
enum {
    QMDPI_DATA_NOTYPE       = 0,   /*!< no type constant value */
    QMDPI_DATA_UINT8        = 1,   /*!< unsigned 1-byte constant value */
    QMDPI_DATA_UINT16       = 2,   /*!< unsigned 2-bytes constant value */
    QMDPI_DATA_UINT32       = 3,   /*!< unsigned 4-bytes constant value */
    QMDPI_DATA_UINT64       = 4,   /*!< unsigned 8-bytes constant value */
    QMDPI_DATA_PTR          = 5,   /*!< pointer constant value (size is CPU dependant) */
    QMDPI_DATA_STRING       = 6,   /*!< string constant value */
    QMDPI_DATA_IP_NET       = 7,   /*!< ip network address constant value */
    QMDPI_DATA_MAC_ADDR     = 8,   /*!< ethernet mac address constant value */
    QMDPI_DATA_IP_ADDR      = 9,   /*!< ip network address and mask constant value */
    QMDPI_DATA_PATH         = 10,  /*!< protocol path constant value, DEPRECATED */
    QMDPI_DATA_TIMEVAL      = 11,  /*!< number of seconds and microseconds constant value */
    QMDPI_DATA_BINARY       = 12,  /*!< binary constant value */
    QMDPI_DATA_BUFFER       = 13,  /*!< binary buffer content */
    QMDPI_DATA_CHAR         = 14,  /*!< 1 character constant value */
    QMDPI_DATA_IP6_ADDR     = 15,  /*!< ip6 address constant value */
    QMDPI_DATA_PORT         = 16,  /*!< DEPRECATED */
    QMDPI_DATA_POINT        = 17,  /*!< point constant value, DEPRECATED */
    QMDPI_DATA_PORT_RANGE   = 18,  /*!< tcp/udp port range constant value, DEPRECATED */
    QMDPI_DATA_DATE         = 19,  /*!< date constant value, DEPRECATED */
    QMDPI_DATA_TIMEARG      = 20,  /*!< time argument constant value, DEPRECATED */
    QMDPI_DATA_STRING_INDEX = 21,  /*!< string index constant value (an association between a string and an integer) */
    QMDPI_DATA_FLOAT        = 22,  /*!< float constant value, DEPRECATED */
    QMDPI_DATA_LAYERID      = 23,  /*!< Layer ID value, DEPRECATED */
    QMDPI_DATA_PARENT       = 25,  /*!< (child_event, is_end_sent) */
    QMDPI_DATA_IP_RANGE     = 26,  /*!< (min_ip, max_ip), DEPRECATED */
    QMDPI_DATA_MATCH        = 27,  /*!< classification_match attribute, DEPRECATED */
    QMDPI_DATA_SM_ANOMALY   = 28,  /*!< statemachine anomaly attributes */
    QMDPI_DATA_IP6_RANGE    = 29,  /*!< (min_ip6, max_ip6), DEPRECATED */
    QMDPI_DATA_INT64        = 30,  /*!< signed 8-bytes constant value */
    QMDPI_DATA_INT32        = 31,  /*!< signed 4-bytes constant value */
    QMDPI_DATA_PEER         = 32,  /*!< information related to a predicted classification <IP address, L4 port, L4 protocol, path> */
    QMDPI_DATA_STRING_SZ    = 33,  /*!< sized string (Reserved for (the)machine), DEPRECATED */
    QMDPI_DATA_LIST         = 34,  /*!< list of fixed size values (Reserved for (the)machine), DEPRECATED */
};

/**
 * @brief Framework configuration flags
 */
enum {
    QMDPI_CONFIG_FLAG_RUNTIME = 0, /*!< config value can be changed at init or runtime */
    QMDPI_CONFIG_FLAG_INIT,        /*!< config value can be changed at init (and NOT at runtime) */
};

/**
 * @addtogroup error
 * @{
 */

/**
 * @brief Error values
 */
enum {
    QMDPI_SUCCESS = 0,             /*!< success */

    QMDPI_EINVAL = -1,             /*!< invalid param */
    QMDPI_ENOSYS = -2,             /*!< function not implemented */

    QMDPI_ENOMEM = -5,             /*!< out of memory */
    QMDPI_EBUSY = -6,              /*!< device or resource busy */
    QMDPI_EEXIST = -7,             /*!< entry already exists */
    QMDPI_ENOEXIST = -8,           /*!< entry does not exist */
    QMDPI_ENOENT = -8,             /*!< alias for QMDPI_ENOEXIST */

    QMDPI_EPERM = -15,             /*!< permission denied */

    QMDPI_EBUNDLE_DEFAULT_UNAVAILABLE = -17, /*!< default bundle not available */
    QMDPI_EBUNDLE_DEFAULT_LOADED = -18,      /*!< default bundle already loaded */
    QMDPI_EEXTRALIB_DEFAULT_LOADED = -18,    /*!< default extralib already loaded */
    QMDPI_EBUNDLE_LOAD = -19,                /*!< generic bundle loading error */
    QMDPI_EEXTRALIB_LOAD = -19,              /*!< generic extralib loading error */
    QMDPI_EBUNDLE_SYM_NOT_FOUND = -20,       /*!< ubundle_get symbol not found */
    QMDPI_EEXTRALIB_SYM_NOT_FOUND = -20,     /*!< extralib entry point not found */
    QMDPI_EBUNDLE_BUNDLE_NOT_FOUND = -21,    /*!< bundle not found */
    QMDPI_EEXTRALIB_EXTRALIB_NOT_FOUND = -21,/*!< extralib not found */
    QMDPI_EBUNDLE_DEFAULT_USED = -22,        /*!< bundle already used */
    QMDPI_EBUNDLE_INIT = -23,                /*!< generic bundle initialization error */
    QMDPI_EBUNDLE_SWAPPABLE = -24,           /*!< bundle can't be swapped */
    QMDPI_EBUNDLE_INVAL = -25,               /*!< bad bundle provided */
    QMDPI_ENOARENA = -26,                    /*!< arena not defined */
    QMDPI_EOVERFLOW = -27,                   /*!< value too large */
    QMDPI_EINIT = -28,                       /*!< module not initialized */
    QMDPI_EBUNDLE_VERSION = -29,             /*!< invalid bundle version */
    QMDPI_EFLOW_LOOKUP = -30,                /*!< flow could not be created */
    QMDPI_EUNAVAIL = -31,                    /*!< data not available */
    QMDPI_EBUNDLE_RULE_DOES_NOT_EXIST = -32, /*!< rule does not exist */
};
/** @} */

/**
 * @addtogroup exception
 * @{
 */

/**
 * @brief Exception type
 */
enum {
    /**
     * DEPRECATED
     * Triggered when per worker dynamic allocation exceed max_worker_memory param
     * - (no param supplied)
     */
    QMDPI_EXCEPTION_NO_MEMORY = 0,

    /**
     * DEPRECATED
     * Triggered when worker count exceed maximum number of workers (currently, 72)
     * - (no param supplied)
     */
    QMDPI_EXCEPTION_CPUID_OVERFLOW = 2,

    /**
     * Triggered when classification path legnth exceeds maximum length (currently, 16 entries)
     * - param supplied: qmdpi_flow that triggered the exception
     */
    QMDPI_EXCEPTION_PATHLEN_OVERFLOW = 3,

    /**
     * Triggered when parent key length exceeds maximum length (currently, 64 bytes)
     * - param supplied: key (an unsigned char buffer)
     */
    QMDPI_EXCEPTION_PARENTKEY_OVERFLOW = 4,

    /**
     * DEPRECATED
     * Triggered when the number of internal flow context pairs per qmdpi_flow exceeds 1
     * - param supplied: qmdpi_flow that triggered the exception
     */
    QMDPI_EXCEPTION_TOOMANY_TUNNEL = 8,

    /**
     * Triggered when engine is trying to use a qmdpi_flow marked as used by another worker
     * (Stream-Injection Mode only)
     * - param supplied: qmdpi_flow that triggered the exception
     */
    QMDPI_EXCEPTION_FLOW_USED = 9,

    /**
     * Triggered when flat buffer length (used when uncompressing the HTTP server payload) exceeds maximum length (default, 151520 bytes)
     * - param supplied: qmdpi_flow that triggered the exception
     */
    QMDPI_EXCEPTION_FLAT_BUFFER_OVERFLOW = 11,

    /**
     * Triggered when resource is busy
     * - no param supplied
     */
    QMDPI_EXCEPTION_EBUSY = 12,

    /** Triggered when the array that collects pdata items during pattern
    * matching process overflows.
    * - param supplied: index that overflows.
    */
    QMDPI_EXCEPTION_PDATA_ITEMS_OVERFLOW = 13,
};
/** @} */

/**
 * @brief Pool type
 */
typedef enum qmdpi_pool_type {
    QMDPI_POOL_TYPE_MIN = 0,
    QMDPI_POOL_TYPE_INTERNAL_FLOW_CTX = 0,
    QMDPI_POOL_TYPE_32 = 1,
    QMDPI_POOL_TYPE_64 = 2,
    QMDPI_POOL_TYPE_128 = 3,
    QMDPI_POOL_TYPE_256 = 4,
    QMDPI_POOL_TYPE_512 = 5,
    QMDPI_POOL_TYPE_MAX = 6,

} qmdpi_pool_type_t;

/**
 * @brief 'table_id' for the function qmdpi_tdm_dump.
 */
enum {
    QMDPI_REPORT_TABLE_PROTO_DISTRIB,
    QMDPI_REPORT_TABLE_NEW,
    QMDPI_REPORT_TABLE_COMBINED,
    QMDPI_REPORT_TABLE_UNKNOWN,
};

/**
 * @brief 'attribute scope' values.
 **/
enum {
    QMDPI_SCOPE_PKT = 0,      /*!< Attribute value may change in each packet */
    QMDPI_SCOPE_FLOW_MOD = 1, /*!< Attribute value is set for the whole flow lifetime but may change */
    QMDPI_SCOPE_FLOW_FIX = 2, /*!< Attribute value is fixed for the whole flow lifetime */
    QMDPI_SCOPE_FLOW_PRT = 3, /*!< Attribute value is fixed in the context of the parent attribute, but another value can be set in the context of any instance of the parent attribute */
};

/**
 * @brief Flow table synchronization flag values.
 **/
enum {
    QMDPI_BLOB_FLAG_FLOW_PATH   = (1 << 0),     /*!< Get the flow path in the blob */
    QMDPI_BLOB_FLAG_FLOW_STATS  = (1 << 1),     /*!< Get the flow stats in the blob */
};

/**
 * @brief flow table allocation mode
 **/
enum {
    QMDPI_FT_ALLOC_DYNAMIC,          /*!< Flow manager dynamic allocation mode */
    QMDPI_FT_ALLOC_STATIC,           /*!< Flow manager static allocation mode */
    QMDPI_FT_ALLOC_STATIC_NO_REUSE,  /*!< Flow manager static allocation mode with flow reuse */
    QMDPI_FT_ALLOC_HYBRID,           /*!< Flow manager hybrid allocation mode */
    QMDPI_FT_ALLOC_END,
};

/**
 * @brief Flow state
 **/
enum {
    QMDPI_FLOW_STATE_STARTED      = 0,
    QMDPI_FLOW_STATE_SYN          = 1,
    QMDPI_FLOW_STATE_SYNACK       = 2,
    QMDPI_FLOW_STATE_ESTABLISHED  = 3,
    QMDPI_FLOW_STATE_FIN          = 4,
    QMDPI_FLOW_STATE_FINACK       = 5,
    QMDPI_FLOW_STATE_RST          = 6,
    QMDPI_FLOW_STATE_FINACK_ACK   = 7,
    QMDPI_FLOW_STATE_MAX          = 8,
};

/**
 * @brief Flow metrics
 **/

enum {
    QMDPI_FLOW_METRIC_ALL                   = (long)0xFFFFFFFFFFFFFFFFULL,
    QMDPI_FLOW_METRIC_PACKETS_CTS           = (1 << 0),
    QMDPI_FLOW_METRIC_PACKETS_STC           = (1 << 1),
    QMDPI_FLOW_METRIC_PACKETS               = QMDPI_FLOW_METRIC_PACKETS_CTS | QMDPI_FLOW_METRIC_PACKETS_STC,
    QMDPI_FLOW_METRIC_PACKETS_PROCESSED_CTS = (1 << 2),
    QMDPI_FLOW_METRIC_PACKETS_PROCESSED_STC = (1 << 3),
    QMDPI_FLOW_METRIC_PACKETS_PROCESSED     = QMDPI_FLOW_METRIC_PACKETS_PROCESSED_CTS | QMDPI_FLOW_METRIC_PACKETS_PROCESSED_STC,
    QMDPI_FLOW_METRIC_BYTES_CTS             = (1 << 4),
    QMDPI_FLOW_METRIC_BYTES_STC             = (1 << 5),
    QMDPI_FLOW_METRIC_BYTES                 = QMDPI_FLOW_METRIC_BYTES_CTS | QMDPI_FLOW_METRIC_BYTES_STC,
    QMDPI_FLOW_METRIC_BYTES_PROCESSED_CTS   = (1 << 6),
    QMDPI_FLOW_METRIC_BYTES_PROCESSED_STC   = (1 << 7),
    QMDPI_FLOW_METRIC_BYTES_PROCESSED       = QMDPI_FLOW_METRIC_BYTES_PROCESSED_CTS | QMDPI_FLOW_METRIC_BYTES_PROCESSED_STC,
    QMDPI_FLOW_METRIC_START_TIME            = (1 << 8),
    QMDPI_FLOW_METRIC_LAST_PACKET_TIME      = (1 << 9),
    QMDPI_FLOW_METRIC_DURATION              = QMDPI_FLOW_METRIC_START_TIME | QMDPI_FLOW_METRIC_LAST_PACKET_TIME,
    QMDPI_FLOW_METRIC_TCP_SYN               = (1 << 10),
    QMDPI_FLOW_METRIC_TCP_SYNACK            = (1 << 11),
    QMDPI_FLOW_METRIC_TCP_FIN               = (1 << 12),
    QMDPI_FLOW_METRIC_TCP_FINACK            = (1 << 13),
    QMDPI_FLOW_METRIC_TCP_ACK               = (1 << 14),
    QMDPI_FLOW_METRIC_TCP_RST               = (1 << 15),
};

/**
 * @brief ioctl commands
 **/
enum {
    QMDPI_CTL_PORT_LIST_SET = 0, /* qmdpi_bundle_signature_ctl_set() arg format: "port_number;..;port_number" */
    QMDPI_CTL_IPV4_DEVICE_ENTRY_REMOVE = 1, /* qmdpi_worker_signature_ctl_set() signature in {"ultrasurf", "hotspot_shield", "hexatech", "anchorfree", "hoxx_vpn", "streamroot", "viber", "ssl", "dns", "pubg", "webrtc", NULL}; arg="virtual_id;ipv4_address;...;ipv4_address"; max ipv4_address entries: 16*/
    QMDPI_CTL_IPV6_DEVICE_ENTRY_REMOVE = 2, /* qmdpi_worker_signature_ctl_set() signature in {"ultrasurf", "hotspot_shield", "hexatech", "anchorfree", "hoxx_vpn", "streamroot", "viber", "ssl", "dns", "pubg", "webrtc", NULL}; arg="virtual_id;ipv6_address;...;ipv6_address"; max ipv6_address entries: 16*/
    QMDPI_CTL_FRAUD_SIGNATURE_LIST_ADD = 3, /* qmdpi_worker_signature_ctl_set() signature in {"evasive_protocol"}; arg="signature_name;signature_name;...;signature_name"; max entries: 64*/
    QMDPI_CTL_FRAUD_SIGNATURE_LIST_REMOVE = 4, /* qmdpi_worker_signature_ctl_set() signature in {"evasive_protocol"}; arg="signature_name;signature_name;...;signature_name"; max entries: 64*/
};

/**
 * @brief N-tuple hashkey
 */
enum {
    QMDPI_5TUPLE_HASHKEY    = 0,
    QMDPI_2TUPLE_HASHKEY    = 1,
};

/**
 * @brief Extra library types
 */
enum {
    QMDPI_EXTRALIB_LIBQMMLETC = 1,
};

/**
 * @brief ICMP Message Type
 */
enum {
    QMDPI_ICMP_ECHO_REPLY = 0,
    QMDPI_ICMP_ECHO = 8,
    QMDPI_ICMP_RTR_ADVERT = 9,
    QMDPI_ICMP_RTR_SOLICIT = 10,
    QMDPI_ICMP_TSTAMP = 13,
    QMDPI_ICMP_TSTAMP_REPLY = 14,
    QMDPI_ICMP_INFO = 15,
    QMDPI_ICMP_INFO_REPLY = 16,
    QMDPI_ICMP_MASK = 17,
    QMDPI_ICMP_MASK_REPLY = 18,
};

/**
 * @brief ICMP6 Message Type
 */
enum {
    QMDPI_ICMPV6_ECHO_REQUEST = 128,
    QMDPI_ICMPV6_ECHO_REPLY = 129,
    QMDPI_ICMPV6_MCST_LISTENER_QUERY = 130,
    QMDPI_ICMPV6_MCST_LISTENER_REPORT = 131,
    QMDPI_ICMPV6_ROUTER_SOLICIT = 133,
    QMDPI_ICMPV6_ROUTER_ADVERT = 134,
    QMDPI_ICMPV6_NEIGHBOR_SOLICIT = 135,
    QMDPI_ICMPV6_NEIGHBOR_ADVERT = 136,
    QMDPI_ICMPV6_NODE_INFO_QUERY  = 139,
    QMDPI_ICMPV6_NODE_INFO_REPLY = 140,
    QMDPI_ICMPV6_HAAD_REQUEST = 144,
    QMDPI_ICMPV6_HAAD_REPLY = 145,
};

/**
 * @brief Max size of the "Community ID"
 */
enum {
    QMDPI_CID_MAX_SIZE  = 256,
};

#endif /* __QMDPI_CONST_H__ */
