/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2000-2016 - All rights reserved
 */

/* standard headers */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <inttypes.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;

struct bundle_ctx {
    struct qmdpi_bundle *bundle;
    int bundle_id;
    const char *bundle_version;
};

struct bundle_ctx bundle_ctx[2];

static unsigned int packet_number;
static unsigned int swapat;
static char *pcapfilename;
static char *soname1, *soname2;
static time_t last_packet_ts;

static int force_cleanup = 0;

static void bundle_swap_usage(const char *prg_name)
{
    fprintf(stderr,
            "%s --swapat=<pkt_num> [--force-unload] <bundle_1> <bundle_2> <pcap_file>\n"
            , prg_name);
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "\t--force-unload : when we manually unload bundle1, "
            "use to option to force unload(garbage collector)\n");
    fprintf(stderr,
            "\t--swapat=<pkt_num> : load bundle_2 at packet number=<pkt_num>\n");
}

static int bundle_swap_parse_param(int argc, char *argv[])
{
    if (argc < 4) {
        bundle_swap_usage(argv[0]);
        exit(1);
    }

    static struct option long_options[] = {
        {"swapat", 1, 0, 0},
        {"force-unload", 0, 0, 0},
        {0, 0, 0, 0},
    };

    while (1) {
        int option_index = 0;
        int c = getopt_long(argc, argv, "", long_options, &option_index);
        if (c == -1) {
            break;
        }
        switch (c) {
            case 0:
                if (!strcmp(long_options[option_index].name, "force-unload")) {
                    force_cleanup = 1;
                } else if (!strcmp(long_options[option_index].name, "swapat")) {
                    if (sscanf(optarg, "%u", &swapat) != 1) {
                        fprintf(stderr, "unable to read pkt number\n");
                        bundle_swap_usage(*argv);
                        exit(-1);
                    }
                }
        }
    }

    pcapfilename = argv[argc - 1];
    soname2      = argv[argc - 2];
    soname1      = argv[argc - 3];

    return 0;
}

static void bundle_swap_play_with_result(struct qmdpi_result *result)
{
    char path_string[512];
    char *active_string;
    struct qmdpi_path *path = qmdpi_result_path_get(result);
    struct qmdpi_bundle *b = qmdpi_flow_bundle_get(qmdpi_result_flow_get(result));
    if (path == NULL || b == NULL) {
        return;
    }
    qmdpi_data_path_to_buffer(b, path_string, sizeof(path_string), path);
    active_string = qmdpi_bundle_is_active(b) ? "Active" : "Inactive";
    fprintf(stdout,
            "Packet #%d processed by bundle ID %d, version %s (%s, %"PRIu64" flow contexts left) flow(%p) path=%s\n",
            packet_number,
            qmdpi_bundle_id_get(b),
            qmdpi_bundle_version_get_string(b),
            active_string,
            qmdpi_bundle_flow_get_count(b),
            qmdpi_result_flow_get(result),
            path_string);

}

static int unload = 0;

static void bundle_swap_process(u_char *user, const struct pcap_pkthdr *phdr,
                                const u_char *pdata)
{

    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* swap bundles */
    if (packet_number == swapat) {
        /* configure and activate bundle 1 */
        qmdpi_bundle_signature_enable_all(bundle_ctx[1].bundle);
        /* activate bundle */
        ret = qmdpi_bundle_activate(bundle_ctx[1].bundle);
        if (ret < 0) {
            fprintf(stderr, "cannot activate bundle %d\n",
                    qmdpi_bundle_id_get(bundle_ctx[1].bundle));
            return;
        }
        printf("bundle %d activated\n", qmdpi_bundle_id_get(bundle_ctx[1].bundle));

        unload = 1;

        if (force_cleanup) {
            if (qmdpi_bundle_flow_get_count(bundle_ctx[0].bundle) != 0) {
                struct qmdpi_flow *flow;
                flow = qmdpi_bundle_get_gc_head(worker, bundle_ctx[0].bundle);
                while (flow) {
                    struct qmdpi_flow *next = qmdpi_bundle_get_gc_next(flow);
                    while (qmdpi_flow_destroy(worker, flow, &result) == QMDPI_PROCESS_MORE);
                    bundle_swap_play_with_result(result);
                    flow = next;
                }
            }
        }

        /* set bundle 0 and all its remaining flow contexts for destruction */
        qmdpi_bundle_destroy(bundle_ctx[0].bundle);
    }

    if (unload && !force_cleanup) {
        if (qmdpi_bundle_flow_get_count(bundle_ctx[0].bundle) == 0) {
            qmdpi_bundle_destroy(bundle_ctx[0].bundle);
        }
    }

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%u\n", packet_number);
            continue;
        }

        /* play with result */
        bundle_swap_play_with_result(result);

    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows */
    if (last_packet_ts < phdr->ts.tv_sec) {

        int nb_remaining = 100;
        do {
            ret = qmdpi_flow_expire_next(worker, &phdr->ts, &result);
            if (ret != 0) {
                break;
            }

            nb_remaining --;

            /* play with result */
            bundle_swap_play_with_result(result);

        } while (nb_remaining);
        if (nb_remaining) {
            last_packet_ts = phdr->ts.tv_sec;
        }
    }
}


/* simply reset pointer to bundle after bundle destruction */
static void bundle_swap_destroy_cb(struct qmdpi_engine *engine,
                                   void *arg)
{
    (void)engine;
    struct bundle_ctx *bundle_ctx = (struct bundle_ctx *)arg;
    printf("Packet #%d bundle %d has been destroyed\n", packet_number,
           bundle_ctx->bundle_id);
    bundle_ctx->bundle = NULL;
}


static int bundle_swap_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance\n");
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance\n");
        goto error_engine;
    }

    /* create 1st bundle instance */
    bundle_ctx[0].bundle = qmdpi_bundle_create_from_file(engine, soname1);
    if (bundle_ctx[0].bundle == NULL) {
        fprintf(stderr, "cannot create 1st bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }
    bundle_ctx[0].bundle_id = qmdpi_bundle_id_get(bundle_ctx[0].bundle);
    bundle_ctx[0].bundle_version = qmdpi_bundle_version_get_string(
                                       bundle_ctx[0].bundle);
    printf("bundle %d created, version %s\n", bundle_ctx[0].bundle_id,
           bundle_ctx[0].bundle_version);

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle_ctx[0].bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols\n");
        goto error_bundle;
    }

    /* set destroy callback for bundle 0 */
    qmdpi_bundle_destroy_callback_set(bundle_ctx[0].bundle, bundle_swap_destroy_cb,
                                      &bundle_ctx[0]);

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle_ctx[0].bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle\n");
        goto error_bundle;
    }
    printf("bundle %d activated\n", qmdpi_bundle_id_get(bundle_ctx[0].bundle));

    /* create 2nd bundle instance */
    bundle_ctx[1].bundle = qmdpi_bundle_create_from_file(engine, soname2);
    if (bundle_ctx[1].bundle == NULL) {
        fprintf(stderr, "cannot create 2nd bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_bundle;
    }
    bundle_ctx[1].bundle_id = qmdpi_bundle_id_get(bundle_ctx[1].bundle);
    bundle_ctx[1].bundle_version = qmdpi_bundle_version_get_string(
                                       bundle_ctx[1].bundle);
    printf("bundle %d created, version %s\n", bundle_ctx[1].bundle_id,
           bundle_ctx[1].bundle_version);

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle_ctx[0].bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;

}

int main(int argc, char **argv)
{
    char errbuf[PCAP_ERRBUF_SIZE];
    pcap_t *pcap;
    int ret, i;

    packet_number = 0;

    /* check arguments */
    if (argc < 5) {
        bundle_swap_usage(argv[0]);
        return 1;
    }

    /* command line parsing */
    bundle_swap_parse_param(argc, argv);


    /* init Qosmos ixEngine */
    ret = bundle_swap_init();
    if (ret < 0) {
        fprintf(stderr, "cannot initialize application\n");
        return 1;
    }

    /* open pcap */
    pcap = pcap_open_offline(pcapfilename, errbuf);
    if (!pcap) {
        fprintf(stderr, "pcap_open: %s\n", errbuf);
        return 1;
    }

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, bundle_swap_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        return 1;
    }

    printf("Cleaning up...\n");
    pcap_close(pcap);


    for (i = 0 ; i < 2 ; i++) {
        if (bundle_ctx[i].bundle) {
            qmdpi_bundle_destroy(bundle_ctx[i].bundle);
        }
    }

    /* expire remaining flow contexts, do not care about processing result */
    while (qmdpi_flow_expire_next(worker, NULL, NULL) == 0);

    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();

    return 0;
}
