/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2000-2016 - All rights reserved
 */

/* standard headers */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle header */
#include <qmdpi_bundle_api.h>

/* L2-L4 protocol headers */
#include <packet_helper.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;

static unsigned packet_number;


/* option management */
static int use_tdm_ext = 0;
static int use_classification_cache = 0;
static unsigned print_method_mode;

static void classification_print_sig_ip(void *header)
{
    char ip1_str[INET_ADDRSTRLEN];
    char ip2_str[INET_ADDRSTRLEN];
    struct qm_ip_hdr *ip_hdr = (struct qm_ip_hdr *)header;

    struct in_addr ip1, ip2;
    ip1.s_addr = ip_hdr->saddr;
    ip2.s_addr = ip_hdr->daddr;
    inet_ntop(AF_INET, &ip1, ip1_str, INET_ADDRSTRLEN);
    inet_ntop(AF_INET, &ip2, ip2_str, INET_ADDRSTRLEN);

    fprintf(stdout, "[IPv4: %s, -> %s, IP Proto:%u] ", ip1_str,
            ip2_str, ip_hdr->protocol);
}

static void classification_print_sig_tcp(void *header)
{
    struct qm_tcp_hdr *tcp = (struct qm_tcp_hdr *)header;
    fprintf(stdout, "[TCP: %u -> %u] ", ntohs(tcp->source),
            ntohs(tcp->dest));
}

static void classification_print_sig_udp(void *header)
{
    struct qm_udp_hdr *udp = (struct qm_udp_hdr *)header;
    fprintf(stdout, "[UDP: %u -> %u] ", ntohs(udp->source),
            ntohs(udp->dest));
}

static void classification_print_sig_ip6(void *header)
{
    char ip1_str[INET6_ADDRSTRLEN];
    char ip2_str[INET6_ADDRSTRLEN];
    struct qm_ip6_hdr *ip6_hdr = (struct qm_ip6_hdr *)header;
    inet_ntop(AF_INET6, (struct in6_addr *)&ip6_hdr->saddr, ip1_str,
              INET6_ADDRSTRLEN);
    inet_ntop(AF_INET6, (struct in6_addr *)&ip6_hdr->daddr, ip2_str,
              INET6_ADDRSTRLEN);

    fprintf(stdout, "[IPv6: %s -> %s Next Proto:%u] ", ip1_str,
            ip2_str, ip6_hdr->nexthdr);
}

static void classification_print_sig(int proto_id, void *header)
{
    switch (proto_id) {
        case Q_PROTO_IP:
            classification_print_sig_ip(header);
            break;
        case Q_PROTO_IP6:
            classification_print_sig_ip6(header);
            break;
        case Q_PROTO_TCP:
            classification_print_sig_tcp(header);
            break;
        case Q_PROTO_UDP:
            classification_print_sig_udp(header);
            break;
    }
}

#define TDM_EXTENSION_LEN 32

static void classification_method_print(char method)
{
    switch (method) {
        case QMDPI_CLASSIF_METHOD_IMPLICIT:
            printf("implicit");
            break;
        case QMDPI_CLASSIF_METHOD_EXPLICIT:
            printf("explicit");
            break;
        case QMDPI_CLASSIF_METHOD_PDATA:
            printf("pdata");
            break;
        case QMDPI_CLASSIF_METHOD_PDATA_IP:
            printf("pdata from IP address");
            break;
        case QMDPI_CLASSIF_METHOD_PDATA_DNS:
            printf("pdata from DNS");
            break;
        case QMDPI_CLASSIF_METHOD_PREDICTIVE:
            printf("predictive");
            break;
        case QMDPI_CLASSIF_METHOD_SPID:
            printf("spid");
            break;
        case QMDPI_CLASSIF_METHOD_BEHAVIORAL:
            printf("behavioral");
            break;
        case QMDPI_CLASSIF_METHOD_PORT_BASED:
            printf("port based");
            break;
        case QMDPI_CLASSIF_METHOD_DOMAIN_FRONTING:
            printf("domain fronting");
            break;
        default:
            printf("%d: unknown method", method);
            break;
    }
}

static void classification_method_result_display(struct qmdpi_bundle *bundle,
                                                 struct qmdpi_result *result)
{
    int attr_id, attr_len, attr_flags, proto_id;
    char const *attr_value;
    struct qmdpi_signature *signature;
    const char *layername;

    printf("Classification methods used for classified protocols: \n");
    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &attr_value, &attr_len, &attr_flags) == 0) {
        if (attr_id != Q_GEN_CLASSIFICATION_METHOD) {
            continue;
        }
        signature = qmdpi_bundle_signature_get_byid(bundle, proto_id);
        layername = qmdpi_signature_name_get(signature);
        printf("%s: ", layername);
        classification_method_print(*attr_value);
        printf("\n");
    }
}

static void classification_result_display(struct qmdpi_worker *worker,
                                          struct qmdpi_result *result)
{
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);
    struct qmdpi_flow *flow;
    struct qmdpi_flow_info *flow_info;
    struct qmdpi_path *path;
    struct qmdpi_bundle *b;
    char buffer[4096];
    int proto_id;
    int ind;


    /* print classification only if protocol path has changed
     * or if an extension lookup can be performed using tdm
     */
    int tdm_extension_lookup = use_tdm_ext &&
                               QMDPI_RESULT_FLAGS_NEED_CLASSIF_EXT(result_flags) &&
                               QMDPI_RESULT_FLAGS_CLASSIFIED_STATE(result_flags);

    flow = qmdpi_result_flow_get(result);
    flow_info = qmdpi_flow_info_get(flow);

    int classif_cache_need = use_classification_cache &&
                             (flow_info != NULL) &&
                             QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags) &&
                             QMDPI_FLOW_INFO_CLASSIF_CACHED(flow_info);

    /* print classification test */
    if ((QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags) == 0) &&
            (classif_cache_need == 0) &&
            (tdm_extension_lookup == 0)) {
        return;
    }

    if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags) || (tdm_extension_lookup ||
                                                          classif_cache_need)) {

        printf("Packet #%u\n", packet_number);
        printf("N-tuple info:\t\t");

        for (ind = 0; (proto_id = qmdpi_worker_pdu_ntuple_get_proto(worker, ind)) > 0;
                ind++) {
            void *header = qmdpi_worker_pdu_ntuple_get_header(worker, ind);
            classification_print_sig(proto_id, header);
        }

        /* print IP adresses and ports stored in signature */
        printf("\n");

        path = qmdpi_result_path_get(result);
        b = qmdpi_flow_bundle_get(flow);

        qmdpi_data_path_to_buffer(b, buffer, sizeof(buffer), path);

        printf("Classification path:\t%s\n", buffer);

        if (print_method_mode) {
            classification_method_result_display(bundle, result);
        }

        if (tdm_extension_lookup) {
            char tdm_ext[TDM_EXTENSION_LEN];
            tdm_ext[0] = '\0';
            int res_lookup = qmdpi_flow_classif_ext_get(flow, tdm_ext, TDM_EXTENSION_LEN);
            if (res_lookup == QMDPI_EOVERFLOW) {
                printf("TDM truncated result, a buffer size of %d is too small\n",
                       TDM_EXTENSION_LEN);
                printf("TDM extended path:\t%s.%.*s\n", buffer, res_lookup, tdm_ext);
            } else if (res_lookup > 0) {
                printf("TDM extended path:\t%s.%.*s\n", buffer, res_lookup, tdm_ext);
            } else {
                /* errors */
                fprintf(stderr, "cannot get tdm classification extension: %s\n",
                        qmdpi_error_get_string(NULL, res_lookup));
            }
        }
        if (classif_cache_need) {
            b = qmdpi_flow_bundle_get(flow);
            path = qmdpi_result_cached_path_get(result);
            *buffer = '\0';
            qmdpi_data_path_to_buffer(b, buffer, sizeof(buffer), path);
            if (*buffer) {
                printf("Cached classification path:\t%s\n", buffer);
            } else {
                printf("Cached classification path:\t(none)\n");
            }
        }
        printf("\n\n");
    }
}

/* open pcap file */
static pcap_t *classification_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];

    fd = pcap_open_offline(filename, errbuf);
    if (fd == NULL) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

/* close pcap file */
static void classification_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/* Config options */
#define CC_CACHE_CFG ";classification_cache_enable=1"
#define TDM_EXT_CFG ";fm_flow_metrics_enable=1"

/* initialize Qosmos ixEngine */
static int classification_engine_init(void)
{
    int ret = 0;
    char config[2048] = "injection_mode=packet;nb_workers=1;nb_flows=200000";

    /* Append configurations options to the config of the engine */
    if (use_classification_cache) {
        strcat(config, CC_CACHE_CFG);
    }
    if (use_tdm_ext) {
        strcat(config, TDM_EXT_CFG);
    }

    /* create engine instance */
    engine = qmdpi_engine_create((char const *) config);

    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto error_bundle;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle: %s\n", qmdpi_error_get_string(bundle,
                                                                               ret));
        goto error_bundle;

    }

    /* activate classification method extraction */
    if (print_method_mode) {
        qmdpi_bundle_attr_register(bundle, NULL, "classification_method");
    }

    if (use_tdm_ext) {
        if ((ret = qmdpi_report_init(engine, NULL))) {
            fprintf(stderr, "tdm report init error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
            goto error_bundle;
        }
        if ((ret = qmdpi_report_start(engine))) {
            fprintf(stderr, "tdm report start error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
            goto error_bundle;
        }
    }

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;
}

static void classification_engine_exit(void)
{
    struct qmdpi_result *result;

    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        classification_result_display(worker, result);
    }

    if (use_tdm_ext) {
        int ret;
        if ((ret = qmdpi_report_stop(engine))) {
            fprintf(stderr, "tdm report stop error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
        }
        if ((ret = qmdpi_report_cleanup(engine))) {
            fprintf(stderr, "tdm report cleanup error: %s\n",
                    qmdpi_error_get_string(bundle, ret));
        }
    }
    qmdpi_bundle_destroy(bundle);

    qmdpi_classification_cache_reset(engine);
    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void classification_process(u_char *user, const struct pcap_pkthdr *phdr,
                                   const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%u\n", packet_number);
        } else {
            /* play with result */
            classification_result_display(worker, result);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows and provide DPI result for each of them */
    int nb_remaining = 10;
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        nb_remaining --;
        /* play with result */
        classification_result_display(worker, result);
    } while (nb_remaining);
}

/**
* Parse options
*/
#define CC_CACHE_STR "--classification-cache-enable"
#define TDM_EXT_STR  "--tdm-classif-ext"
#define PRINT_METHOD_STR  "--print-method"

static void print_usage(const char *argv)
{
    fprintf(stderr, "Usage: %s [options] <pcap_file>\n", argv);
    fprintf(stderr, "options:\n");
    fprintf(stderr,
            "\t%s : activate tdm and give classification extension on unknown protocols if they exist into tdm tables\n",
            TDM_EXT_STR);
    fprintf(stderr,
            "\t%s:    print the classification method\n", PRINT_METHOD_STR);
    fprintf(stderr,
            "\t%s : enable classification in 1 packet using caches\n", CC_CACHE_STR);
}

static int pcap_args_parse(int argc, char const **argv)
{
    int ac;
    char const **p;

    if (argc < 2) {
        return -1;
    }

    for (ac = 1, p = argv + 1; ac < argc; ++ac, ++p) {
        if (strncmp(*p, CC_CACHE_STR, sizeof(CC_CACHE_STR) - 1) == 0) {
            use_classification_cache = 1;
        } else if (strncmp(*p, TDM_EXT_STR, sizeof(TDM_EXT_STR) - 1) == 0) {
            use_tdm_ext = 1;
        } else if (strncmp(*p, PRINT_METHOD_STR, sizeof(PRINT_METHOD_STR) - 1) == 0) {
            print_method_mode = 1;
        } else {
            break;
        }
    }

    if (ac == argc) {
        return -1;
    }

    return ac;
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc < 2 || argc > 5) {
        print_usage(*argv);
        return 1;
    }

    if (pcap_args_parse(argc, argv) < 0) {
        print_usage(*argv);
        return 1;
    }

    /* check pcap */
    pcap = classification_pcap_open(argv[argc - 1]);
    if (pcap == NULL) {
        print_usage(*argv);
        return 1;
    }

    /* init Qosmos ixEngine */
    ret = classification_engine_init();
    if (ret < 0) {
        return 1;
    }

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, classification_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        classification_engine_exit();
        classification_pcap_close(pcap);
        return 1;
    }

    classification_engine_exit();
    classification_pcap_close(pcap);

    return 0;
}
