/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2021 All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>

/* DPDK headers */
#include <rte_config.h>
#include <rte_eal.h>
#include <rte_mempool.h>
#include <rte_ethdev.h>
#include <rte_pci.h>
#include <rte_launch.h>
#include <rte_cycles.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>

#include "customize.h"

#define MBUFSZ (2048 + sizeof(struct rte_mbuf) + RTE_PKTMBUF_HEADROOM)
/**
 * Set the number of queue descriptor for lcores' software ring queue
 */
#define LCORE_QUEUESZ 1024 * 32

/**
 * Set number of queue descriptor for port's Hardware receive ring queue
 * 3000 is max number of packet in HW queue
 */
#define RXDNB 128
#define TXDNB 512
#define PKTBURSTNB 32

#define QNAMEPREFIX "LCOREQ"
/**
 * Safe for 10^20 lcores
 */
#define QNAMESZ (sizeof(QNAMEPREFIX) + 20)

char config_line[CONFIG_STR_SIZE] = "injection_mode=packet;";

static const struct rte_eth_conf ethconf = {
    .rxmode = {
        .split_hdr_size = 0,    /**< Header Split buffer size */
        .max_rx_pkt_len = 9600, /**< MTU to the max */
    },
    .txmode = {
        .mq_mode = ETH_MQ_TX_NONE,
    },
};

/* Timestamp */
static uint64_t starting_cycles;
static uint64_t hz;
static struct timeval  start_time;

/* Timer for stats */
static int64_t timer_period = 3;      /* default period is 3 seconds */
static uint64_t timer_cycles;

/**
 * Structure describing a lcore for packet processing
 */
static uint64_t dispatch_pkts_rx =
    0;        /* number of pktx rx on dispatch thread */
struct lcore {
    struct rte_ring *queue;                  /* lcore_queue */
    uint64_t ring_drop;                      /* #pkts drop on ring */
    uint64_t ring_enqueued;                  /* #pkts drop on ring */
    uint64_t worker_pkts_rx;                 /* #pkts received on worker */
    uint64_t worker_pkts_processed;          /* #pkts processed by worker */
    uint64_t worker_pkts_errors;             /* #pkts erronoeous on dpi */

    uint64_t worker_flows_expired;           /* #flow expired */
    uint64_t worker_bytes_processed;         /* #bytes processed */

    unsigned int id;                         /* worker_id */
};

/**
 * Temporary packet per thread cache in order to use more efficient
 * rte_ring_sp_enqueue_bulk
 */
struct mbuf_queue {
    struct rte_mbuf *pkt[PKTBURSTNB];
    unsigned int size;
};


static struct rte_mempool *pktmbuf_pool = NULL;
static unsigned long hwsize = RXDNB;
static unsigned long swsize = LCORE_QUEUESZ;

static int noprint = 0;
static int monitoring = 0;
static int nb_flows_arg;

static volatile char di_stop = 0;

static struct qmdpi_engine *engine;
struct qmdpi_bundle *bundle;

static inline void usage(char const *p)
{
    fprintf(stderr, "Usage:\n%s <dpdk_cmdline> -- [options]\n"
            "Options:\n"
            "\t--nic_ring_size: Size of hardware queue\n"
            "\t--dpi_ring_size: Size of software queue\n"
            "\t--enable-monitoring: Enable performance monitoring. Resulting statistics (in terms of memory and processing time) are displayed at exit.\n"
            "\t--no-print: Do not print classification\n"
            "\t--timer: time between stats printing (s)\n"
            "\t--nb_flows: maximal number of flow context per worker\n",
            p);
}

/**
 * Stop application when SIGINT is received
 */
void di_sig_stop(int s)
{
    (void)s;
    di_stop = 1;
}

/**
 * Helper functions
 */
static inline unsigned long int
hash_compute(unsigned char *buffer, unsigned int len)
{
    return qmdpi_packet_hashkey_get((const char *)buffer, len, QMDPI_PROTO_ETH);
}

static inline
void calculate_timestamp(struct timeval *ts)
{
    uint64_t       cycles;
    struct timeval cur_time;

    cycles = rte_get_timer_cycles() - starting_cycles;
    cur_time.tv_sec = cycles / hz;
    cur_time.tv_usec = ((cycles % hz) * 1000000) / hz;

    timeradd(&start_time, &cur_time, ts);

    if (unlikely(cur_time.tv_sec > 5 * 60)) {
        /* Correct time every 5 min. */
        starting_cycles = rte_get_timer_cycles();
        gettimeofday(&start_time, NULL);

#ifdef DPDK_CLOCK_DEBUG
        /*
        ** show drift from real time.
        */
        LOG(stderr, "calibrating clocks...\n");

        LOG(stderr,
            "our time <%10ld.%06ld>\n"
            "current  <%10ld.%06ld> \n"
            "diff     <%10ld.%06ld> \n",
            (long int)ts->tv_sec, (long int)ts->tv_usec,
            (long int)start_time.tv_sec, (long int) start_time.tv_usec,
            (long int)start_time.tv_sec - (long int) ts->tv_sec,
            (long int)start_time.tv_usec - (long int) ts->tv_usec);
#endif /* DPDK_CLOCK_DEBUG */
    }
}

/**
 * Print dpdk queue statistics
 */
static void
di_stats_print(struct lcore const *lcore, unsigned int lcorenb,
               uint16_t portnb)
{
    struct rte_eth_stats st;
    unsigned int lc;
    uint16_t p;
    uint64_t total_pkts_processed = 0;
    uint64_t total_bytes_processed = 0;
    static uint64_t old_pkts_processed = 0;
    static uint64_t old_bytes_processed = 0;
    uint64_t total_flows_expired = 0;

    const char clr[] = { 27, '[', '2', 'J', '\0' };
    const char topLeft[] = { 27, '[', '1', ';', '1', 'H', '\0' };


    fprintf(stderr, "%s%s", clr, topLeft);
    fprintf(stderr, "\nStatistics ======================================\n\n");

    fprintf(stderr, "NIC; (total pkts rx: %lu) \n", dispatch_pkts_rx);

    for (p = 0; p < portnb; ++p) {
        rte_eth_stats_get(p, &st);
        fprintf(stderr, " Port #%u: %lu received / %lu errors / %lu missed\n",
                p, st.ipackets, st.ierrors, st.imissed);
    }

    fprintf(stderr, "\nSoftware Ring\n");
    for (lc = 0; lc < lcorenb; ++lc) {
        fprintf(stderr, " Core #%u: %lu enqueued / %lu dropped\n",
                lcore[lc].id,
                lcore[lc].ring_enqueued, lcore[lc].ring_drop);
    }

    fprintf(stderr, "\nDPI :\n");

    for (lc = 0; lc < lcorenb; ++lc) {
        fprintf(stderr,
                " Core #%u: %lu rx / %lu processed / %lu err  \n",
                lcore[lc].id,
                lcore[lc].worker_pkts_rx,
                lcore[lc].worker_pkts_processed,
                lcore[lc].worker_pkts_errors);
        total_pkts_processed += lcore[lc].worker_pkts_processed;
        total_bytes_processed += lcore[lc].worker_bytes_processed;
    }
    fprintf(stderr, "\n");

    for (lc = 0; lc < lcorenb; ++lc) {
        fprintf(stderr,
                " Core #%u: %lu flows expired\n",
                lcore[lc].id,
                lcore[lc].worker_flows_expired);
        total_flows_expired += lcore[lc].worker_flows_expired;
    }
    fprintf(stderr, "\n");

    long double pkt_rate = (total_pkts_processed - old_pkts_processed) /
                           timer_period;
    long double byte_rate = (total_bytes_processed - old_bytes_processed) /
                            timer_period;

    fprintf(stderr,
            "\nAggregate statistics :\n"
            "\nTotal pkts processed: %18"PRIu64
            "\nTotal pps : %28"PRIu64
            "\npps/core : %29"PRIu64
            "\nTotal rate (KB/s) : %20"PRIu64
            "\nrate (KB) /core : %22"PRIu64
            "\nTotal flows expired: %19"PRIu64,
            total_pkts_processed,
            (uint64_t)pkt_rate,
            (uint64_t)pkt_rate / (rte_lcore_count() - 1),
            (uint64_t)byte_rate / 1000,
            (uint64_t)byte_rate / (rte_lcore_count() - 1) / 1000,
            total_flows_expired);

    fprintf(stderr,
            "\n====================================================\n");

    old_pkts_processed = total_pkts_processed;
    old_bytes_processed = total_bytes_processed;
}


/**
 * Get packet and dispatch them to each lcore
 */
static int di_pkt_dispatch(struct lcore *lc, uint16_t portnb)
{
    struct rte_mbuf *pkt[PKTBURSTNB];
    struct timeval *ts = NULL;
    unsigned int slavenb;
    unsigned int ret;
    uint64_t prev_tsc, diff_tsc, cur_tsc, timer_tsc;
    uint16_t nbrx;
    uint16_t p;
    uint16_t i;

    slavenb = rte_lcore_count() - 1;
    prev_tsc = 0;
    timer_tsc = 0;

    for (p = 0; p < portnb; p++) {
        if (rte_eth_dev_socket_id(p) > 0 &&
                rte_eth_dev_socket_id(p) !=
                (int)rte_socket_id()) {
            fprintf(stderr, "WARNING, port %u is on remote NUMA node to "
                    "polling thread.\n\tPerformance will "
                    "not be optimal.\n", p);
        }
    }

    while (!di_stop) {
        cur_tsc = rte_rdtsc();
        diff_tsc = cur_tsc - prev_tsc;

        if (timer_cycles > 0) {   /* timer enabled */
            timer_tsc += diff_tsc;
            if (unlikely(timer_tsc >= timer_cycles)) { /* timeout */
                di_stats_print(lc, slavenb, portnb);
                timer_tsc = 0;
            }
        }
        prev_tsc = cur_tsc;

        for (p = 0; p < portnb; ++p) {
            nbrx = rte_eth_rx_burst(p, 0, pkt, PKTBURSTNB);
            if (unlikely(nbrx == 0)) {
                continue;
            }

            for (i = 0; i < nbrx; i++) {
                dispatch_pkts_rx++;

                /* Software hash for dispatching */
                unsigned long int hash = hash_compute(rte_pktmbuf_mtod(pkt[i], unsigned char *),
                                                      rte_pktmbuf_data_len(pkt[i]));
                int soft_queue = hash % slavenb;
                ts = (struct timeval *)rte_pktmbuf_prepend(pkt[i],
                                                           (uint16_t)sizeof(struct timeval));
                calculate_timestamp(ts);

                ret = rte_ring_enqueue_bulk(lc[soft_queue].queue, (void *const *)&pkt[i], 1,
                                            NULL);
                if (unlikely(ret == 0)) {
                    rte_pktmbuf_free(pkt[i]);
                    lc[soft_queue].ring_drop += 1;
                } else {
                    lc[soft_queue].ring_enqueued += 1;
                }
            }
        }
    }

    return 0;
}

static void di_result_display(struct lcore *lc,
                              struct qmdpi_result *result,
                              struct qmdpi_worker *worker)
{
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
        char buffer[512];
        struct qmdpi_flow *flow = qmdpi_result_flow_get(result);
        struct qmdpi_path *path = qmdpi_result_path_get(result);
        struct qmdpi_bundle *b = qmdpi_flow_bundle_get(flow);
        if (path) {
            qmdpi_worker_data_path_to_buffer(worker, b, buffer, sizeof(buffer), path);
        }

        fprintf(stdout, "Core #%u\t\tClassification Path: %s\n",
                lc->id, buffer);
    }
}

/* This function must be invoked at the end of every thread */
void dpi_expire_all_flows(struct lcore *lc, struct qmdpi_worker *worker)
{
    if (!worker) {
        return;
    }

    struct qmdpi_result *results = NULL;
    while (qmdpi_flow_expire_next(worker, NULL, &results) == 0) {
        di_result_display(lc, results, worker);
    }
}

/**
 * Send packet to ixe
 */
static int di_pkt_loop(void *arg)
{
    struct lcore *lc = (struct lcore *)arg;
    struct rte_mbuf *mbuf[PKTBURSTNB];
    int ret;
    unsigned int nb, i;

    qmdpi_stats_init_end(engine);
    struct qmdpi_worker *worker = qmdpi_worker_create(engine);

    while (!di_stop) {
        nb = rte_ring_dequeue_burst(lc->queue,
                                    (void **)mbuf,
                                    PKTBURSTNB,
                                    NULL);
        if (unlikely(nb == 0)) {
            continue;
        }

        lc->worker_pkts_rx += nb;

        for (i = 0; i < nb; ++i) {
            if (rte_pktmbuf_data_len(mbuf[i]) == 0) {
                rte_pktmbuf_free(mbuf[i]);
                continue;
            }

            struct timeval ts;
            memcpy(&ts, rte_pktmbuf_mtod(mbuf[i], char *), sizeof(struct timeval));
            rte_pktmbuf_adj(mbuf[i], (uint16_t) sizeof(struct timeval));

            if (qmdpi_worker_pdu_set(worker, rte_pktmbuf_mtod(mbuf[i], void *),
                                     rte_pktmbuf_data_len(mbuf[i]), &ts,
                                     QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
                ++lc->worker_pkts_errors;
                rte_pktmbuf_free(mbuf[i]);
                continue;
            }

            do {
                struct qmdpi_result *result;
                ret = qmdpi_worker_process(worker, NULL, &result);
                if (ret < 0) {
                    break;
                }

                if (likely(!noprint)) {
                    di_result_display(lc, result, worker);
                }
            } while (ret == QMDPI_PROCESS_MORE);

            if (unlikely(ret < 0)) {
                ++lc->worker_pkts_errors;
            } else {
                lc->worker_bytes_processed += rte_pktmbuf_data_len(mbuf[i]);
                lc->worker_pkts_processed++;
            }

            struct qmdpi_result *result;
            int nb_remaining = 10;
            do {
                ret = qmdpi_flow_expire_next(worker, &ts, &result);
                if (ret != 0) {
                    break;
                }
                lc->worker_flows_expired++;
                nb_remaining--;
                if (likely(!noprint)) {
                    di_result_display(lc, result, worker);
                }
            } while (nb_remaining);

            rte_pktmbuf_free(mbuf[i]);
        }
    }

    /*
     * Packet processing has been stopped but packets may be pending in ixE.
     * Expire corresponding flows.
     */
    dpi_expire_all_flows(lc, worker);

    /* Destroy ixEngine worker */
    if (worker) {
        qmdpi_worker_destroy(worker);
    }

    return 0;
}

/**
 * Initialize dpdk application.
 * Should be called on master lcore as dpdk application initialization functions
 * are not thread safe
 */
static int di_app_init(int argc, char **argv)
{
    int ret = 0;
    struct sigaction sighdl = {
        .sa_handler = di_sig_stop,
        .sa_flags = 0
    };
    sigemptyset(&sighdl.sa_mask);

    /**
     * Initialize Environment Abstraction Layer
     * This manages hardware resources (memory, PCI devices, timers, ...) and
     * threads.
     */
    ret = rte_eal_init(argc, argv);
    if (ret < 0) {
        fprintf(stderr, "EAL initialization fail : %d (%s)\n", rte_errno,
                rte_strerror(rte_errno));
        goto out;
    }

    if (rte_lcore_count() <= 1) {
        ret = -1;
        fprintf(stderr,
                "Number of cores should be > 1 (please check your DPDK command line arguments)\n");
        goto out;
    }

    /**
     * Create the mbuf pool:
     */
    pktmbuf_pool = rte_mempool_create("mbuf_pool",
                                      hwsize + swsize * (rte_lcore_count() - 1), MBUFSZ, 32,
                                      sizeof(struct rte_pktmbuf_pool_private),
                                      rte_pktmbuf_pool_init, NULL,
                                      rte_pktmbuf_init, NULL, SOCKET_ID_ANY, 0);

    if (pktmbuf_pool == NULL) {
        ret = -ENOMEM;
        fprintf(stderr, "Mempool creation fail : %d (%s)\n", rte_errno,
                rte_strerror(rte_errno));
        goto out;
    }

    /**
     * Install signal handler on SIGINT to stop the application
     */
    ret = sigaction(SIGINT, &sighdl, NULL);

out:
    return ret;
}

/**
 * Exit dpdk application.
 */
static int di_app_exit(void)
{
    /* clean up the EAL */
    return rte_eal_cleanup();
}

/**
 * Configure an ethernet port
 */
static int di_ethport_init(int16_t port)
{
    int ret;

    /**
     * The ethernet device is configured to have 1 receive queue and no
     * transmit queue
     */
    ret = rte_eth_dev_configure(port, 1, 1, &ethconf);
    if (ret < 0) {
        fprintf(stderr, "Ethernet configuration fail\n");
        goto out;
    }

    /**
     * Then allocate and set up the receive queues for this Ethernet device
     */
    ret = rte_eth_rx_queue_setup(port, 0, hwsize, SOCKET_ID_ANY, NULL,
                                 pktmbuf_pool);
    if (ret < 0) {
        fprintf(stderr, "Ethernet rx queue setup fail\n");
        goto out;
    }

    /**
     * Then allocate and set up the transmit queue for this Ethernet device
     * This is needed by dpdk even if not used
     */
    ret = rte_eth_tx_queue_setup(port, 0, TXDNB, SOCKET_ID_ANY, NULL);
    if (ret < 0) {
        fprintf(stderr, "Ethernet tx queue setup fail\n");
        goto out;
    }

    /**
     * Let's start the ethernet device and begin to receive packets
     */
    ret = rte_eth_dev_start(port);
    if (ret < 0) {
        fprintf(stderr, "Ethernet device start fail\n");
        goto out;
    }

    /**
     * Then go into promiscuous mode
     */
    rte_eth_promiscuous_enable(port);

    /**
     * Then receive any multicast ethernet frame
     */
    rte_eth_allmulticast_enable(port);

out:
    return ret;
}

/**
 * Destroy an ethernet port
 */
static int di_ethport_exit(uint16_t port)
{
    rte_eth_dev_close(port);
    return 0;
}


/**
 * Launch packet processing on each lcore.
 * XXX Should be run be MASTER only
 */
static int di_run(void)
{
    struct lcore *lcore;
    unsigned int slavenb, lcoreid, lcid = 0;
    int ret = 0;
    uint16_t p, portnb;
    char qname[QNAMESZ] = {};

    /* convert to number of cycles */
    timer_cycles = timer_period * rte_get_timer_hz();

    slavenb = rte_lcore_count() - 1;

    /**
     * Get the number of ethernet NIC ports
     */
    portnb = rte_eth_dev_count_avail();
    if (portnb == 0) {
        fprintf(stderr, "No probed ether devices check config\n");
        ret = -ENODEV;
        goto out;
    }

    /**
     * Enable all dpdk port
     */
    for (p = 0; p < portnb; ++p) {
        ret = di_ethport_init(p);
        if (ret < 0) {
            portnb = p;
            goto out;
        }
    }

    lcore = calloc(slavenb, sizeof(*lcore));
    if (lcore == NULL) {
        goto out;
    }

    hz = rte_get_timer_hz();
    gettimeofday(&start_time, NULL);
    starting_cycles = rte_get_timer_cycles();

    /**
     * Launch packet loop on each lcore except MASTER. This function will
     * first check that each SLAVE lcore are in WAIT state the call
     * rte_eal_remote_launch() for each one.
     **/
    RTE_LCORE_FOREACH_WORKER(lcoreid) {
        snprintf(qname, QNAMESZ - 1, QNAMEPREFIX"%u", lcid);
        lcore[lcid].id = lcid;
        lcore[lcid].queue = rte_ring_create(qname, swsize, SOCKET_ID_ANY,
                                            RING_F_SP_ENQ | RING_F_SC_DEQ);
        if (lcore[lcid].queue == NULL) {
            goto out;
        }
        rte_eal_remote_launch(di_pkt_loop, (void *)&lcore[lcid], lcoreid);
        ++lcid;
    }

    /**
     * Dispatch packet for every body
     */
    di_pkt_dispatch(lcore, portnb);

    /**
     * Wait all lcore to finish and put them into WAIT state
     */
    rte_eal_mp_wait_lcore();

    di_stats_print(lcore, slavenb, portnb);

    free(lcore);

out:

    /**
     * Disable all dpdk port
     */
    for (p = 0; p < portnb; ++p) {
        di_ethport_exit(p);
    }

    return ret;
}

static int di_dpi_init(void)
{
    int ret;

    int nb_flows;
    if (nb_flows_arg) {
        nb_flows = nb_flows_arg;
    } else {
        /* default number of flows is 100000 */
        nb_flows = 100000;
    }
    di_config_line_add(config_line, "nb_flows", nb_flows);
    fprintf(stdout, "DPI engine : %d flows per worker\n", nb_flows);

    di_config_line_add(config_line, "nb_workers", rte_lcore_count() - 1);
    fprintf(stdout, "DPI engine : nb_workers used = %d\n", rte_lcore_count() - 1);
    if (monitoring) {
        di_config_line_add(config_line, "monitoring_enable", 1);
    }

    if (di_custom_engine_configuration() < 0) {
        return -1;
    }

    engine = qmdpi_engine_create(config_line);
    if (engine == NULL) {
        fprintf(stderr, "error initializing DPI engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_engine;
    }

    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot load bundle: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_bundle;
    }

    qmdpi_bundle_activate(bundle);

    if ((ret = qmdpi_bundle_signature_enable_all(bundle)) < 0) {
        fprintf(stderr, "Failed to enable all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto err_sig;
    }

    if (di_custom_tune_configuration() < 0) {
        return -1;
    }

    if (di_custom_attribute_configuration() < 0) {
        return -1;
    }

    if (monitoring) {
        qmdpi_stats_init_start(engine);
    }
    return 0;

err_sig:
    qmdpi_bundle_destroy(bundle);

err_bundle:
    qmdpi_engine_destroy(engine);

err_engine:
    return -1;
}

static int di_dpi_exit(void)
{
    if (monitoring) {
        qmdpi_stats_dump(engine, stdout, (qmdpi_output_fn_t)fprintf);
        qmdpi_stats_exit(engine);
    }

    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
    return 0;
}


#define MAX_TIMER_PERIOD 86400 /* 1 day max */
static int
di_parse_timer_period(const char *q_arg)
{
    char *end = NULL;
    int n;

    /* parse number string */
    n = strtol(q_arg, &end, 10);
    if ((q_arg[0] == '\0') || (end == NULL) || (*end != '\0')) {
        return -1;
    }

    if (n >= MAX_TIMER_PERIOD) {
        return -1;
    }

    return n;
}

/**
 * Parse non dpdk only arguments
 */
static int di_parse_args(int argc, char **argv)
{
    int i;

    if (argc < 2) {
        return -1;
    }

    for (i = 0; i < argc; ++i) {
        if (strcmp(argv[i], "--") == 0) {
            ++i;
            break;
        }
    }

    if (i >= argc) {
        return 0;
    }

    for (; i < argc; ++i) {
        if (strcmp(argv[i], "--nic_ring_size") == 0) {
            ++i;
            if (i >= argc) {
                return -1;
            }
            hwsize = atoi(argv[i]);
            continue;
        } else if (strcmp(argv[i], "--dpi_ring_size") == 0) {
            ++i;
            if (i >= argc) {
                return -1;
            }
            swsize = atoi(argv[i]);
            continue;
        } else if (strcmp(argv[i], "--nb_flows") == 0) {
            ++i;
            if (i >= argc) {
                return -1;
            }
            nb_flows_arg = atoi(argv[i]);
            continue;
        } else if (strcmp(argv[i], "--enable-monitoring") == 0) {
            monitoring = 1;
            continue;
        } else if (strcmp(argv[i], "--no-print") == 0) {
            noprint = 1;
            continue;
        } else if (strcmp(argv[i], "--timer") == 0) {
            ++i;
            if (i >= argc) {
                return -1;
            }
            timer_period = di_parse_timer_period(argv[i]);
            if (timer_period < 0) {
                fprintf(stderr, "invalid timer period\n");
                return -1;
            }
            continue;
        } else {
            return -1;
        }
    }

    return 0;
}

int main(int argc, char **argv)
{
    int ret = 0;

    ret = di_parse_args(argc, argv);
    if (ret != 0) {
        usage(argv[0]);
        ret = 1;
        goto exit;
    }

    ret = di_app_init(argc, argv);
    if (ret != 0) {
        ret = 1;
        goto exit;
    }

    ret = di_dpi_init();
    if (ret != 0) {
        ret = 1;
        goto appexit;
    }

    di_run();

    di_dpi_exit();

appexit:
    ret = di_app_exit();
    if (ret < 0) {
        return 1;
    }

exit:
    return ret;
}
