/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2018-2018 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <inttypes.h>
#include <sys/time.h>
#include <sys_queue.h>

#include "common.h"

struct table_entry {
    SLIST_ENTRY(table_entry) next;
    struct file_context   *file_context;
    uint32_t               file_id;
} table_entry_t;

#define TABLE_ENTRY_LOOKUP_HASHSZ (1 << 10)
static SLIST_HEAD(, table_entry) table_hash[TABLE_ENTRY_LOOKUP_HASHSZ];

static inline uint32_t get_table_entry_hash_key(uint32_t id)
{
    return id % TABLE_ENTRY_LOOKUP_HASHSZ;
}

void fr_table_init(void)
{
    int i;
    for (i = 0; i < TABLE_ENTRY_LOOKUP_HASHSZ; ++i) {
        SLIST_INIT(&table_hash[i]);
    }
}

void fr_table_destroy(void)
{
    int i;
    struct table_entry *entry = NULL;

    for (i = 0; i < TABLE_ENTRY_LOOKUP_HASHSZ; ++i) {
        entry = SLIST_FIRST(&table_hash[i]);
        /* Go through the linked list of contexts */
        while (entry) {
            struct table_entry *next_entry = SLIST_NEXT(entry, next);

            /* Remove entry */
            SLIST_REMOVE(&table_hash[i], entry, table_entry, next);
            free(entry);
            entry = next_entry;
        }
    }
}

int fr_table_add_entry(uint32_t file_id, struct file_context *fc)
{
    int ret = 0;
    uint32_t hash_key = get_table_entry_hash_key(file_id);
    struct table_entry *entry = NULL;

    SLIST_FOREACH(entry, &table_hash[hash_key], next) {
        if (entry->file_id == file_id) {
            return -1;
        }
    }

    /* The table does not contain an entry for file_id, create one. */
    struct table_entry *new_entry = (struct table_entry *)malloc(sizeof(
                                                                     struct table_entry));
    if (new_entry == NULL) {
        PRINT_ERR("ERROR: cannot allocate entry\n");
        return -1;
    }

    memset(new_entry, 0, sizeof(struct table_entry));
    new_entry->file_id = file_id;
    new_entry->file_context = fc;

    SLIST_INSERT_HEAD(&table_hash[hash_key], new_entry, next);

    return ret;
}

int fr_table_get_entry(uint32_t file_id, struct file_context **fc)
{
    uint32_t hash_key = get_table_entry_hash_key(file_id);
    struct table_entry *entry = NULL;

    SLIST_FOREACH(entry, &table_hash[hash_key], next) {
        if (entry->file_id == file_id) {
            *fc = entry->file_context;
            return 0;
        }
    }

    return -1;
}

int fr_table_remove_entry(uint32_t file_id)
{
    uint32_t hash_key = get_table_entry_hash_key(file_id);
    struct table_entry *entry = NULL;

    entry = SLIST_FIRST(&table_hash[hash_key]);
    /* Go through the linked list of elements */
    while (entry) {
        struct table_entry *next_entry = SLIST_NEXT(entry, next);

        /* Remove element */
        if (entry->file_id == file_id) {
            SLIST_REMOVE(&table_hash[hash_key], entry, table_entry, next);
            free(entry);
            return 0;
        }

        entry = next_entry;
    }

    return -1;
}
