/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2000-2016 - All rights reserved
 */

/* standard headers */
#include <stdio.h>
#include <string.h>

#include <ctype.h>
/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine headers */
#include <qmdpi.h>
#include <qmdpi_bundle_api.h>

/* Inheritance management algorithm:
 *
 * Each time an FTP flow triggers an inheritance key (attribute Q_GEN_INHERIT_PARENT),
 * we store it in a table, along with its flow pointer. Later when the FTP
 * flow triggers a filename attribute (Q_FTP_FILENAME), we add it to entries
 * with matching flow pointers.
 * When an FTP DATA flow is classified, the inheritance key (Q_GEN_INHERIT_KEY) will be triggered
 * and we will lookup for this key in our table. If inherit key matches,
 * we print the filename for that FTP DATA flow.
 *
 * Table structure:
 *
 *  ------------------------------------------------
 * | Flow pointer | Inheritance key | Filename      |
 * |              |                 |               |
 * |              |                 |               |
 *  ------------------------------------------------
 */

static unsigned packet_number;

static struct qmdpi_engine *engine;
static struct qmdpi_bundle *bundle;
static struct qmdpi_worker *worker;

#define INHERITANCE_TABLE_KEY_SIZE 64
#define INHERITANCE_TABLE_MAX_ENTRIES 256
#define INHERITANCE_TABLE_MAX_PRIV_SIZE 1024

struct fh_entry {
    struct qmdpi_flow *dpi_flow;
    uint8_t key[INHERITANCE_TABLE_KEY_SIZE];
    uint32_t len;
    uint8_t priv[INHERITANCE_TABLE_MAX_PRIV_SIZE];
};

struct fh_table {
    struct fh_entry entries[INHERITANCE_TABLE_MAX_ENTRIES];
    uint8_t len;
};

struct fh_table table;

static void fh_table_init(struct fh_table *t)
{
    memset(t, 0, sizeof(*t));
}

/* add entry with key and dpi_flow pointer */
static void fh_table_add_key(struct fh_table *t, struct qmdpi_flow *dpi_flow,
                             char const *key, uint32_t key_len)
{
    memcpy(t->entries[t->len].key, key, key_len);
    t->entries[t->len].len = key_len;
    t->entries[t->len].dpi_flow = dpi_flow;
    t->entries[t->len].priv[0] = '\0';
    t->len++;
}

/* add filename to entries matching dpi_flow */
static void fh_table_add_priv(struct fh_table *t, struct qmdpi_flow *dpi_flow,
                              char const *priv, uint32_t priv_len)
{
    int i;

    for (i = 0; i < INHERITANCE_TABLE_MAX_ENTRIES; i++) {
        if (t->entries[i].dpi_flow == dpi_flow) {
            memcpy(t->entries[i].priv, priv, priv_len);
        }
    }
}

/* expire entries */
static void fh_expire_flow(struct fh_table *t,
                           struct qmdpi_flow *dpi_flow)
{
    int i;

    for (i = 0; i < INHERITANCE_TABLE_MAX_ENTRIES; i++) {
        if (t->entries[i].dpi_flow == dpi_flow) {
            t->entries[i].len     = 0;
            t->entries[i].priv[0] = '\0';
        }
    }
}

/* lookup */
static int fh_table_find(struct fh_table *t, char const *key, uint32_t key_len,
                         uint8_t **priv)
{
    int i = 0;
    int ret = 0;

    for (i = 0; i < INHERITANCE_TABLE_MAX_ENTRIES; i++) {
        if (t->entries[i].len != key_len) {
            continue;
        }
        ret = memcmp(t->entries[i].key, key, key_len);
        /* found entry */
        if (ret == 0) {
            *priv = t->entries[i].priv;
            return 1;
        }
    }
    /* nothing was found */
    return 0;
}

/* open pcap file */
static pcap_t *fh_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];

    fd = pcap_open_offline(filename, errbuf);
    if (fd == NULL) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

/* close pcap file */
static void fh_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/* store FTP flow file name in the table */
static void fh_ftp_filename(char const *data, int datalen, struct qmdpi_flow *f)
{
    fh_table_add_priv(&table, f, data, datalen);
}

/* recover FTP parent flow of ftp_data flow and print stored filename */
static void fh_ftp_data_inherit_key(char const *data, int datalen,
                                    struct qmdpi_flow *f)
{
    int ret = 0;
    uint8_t *filename;

    /* match child key with parent keys in the table */
    ret = fh_table_find(&table, data, datalen, &filename);
    if (ret > 0 && filename[0] != '\0') {
        fprintf(stdout, "Found parent FTP flow for flow address %p\n", (void *)f);
        fprintf(stdout, " FTP filename: %s\n", filename);
    }
}

/**
 * Store parent key in table
 */
static void fh_ftp_data_inherit_parent(char const *data, int datalen,
                                       struct qmdpi_flow *f)
{
    fh_table_add_key(&table, f, data, datalen);
}

/* metadata callback */
static void fh_result_process(struct qmdpi_result *result)
{
    struct qmdpi_flow *f = qmdpi_result_flow_get(result);
    char const *data;
    int datalen;
    int proto_id;
    int attr_id;
    int flags;
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (f == NULL) {
        return;
    }

    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id, &data, &datalen,
                                     &flags) == 0) {
        switch (attr_id) {
            case Q_FTP_FILENAME:
                fh_ftp_filename(data, datalen, f);
                break;
            case Q_MPA_INHERIT_KEY:
                fh_ftp_data_inherit_key(data, datalen, f);
                break;
            case Q_MPA_INHERIT_PARENT:
                fh_ftp_data_inherit_parent(data, datalen, f);
                break;
            default:
                fprintf(stdout, "unknown attribute id %d\n", attr_id);
        }
    }

    /* if flow is expired, delete corresponding entry in the table */
    if (QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags) &&
            qmdpi_flow_user_handle_get(f) != NULL) {
        fh_expire_flow(&table, f);
    }
}

struct ftp_proto_attr {
    char *proto;
    char *attr;
};

static struct ftp_proto_attr ftp_proto_attr[] = {
    /* this metadata is needed to get parent key */
    { "ftp", "inherit_parent" },
    /* this metadata us needed to get the filename being transfered */
    { "ftp", "filename" },
    /* this metadata is needed to enable inheritance */
    { "ftp_data", "gen_parent_layer" },
    /* this metadata is needed to get child key */
    { "ftp_data", "inherit_key" },
    { NULL, NULL },
};

static int fh_ftp_inherit_init(void)
{
    int32_t i;

    for (i = 0; ftp_proto_attr[i].proto != NULL; i++) {
        if (qmdpi_bundle_attr_register(bundle,
                                       ftp_proto_attr[i].proto,
                                       ftp_proto_attr[i].attr) < 0) {
            fprintf(stderr, "cannot add metadata %s:%s\n",
                    ftp_proto_attr[i].proto,
                    ftp_proto_attr[i].attr);
            return -1;
        }
    }

    return 0;
}

static void fh_ftp_inherit_cleanup(void)
{
    int32_t i;

    for (i = 0; ftp_proto_attr[i].proto != NULL; i++) {
        if (qmdpi_bundle_attr_register(bundle,
                                       ftp_proto_attr[i].proto,
                                       ftp_proto_attr[i].attr) < 0) {
            fprintf(stderr, "cannot remove metadata %s:%s\n",
                    ftp_proto_attr[i].proto,
                    ftp_proto_attr[i].attr);
        }
    }
}

/* initialize Qosmos ixEngine */
static int fh_engine_init(void)
{
    int ret;

    engine = qmdpi_engine_create("nb_workers=1;nb_flows=20000");
    if (engine == NULL) {
        fprintf(stderr, "Cannot initialize engine: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_engine;
    }

    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "Cannot load bundle: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto err_bundle;
    }

    qmdpi_bundle_activate(bundle);

    if ((ret = qmdpi_bundle_signature_enable_all(bundle)) < 0) {
        fprintf(stderr, "Failed to enable all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto err_sig;
    }

    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "Cannot initialize worker: %s\n",
                qmdpi_error_get_string(bundle, qmdpi_error_get()));
        goto err_worker;
    }

    /* request metadata (inheritance keys) */
    if (fh_ftp_inherit_init() < 0) {
        goto err_init;
    }

    return 0;

err_init:
    qmdpi_worker_destroy(worker);
err_worker:
err_sig:
    qmdpi_bundle_destroy(bundle);
err_bundle:
    qmdpi_engine_destroy(engine);

err_engine:
    return -1;
}

static void fh_engine_exit(void)
{
    fh_ftp_inherit_cleanup();

    qmdpi_bundle_destroy(bundle);
    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void fh_process(u_char *user,
                       const struct pcap_pkthdr *phdr,
                       const u_char *pdata)
{
    int ret;

    (void)user;
    packet_number++;

    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    do {
        struct qmdpi_result *result;

        /* feed packet to the engine */
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stdout, "%d DPI failure\n", packet_number);
            continue;
        }

        fh_result_process(result);
    } while (ret == QMDPI_PROCESS_MORE);
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    packet_number = 0;

    /* check arguments */
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = fh_pcap_open(argv[1]);
    if (pcap == NULL) {
        return 1;
    }

    /* init inheritance hashtable */
    fh_table_init(&table);


    /* init Qosmos ixEngine */
    ret = fh_engine_init();
    if (ret < 0) {
        return 1;
    }

    ret = pcap_loop(pcap, -1, fh_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        fh_engine_exit();
        fh_pcap_close(pcap);
        return 1;
    }

    fh_engine_exit();
    fh_pcap_close(pcap);

    return 0;
}
