/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle header */
#include <qmdpi_bundle_api.h>
/* Qosmos Rule Engine header */
#include <qm_rule_engine.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;


/* Qosmos Rule Engine main objects */
struct qm_rule_engine *rule_engine = NULL;

#define KEYWORD_ID_UA           1
#define KEYWORD_ID_IP       2
#define IP4_ADDR_SIZE           4

static unsigned packet_number;

static void rei_play_with_result(struct qmdpi_worker *worker,
                                 struct qmdpi_result *result)
{
    struct qm_rule_engine_ctx *rule_engine_ctx = NULL;
    char const *attr_value;
    int proto_id, attr_id, attr_len, attr_flags;
    unsigned int rule_id;
    int ret;

    struct qmdpi_flow *f = qmdpi_result_flow_get(result);
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    rule_engine_ctx = (struct qm_rule_engine_ctx *)qmdpi_flow_user_handle_get(f);

    if (f == NULL || result_flags == NULL) {
        return;
    }

    rule_engine_ctx = (struct qm_rule_engine_ctx *)qmdpi_flow_user_handle_get(f);

    if (rule_engine_ctx == NULL) {
        ret = qm_rule_engine_ctx_create(rule_engine, &rule_engine_ctx);
        if (ret < 0) {
            fprintf(stderr, "could not create rule engine context\n");
            return;
        }
        qmdpi_flow_user_handle_set(f, rule_engine_ctx);
    }

    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &attr_value, &attr_len, &attr_flags) == 0) {
        switch (attr_id) {
            case Q_HTTP_USER_AGENT:
                qm_rule_engine_data_inject(rule_engine_ctx, (void *)attr_value,  attr_len,
                                           KEYWORD_ID_UA);
                break;
            default:
                printf("unknown attribute ID %d\n", attr_id);
        }
    }

    if (QMDPI_RESULT_FLAGS_FLOW_EXPIRED(result_flags)) {
        while (qm_rule_engine_matched_getnext(rule_engine_ctx, &rule_id) == 0) {
            printf("rule_id #%d matches the flow %p\n", rule_id, f);
        }
        qm_rule_engine_ctx_destroy(rule_engine_ctx);
        qmdpi_flow_user_handle_set(f, NULL);
    }
}

static pcap_t *rei_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];
    fd = pcap_open_offline(filename, errbuf);
    if (0 == fd) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

static void rei_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/* initialize Qosmos ixEngine */
static int rei_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance\n");
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance\n");
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance\n");
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle\n");
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols\n");
        goto error_bundle;
    }

    /* register attributes */
    ret = qmdpi_bundle_attr_register(bundle, "http", "user_agent");
    if (ret < 0) {
        fprintf(stderr, "cannot add metadata http:user_agent\n");
        goto error_bundle;
    }

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;

}

static void rei_engine_exit(void)
{
    struct qmdpi_result *result;

    qmdpi_bundle_destroy(bundle);

    /* expire remaining flow contexts */
    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        rei_play_with_result(worker, result);
    }

    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

/* initialize Rule Engine */
static int rei_rule_engine_init(const char *rule_file)
{

    char *buffer = NULL;
    char *error_string = NULL;
    int read_size = 0;
    struct stat buf;
    int ret, fd;

    /* put ruleset file into a buffer */
    fd = open(rule_file, O_RDONLY);
    if (fd < 0) {
        fprintf(stderr, "could not open rule file\n");
        return -1;
    }
    fstat(fd, &buf);
    read_size = buf.st_size;

    buffer = (char *) malloc(sizeof(char) * read_size);
    if (buffer == NULL) {
        return -1;
    }
    read_size = read(fd, buffer, read_size);
    close(fd);

    ret = qm_rule_engine_create(&rule_engine, buffer, read_size,
                                (const char **)&error_string);
    if (ret < 0) {
        fprintf(stderr, "could not create rule engine:\n%s\n", error_string);
        return -1;
    }

    free(buffer);

    return 0;
}

static int rei_rule_engine_exit(void)
{
    return qm_rule_engine_destroy(rule_engine);
}

static void rei_process(u_char *user,
                        const struct pcap_pkthdr *phdr,
                        const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%u\n", packet_number);
        } else {
            /* play with result */
            rei_play_with_result(worker, result);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows */
    int nb_remaining = 10;
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        rei_play_with_result(worker, result);
        nb_remaining --;
    } while (nb_remaining);
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc != 3) {
        fprintf(stderr, "Usage: %s <rule_file> <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = rei_pcap_open(argv[2]);
    if (pcap == NULL) {
        return 1;
    }

    /* init Qosmos ixEngine */
    ret = rei_engine_init();
    if (ret < 0) {
        return 1;
    }

    /* init Rule Engine */
    ret = rei_rule_engine_init(argv[1]);
    if (ret < 0) {
        return 1;
    }

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, rei_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        rei_engine_exit();
        rei_pcap_close(pcap);
        return 1;
    }

    rei_engine_exit();
    rei_rule_engine_exit();
    rei_pcap_close(pcap);

    return 0;
}
