/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle header */
#include <qmdpi_bundle_api.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_worker *worker = NULL;
struct qmdpi_bundle *bundle = NULL;

static unsigned packet_number;
static unsigned found = 0;

/* print found transaction with flow info */
static void trans_dpi_play_with_result(struct qmdpi_result *result)
{
    char const *attr_value;
    int flow_id, proto_id, attr_id, attr_len, attr_flags;
    struct qmdpi_flow *flow;
    struct qmdpi_path *path;
    struct qmdpi_bundle *bundle;
    char buffer[4096];
    const char *transaction_name = NULL;
    int transaction_name_len;

    while (qmdpi_result_attr_getnext(result, &proto_id, &attr_id,
                                     &attr_value, &attr_len, &attr_flags) == 0) {
        switch (attr_id) {
            case Q_MPA_TRANSACTION:
                found++;
                flow = qmdpi_result_flow_get(result);
                flow_id = qmdpi_flow_id_get(flow);
                path = qmdpi_result_path_get(result);
                bundle = qmdpi_flow_bundle_get(flow);

                qmdpi_data_path_to_buffer(bundle, buffer, sizeof(buffer), path);
                printf("Flow #%d (Packet #%d) - Classification path:\t%s\n", flow_id,
                       packet_number, buffer);

                qmdpi_data_transaction_get_name(bundle, *(uint32_t *)attr_value,
                                                &transaction_name, &transaction_name_len);
                printf(" Transaction name : %s (%d)\n", transaction_name,
                       *(uint32_t *)attr_value);

                break;
            default:
                printf("unknown attribute ID %d\n", attr_id);
        }
    }
}

static pcap_t *trans_dpi_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];
    fd = pcap_open_offline(filename, errbuf);
    if (0 == fd) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

static void trans_dpi_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

int register_signature(struct qmdpi_bundle *bundle,
                       struct qmdpi_signature *sig,
                       struct qmdpi_attr *attr,
                       void *arg)
{
    char const *sig_name = qmdpi_signature_name_get(sig);
    char const *attr_name = qmdpi_attr_name_get(attr);

    (void)arg;
    printf("Registering %s attribute on signature %s ... ", attr_name, sig_name);
    int ret = qmdpi_bundle_attr_register(bundle, sig_name, attr_name);
    if (ret < 0) {
        printf("failed (%s)\n", qmdpi_error_get_string(bundle, ret));
    } else {
        printf("OK\n");
    }

    return 0;
}

/* initialize Qosmos ixEngine */
static int trans_dpi_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance\n");
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance\n");
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance\n");
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle\n");
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols\n");
        goto error_bundle;
    }

    /* register Q_MPA_TRANSACTION attributes on protocols that support it */
    ret = qmdpi_bundle_signature_foreach_enabled_by_mpa(bundle, "transaction",
                                                        register_signature, NULL);
    if (ret < 0) {
        goto error_bundle;
    }

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;

}

static void trans_dpi_engine_exit(void)
{
    struct qmdpi_result *result;

    qmdpi_bundle_destroy(bundle);

    /* expire remaining flow contexts */
    while (qmdpi_flow_expire_next(worker, NULL, &result) == 0) {
        trans_dpi_play_with_result(result);
    }

    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void trans_dpi_process(u_char *user,
                              const struct pcap_pkthdr *phdr,
                              const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            fprintf(stderr, "DPI processing failure at packet #%u\n", packet_number);
        } else {
            /* play with result */
            trans_dpi_play_with_result(result);
        }
    } while (ret == QMDPI_PROCESS_MORE);

    /* destroy expired flows */
    int nb_remaining = 10;
    do {
        if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
            break;
        }
        trans_dpi_play_with_result(result);
        nb_remaining --;
    } while (nb_remaining);


}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = trans_dpi_pcap_open(argv[1]);
    if (pcap == NULL) {
        return 1;
    }


    /* init Qosmos ixEngine */
    ret = trans_dpi_engine_init();
    if (ret < 0) {
        trans_dpi_pcap_close(pcap);
        return 1;
    }

    printf("Processing...\n");

    /* DPI processing loop  */
    ret = pcap_loop(pcap, -1, trans_dpi_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        trans_dpi_engine_exit();
        trans_dpi_pcap_close(pcap);
        return 1;
    }

    if (found == 0) {
        printf("No transaction attribute found in this pcap.\n");
    }

    trans_dpi_engine_exit();
    trans_dpi_pcap_close(pcap);

    printf("Exiting.\n");

    return 0;
}
