/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/

/* standard headers */
#include <stdio.h>
#include <string.h>
#include <inttypes.h>

/* libpcap header for capture */
#include <pcap.h>

/* Qosmos ixEngine headers */
#include <qmdpi.h>
#include <qmdpi_bundle_api.h>

static uint64_t packet_number;


static struct qmdpi_engine *engine;
static struct qmdpi_worker *worker;
static struct qmdpi_bundle *bundle;

/* open pcap file */
static pcap_t *uhh_pcap_open(const char filename[])
{
    pcap_t *fd;
    char errbuf[PCAP_ERRBUF_SIZE];
    fd = pcap_open_offline(filename, errbuf);
    if (0 == fd) {
        fprintf(stderr, "cannot open pcap file: %s\n", errbuf);
        return NULL;
    }
    return fd;
}

/* close pcap file */
static void uhh_pcap_close(pcap_t *p)
{
    if (p) {
        pcap_close(p);
    }
}

/* user upper HTTP/HTTPS API to add protocol support */
static int uhh_add_protocols(void)
{
    int ret;

    ret = qmdpi_bundle_signature_upper_add(bundle, "http", "qosmos", "Qosmos", 0,
                                           "qosmos.com");
    if (ret < 0) {
        fprintf(stderr, "Cannot register protocol `qosmos': %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto out;
    }

    ret = qmdpi_bundle_signature_upper_add(bundle, "https", "httpbin", "httpbin", 0,
                                           "httpbin.org");
    if (ret < 0) {
        fprintf(stderr, "Cannot register protocol `httpbin': %s\n",
                qmdpi_error_get_string(bundle, ret));
    }

out:

    return ret;
}

static int uhh_del_protocols(void)
{
    int ret;

    ret = qmdpi_bundle_signature_upper_del(bundle, "qosmos");
    if (ret < 0) {
        fprintf(stderr, "Cannot remove protocol `qosmos': %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto out;
    }

    ret = qmdpi_bundle_signature_upper_del(bundle, "httpbin");
    if (ret < 0) {
        fprintf(stderr, "Cannot remove protocol `httpbin': %s\n",
                qmdpi_error_get_string(bundle, ret));
    }

out:

    return ret;
}


static int uhh_engine_init(void)
{
    int ret = 0;

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=1000");
    if (engine == NULL) {
        fprintf(stderr, "cannot create engine instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        return -1;
    }

    /* create worker instance */
    worker = qmdpi_worker_create(engine);
    if (worker == NULL) {
        fprintf(stderr, "cannot create worker instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_engine;

    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        fprintf(stderr, "cannot create bundle instance: %s\n",
                qmdpi_error_get_string(NULL, qmdpi_error_get()));
        goto error_worker;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        fprintf(stderr, "cannot activate bundle: %s\n", qmdpi_error_get_string(bundle,
                                                                               ret));
        goto error_bundle;

    }

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(bundle);
    if (ret < 0) {
        fprintf(stderr, "error enabling all protocols: %s\n",
                qmdpi_error_get_string(bundle, ret));
        goto error_bundle;
    }

    /* add new upper HTTP/HTTPS protocols */
    if (uhh_add_protocols() < 0) {
        goto error_bundle;
    }
    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_worker:
    qmdpi_worker_destroy(worker);
error_engine:
    qmdpi_engine_destroy(engine);

    return -1;
}

static void uhh_result_display(struct qmdpi_result *result)
{
    struct qmdpi_result_flags const *result_flags = qmdpi_result_flags_get(result);

    if (!QMDPI_RESULT_FLAGS_PATH_CHANGED(result_flags)) {
        return;
    }

    char buffer[4096];
    unsigned int i;
    int pos = 0;

    struct qmdpi_flow *flow = qmdpi_result_flow_get(result);
    struct qmdpi_path *path = qmdpi_result_path_get(result);
    struct qmdpi_bundle *b = qmdpi_flow_bundle_get(flow);
    struct qmdpi_signature *signature;
    char const *name;

    for (i = 0; i < path->qp_len; i++) {
        signature = qmdpi_bundle_signature_get_byid(b, path->qp_value[i]);
        name = qmdpi_signature_name_get(signature);
        pos += sprintf(buffer + pos, "%s%s", i > 0 ? "." : "", name);
    }

    printf("%"PRIu64" f(%"PRIu64")/path=%s\n", packet_number,
           qmdpi_flow_id_get(flow), buffer);
}

static void uhh_engine_exit(void)
{
    uhh_del_protocols();

    qmdpi_bundle_destroy(bundle);

    do {
        struct qmdpi_result *result;

        if (qmdpi_flow_expire_next(worker, NULL, &result) != 0) {
            break;
        }

        /* play with result */
        uhh_result_display(result);
    } while (1);

    qmdpi_worker_destroy(worker);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
}

static void uhh_process(u_char *user,
                        const struct pcap_pkthdr *phdr,
                        const u_char *pdata)
{
    struct qmdpi_result *result;
    int ret;

    (void)user;
    packet_number++;

    /* set PDU information to be processed by the worker */
    if (qmdpi_worker_pdu_set(worker, pdata, phdr->caplen, &phdr->ts,
                             QMDPI_PROTO_ETH, QMDPI_DIR_DEFAULT, 0) != 0) {
        return;
    }

    /* process packet with worker and provide DPI result */
    do {
        ret = qmdpi_worker_process(worker, NULL, &result);
        if (ret < 0) {
            printf("DPI processing failure at packet #%"PRIu64"\n", packet_number);
        } else {
            uhh_result_display(result);
        }
        /* loop if the packet contains multiple flows */
    } while (ret == QMDPI_PROCESS_MORE);

    if ((packet_number % 1000) == 0) {
        /* destroy expired flows and provide DPI result for each of them */
        int nb_remaining = 100;
        do {
            if (qmdpi_flow_expire_next(worker, &phdr->ts, &result) != 0) {
                break;
            }

            nb_remaining --;
            /* play with result */
            uhh_result_display(result);
        } while (nb_remaining);
    }
}

int main(int argc, const char **argv)
{
    pcap_t *pcap;
    int ret;

    /* check arguments */
    if (argc != 2) {
        fprintf(stderr, "Usage: %s <pcap_file>\n", *argv);
        return 1;
    }

    /* check pcap */
    pcap = uhh_pcap_open(argv[1]);
    if (pcap == NULL) {
        return 1;
    }


    /* init Qosmos ixEngine */
    ret = uhh_engine_init();
    if (ret < 0) {
        return 1;
    }

    ret = pcap_loop(pcap, -1, uhh_process, NULL);
    if (ret == -1) {
        fprintf(stderr, "error: pcap_loop\n");
        uhh_engine_exit();
        uhh_pcap_close(pcap);
        return 1;
    }

    uhh_engine_exit();
    uhh_pcap_close(pcap);

    return 0;
}
