/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2022 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/

#include <vnet/vnet.h>
#include <vnet/plugin/plugin.h>
#include <dpi/dpi.h>

#include <vlibapi/api.h>
#include <vlibmemory/api.h>
#include <stdbool.h>

#include <qmdpi.h>

dpi_main_t dpi_main;

char ixe_config[128] = "";

/* initialize Qosmos ixEngine */
static clib_error_t *classification_engine_init(dpi_main_t *dpi, u32 nb_workers)
{
    int ret = 0;
    int i;

    if (nb_workers > MAX_NB_WORKERS)
        return clib_error_return(0, "Too many workers requested (%d, max=%d)\n",
                                 nb_workers, MAX_NB_WORKERS);

    /* create engine instance */
    sprintf(ixe_config, "injection_mode=stream;nb_flows=100000;nb_workers=%d;",
            nb_workers);
    dpi_print(dpi->vlib_main, "[DPI] ixe configuration: %s\n", ixe_config);

    dpi->engine = qmdpi_engine_create((char const *) ixe_config);
    if (dpi->engine == NULL)
        return clib_error_return(0, "Cannot create engine instance: %s\n",
                                 qmdpi_error_get_string(NULL, qmdpi_error_get()));

    /* create workers instances */
    for (i = 0; i < nb_workers; i++) {
        dpi->workers_table[i] = qmdpi_worker_create(dpi->engine);
        if (dpi->workers_table[i] == NULL)
            return clib_error_return(0, "Cannot create worker instance: %s\n",
                                     qmdpi_error_get_string(NULL, qmdpi_error_get()));
    }

    /* create bundle instance */
    dpi->bundle = qmdpi_bundle_create_from_file(dpi->engine, "libqmbundle.so");
    if (dpi->bundle == NULL)
        return clib_error_return(0, "Cannot create bundle instance: %s\n",
                                 qmdpi_error_get_string(NULL, qmdpi_error_get()));

    /* enable all signatures on bundle */
    ret = qmdpi_bundle_signature_enable_all(dpi->bundle);
    if (ret < 0)
        return clib_error_return(0, "Error enabling all protocols: %s\n",
                                 qmdpi_error_get_string(dpi->bundle, ret));

    /* activate bundle */
    ret = qmdpi_bundle_activate(dpi->bundle);
    if (ret < 0)
        return clib_error_return(0, "Cannot activate bundle: %s\n",
                                 qmdpi_error_get_string(dpi->bundle, ret));

    return NULL;
}

#if (defined DEBUG) && (DEBUG >= 1)
#define DPI_LOG_DESC "dpi log [enable|disable]"
static clib_error_t *
dpi_log_command_fn(vlib_main_t          *vm,
                   unformat_input_t     *input,
                   vlib_cli_command_t   *cmd)
{
    while (unformat_check_input(input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(input, "enable")) {
            dpi_main.log_enable = 1;
        } else if (unformat(input, "disable")) {
            dpi_main.log_enable = 0;
        } else {
            break;
        }
    }

    if (dpi_main.log_enable) {
        dpi_print(vm, "DPI logging is enabled\n");
    } else {
        dpi_print(vm, "DPI logging is disabled\n");
    }
    return 0;
}
#endif  /* DEBUG */

#define DPI_ADD_DESC "dpi add next-node <Node>"
static clib_error_t *
dpi_add_command_fn(vlib_main_t          *vm,
                   unformat_input_t     *input,
                   vlib_cli_command_t   *cmd)
{
    unformat_input_t _line_input, *line_input = &_line_input;
    u32 next_node_index = ~0;
    u32 next_index = ~0;

    /* Get a line of input. */
    if (!unformat_user(input, unformat_line_input, line_input)) {
        return clib_error_return(0, "Invalid argument\nUsage:\n\t"DPI_ADD_DESC"\n");
    }

    while (unformat_check_input(line_input) != UNFORMAT_END_OF_INPUT) {
        if (unformat(line_input, "next-node %U", unformat_vlib_node,
                     dpi_main.vlib_main, &next_node_index)) {
            continue;
        } else {
            return clib_error_return(0, "Invalid argument\nUsage:\n\t"DPI_ADD_DESC"\n");
        }
    }

    if (next_node_index == ~0) {
        return clib_error_return(0, "Invalid argument\nUsage:\n\t"DPI_ADD_DESC"\n");
    }

    next_index = vlib_node_get_next(dpi_main.vlib_main,
                                    dpi_main.dpi_index,
                                    next_node_index);
    if (next_index == (~0)) /* not found */
        next_index = vlib_node_add_next(dpi_main.vlib_main,
                                        dpi_main.dpi_index,
                                        next_node_index);
    dpi_main.next_node_index = next_index;

    return 0;
}

#define DPI_INFO_DESC "dpi info"
static clib_error_t *
dpi_info_command_fn(vlib_main_t          *vm,
                    unformat_input_t     *input,
                    vlib_cli_command_t   *cmd)
{
    dpi_main_t *dpi = &dpi_main;
    vlib_node_t *node;

    (void)input;
    (void)cmd;

    dpi_print(vm,
              "The dpi plugin integrates the ixEngine libraries in stream mode and \n"
              "provides flow classification results for the processed packets.\n"
              "This module is used in conjunction with the flowtable module.\n\n");

    dpi_print(vm, "DPI plugin version: %s\n", DPI_PLUGIN_VER);
    dpi_print(vm, "ixEngine version: %s\n",
              dpi->engine ? qmdpi_engine_version_get_string(dpi->engine) : "Unknown");
    dpi_print(vm, "Protocol bundle version: %s\n\n",
              dpi->bundle ? qmdpi_bundle_version_get_string(dpi->bundle) : "Unknown");

    /* IXE configuration */
    dpi_print(vm, "ixEngine configuration: \"%s\"\n", ixe_config);

    /* Next node */
    node = vlib_get_next_node(vm, dpi->dpi_index, dpi->next_node_index);
    if (node) {
        dpi_print(vm, "Next-node: %s\n", node->name);
    }
    dpi_print(vm, "\n");

    dpi_print(vm, "CLI commands:\n");
    dpi_print(vm, "\t"DPI_ADD_DESC"\n");
    dpi_print(vm, "\t"DPI_INFO_DESC"\n");
#if (defined DEBUG) && (DEBUG >= 1)
    dpi_print(vm, "\t"DPI_LOG_DESC"\n");
#endif /* DEBUG */

    return 0;
}

#if (defined DEBUG) && (DEBUG >= 1)
VLIB_CLI_COMMAND(dpi_log_command, static) = {
    .path = "dpi log",
    .short_help = DPI_LOG_DESC,
    .function = dpi_log_command_fn,
};
#endif /* DEBUG */

VLIB_CLI_COMMAND(dpi_add_command, static) = {
    .path = "dpi add",
    .short_help = DPI_ADD_DESC,
    .function = dpi_add_command_fn,
};

VLIB_CLI_COMMAND(dpi_info_command, static) = {
    .path = "dpi info",
    .short_help = DPI_INFO_DESC,
    .function = dpi_info_command_fn,
};

/* *INDENT-ON* */

__clib_export void dpi_get_ixe_elements(struct qmdpi_engine **engine,
                                        struct qmdpi_bundle **bundle,
                                        struct qmdpi_worker ***workers_table)
{
    *engine = dpi_main.engine;
    *bundle = dpi_main.bundle;
    *workers_table = &(dpi_main.workers_table[0]);
}

static clib_error_t *dpi_init(vlib_main_t *vm)
{
    clib_error_t *error = NULL;
    dpi_main_t *dpi = &dpi_main;

    dpi->vlib_main = vm;
    dpi->vnet_main = vnet_get_main();
    dpi->clocks_per_second = vm->clib_time.clocks_per_second;
    dpi->dpi_index = dpi_node.index;
    dpi->next_node_index = DPI_NEXT_ETHERNET_INPUT;

    vlib_thread_main_t *tm = vlib_get_thread_main();

    int thread_count = tm->n_threads;
    if (thread_count == 0) {
        thread_count = 1;
    }
    error = classification_engine_init(dpi, thread_count + 1);
    if (error) {
        dpi_print(vm, "[DPI] classification engine initialization failure\n");
        return error;
    }

    dpi_print(vm, "[DPI] plugin initialized: %d threads\n", thread_count);

    return NULL;
}

VLIB_INIT_FUNCTION(dpi_init);

/* *INDENT-OFF* */
VNET_FEATURE_INIT(dpi, static) = {
    .arc_name = "device-input",
    .node_name = "dpi",
    .runs_before = VNET_FEATURES("ethernet-input"),
};
/* *INDENT-ON */

/* *INDENT-OFF* */
VLIB_PLUGIN_REGISTER() = {
    .version = DPI_PLUGIN_VER,
    .description = "ixEngine-based DPI plugin",
};
/* *INDENT-ON* */

/*
 * fd.io coding-style-patch-verification: ON
 *
 * Local Variables:
 * eval: (c-set-style "gnu")
 * End:
 */
