/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2022 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/

#include <arpa/inet.h>

#include <vppinfra/os.h>

#include "flowtable.h"
#include <qmdpi.h>

#define FINISH                                  \
    vec_add1(s, 0);                            \
    ft_print(handle, (char *)s);               \
    vec_free(s)

static u8 *dump_ip4_address(u8 *s, ip4_address_t *addr)
{
    s = format(s, "%d.%d.%d.%d",
               addr->as_u8[0], addr->as_u8[1],
               addr->as_u8[2], addr->as_u8[3]);

    return s;
}

static u8 *dump_ip6_address(u8 *s, ip6_address_t *addr)
{
    int     i;

    for (i = 0; i < 16; i++) {
        s = format(s, "%02x", addr->as_u8[i]);
    }

    return s;
}

static u8 *dump_path(u8 *s, flow_entry_t *flow)
{
    struct qmdpi_bundle *bndl;
    struct qmdpi_path   *path;

    if (flow && flow->infos.data.dpi_flow) {
        struct qmdpi_flow *dpi_flow = flow->infos.data.dpi_flow;
        bndl = qmdpi_flow_bundle_get(dpi_flow);
        path = qmdpi_flow_path_get(dpi_flow);

        char buffer[200];
        qmdpi_data_path_to_buffer(bndl, buffer, sizeof(buffer), path);
        s = format(s, "%s", buffer);
    }

    return s;
}

static u8 *dump_sig(u8 *s, flow_signature_t *sig)
{
    if (sig->len == sizeof(struct ip4_sig)) {
        ip4_address_t addr;

        s = format(s, "IP4 ");
        addr = sig->s.ip4.src;
        s = dump_ip4_address(s, &addr);
        s = format(s, ":%d ", ntohs(sig->s.ip4.port_src));
        s = format(s, " <-> ");
        addr = sig->s.ip4.dst;
        s = dump_ip4_address(s, &addr);
        s = format(s, ":%d ", ntohs(sig->s.ip4.port_dst));
    } else {
        s = format(s, "IP6 ");
        s = dump_ip6_address(s, &sig->s.ip6.src);
        s = format(s, ":%d ", ntohs(sig->s.ip6.port_src));
        s = format(s, " <-> ");
        s = dump_ip6_address(s, &sig->s.ip6.dst);
        s = format(s, ":%d ", ntohs(sig->s.ip6.port_dst));
    }

    return s;
}

void flowtable_dump_flow(vlib_main_t *handle, flow_entry_t *flow)
{
    u8 *s = NULL;

    s = format(0, "[FT ] CPU %d Flow ", (int) os_get_thread_index());

    s = dump_sig(s, &flow->sig);
    s = format(s, " ");

    if (flow->infos.data.classified) {
        s = dump_path(s, flow);
    } else {
        s = format(s, "Not classified");
    }

    if (flow->infos.data.offloaded) {
        s = format(s, " Offloaded");
    }

    s = format(s, "\n");
    FINISH;
}

