/*
  This file is a part of Qosmos ixEngine.

   Copyright  Qosmos 2022 - All rights reserved

  This computer program and all its components are protected by
  authors' rights and copyright law and by international treaties.
  Any representation, reproduction, distribution or modification
  of this program or any portion of it is forbidden without
  Qosmos explicit and written agreement and may result in severe
  civil and criminal penalties, and will be prosecuted
  to the maximum extent possible under the law.
*/

#include <vppinfra/os.h>

#include "flowtable.h"
#include <qmdpi.h>

#define FINISH                                  \
    vec_add1(s, 0);                            \
    ft_print(handle, (char *)s);               \
    vec_free(s)

void flowtable_stats_reset(vlib_main_t *vm,
                           flowtable_main_t *fm)
{
    flowtable_main_per_cpu_t *fmt;
    vlib_thread_main_t *tm = vlib_get_thread_main();
    u32 cpu_index;

    (void)vm;

    for (cpu_index = 0; cpu_index < tm->n_vlib_mains; cpu_index++) {
        fmt = &fm->per_cpu[cpu_index];

        /* L3 */
        if (fmt->stat_l3) {
            vec_free(fmt->stat_l3);
        }
        fmt->stat_l3 = vec_new(flowtable_stat_t, 0);

        /* L4 */
        if (fmt->stat_l4) {
            vec_free(fmt->stat_l4);
        }
        fmt->stat_l4 = vec_new(flowtable_stat_t, 0);

        /* L5 */
        if (fmt->stat_l5) {
            vec_free(fmt->stat_l5);
        }
        fmt->stat_l5 = vec_new(flowtable_stat_t, 0);

        /* Application */
        if (fmt->stat_appli) {
            vec_free(fmt->stat_appli);
        }
        fmt->stat_appli = vec_new(flowtable_stat_t, 0);
    }
}

static void stat_add(flowtable_stat_t **stat_vec, u32 id, u32 count, u64 volume,
                     clib_rwlock_t *lock)
{
    flowtable_stat_t stat = {0};
    u32 i;

    for (i = 0; i < vec_len((*stat_vec)); i++)
        if ((*stat_vec)[i].id == id) {
            (*stat_vec)[i].count += count;
            (*stat_vec)[i].volume += volume;
            return;
        }

    stat.id = id;
    stat.count = count;
    stat.volume = volume;
    if (lock) {
        clib_rwlock_writer_lock(lock);
    }
    vec_add1((*stat_vec), stat);
    if (lock) {
        clib_rwlock_writer_unlock(lock);
    }
}

void flowtable_stats_add(vlib_main_t *vm,
                         flowtable_main_t *fm,
                         flowtable_main_per_cpu_t *fmt,
                         flow_entry_t *flow,
                         vlib_buffer_t *b)
{

    struct qmdpi_path *path;
    flowtable_stat_t  *stats;
    uint32_t i, l3 = 0;
    u16 pkt_cnt = 1;
    u16 pkt_volume = b->current_length + b->current_data;

    (void)vm;
    (void)fm;

    if (!flow || !flow->infos.data.dpi_flow) {
        return;
    }

    if (!flow->infos.data.classified) {
        flow->pkt_cnt += pkt_cnt;
        flow->pkt_volume += pkt_volume;
        return;
    }

    /* Add cached packet stats */
    if (flow->pkt_cnt) {
        pkt_cnt += flow->pkt_cnt;
        flow->pkt_cnt = 0;
        pkt_volume += flow->pkt_volume;
        flow->pkt_volume = 0;
    }

    path = qmdpi_flow_path_get(flow->infos.data.dpi_flow);

    if (path->qp_len < 2) {
        return;
    }

    for (i = 0; i < path->qp_len; i++)
        if (path->qp_value[i] == Q_PROTO_IP ||
                path->qp_value[i] == Q_PROTO_IP6 ||
                path->qp_value[i] == Q_PROTO_ARP) {
            l3 = i;
        }

    if (l3 == 0) {
        l3 = 1;
    }

    /* L3 */
    if (path->qp_len > l3)
        stat_add(&fmt->stat_l3,
                 path->qp_value[l3],
                 pkt_cnt, pkt_volume,
                 &fmt->stat_lock);

    /* L4 */
    if (path->qp_len > l3 + 1)
        stat_add(&fmt->stat_l4,
                 path->qp_value[l3 + 1],
                 pkt_cnt, pkt_volume,
                 &fmt->stat_lock);

    /* L5 */
    if (path->qp_len > l3 + 2)
        stat_add(&fmt->stat_l5,
                 path->qp_value[l3 + 2],
                 pkt_cnt, pkt_volume,
                 &fmt->stat_lock);

    /* Application */
    if (path->qp_len > l3 + 3)
        stat_add(&fmt->stat_appli,
                 path->qp_value[path->qp_len - 1],
                 pkt_cnt, pkt_volume,
                 &fmt->stat_lock);
}

static void flowtable_stats_show_vec(vlib_main_t *handle,
                                     flowtable_main_t *fm,
                                     flowtable_stat_t **stat_array,
                                     u32 stat_array_size,
                                     const char *stat_name)
{
    u8 *s = NULL;
    struct qmdpi_bundle *bundle = fm->bundle;
    struct qmdpi_signature *signature;
    flowtable_stat_t *stat_vec;
    u32 count_all = 0;
    u64 volume_all = 0;
    u32 proto_index, i;

    s = format(0, "%-20sPackets             Bytes\n", stat_name);
    s = format(s, "--------            -------             -----\n");
    for (i = 0; i < stat_array_size; i++) {
        stat_vec = stat_array[i];
        for (proto_index = 0; proto_index < vec_len(stat_vec); proto_index++) {
            signature = qmdpi_bundle_signature_get_byid(bundle, stat_vec[proto_index].id);
            if (!signature) {
                continue;
            }

            s = format(s, "%-20s%-20ld%-20ld\n",
                       qmdpi_signature_name_get(signature),
                       stat_vec[proto_index].count,
                       stat_vec[proto_index].volume);
            if (i == stat_array_size - 1) {
                count_all += stat_vec[proto_index].count;
                volume_all += stat_vec[proto_index].volume;
            }
        }
    }
    s = format(s, "--------            -------             -----\n");
    s = format(s, "%-20s%-20ld%-20ld\n\n", "Total", count_all, volume_all);
    FINISH;
}

clib_error_t *flowtable_stats_show(vlib_main_t *handle,
                                   flowtable_main_t *fm)
{
    flowtable_stat_t *stat_array[3];
    flowtable_stat_t *stat_all = NULL;
    flowtable_stat_t *stat_l3 = NULL;
    flowtable_stat_t *stat_l4 = NULL;
    flowtable_stat_t *stat_fmt;
    vlib_thread_main_t *tm = vlib_get_thread_main();
    u32 cpu_index, proto_index;

    if (!handle || !fm) {
        return clib_error_return(0, "Invalid argument\n");
    }

    if (!fm->bundle) {
        return clib_error_return(0, "Stats not available\n");
    }

    /* Protocols */
    stat_all = vec_new(flowtable_stat_t, 0);
    stat_l3 = vec_new(flowtable_stat_t, 0);
    stat_l4 = vec_new(flowtable_stat_t, 0);
    stat_all = vec_new(flowtable_stat_t, 0);
    for (cpu_index = 0; cpu_index < tm->n_vlib_mains; cpu_index++) {
        clib_rwlock_reader_lock(&fm->per_cpu[cpu_index].stat_lock);

        /* L3 */
        stat_fmt = fm->per_cpu[cpu_index].stat_l3;
        for (proto_index = 0; proto_index < vec_len(stat_fmt); proto_index++)
            stat_add(&stat_l3,
                     stat_fmt[proto_index].id,
                     stat_fmt[proto_index].count,
                     stat_fmt[proto_index].volume,
                     NULL);

        /* L4 */
        stat_fmt = fm->per_cpu[cpu_index].stat_l4;
        for (proto_index = 0; proto_index < vec_len(stat_fmt); proto_index++)
            stat_add(&stat_l4,
                     stat_fmt[proto_index].id,
                     stat_fmt[proto_index].count,
                     stat_fmt[proto_index].volume,
                     NULL);

        /*L5 Proto*/
        stat_fmt = fm->per_cpu[cpu_index].stat_l5;
        for (proto_index = 0; proto_index < vec_len(stat_fmt); proto_index++)
            stat_add(&stat_all,
                     stat_fmt[proto_index].id,
                     stat_fmt[proto_index].count,
                     stat_fmt[proto_index].volume,
                     NULL);
        clib_rwlock_reader_unlock(&fm->per_cpu[cpu_index].stat_lock);
    }
    stat_array[0] = stat_l3;
    stat_array[1] = stat_l4;
    stat_array[2] = stat_all;
    flowtable_stats_show_vec(handle, fm, stat_array, 3, "Protocol");
    vec_free(stat_l3);
    vec_free(stat_l4);
    vec_free(stat_all);

    /* Application */
    stat_all = vec_new(flowtable_stat_t, 0);
    for (cpu_index = 0; cpu_index < tm->n_vlib_mains; cpu_index++) {
        stat_fmt = fm->per_cpu[cpu_index].stat_appli;
        clib_rwlock_reader_lock(&fm->per_cpu[cpu_index].stat_lock);
        for (proto_index = 0; proto_index < vec_len(stat_fmt); proto_index++)
            stat_add(&stat_all,
                     stat_fmt[proto_index].id,
                     stat_fmt[proto_index].count,
                     stat_fmt[proto_index].volume,
                     NULL);
        clib_rwlock_reader_unlock(&fm->per_cpu[cpu_index].stat_lock);
    }
    stat_array[0] = stat_all;
    flowtable_stats_show_vec(handle, fm, stat_array, 1, "Application");
    vec_free(stat_all);

    return NULL;
}
