#ifndef __PACKET_HELPER_H__
#define __PACKET_HELPER_H__

#include <stdint.h>
#include "dpi/protodef.h"

#define QM_ETHERTYPE_IP    0x0800
#define QM_ETHERTYPE_ETH1Q 0x8100
#define QM_ETHERTYPE_IP6   0x86dd

#pragma pack(1)
struct qm_mac_addr {
    uint8_t addr[6]; /**< Ethernet address on 6 bytes. */
};
#pragma pack()

#pragma pack(1)
struct qm_eth_hdr {
    struct qm_mac_addr dest;
    struct qm_mac_addr src;
    uint16_t      h_proto;
};
#pragma pack()

#pragma pack(1)
struct qm_eth1q_hdr {
    uint16_t        tci;
    uint16_t        h_proto;
};
#pragma pack()

#pragma pack(1)
struct qm_vxlan_hdr {
    uint8_t flags;
    uint8_t reserved0[2];
    uint8_t protocol; /* used by vxlan-gpe */
    uint8_t vni[3];
    uint8_t reserverd1;
};
#pragma pack()

#pragma pack(1)
struct qm_arp_hdr {
    uint16_t h_type;
    uint16_t p_type;
    uint8_t  h_size;
    uint8_t  p_size;
    uint16_t opcode;
};
#pragma pack()

enum {
    PPPOE_VERSION = 1,
    PPPOE_TYPE,
    PPPOE_CODE,
    PPPOE_SESSION_ID
};

#pragma pack(1)
struct qm_pppoe_hdr {
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
    uint8_t      type: 4,
                 version: 4;
#else
#if __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
    uint8_t      version: 4,
                 type: 4;
#else
#error     "Please fix LITTLE/BIG_ENDIAN define"
#endif
#endif

    uint8_t      code;
    uint16_t     session_id;
    uint16_t     payload_len;
};
#pragma pack()


#pragma pack(1)
struct qm_ipx_hdr {
    uint16_t    checksum;
    uint16_t    pkt_size;
    uint8_t     tctrl;
    uint8_t     type;
    uint32_t    dest_net;
    uint8_t     dest_node[6];
    uint16_t    dest_socket;
    uint32_t    src_net;
    uint8_t     src_node[6];
    uint16_t    src_socket;
};
#pragma pack()

#pragma pack(1)
struct qm_ip_hdr {
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
    uint8_t    ihl: 4,
               version: 4;
#else
#if __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
    uint8_t    version: 4,
               ihl: 4;
#else
#error    "Please fix LITTLE/BIG_ENDIAN define"
#endif
#endif

    uint8_t    tos;
    uint16_t   tot_len;
    uint16_t   id;
    uint16_t   frag_off;
    uint8_t    ttl;
    uint8_t    protocol;
    uint16_t   check;
    uint32_t   saddr;
    uint32_t   daddr;
};
#pragma pack()

#define MAPPING_UNDEFINED 0

/* Map IP protocol to Qosmos signature ID */
#define MAP_IPPROTO_TO_QPROTO(v,p) (v) = protocol_mapping_get(p)
static __inline int protocol_mapping_get(uint8_t protocol_id)
{
    static const int protocol_mapping[256] = {
        Q_PROTO_HOPOPT,         /* 0 */
        Q_PROTO_ICMP,           /* 1 */
        Q_PROTO_IGMP,           /* 2 */
        Q_PROTO_GGP,            /* 3 */
        Q_PROTO_IP_IN_IP,       /* 4 */
        Q_PROTO_ST,             /* 5 */
        Q_PROTO_TCP,            /* 6 */
        Q_PROTO_CBT,            /* 7 */
        Q_PROTO_EGP,            /* 8 */
        Q_PROTO_IGP,            /* 9 */
        Q_PROTO_BBN_RCC_MON,    /* 10 */
        Q_PROTO_NVP_II,         /* 11 */
        Q_PROTO_PUP,            /* 12 */
        Q_PROTO_ARGUS,          /* 13 */
        Q_PROTO_EMCON,          /* 14 */
        Q_PROTO_XNET,           /* 15 */
        Q_PROTO_CHAOS,          /* 16 */
        Q_PROTO_UDP,            /* 17 */
        Q_PROTO_MUX,            /* 18 */
        Q_PROTO_DCN_MEAS,       /* 19 */
        Q_PROTO_HMP,            /* 20 */
        Q_PROTO_PRM,            /* 21 */
        Q_PROTO_XNS_IDP,        /* 22 */
        Q_PROTO_TRUNK_1,        /* 23 */
        Q_PROTO_TRUNK_2,        /* 24 */
        Q_PROTO_LEAF_1,         /* 25 */
        Q_PROTO_LEAF_2,         /* 26 */
        Q_PROTO_IP_RDP,         /* 27 */
        Q_PROTO_IRTP,           /* 28 */
        Q_PROTO_ISO_TP4,        /* 29 */
        Q_PROTO_NETBLT,         /* 30 */
        Q_PROTO_MFE_NSP,        /* 31 */
        Q_PROTO_MERIT_INP,      /* 32 */
        Q_PROTO_DCCP,           /* 33 */
        Q_PROTO_3PC,            /* 34 */
        Q_PROTO_IDPR,           /* 35 */
        Q_PROTO_XTP,            /* 36 */
        Q_PROTO_DDP,            /* 37 */
        Q_PROTO_IDPR_CMTP,      /* 38 */
        Q_PROTO_TP_PLUS_PLUS,   /* 39 */
        Q_PROTO_IL,             /* 40 */
        Q_PROTO_IP6,            /* 41 */
        Q_PROTO_SDRP,           /* 42 */
        MAPPING_UNDEFINED,      /* 43 */
        MAPPING_UNDEFINED,      /* 44 */
        Q_PROTO_IDRP,           /* 45 */
        Q_PROTO_RSVP,           /* 46 */
        Q_PROTO_GRE,            /* 47 */
        Q_PROTO_DSR,            /* 48 */
        Q_PROTO_BNA,            /* 49 */
        Q_PROTO_ESP,            /* 50 */
        Q_PROTO_AH,             /* 51 */
        Q_PROTO_I_NLSP,         /* 52 */
        Q_PROTO_SWIPE,          /* 53 */
        Q_PROTO_NARP,           /* 54 */
        Q_PROTO_MOBILE,         /* 55 */
        Q_PROTO_TLSP,           /* 56 */
        Q_PROTO_SKIP,           /* 57 */
        Q_PROTO_ICMP6,          /* 58 */
        MAPPING_UNDEFINED,      /* 59 */
        MAPPING_UNDEFINED,      /* 60 */
        Q_PROTO_HOST,           /* 61 */
        Q_PROTO_CFTP,           /* 62 */
        Q_PROTO_LAN,            /* 63 */
        Q_PROTO_SAT_EXPAK,      /* 64 */
        Q_PROTO_KRYPTOLAN,      /* 65 */
        Q_PROTO_RVD,            /* 66 */
        Q_PROTO_IPPC,           /* 67 */
        Q_PROTO_DFS,            /* 68 */
        Q_PROTO_SAT_MON,        /* 69 */
        Q_PROTO_VISA,           /* 70 */
        Q_PROTO_IPCV,           /* 71 */
        Q_PROTO_CPNX,           /* 72 */
        Q_PROTO_CPHB,           /* 73 */
        Q_PROTO_WSN,            /* 74 */
        Q_PROTO_PVP,            /* 75 */
        Q_PROTO_BR_SAT_MON,     /* 76 */
        Q_PROTO_SUN_ND,         /* 77 */
        Q_PROTO_WB_MON,         /* 78 */
        Q_PROTO_WB_EXPAK,       /* 79 */
        Q_PROTO_ISO_IP,         /* 80 */
        Q_PROTO_VMTP,           /* 81 */
        Q_PROTO_SECURE_VMTP,    /* 82 */
        Q_PROTO_VINES,          /* 83 */
        Q_PROTO_IPTM,           /* 84 */
        Q_PROTO_NSFNET_IGP,     /* 85 */
        Q_PROTO_DGP,            /* 86 */
        Q_PROTO_TCF,            /* 87 */
        Q_PROTO_EIGRP,          /* 88 */
        Q_PROTO_OSPF,           /* 89 */
        Q_PROTO_SPRITE_RPC,     /* 90 */
        Q_PROTO_LARP,           /* 91 */
        Q_PROTO_MTP,            /* 92 */
        Q_PROTO_AX25,           /* 93 */
        MAPPING_UNDEFINED,      /* 94 */
        Q_PROTO_MICP,           /* 95 */
        Q_PROTO_SCC_SP,         /* 96 */
        Q_PROTO_ETHERIP,        /* 97 */
        Q_PROTO_ENCAP,          /* 98 */
        Q_PROTO_PRIVATE_ENC,    /* 99 */
        Q_PROTO_GMTP,           /* 100 */
        Q_PROTO_IFMP,           /* 101 */
        Q_PROTO_PNNI,           /* 102 */
        Q_PROTO_PIM,            /* 103 */
        Q_PROTO_ARIS,           /* 104 */
        Q_PROTO_SCPS,           /* 105 */
        Q_PROTO_QNX,            /* 106 */
        Q_PROTO_ACTIVENET,      /* 107 */
        Q_PROTO_IPCOMP,         /* 108 */
        Q_PROTO_SNP,            /* 109 */
        Q_PROTO_COMPAQ_PEER,    /* 110 */
        Q_PROTO_IPX_IN_IP,      /* 111 */
        Q_PROTO_VRRP,           /* 112 */
        Q_PROTO_PGM,            /* 113 */
        Q_PROTO_ANY_0HOP,       /* 114 */
        Q_PROTO_L2TP,           /* 115 */
        Q_PROTO_DDX,            /* 116 */
        Q_PROTO_IATP,           /* 117 */
        Q_PROTO_SCHEDULED_TP,   /* 118 */
        Q_PROTO_SRP,            /* 119 */
        Q_PROTO_UTI,            /* 120 */
        Q_PROTO_SIMPLE_MSG_P,   /* 121 */
        Q_PROTO_SM,             /* 122 */
        Q_PROTO_PTP,            /* 123 */
        Q_PROTO_ISIS,           /* 124 */
        Q_PROTO_FIRE,           /* 125 */
        Q_PROTO_CRTP,           /* 126 */
        Q_PROTO_CRUDP,          /* 127 */
        Q_PROTO_SSCOPMCE,       /* 128 */
        Q_PROTO_IPLT,           /* 129 */
        Q_PROTO_SPS,            /* 130 */
        Q_PROTO_PIPE,           /* 131 */
        Q_PROTO_SCTP,           /* 132 */
        Q_PROTO_FIBRE_CHANNEL,  /* 133 */
        Q_PROTO_RSVP_E2E_IGNORE, /* 134 */
        Q_PROTO_MOBILEHDR,      /* 135 */
        Q_PROTO_UDPLITE,        /* 136 */
        Q_PROTO_MPLS_IN_IP,     /* 137 */
        Q_PROTO_MANET,          /* 138 */
        Q_PROTO_HIP,            /* 139 */
        Q_PROTO_SHIM6,          /* 140 */
        Q_PROTO_WESP,           /* 141 */
        Q_PROTO_ROHC,           /* 142 */
        Q_PROTO_UNASSIGNED_IP_PROT_143, /* 143 */
        Q_PROTO_UNASSIGNED_IP_PROT_144, /* 144 */
        Q_PROTO_UNASSIGNED_IP_PROT_145, /* 145 */
        Q_PROTO_UNASSIGNED_IP_PROT_146, /* 146 */
        Q_PROTO_UNASSIGNED_IP_PROT_147, /* 147 */
        Q_PROTO_UNASSIGNED_IP_PROT_148, /* 148 */
        Q_PROTO_UNASSIGNED_IP_PROT_149, /* 149 */
        Q_PROTO_UNASSIGNED_IP_PROT_150, /* 150 */
        Q_PROTO_UNASSIGNED_IP_PROT_151, /* 151 */
        Q_PROTO_UNASSIGNED_IP_PROT_152, /* 152 */
        Q_PROTO_UNASSIGNED_IP_PROT_153, /* 153 */
        Q_PROTO_UNASSIGNED_IP_PROT_154, /* 154 */
        Q_PROTO_UNASSIGNED_IP_PROT_155, /* 155 */
        Q_PROTO_UNASSIGNED_IP_PROT_156, /* 156 */
        Q_PROTO_UNASSIGNED_IP_PROT_157, /* 157 */
        Q_PROTO_UNASSIGNED_IP_PROT_158, /* 158 */
        Q_PROTO_UNASSIGNED_IP_PROT_159, /* 159 */
        Q_PROTO_UNASSIGNED_IP_PROT_160, /* 160 */
        Q_PROTO_UNASSIGNED_IP_PROT_161, /* 161 */
        Q_PROTO_UNASSIGNED_IP_PROT_162, /* 162 */
        Q_PROTO_UNASSIGNED_IP_PROT_163, /* 163 */
        Q_PROTO_UNASSIGNED_IP_PROT_164, /* 164 */
        Q_PROTO_UNASSIGNED_IP_PROT_165, /* 165 */
        Q_PROTO_UNASSIGNED_IP_PROT_166, /* 166 */
        Q_PROTO_UNASSIGNED_IP_PROT_167, /* 167 */
        Q_PROTO_UNASSIGNED_IP_PROT_168, /* 168 */
        Q_PROTO_UNASSIGNED_IP_PROT_169, /* 169 */
        Q_PROTO_UNASSIGNED_IP_PROT_170, /* 170 */
        Q_PROTO_UNASSIGNED_IP_PROT_171, /* 171 */
        Q_PROTO_UNASSIGNED_IP_PROT_172, /* 172 */
        Q_PROTO_UNASSIGNED_IP_PROT_173, /* 173 */
        Q_PROTO_UNASSIGNED_IP_PROT_174, /* 174 */
        Q_PROTO_UNASSIGNED_IP_PROT_175, /* 175 */
        Q_PROTO_UNASSIGNED_IP_PROT_176, /* 176 */
        Q_PROTO_UNASSIGNED_IP_PROT_177, /* 177 */
        Q_PROTO_UNASSIGNED_IP_PROT_178, /* 178 */
        Q_PROTO_UNASSIGNED_IP_PROT_179, /* 179 */
        Q_PROTO_UNASSIGNED_IP_PROT_180, /* 180 */
        Q_PROTO_UNASSIGNED_IP_PROT_181, /* 181 */
        Q_PROTO_UNASSIGNED_IP_PROT_182, /* 182 */
        Q_PROTO_UNASSIGNED_IP_PROT_183, /* 183 */
        Q_PROTO_UNASSIGNED_IP_PROT_184, /* 184 */
        Q_PROTO_UNASSIGNED_IP_PROT_185, /* 185 */
        Q_PROTO_UNASSIGNED_IP_PROT_186, /* 186 */
        Q_PROTO_UNASSIGNED_IP_PROT_187, /* 187 */
        Q_PROTO_UNASSIGNED_IP_PROT_188, /* 188 */
        Q_PROTO_UNASSIGNED_IP_PROT_189, /* 189 */
        Q_PROTO_UNASSIGNED_IP_PROT_190, /* 190 */
        Q_PROTO_UNASSIGNED_IP_PROT_191, /* 191 */
        Q_PROTO_UNASSIGNED_IP_PROT_192, /* 192 */
        Q_PROTO_UNASSIGNED_IP_PROT_193, /* 193 */
        Q_PROTO_UNASSIGNED_IP_PROT_194, /* 194 */
        Q_PROTO_UNASSIGNED_IP_PROT_195, /* 195 */
        Q_PROTO_UNASSIGNED_IP_PROT_196, /* 196 */
        Q_PROTO_UNASSIGNED_IP_PROT_197, /* 197 */
        Q_PROTO_UNASSIGNED_IP_PROT_198, /* 198 */
        Q_PROTO_UNASSIGNED_IP_PROT_199, /* 199 */
        Q_PROTO_UNASSIGNED_IP_PROT_200, /* 200 */
        Q_PROTO_UNASSIGNED_IP_PROT_201, /* 201 */
        Q_PROTO_UNASSIGNED_IP_PROT_202, /* 202 */
        Q_PROTO_UNASSIGNED_IP_PROT_203, /* 203 */
        Q_PROTO_UNASSIGNED_IP_PROT_204, /* 204 */
        Q_PROTO_UNASSIGNED_IP_PROT_205, /* 205 */
        Q_PROTO_UNASSIGNED_IP_PROT_206, /* 206 */
        Q_PROTO_UNASSIGNED_IP_PROT_207, /* 207 */
        Q_PROTO_UNASSIGNED_IP_PROT_208, /* 208 */
        Q_PROTO_UNASSIGNED_IP_PROT_209, /* 209 */
        Q_PROTO_UNASSIGNED_IP_PROT_210, /* 210 */
        Q_PROTO_UNASSIGNED_IP_PROT_211, /* 211 */
        Q_PROTO_UNASSIGNED_IP_PROT_212, /* 212 */
        Q_PROTO_UNASSIGNED_IP_PROT_213, /* 213 */
        Q_PROTO_UNASSIGNED_IP_PROT_214, /* 214 */
        Q_PROTO_UNASSIGNED_IP_PROT_215, /* 215 */
        Q_PROTO_UNASSIGNED_IP_PROT_216, /* 216 */
        Q_PROTO_UNASSIGNED_IP_PROT_217, /* 217 */
        Q_PROTO_UNASSIGNED_IP_PROT_218, /* 218 */
        Q_PROTO_UNASSIGNED_IP_PROT_219, /* 219 */
        Q_PROTO_UNASSIGNED_IP_PROT_220, /* 220 */
        Q_PROTO_UNASSIGNED_IP_PROT_221, /* 221 */
        Q_PROTO_UNASSIGNED_IP_PROT_222, /* 222 */
        Q_PROTO_UNASSIGNED_IP_PROT_223, /* 223 */
        Q_PROTO_UNASSIGNED_IP_PROT_224, /* 224 */
        Q_PROTO_UNASSIGNED_IP_PROT_225, /* 225 */
        Q_PROTO_UNASSIGNED_IP_PROT_226, /* 226 */
        Q_PROTO_UNASSIGNED_IP_PROT_227, /* 227 */
        Q_PROTO_UNASSIGNED_IP_PROT_228, /* 228 */
        Q_PROTO_UNASSIGNED_IP_PROT_229, /* 229 */
        Q_PROTO_UNASSIGNED_IP_PROT_230, /* 230 */
        Q_PROTO_UNASSIGNED_IP_PROT_231, /* 231 */
        Q_PROTO_UNASSIGNED_IP_PROT_232, /* 232 */
        Q_PROTO_UNASSIGNED_IP_PROT_233, /* 233 */
        Q_PROTO_UNASSIGNED_IP_PROT_234, /* 234 */
        Q_PROTO_UNASSIGNED_IP_PROT_235, /* 235 */
        Q_PROTO_UNASSIGNED_IP_PROT_236, /* 236 */
        Q_PROTO_UNASSIGNED_IP_PROT_237, /* 237 */
        Q_PROTO_UNASSIGNED_IP_PROT_238, /* 238 */
        Q_PROTO_UNASSIGNED_IP_PROT_239, /* 239 */
        Q_PROTO_UNASSIGNED_IP_PROT_240, /* 240 */
        Q_PROTO_UNASSIGNED_IP_PROT_241, /* 241 */
        Q_PROTO_UNASSIGNED_IP_PROT_242, /* 242 */
        Q_PROTO_UNASSIGNED_IP_PROT_243, /* 243 */
        Q_PROTO_UNASSIGNED_IP_PROT_244, /* 244 */
        Q_PROTO_UNASSIGNED_IP_PROT_245, /* 245 */
        Q_PROTO_UNASSIGNED_IP_PROT_246, /* 246 */
        Q_PROTO_UNASSIGNED_IP_PROT_247, /* 247 */
        Q_PROTO_UNASSIGNED_IP_PROT_248, /* 248 */
        Q_PROTO_UNASSIGNED_IP_PROT_249, /* 249 */
        Q_PROTO_UNASSIGNED_IP_PROT_250, /* 250 */
        Q_PROTO_UNASSIGNED_IP_PROT_251, /* 251 */
        Q_PROTO_UNASSIGNED_IP_PROT_252, /* 252 */
        Q_PROTO_IP_EXP_1,       /* 253 */
        Q_PROTO_IP_EXP_2,       /* 254 */
        Q_PROTO_IPSEC
    };         /* 255 */

    return protocol_mapping[protocol_id];
}

#pragma pack(1)
struct qm_gre_hdr {
    uint16_t flags;
    uint16_t protocol;
};
#pragma pack()


#pragma pack(1)
struct qm_icmp_hdr {
    uint8_t   type;
    uint8_t   code;
    uint16_t  checksum;
    union {
        struct {
            uint16_t  id;
            uint16_t  sequence;
        } echo;
        uint32_t  gateway;
        struct {
            uint16_t  unused;
            uint16_t  mtu;
        } frag;
    } un;
};
#pragma pack()


#pragma pack(1)
struct qm_ip6_addr {
    uint8_t addr[16];
};
#pragma pack()

#pragma pack(1)
struct qm_ip6_hdr {
    uint32_t   vcf;
    uint16_t   payload_len;
    uint8_t    nexthdr;
    uint8_t    hop_limit;
    struct qm_ip6_addr saddr;
    struct qm_ip6_addr daddr;
};
#pragma pack()

#pragma pack(1)
struct qm_tcp_hdr {
    uint16_t    source;
    uint16_t    dest;
    uint32_t    seq;
    uint32_t    ack_seq;
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
    uint16_t
    res1: 4,
          doff: 4,
          fin: 1,
          syn: 1,
          rst: 1,
          psh: 1,
          ack: 1,
          urg: 1,
          ece: 1,
          cwr: 1;
#else
#if __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
    uint16_t
    doff: 4,
          res1: 4,
          cwr: 1,
          ece: 1,
          urg: 1,
          ack: 1,
          psh: 1,
          rst: 1,
          syn: 1,
          fin: 1;
#else
#error  "Please fix LITTLE/BIG_ENDIAN define"
#endif
#endif
    uint16_t    window;
    uint16_t    check;
    uint16_t    urg_ptr;
};
#pragma pack()

#pragma pack(1)
struct qm_udp_hdr {
    uint16_t    source;
    uint16_t    dest;
    uint16_t    len;
    uint16_t    check;
};
#pragma pack()

#pragma pack(1)
struct qm_netbios_hdr {
    uint16_t     len;
    uint16_t     delimeter;
    uint8_t      cmd;
    uint8_t      reserved;
};
#pragma pack()

/* SCTP chunk types */
enum {
    SCTP_TYPE_DATA = 0x00,
    SCTP_TYPE_INIT = 0x01,
    SCTP_TYPE_INIT_ACK = 0x02,
    SCTP_TYPE_SACK = 0x03,
    SCTP_TYPE_HEARTBEAT = 0x04,
    SCTP_TYPE_HEARTBEAT_ACK = 0x05,
    SCTP_TYPE_ABORT = 0x06,
    SCTP_TYPE_SHUTDOWN = 0x07,
    SCTP_TYPE_SHUTDOWN_ACK = 0x08,
    SCTP_TYPE_ERROR = 0x09,
    SCTP_TYPE_COOKIE_ECHO = 0x0a,
    SCTP_TYPE_COOKIE_ACK = 0x0b,
    SCTP_TYPE_SHUTDOWN_COMPLETE = 0x0e,
    SCTP_TYPE_ASCONF_ACK = 0x80,
    SCTP_TYPE_ASCONF = 0xc1,
};


/* The common header before all chunks */
#pragma pack(1)
struct qm_sctp_hdr {
    uint16_t    source;
    uint16_t    dest;
    uint32_t    verif_tag;
    uint32_t    checksum;
};
#pragma pack()

/* The common header in every chunk. */
#pragma pack(1)
struct qm_sctp_chunk_hdr {
    uint8_t    type;
    uint8_t    flags;
    uint16_t   len;
};
#pragma pack()

/* The header for a data chunk. */
#pragma pack(1)
struct qm_sctp_chunk_data {
    uint32_t   tsn;
    uint16_t   stream_id;
    uint16_t   seq_nbr;
    uint32_t   proto_id;
};
#pragma pack()

/* The header for a sack chunk.
 * This chunk is sent to the peer endpoint to acknowledge
 * received DATA chunks and to inform the peer endpoint of
 * gaps in the received subsequences of DATA chunks as
 * represented by their TSNs. */
#pragma pack(1)
struct qm_sctp_chunk_sack {
    uint32_t    cum_tsn_ack;
    uint32_t    a_rwnd;
    uint16_t    nb_ack_blocks;
    uint16_t    nb_dup_tsns;
};
#pragma pack()


/* The header for a init chunk.
 * This chunk is used to initiate a SCTP
 * association between two endpoints. */
#pragma pack(1)
struct qm_sctp_chunk_init {
    uint32_t    init_tag;
    uint32_t    a_rwnd;
    uint16_t    out_streams;
    uint16_t    in_streams;
    uint32_t        initial_tsn;
};
#pragma pack()


/* The header for a init chunk.
 * The INIT ACK chunk is used to acknowledge
 * the initiation of an SCTP association. */
#pragma pack(1)
struct qm_sctp_chunk_init_ack {
    uint32_t    init_tag;
    uint32_t    a_rwnd;
    uint16_t    out_streams;
    uint16_t    in_streams;
    uint32_t        initial_tsn;
};
#pragma pack()

#pragma pack(1)
struct qm_sctp_chunk_hbrt {
    uint16_t param_type;
    uint16_t param_len;
};
#pragma pack()

#pragma pack(1)
struct qm_sctp_chunk_hbrt_ack {
    uint16_t param_type;
    uint16_t param_len;
};
#pragma pack()

#pragma pack(1)
struct qm_sctp_chunk_abort {
    uint16_t param_type;
    uint16_t param_len;
};
#pragma pack()

/**
 * GTP common header to all versions
 */
#pragma pack(1)
struct qm_gtp_globalhdr {
    /**
     * This holds flags general protocol Flags such as gtp version etc ...
     */
    uint8_t   flags;
    /**
     * This holds the type of GTP message
     */
    uint8_t   msgtype;
    /**
     * This holds GTP payload length with optional headers included
     */
    uint16_t  len;
};
#pragma pack()

/* Returns the GTP version from GTP flags */
#define GTP_VER(flags) ((flags) >> 5)
#define GTP_V0 0
#define GTPFLAG_V0 (GTP_V0 << 5)
#define GTP_V1 1
#define GTPFLAG_V1 (GTP_V1 << 5)
#define GTP_V2 2
#define GTPFLAG_V2 (GTP_V2 << 5)

/**
 * GTPv0 Common header
 */
#pragma pack(1)
struct qm_gtp_v0hdr {
    uint8_t flags;
    uint8_t msgtype;
    uint16_t    len;
    uint16_t    seqnum;
    uint16_t    flowlbl;
    uint8_t npdu_llc_number;
    uint8_t npdu_llc_number_suite[3];
    uint8_t tid[8];
};
#pragma pack()

#define GTPv0_FLAG_VERSION    0xE0
#define GTPv0_FLAG_TYPE       0x10
#define GTPv0_FLAG_RSV        0x0E


/* GTPv0 message types */
#define GTPv0_MSG_UNKNOWN               0x00
#define GTPv0_MSG_ECHO_REQ              0x01
#define GTPv0_MSG_ECHO_RESP             0x02
#define GTPv0_MSG_VER_NOT_SUPP          0x03
#define GTPv0_MSG_NODE_ALIVE_REQ        0x04
#define GTPv0_MSG_NODE_ALIVE_RESP       0x05
#define GTPv0_MSG_REDIR_REQ             0x06
#define GTPv0_MSG_REDIR_RESP            0x07
#define GTPv0_MSG_CREATE_PDP_REQ        0x10
#define GTPv0_MSG_CREATE_PDP_RESP       0x11
#define GTPv0_MSG_UPDATE_PDP_REQ        0x12
#define GTPv0_MSG_UPDATE_PDP_RESP       0x13
#define GTPv0_MSG_DELETE_PDP_REQ        0x14
#define GTPv0_MSG_DELETE_PDP_RESP       0x15
#define GTPv0_MSG_CREATE_AA_PDP_REQ     0x16
#define GTPv0_MSG_CREATE_AA_PDP_RESP    0x17
#define GTPv0_MSG_DELETE_AA_PDP_REQ     0x18
#define GTPv0_MSG_DELETE_AA_PDP_RESP    0x19
#define GTPv0_MSG_ERR_IND               0x1A
#define GTPv0_MSG_PDU_NOTIFY_REQ        0x1B
#define GTPv0_MSG_PDU_NOTIFY_RESP       0x1C
#define GTPv0_MSG_PDU_NOTIFY_REJ_REQ    0x1D
#define GTPv0_MSG_PDU_NOTIFY_REJ_RESP   0x1E
#define GTPv0_MSG_SUPP_EXT_HDR          0x1F
#define GTPv0_MSG_SEND_ROUT_INFO_REQ    0x20
#define GTPv0_MSG_SEND_ROUT_INFO_RESP   0x21
#define GTPv0_MSG_FAIL_REP_REQ          0x22
#define GTPv0_MSG_FAIL_REP_RESP         0x23
#define GTPv0_MSG_MS_PRESENT_REQ        0x24
#define GTPv0_MSG_MS_PRESENT_RESP       0x25
#define GTPv0_MSG_IDENT_REQ             0x30

/**
 * GTPv1 Common header
 */
#pragma pack(1)
struct qm_gtp_v1hdr {
    /**
     * This holds flags general protocol flags such as sequence number presence
     * ...
     */
    uint8_t   flags;
    /**
     * This holds the type of GTP message
     */
    uint8_t   msgtype;
    /**
     * This holds GTP payload length with optional headers included
     */
    uint16_t  len;
    /**
     * This holds an identifiant used to multiplex different connections in the
     * same tunnel
     */
    uint32_t  teid;
};
#pragma pack()


/**
 * GTPv1 Header with optional sequence number field
 */
#pragma pack(1)
struct qm_gtp_v1hdr_seq {
    /**
     * This holds flags general protocol flags such as sequence number presence
     * ...
     */
    uint8_t   flags;
    /**
     * This holds the type of GTP message
     */
    uint8_t   msgtype;
    /**
     * This holds GTP payload length with optional headers included
     */
    uint16_t  len;
    /**
     * This holds an identifiant used to multiplex different connection in the
     * same tunnel
     */
    uint32_t  teid;
    /**
     * This holds the sequence number
     */
    uint16_t  seqnum;
};
#pragma pack()


#define GTPv1_FLAG_VERSION                  0xE0
#define GTPv1_FLAG_TYPE                     0x10
#define GTPv1_FLAG_RSV                      0x08
#define GTPv1_FLAG_E                        0x04
#define GTPv1_FLAG_S                        0x02
#define GTPv1_FLAG_PN                       0x01

/* GTPv1 message types */
#define GTPv1_MSG_UNKNOWN                   0x00
#define GTPv1_MSG_ECHO_REQ                  0x01
#define GTPv1_MSG_ECHO_RESP                 0x02
#define GTPv1_MSG_VER_NOT_SUPP              0x03
#define GTPv1_MSG_NODE_ALIVE_REQ            0x04
#define GTPv1_MSG_NODE_ALIVE_RESP           0x05
#define GTPv1_MSG_REDIR_REQ                 0x06
#define GTPv1_MSG_REDIR_RESP                0x07
#define GTPv1_MSG_CREATE_PDP_REQ            0x10
#define GTPv1_MSG_CREATE_PDP_RESP           0x11
#define GTPv1_MSG_UPDATE_PDP_REQ            0x12
#define GTPv1_MSG_UPDATE_PDP_RESP           0x13
#define GTPv1_MSG_DELETE_PDP_REQ            0x14
#define GTPv1_MSG_DELETE_PDP_RESP           0x15
#define GTPv1_MSG_CREATE_AA_PDP_REQ         0x16
#define GTPv1_MSG_CREATE_AA_PDP_RESP        0x17
#define GTPv1_MSG_ERR_IND                   0x1A
#define GTPv1_MSG_PDU_NOTIFY_REQ            0x1B
#define GTPv1_MSG_PDU_NOTIFY_RESP           0x1C
#define GTPv1_MSG_PDU_NOTIFY_REJ_REQ        0x1D
#define GTPv1_MSG_PDU_NOTIFY_REJ_RESP       0x1E
#define GTPv1_MSG_SUPP_EXT_HDR              0x1F
#define GTPv1_MSG_SEND_ROUT_INFO_REQ        0x20
#define GTPv1_MSG_SEND_ROUT_INFO_RESP       0x21
#define GTPv1_MSG_FAIL_REP_REQ              0x22
#define GTPv1_MSG_FAIL_REP_RESP             0x23
#define GTPv1_MSG_MS_PRESENT_REQ            0x24
#define GTPv1_MSG_MS_PRESENT_RESP           0x25
#define GTPv1_MSG_IDENT_REQ                 0x30
#define GTPv1_MSG_IDENT_RESP                0x31
#define GTPv1_MSG_SGSN_CNTXT_REQ            0x32
#define GTPv1_MSG_SGSN_CNTXT_RESP           0x33
#define GTPv1_MSG_SGSN_CNTXT_ACK            0x34
#define GTPv1_MSG_FORW_RELOC_REQ            0x35
#define GTPv1_MSG_FORW_RELOC_RESP           0x36
#define GTPv1_MSG_FORW_RELOC_COMP           0x37
#define GTPv1_MSG_RELOC_CANCEL_REQ          0x38
#define GTPv1_MSG_RELOC_CANCEL_RESP         0x39
#define GTPv1_MSG_FORW_SRNS_CNTXT           0x3A
#define GTPv1_MSG_FORW_RELOC_ACK            0x3B
#define GTPv1_MSG_FORW_SRNS_CNTXT_ACK       0x3C
#define GTPv1_MSG_RAN_RELAY                 0x46
#define GTPv1_MSG_MBMS_NOTIF_REQ            0x60
#define GTPv1_MSG_MBMS_NOTIF_RESP           0x61
#define GTPv1_MSG_MBMS_REJECT_REQ           0x62
#define GTPv1_MSG_MBMS_REJECT_RESP          0x63
#define GTPv1_MSG_MBMS_CNTXT_CREATE_REQ     0x64
#define GTPv1_MSG_MBMS_CNTXT_CREATE_RESP    0x65
#define GTPv1_MSG_MBMS_CNTXT_UPDATE_REQ     0x66
#define GTPv1_MSG_MBMS_CNTXT_UPDATE_RESP    0x67
#define GTPv1_MSG_MBMS_CNTXT_DELETE_REQ     0x68
#define GTPv1_MSG_MBMS_CNTXT_DELETE_RESP    0x69
#define GTPv1_MSG_MBMS_REGISTER_REQ         0x70
#define GTPv1_MSG_MBMS_REGISTER_RESP        0x71
#define GTPv1_MSG_MBMS_UNREGISTER_REQ       0x72
#define GTPv1_MSG_MBMS_UNREGISTER_RESP      0x73
#define GTPv1_MSG_MBMS_SESSION_START_REQ    0x74
#define GTPv1_MSG_MBMS_SESSION_START_RESP   0x75
#define GTPv1_MSG_MBMS_SESSION_STOP_REQ     0x76
#define GTPv1_MSG_MBMS_SESSION_STOP_RESP    0x77
#define GTPv1_MSG_MBMS_SESSION_UPDATE_REQ   0x78
#define GTPv1_MSG_MBMS_SESSION_UPDATE_RESP  0x79
#define GTPv1_MSG_MS_CHANGE_REQ             0x80
#define GTPv1_MSG_MS_CHANGE_RESP            0x81
#define GTPv1_MSG_DATA_TRANSF_REQ           0xF0
#define GTPv1_MSG_DATA_TRANSF_RESP          0xF1
#define GTPv1_MSG_MS_CHANGE_RESP2           0xFE
#define GTPv1_MSG_TPDU                      0xFF

/**
 * GTPv2 Header with optional TEID
 */
#pragma pack(1)
struct qm_gtp_v2hdr {
    /**
     * This holds flags general protocol flags such as TEID presence ...
     */
    uint8_t   flags;
    /**
     * This holds the type of GTP message
     */
    uint8_t   msgtype;
    /**
     * This holds GTP payload length with optional headers included
     */
    uint16_t  len;
    /**
     * This holds an identifiant used to multiplex different connection in the
     * same tunnel
     */
    uint32_t  teid;
    /**
     * This holds the sequence number
     */
    uint32_t  seqnum : 24;

    uint32_t  spare : 8;
};
#pragma pack()


/**
 * GTPv2 Header without optional TEID
 */
#pragma pack(1)
struct qm_gtp_v2hdr_noteid {
    /**
     * This holds flags general protocol flags such as teid presence ...
     */
    uint8_t   flags;
    /**
     * This holds the type of GTP message
     */
    uint8_t   msgtype;
    /**
     * This holds GTP payload length with optional headers included
     */
    uint16_t  len;
    /**
     * This holds the sequence number
     */
    uint32_t  seqnum : 24;

    uint32_t  spare : 8;
};
#pragma pack()

#define GTPv2_FLAG_PIGGY    0x10
#define GTPv2_FLAG_T        0x08
#define GTPv2_FLAG_SPARE    0x03

/* GTPv2 MSG IDs */
#define GTPV2_MSG_ECHO_REQ                          0x01
#define GTPV2_MSG_ECHO_RESP                     0x02
#define GTPV2_MSG_VER_NOT_SUPP                      0x03
#define GTPV2_MSG_CREATE_SESS_REQ                   0x20
#define GTPV2_MSG_CREATE_SESS_RESP                  0x21
#define GTPV2_MSG_MODIFY_BEARER_REQ                 0x22
#define GTPV2_MSG_MODIFY_BEARER_RESP                0x23
#define GTPV2_MSG_DELETE_SESS_REQ                   0x24
#define GTPV2_MSG_DELETE_SESS_RESP                  0x25
#define GTPV2_MSG_CHANGE_NOTIF_REQ                  0x26
#define GTPV2_MSG_CHANGE_NOTIF_RESP                 0x27
#define GTPV2_MSG_MODIFY_BEARER_COMM                0x40
#define GTPV2_MSG_MODIFY_BEARER_FAIL_IND            0x41
#define GTPV2_MSG_DELETE_BEARER_COMM                0x42
#define GTPV2_MSG_DELETE_BEARER_FAIL_IND            0x43
#define GTPV2_MSG_BEARER_RESS_COMM                  0x44
#define GTPV2_MSG_BEARER_FAIL_IND                   0x45
#define GTPV2_MSG_DOWNLINK_DATA_NOTIF_FAIL_IND      0x46
#define GTPV2_MSG_TRACE_SESS_ACTIV                  0x47
#define GTPV2_MSG_TRACE_SESS_DEACT                  0x48
#define GTPV2_MSG_STOP_PAGING_IND                   0x49
#define GTPV2_MSG_CREATE_BEARER_REQ                 0x5F
#define GTPV2_MSG_CREATE_BEARER_RESP                0x60
#define GTPV2_MSG_UPDATE_BEARER_REQ                 0x61
#define GTPV2_MSG_UPDATE_BEARER_RESP                0x62
#define GTPV2_MSG_DELETE_BEARER_REQ                 0x63
#define GTPV2_MSG_DELETE_BEARER_RESP                0x64
#define GTPV2_MSG_DELETE_PDN_CONN_SET_REQ           0x65
#define GTPV2_MSG_DELETE_PDN_CONN_SET_RESP          0x66
#define GTPV2_MSG_PGW_DOWNLINK_TRIG_NOTIF           0x67
#define GTPV2_MSG_PGW_DOWNLINK_TRIG_ACK             0x68
#define GTPV2_MSG_IDENTIFICATION_REQ                0x80
#define GTPV2_MSG_IDENTIFICATION_RESP               0x81
#define GTPV2_MSG_CNTXT_REQ                         0x82
#define GTPV2_MSG_CNTXT_RESP                        0x83
#define GTPV2_MSG_CNTXT_ACK                         0x84
#define GTPV2_MSG_FORW_RELOC_REQ                    0x85
#define GTPV2_MSG_FORW_RELOC_RESP                   0x86
#define GTPV2_MSG_FORW_RELOC_COMPL_NOTIF            0x87
#define GTPV2_MSG_FORW_RELOC_COMPL_ACK              0x88
#define GTPV2_MSG_FORW_ACCESS_CNTXT_NOTIF           0x89
#define GTPV2_MSG_FORW_ACCESS_CNTXT_ACK             0x8A
#define GTPV2_MSG_RELOC_CANCEL_REQ                  0x8B
#define GTPV2_MSG_RELOC_CANCEL_RESP                 0x8C
#define GTPV2_MSG_CONFIG_TRANS_TUNNEL               0x8D
#define GTPV2_MSG_DETACH_NOTIF                      0x95
#define GTPV2_MSG_DETACH_ACK                        0x96
#define GTPV2_MSG_CS_PAGING_IND                     0x97
#define GTPV2_MSG_RAN_INFO_DELAY                    0x98
#define GTPV2_MSG_ALERT_MME_NOTIF                   0x99
#define GTPV2_MSG_ALERT_MME_ACK                     0x9A
#define GTPV2_MSG_UE_ACTIVITY_NOTIF                 0x9B
#define GTPV2_MSG_UE_ACTIVITY_ACK                   0x9C
#define GTPV2_MSG_CREATE_FORW_TUNNEL_REQ            0xA0
#define GTPV2_MSG_CREATE_FORW_TUNNEL_RESP           0xA1
#define GTPV2_MSG_SUSPEND_NOTIF                     0xA2
#define GTPV2_MSG_SUSPEND_ACK                       0xA3
#define GTPV2_MSG_RESUME_NOTIF                      0xA4
#define GTPV2_MSG_RESUME_ACK                        0xA5
#define GTPV2_MSG_CREATE_IND_DATA_FORW_TUNNEL_REQ   0xA6
#define GTPV2_MSG_CREATE_IND_DATA_FORW_TUNNEL_RESP  0xA7
#define GTPV2_MSG_DELETE_IND_DATA_FORW_TUNNEL_REQ   0xA8
#define GTPV2_MSG_DELETE_IND_DATA_FORW_TUNNEL_RESP  0xA9
#define GTPV2_MSG_RELEASE_ACCESS_BEARERS_REQ        0xAA
#define GTPV2_MSG_RELEASE_ACCESS_BEARERS_RESP       0xAB
#define GTPV2_MSG_DOWNLINK_DATA_NOTIF               0xB0
#define GTPV2_MSG_DOWNLINK_DATA_ACK                 0xB1
#define GTPV2_MSG_RES_ALLOCATED_EARLIER             0xB2
#define GTPV2_MSG_PGW_RESTART_NOTIF                 0xB3
#define GTPV2_MSG_PGW_RESTART_NOTIF_ACK             0xB4
#define GTPV2_MSG_UPDATE_PDN_CONN_SET_REQ           0xC8
#define GTPV2_MSG_UPDATE_PDN_CONN_SET_RESP          0xC9
#define GTPV2_MSG_MODIFY_ACCESS_BEARERS_REQ         0xD3
#define GTPV2_MSG_MODIFY_ACCESS_BEARERS_RESP        0xD4
#define GTPV2_MSG_MBMS_SESSION_START_REQ            0xE7
#define GTPV2_MSG_MBMS_SESSION_START_RESP           0xE8
#define GTPV2_MSG_MBMS_SESSION_UPDATE_REQ           0xE9
#define GTPV2_MSG_MBMS_SESSION_UPDATE_RESP          0xEA
#define GTPV2_MSG_MBMS_SESSION_STOP_REQ             0xEB
#define GTPV2_MSG_MBMS_SESSION_STOP_RESP            0xEC



#endif /* __PACKET_HELPER_H__ */
