/*
 *   This file is a part of Qosmos ixEngine.
 *   Copyright  Qosmos 2000-2016 - All rights reserved
 */

/* standard libc headers */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

/* Qosmos ixEngine header */
#include <qmdpi.h>
/* Bundle API */
#include <qmdpi_bundle_api.h>

/* Qosmos ixEngine main objects */
struct qmdpi_engine *engine = NULL;
struct qmdpi_bundle *bundle = NULL;

#define PROG_NAME "license_info"

enum perm_level {
    perm_level_first = 0,
    perm_level_classification = perm_level_first,
    perm_level_metadata,
    perm_level_content,
    perm_level_last = perm_level_content,
    perm_level_num = perm_level_last - perm_level_first + 1,
};

const char tab_str_perm_level[perm_level_num][32] = {
    "Classification",
    "Metadata",
    "Content",
};

const int tab_bit_perm_level[perm_level_num] = {
    0x01,
    0x02,
    0x04,
};

static int dump_signature_list = 0;
static int dump_attribute_list = 0;
static int dump_non_generic_attribute_list = 0;
static int perm_level = 0;
static int license_loaded = 0;
static int quiet = 0;
static int skip_content = 0;
static int skip_metadata = 0;
static int skip_sig = 0;

static int load_license(const char *filename, const char *serial)
{
    char *bytes = NULL;
    int fd, nbytes, r;
    struct stat f_stats;

    nbytes = 0;

#if defined(OS_WINDOWS)
    fd = open(filename, O_RDONLY | O_BINARY);
#else
    fd = open(filename, O_RDONLY);
#endif

    if (fd < 0) {
        fprintf(stderr, "can not open %s\n", filename);
        return -1;
    }
    if (fstat(fd, &f_stats)) {
        fprintf(stderr, "could not fstat %s\n", filename);
        close(fd);
        return -1;
    }
    nbytes = (int) f_stats.st_size;
    bytes = (char *)malloc(nbytes);

    if (NULL == bytes) {
        close(fd);
        return -1;
    }
    if ((read(fd, bytes, f_stats.st_size)) < f_stats.st_size) {
        fprintf(stderr, "could not read %s\n", filename);
        close(fd);
        free(bytes);
        return -1;
    }
    close(fd);
    r = qmdpi_license_load_from_addr(bytes, nbytes, serial);
    free(bytes);
    return r;
}

static int attr_callback(struct qmdpi_bundle *bundle,
                         char const *signature_name,
                         struct qmdpi_attr const *attr, void *arg)
{
    int is_content, is_metadata, ret;
    char const *attr_name = qmdpi_attr_name_get(attr);

    (void)arg;
    if (1 == dump_attribute_list) {
        if ((1 == dump_non_generic_attribute_list)
                && (0 <= qmdpi_attr_id_get(attr))) {
            fprintf(stdout, "   %s:%s\n", signature_name, attr_name);
        } else if (0 == dump_non_generic_attribute_list) {
            fprintf(stdout, "   %s:%s\n", signature_name, attr_name);
        }
    }
    /* get if attribute is content */
    is_metadata = 0;
    is_content = qmdpi_attr_is_content(attr);

    int attr_id = qmdpi_attr_id_get(attr);
    struct qmdpi_signature *signature = qmdpi_bundle_signature_get_byname(bundle,
                                                                          signature_name);
    int sig_id = qmdpi_signature_id_get(signature);

    if (!is_content || ((0 < attr_id) &&
                        (Q_PROTO_BASE != sig_id) &&
                        (Q_PROTO_UNKNOWN != sig_id) &&
                        (Q_PROTO_MALFORMED != sig_id) &&
                        (Q_PROTO_INCOMPLETE != sig_id) &&
                        ((Q_PROTO_HTTP != sig_id) ||
                         (Q_HTTP_HEADER_END != attr_id)))) {
        is_metadata = 1;
    }

    if ((is_content && skip_content) || (is_metadata && skip_metadata)) {
        return 0;
    }

    /* try to enable hook on attribute */
    ret = qmdpi_bundle_attr_register(bundle, signature_name, attr_name);
    if (quiet && ret == QMDPI_EPERM) {
        if (is_content) {
            skip_content = 1;
        } else if (is_metadata) {
            skip_metadata = 1;
        }
    }

    if (is_content && ret == 0)
        perm_level |=
            tab_bit_perm_level[perm_level_content];
    if (is_metadata && ret == 0)
        perm_level |=
            tab_bit_perm_level[perm_level_metadata];

    return 0;
}

static int signature_callback(struct qmdpi_bundle *bundle,
                              struct qmdpi_signature *signature,
                              void *arg)
{
    char const *signature_name = qmdpi_signature_name_get(signature);

    (void)arg;
    if (dump_signature_list) {
        fprintf(stdout, "  %s (%s) - [%s]\n", signature_name,
                qmdpi_signature_longname_get(signature),
                qmdpi_signature_family_get(signature)
               );
    }
    qmdpi_bundle_attr_foreach_permitted(bundle, signature_name, attr_callback,
                                        NULL);

    return 0;
}

static int sig_enable(struct qmdpi_bundle *bundle, struct qmdpi_signature *sig,
                      void *arg)
{
    int ret = 0;
    const char *signame;

    (void)arg;
    signame = qmdpi_signature_name_get(sig);

    if (skip_sig) {
        return 0;
    }

    ret = qmdpi_bundle_signature_enable(bundle, signame);
    if (ret == 0) {
        perm_level |=
            tab_bit_perm_level[perm_level_classification];
    } else if ((ret == QMDPI_EPERM) && quiet) {
        /* skip remaining signatures */
        skip_sig = 1;
    }
    return 0;
}


/* ixEngine init */
static int engine_init(const char *filename, const char *serial)
{
    int ret = 0;

    /* if provided, load the dynamic license */
    if (strlen(serial) && strlen(filename)) {
        printf("License info was given in command line, trying to load this license...\n");
        ret = load_license(filename, serial);
        if (ret < 0) {
            printf("Could not load provided license.\n");
            return -1;
        }
    }

    /* create engine instance */
    engine = qmdpi_engine_create("injection_mode=packet;nb_workers=1;nb_flows=10");
    if (engine == NULL) {
        printf("cannot create engine instance: %s\n", qmdpi_error_get_string(NULL,
                                                                             qmdpi_error_get()));
        return -1;
    }

    license_loaded = qmdpi_license_is_loaded();
    if (license_loaded) {
        printf("License loaded successfully.\n");
    } else {
        printf("Could not load license.\n");
        return 0;
    }

    /* create bundle instance */
    bundle = qmdpi_bundle_create_from_file(engine, NULL);
    if (bundle == NULL) {
        printf("cannot create bundle instance: %s\n", qmdpi_error_get_string(NULL,
                                                                             qmdpi_error_get()));
        goto error_engine;
    }

    /* activate bundle */
    ret = qmdpi_bundle_activate(bundle);
    if (ret < 0) {
        printf("cannot activate bundle: %s\n", qmdpi_error_get_string(bundle, ret));
        goto error_bundle;

    }

    /* enable all permitted signatures on bundle */
    qmdpi_bundle_signature_foreach_permitted(bundle, sig_enable, NULL);

    return 0;

error_bundle:
    qmdpi_bundle_destroy(bundle);
error_engine:
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();
    return -1;
}

int main(int argc, char **argv)
{
    int i, ret;
    char serial[256];
    char filename[256];

    int args_error = 0;
    int tdm_is_enabled = 0;

    memset(serial, 0, 256);
    memset(filename, 0, 256);

    for (i = 1; i < argc; i++) {
        if (0 == strcmp("--list-signatures", argv[i])) {
            dump_signature_list = 1;
        } else if (0 == strcmp("--list-attributes", argv[i])) {
            dump_signature_list = 1;
            dump_attribute_list = 1;
        } else if (0 == strcmp("--list-non-generic-attributes", argv[i])) {
            dump_signature_list = 1;
            dump_attribute_list = 1;
            dump_non_generic_attribute_list = 1;
        } else if (0 == strcmp("--serial-number", argv[i])) {
            if (argc > i + 1) {
                strncpy(serial, argv[i + 1], 255);
                i++;
            } else {
                args_error = 1;
            }
        } else if (0 == strcmp("--license-filename", argv[i])) {
            if (argc > i + 1) {
                strncpy(filename, argv[i + 1], 255);
                i++;
            } else {
                args_error = 1;
            }
        } else if (0 == strcmp("-q", argv[i])) {
            quiet = 1;
        } else {
            args_error = 1;
        }
    }

    if (args_error) {
        printf("Usage:\n\t%s [option]\n", argv[0]);
        printf("options:\n");
        printf
        ("\t-q : quiet mode: abort on first permission error\n");
        printf
        ("\t--list-signatures : display every enabled signatures in the license\n");
        printf
        ("\t--list-attributes : display every enabled attributes & signatures in the license\n");
        printf
        ("\t--list-non-generic-attributes : display every enabled non-generic attributes & signatures in the license\n");
        printf
        ("\t--serial-number serial_number : set the license serial number for dynamic license loading. Also requires --license-filename.\n");
        printf
        ("\t--license-filename filename : provide the path to access the file containing the license.\n");
        printf("examples:\n");
        printf("\t%s\n", argv[0]);
        printf("\t%s --list-signatures\n", argv[0]);
        printf("\t%s --list-attributes\n", argv[0]);
        printf("\t%s --list-non-generic-attributes\n", argv[0]);
        printf("\t%s --serial-number MY_SERIAL --license-filename license_file\n",
               argv[0]);
        exit(1);
    }


    /* init dpi engine */
    ret = engine_init(filename, serial);
    if (ret < 0) {
        return 1;
    }

    if (!license_loaded) {
        goto err0;
    }

    printf("**********************************************\n");
    printf("*            ixEngine info                   *\n");
    printf("**********************************************\n");

    /* get DPI engine version */
    printf("Engine Version: %s\n", qmdpi_engine_version_get_string(engine));
    printf("Protocol Bundle Version: %s\n",
           qmdpi_bundle_version_get_string(bundle));

    printf("\n*** License info ***\n");
    qmdpi_license_dump(stdout, (qmdpi_output_fn_t)fprintf);
    printf("\n");

    /* display list of enabled signatures */
    if (dump_signature_list) {
        printf("Signature list:\n");
    }
    ret = qmdpi_bundle_signature_foreach_permitted(bundle, signature_callback,
                                                   NULL);
    if (ret < 0) {
        fprintf(stderr, "%s\n", qmdpi_error_get_string(bundle, ret));
        goto err0;
    }

    /* display permission level */
    printf("*** Permission level ***\n");
    for (i = 0; i < perm_level_num; i++) {
        printf("%-20s", tab_str_perm_level[i]);
        if (0 != (perm_level & tab_bit_perm_level[i])) {
            printf(" [X]");
        } else {
            printf(" [ ]");
        }
        printf("\n");
    }
    printf("\n");

    printf("*** Enabled modules ***\n");
    printf("%-40s", "Custom Signature Module");
    if (qmdpi_bundle_pdb_compile_and_activate(bundle, NULL, 0, NULL,
                                              0) != QMDPI_EPERM) {
        printf(" [X]");
    } else {
        printf(" [ ]");
    }
    printf("\n");

    printf("%-40s", "Traffic Distribution Module");
    if (qmdpi_report_init(engine, NULL) != QMDPI_EPERM) {
        printf(" [X]");
        tdm_is_enabled = 1;
    } else {
        printf(" [ ]");
    }
    printf("\n");

    printf("%-40s", "Classification Extension");
    if (tdm_is_enabled) {
        if (qmdpi_flow_classif_ext_get(NULL, NULL, 0) != QMDPI_EPERM) {
            printf(" [X]");
        } else {
            printf(" [ ]");
        }
        printf("\n");
    } else {
        /* should never happen since TDM does not require any license activation from now on */
        printf(" [ ] (TDM is required for this feature !)\n");
    }

    printf("%-40s", "Transactional DPI");
    if (qmdpi_bundle_attr_register(bundle, "box_net", "transaction") == 0) {
        printf(" [X]");
    } else {
        printf(" [ ]");
    }
    printf("\n");

    ret = 0;

    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();

    return ret;

err0:
    ret = 1;
    qmdpi_bundle_destroy(bundle);
    qmdpi_engine_destroy(engine);
    qmdpi_license_destroy();

    return ret;
}
