#!/usr/bin/env perl

use strict;
use File::Copy "cp";
use File::Basename;
use Getopt::Std;

# getopts will consume any argument starting with -
my %options=();
getopts("f", \%options);

my $arg_num = @ARGV;
print("usage: $0 [-f] <license file> <binary file> [serial]\n"), exit if ($arg_num != 2 && $arg_num != 3);

my $license_filename = $ARGV[0];
my $binary_filename = $ARGV[1];
my $serial;

if (defined($ARGV[2])) {
    $serial = $ARGV[2];
} else {
    $serial = basename($license_filename);
    $serial =~ s/^(.*)-[0-9]{8}\.bin/$1/;
}

print "serial: $serial\n";

# Serial and license section patterns
my $serial_prefix = "<magic!section>.qm.licence.uuid\0";
my $license_prefix = "<magic!section>.qm.licence\0";

$serial = $serial_prefix . $serial . "\0";

# Read license file
my $license_content = $license_prefix;
open(my $l_fh, "<$license_filename") || die "cannot open $license_filename: $!";
binmode $l_fh;

my $buffer;
while(read($l_fh, $buffer, 64 * 1024)) {
  $license_content .= $buffer;
}
close($l_fh);
$license_content .= "\0";

# Read binary file
my $binary_content = "";
open(my $in_fh, "<$binary_filename") || die "cannot open $binary_filename: $!";
binmode $in_fh;
while(read($in_fh, $buffer, 64 * 1024)) {
  $binary_content .= $buffer;
}
close($in_fh);

# Detect serial and license lengths in binary file
my $serial_orig_length = 0;
my $license_orig_length = 0;

if ($binary_content =~ /($serial_prefix\0*)/) {
  $serial_orig_length = length($1);
}
if ($binary_content =~ /($license_prefix\0*)/) {
  $license_orig_length = length($1);
}

my $license_length = length($license_content);
my $serial_length = length($serial);

# Check license and serial lengths
if ($serial_orig_length == 0 || $license_orig_length == 0) {
  die "$binary_filename is not a valid binary file.";
}
if($license_orig_length == length($license_prefix) || $serial_orig_length == length($serial_prefix)) {
  if (not defined $options{f}) {
    die "$binary_filename already patched. Use -f to force.";
  }
}

if($license_length > $license_orig_length && not defined $options{f}) {
  die "License content has the wrong length: $license_length bytes";
}
if($serial_length > $serial_orig_length && not defined $options{f}) {
  die "Serial too long: $serial_length bytes";
}

cp($binary_filename, $binary_filename.".bak");

# Patch binary
my $serial_prefix_offset = index($binary_content, $serial_prefix);
$binary_content = substr($binary_content, 0, $serial_prefix_offset).$serial.substr($binary_content, $serial_prefix_offset + length($serial));

my $license_prefix_offset = index($binary_content, $license_prefix);
$binary_content = substr($binary_content, 0, $license_prefix_offset).$license_content.substr($binary_content, $license_prefix_offset + $license_length);

print("Opening `$binary_filename' for writing.\n");
open(my $out_fh, ">$binary_filename") || die "cannot open $binary_filename for writing: $!";
binmode $out_fh;
print $out_fh $binary_content;
close($out_fh);

chmod(0755, $binary_filename);
print ("Done.\n");
