/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2016 - All rights reserved
*/
#include <stdio.h>
#include <stdlib.h>
#include <qmdpi.h>
#include <string.h>

#include "hooks.h"
#include "pcap_analyzer_getter.h"

int hook_base_path_end = 0;
int hook_base_final_path = 0;
int hook_base_classified = 0;
int hook_base_session_end = 0;

/**
 * Function that checks if a caracter is a procotol/attribute wildchar (./?)
 *
 * @param c : the character to test
 * @return 1 if c if a wildchar, 0 otherwise
 */
static inline int pa_wildchar(char c)
{
    return c == '.' || c == '?';
}


/* register a callback for an attribute of one protocol */
static int register_one_attribute_of_one_protocol(struct qmdpi_bundle *bundle,
                                                  char const *signature_name,
                                                  struct qmdpi_attr const *attr,
                                                  void *arg)
{
    (void)arg;
    if (strcmp(signature_name, "base") == 0) {
        if (strcmp(qmdpi_attr_name_get(attr), "path") == 0) {
            hook_base_path_end = 1;
        } else if (strcmp(qmdpi_attr_name_get(attr), "path_end") == 0) {
            hook_base_path_end = 1;
        } else if (strcmp(qmdpi_attr_name_get(attr), "classified") == 0) {
            hook_base_classified = 1;
        } else if (strcmp(qmdpi_attr_name_get(attr), "session_end") == 0) {
            hook_base_session_end = 1;
        } else {
            qmdpi_bundle_attr_register(bundle, signature_name, qmdpi_attr_name_get(attr));
        }
    } else {
        qmdpi_bundle_attr_register(bundle, signature_name, qmdpi_attr_name_get(attr));
    }

    return 0;
}

/* unregister a callback for an attribute of one protocol */
static int unregister_one_attribute_of_one_protocol(struct qmdpi_bundle *bundle,
                                                    char const *signature_name,
                                                    struct qmdpi_attr const *attr,
                                                    void *arg)
{
    (void)arg;
    qmdpi_bundle_attr_unregister(bundle, signature_name, qmdpi_attr_name_get(attr));

    return 0;
}

/* register a callback for all attributes of one protocol */
int register_all_attributes_of_one_protocol(struct qmdpi_bundle *bundle,
                                            char const *signature_name,
                                            void *arg)
{
    (void)arg;
    qmdpi_bundle_attr_foreach(bundle,
                              signature_name,
                              register_one_attribute_of_one_protocol,
                              NULL);
    return 0;
}

int register_all_attributes_of_one_protocol_cb(struct qmdpi_bundle *bundle,
                                               struct qmdpi_signature *signature,
                                               void *arg)
{
    (void)arg;
    return register_all_attributes_of_one_protocol(bundle,
                                                   qmdpi_signature_name_get(signature),
                                                   NULL);
}

/* unregister a callback for all attributes of one protocol */
int unregister_all_attributes_of_one_protocol(struct qmdpi_bundle *bundle,
                                              char const *signature_name,
                                              void *arg)
{
    (void)arg;
    qmdpi_bundle_attr_foreach(bundle,
                              signature_name,
                              unregister_one_attribute_of_one_protocol,
                              NULL);
    return 0;
}

int unregister_all_attributes_of_one_protocol_cb(struct qmdpi_bundle *bundle,
                                                 struct qmdpi_signature *signature,
                                                 void *arg)
{
    (void)arg;
    return unregister_all_attributes_of_one_protocol(bundle,
                                                     qmdpi_signature_name_get(signature),
                                                     NULL);
}

/**
 * Function that registers all layer hooks in a hook store
 *
 * @param hs : The hook store
 */
int pa_metadata_add(struct hook_store *hs)
{
    char *layer;
    char *attr;
    struct hook *h;
    unsigned int i;
    int wild;
    int wild2;

    for (i = 0; i < hs->nb; ++i) {
        h = HOOK_STORE_GET(hs, i);
        layer = HOOK_LAYER(h);
        attr = HOOK_ATTR(h);

        wild = pa_wildchar(layer[0]);
        wild2 = pa_wildchar(attr[0]);

        if (!wild) {
            if (qmdpi_bundle_signature_get_byname(pcap_analyzer_bundle_get(),
                                                  layer) == NULL) {
                printf("Protocol %s not found\n", layer);
                exit(1);
            }

            if (wild2) {
                register_all_attributes_of_one_protocol(pcap_analyzer_bundle_get(), layer,
                                                        NULL);
            } else {
                if (strcmp(layer, "base") == 0) {
                    if (strcmp(attr, "path") == 0) {
                        fprintf(stderr,
                                "Warning: base:path doesn't exist anymore will enable base:path_end\n");
                        hook_base_path_end = 1;
                    } else if (strcmp(attr, "path_end") == 0) {
                        hook_base_path_end = 1;
                    } else if (strcmp(attr, "final_path") == 0) {
                        hook_base_final_path = 1;
                    } else if (strcmp(attr, "classified") == 0) {
                        hook_base_classified = 1;
                    } else if (strcmp(attr, "session_end") == 0) {
                        hook_base_session_end = 1;
                    } else {
                        qmdpi_bundle_attr_register(pcap_analyzer_bundle_get(), layer, attr);
                    }
                } else {
                    qmdpi_bundle_attr_register(pcap_analyzer_bundle_get(), layer, attr);
                }
            }
        } else {
            if (wild2) {
                qmdpi_bundle_signature_foreach(pcap_analyzer_bundle_get(),
                                               register_all_attributes_of_one_protocol_cb, NULL);
            }
        }
    }

    return 0;
}



/**
 * Function that unregisters all layer hooks in a hook store
 *
 * @param hs : The hook store
 */
void pa_metadata_del(struct hook_store const *hs)
{
    char *layer;
    char *attr;
    struct hook const *h;
    unsigned int i;
    int wild;
    int wild2;

    for (i = 0; i < hs->nb; ++i) {
        h = HOOK_STORE_GET(hs, i);
        layer = HOOK_LAYER(h);
        attr = HOOK_ATTR(h);

        wild = pa_wildchar(layer[0]);
        wild2 = pa_wildchar(attr[0]);

        if (!wild) {
            if (wild2) {
                unregister_all_attributes_of_one_protocol(pcap_analyzer_bundle_get(), layer,
                                                          NULL);
            } else {
                qmdpi_bundle_attr_unregister(pcap_analyzer_bundle_get(), layer, attr);
            }
        } else {
            if (wild2) {
                qmdpi_bundle_signature_foreach(pcap_analyzer_bundle_get(),
                                               unregister_all_attributes_of_one_protocol_cb, NULL);
            }
        }
    }
}
