/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#include <stdlib.h>
#include <string.h>
#include <dpi/protodef.h>

#include "fasthash.h"
#include "attributes.h"

/**
 * Supported attribute list and related data storage rules.
 * The order in this list defines the CSV column order in output log.
 */
struct attribute ATTRIBUTES[] = {
    { Q_PROTO_TCP,   Q_MPA_STREAM,           LOG_DATA_STREAM,   LOG_CLASS_L4,   0},
    { Q_PROTO_UDP,   Q_MPA_STREAM,           LOG_DATA_STREAM,   LOG_CLASS_L4,   0},
    { Q_PROTO_IP,    Q_IP_RESOLV_NAME,       LOG_DATA_HOSTNAME, LOG_CLASS_DNS,  0},
    { Q_PROTO_IP,    Q_IP_ADDRESS,           LOG_DATA_IP4,      LOG_CLASS_L3,   0},
    { Q_PROTO_SSL,   Q_SSL_COMMON_NAME,      LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_SSL,   Q_SSL_SERVER_NAME,      LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_SSL,   Q_SSL_SUBJECT_ALT_NAME, LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_QUIC,  Q_QUIC_SERVER_NAME,     LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_QUIC,  Q_MPA_USER_AGENT,       LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_RTMP,  Q_RTMP_PAGE_URL,        LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_RTMP,  Q_RTMP_STREAM_URL,      LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_STUN,  Q_STUN_REALM,           LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_STUN,  Q_STUN_SOFTWARE,        LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_SERVER,          LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_MIME_TYPE,       LOG_DATA_STD,      LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_REFERER_SERVER,  LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_REFERER,         LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_LOCATION,        LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_SERVER_AGENT,    LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_MPA_USER_AGENT,       LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_URI_PATH,        LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP,  Q_HTTP_URI,             LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_SPDY,  Q_SPDY_HOST,            LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_SPDY,  Q_SPDY_URI_RAW,         LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_SPDY,  Q_MPA_USER_AGENT,       LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_SPDY,  Q_SPDY_MIME_TYPE,       LOG_DATA_STD,      LOG_CLASS_L7,   0},
    { Q_PROTO_SPDY,  Q_SPDY_LOCATION,        LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_SPDY,  Q_SPDY_SERVER_AGENT,    LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_SIP,   Q_MPA_USER_AGENT,       LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_HTTP2_HOST,           LOG_DATA_HOSTNAME, LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_MPA_USER_AGENT,       LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_HTTP2_MIME_TYPE,      LOG_DATA_STD,      LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_HTTP2_LOCATION,       LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_HTTP2_SERVER_AGENT,   LOG_DATA_AGENT,    LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_HTTP2_URI_RAW,        LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
    { Q_PROTO_HTTP2, Q_HTTP2_REFERER,        LOG_DATA_PREFIX,   LOG_CLASS_L7,   0},
};

static inline int attribute_item_cmp(struct attribute_item *i1,
                                     struct attribute_item *i2)
{
    return ((i1->value_len != i2->value_len) ||
            (i1->attr_id != i2->attr_id) ||
            (i1->proto_id != i2->proto_id) ||
            (memcmp(i1->value_data, i2->value_data, i1->value_len)));
}

int
attribute_list_item_add(attribute_list_t **list, struct attribute_item *item)
{
    while (*list != NULL && attribute_item_cmp(&((*list)->item), item)) {
        list = &((*list)->next);
    }

    if (*list != NULL) {
        /* this attribute/value already exists in this flow's context */
        return 1;
    }

    attribute_list_t *new_attr = malloc(sizeof(attribute_list_t));
    if (new_attr == NULL) {
        return 0;
    }
    new_attr->next = *list;
    *list = new_attr;

    new_attr->item.attr_id = item->attr_id;
    new_attr->item.proto_id = item->proto_id;
    new_attr->item.value_len = item->value_len;
    new_attr->item.value_data = malloc(item->value_len);
    memcpy(new_attr->item.value_data, item->value_data, item->value_len);

    return 1;
}

void
attribute_list_item_clean(struct attribute_item *item)
{
    if (item->value_data != NULL) {
        free(item->value_data);
        item->value_data = NULL;
    }
}

struct attribute *
attribute_def_get_by_index(int index)
{
    if (index >= (int)(sizeof(ATTRIBUTES) / sizeof(struct attribute))) {
        return NULL;
    }
    return &(ATTRIBUTES[index]);
}

void
attribute_def_init(void)
{
    unsigned int attr_index = 0;

    while (attr_index < (sizeof(ATTRIBUTES) / sizeof(struct attribute))) {
        ATTRIBUTES[attr_index].activated = 0;
        attr_index++;
    }
}

struct attribute *
attribute_def_get_by_ids(int protoid, int attrid)
{
    unsigned int attr_index = 0;

    while (attr_index < (sizeof(ATTRIBUTES) / sizeof(struct attribute))) {
        if (ATTRIBUTES[attr_index].proto_id == protoid &&
                ATTRIBUTES[attr_index].attr_id == attrid) {
            return &ATTRIBUTES[attr_index];
        }
        attr_index++;
    }

    return NULL;
}

