/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#ifndef __ATTRIBUTES__
#define __ATTRIBUTES__

#define ATTRIBUTE_HT_SIZE 2048
#define ATTRIBUTE_NAME_SZ 64

/* An attribute of a given proto:attr ID having
 * a unique data value */
struct attribute_item {
    uint8_t *value_data;
    uint32_t value_len;
    int32_t proto_id;
    int32_t attr_id;
};

/* Attributes data formatting rules */
typedef enum attribute_data {
    LOG_DATA_STD,       /* standard JSON/PDD escaping */
    LOG_DATA_AGENT,     /* for prefix-based user/server agents */
    LOG_DATA_PREFIX,    /* for prefix-based patterns */
    LOG_DATA_HOSTNAME,  /* for suffix-based hostnames */
    LOG_DATA_STREAM,    /* for raw payload with prefix matching */
    LOG_DATA_IP4,       /* special formatting for 32-bit IPv4 addresses */
    LOG_DATA_IP6,       /* special formatting for 128-bit IPv6 addresses */
    LOG_DATA_PROTOID,   /* protocol ID to protocol name (string) conversion */
} attribute_data_t;

/* Attributes class */
typedef enum attribute_class {
    LOG_CLASS_L7,       /* all L7 protocols metadata */
    LOG_CLASS_L4,       /* TCP/UDP ports and payload */
    LOG_CLASS_L3,       /* IP addresses, etc. */
    LOG_CLASS_DNS,      /* should be dns.query */
} attribute_class_t;

/* Attributes types definition */
struct attribute {
    int               proto_id;
    int               attr_id;
    attribute_data_t  data_type;
    attribute_class_t data_class;
    unsigned int      activated: 1; /* is this attribute requested ? */
};

/*************************************/
/**** ATTRIBUTES definition array ****/
/*************************************/

/**
 * Mandatory attributes definition list initialization
 */
void attribute_def_init(void);

/**
 * Get the attribute type definition element stored
 * at a given <index> in the ATTRIBUTES definition array.
 */
struct attribute *attribute_def_get_by_index(int index);

/**
 * Get the attribute type definition element
 * corresponding to a given <protoid> and <attrid>.
 */
struct attribute *attribute_def_get_by_ids(int protoid, int attrid);

/******************************/
/**** Attribute item lists ****/
/******************************/

/* A list of attribute items */
typedef struct attribute_list {
    struct attribute_item  item;
    struct attribute_list *next;
} attribute_list_t;

/**
 * Add a UNIQUE attribute <item> in an attribute <list>.
 *
 * Returns: 1 if the adding is a success,
 *          0 if it failed.
 */
int attribute_list_item_add(attribute_list_t **list,
                            struct attribute_item *item);

/**
 * Clean an <attribute_item>.
 */
void attribute_list_item_clean(struct attribute_item *attribute_item);

#endif
