/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#ifndef __DATA_FORMAT__
#define __DATA_FORMAT__

#include <qmdpi.h>
#include <string.h>
#include <stdio.h>

#include "attributes.h"

#define DATA_PATH_SEP '.'
#define DATA_LOG_SEP ','
#define DATA_VAL_SEP ';'
#define DATA_STR_SEP '\"'
#define DATA_STR_SEP2 '\''
#define DATA_CFG_SEP ':'

#define DATA_IP_ADDR_LEN 16
#define DATA_VALUE_LEN 256

extern struct qmdpi_bundle *bundle;

static inline uint32_t
data_umin32(uint32_t a, uint32_t b)
{
    return a < b ? a : b;
}

static inline uint64_t
data_umin64(uint64_t a, uint64_t b)
{
    return a < b ? a : b;
}

static inline int
data_str_end_with(const char *string1, unsigned int len1,
                  const char *string2, unsigned int len2)
{
    if (len1 < len2) {
        return 0;
    }

    return (memcmp(string1 + (len1 - len2), string2, len2) == 0);
}

static inline int
data_str_start_with(const char *string1, unsigned int len1,
                    const char *string2, unsigned int len2)
{
    if (len1 < len2) {
        return 0;
    }

    return (memcmp(string1, string2, len2) == 0);
}

static inline int
data_str_find(const char *bigstring, const unsigned int blen,
              const char *smallstring, const unsigned int slen)
{
    const char *end = bigstring + blen;
    const char *p = bigstring;

    while (p <= end - slen) {
        if (*p == *smallstring && !memcmp(p, smallstring, slen)) {
            return 1;
        }
        p++;
    }

    return 0;
}

static inline const char *
data_signame_get_byid(uint32_t sig_id)
{
    struct qmdpi_signature *sig = qmdpi_bundle_signature_get_byid(bundle, sig_id);
    return qmdpi_signature_name_get(sig);
}

static inline const char *
data_attrname_get_byid(uint32_t sig_id, uint32_t attr_id)
{
    struct qmdpi_attr *attr = qmdpi_bundle_attr_get_byid(bundle, sig_id, attr_id);
    return qmdpi_attr_name_get(attr);
}

static inline int
data_signame_get_byname(const char *sig_name)
{
    struct qmdpi_signature *sig = qmdpi_bundle_signature_get_byname(bundle,
                                                                    sig_name);
    return qmdpi_signature_id_get(sig);
}

static inline int
data_attrname_get_byname(int sig_id, const char *attr_name)
{
    struct qmdpi_attr *attr = qmdpi_bundle_attr_get_byname(bundle,
                                                           data_signame_get_byid(sig_id),
                                                           attr_name);
    return qmdpi_attr_id_get(attr);
}

static inline int
data_is_printable(uint8_t c)
{
    return ((c) >= 32 && (c) <= 126);
}

static inline unsigned int
data_format_stream(FILE *out, uint8_t *data, unsigned int data_len)
{
    unsigned int cnt = 0;
    unsigned int index = 0;
    uint8_t byte;

    while (index < data_len) {
        byte = data[index];
        if (byte == 0) {
            cnt += fprintf(out, "\\0");
        } else if (data_is_printable(byte) &&
                   byte != DATA_STR_SEP &&
                   byte != DATA_STR_SEP2 &&
                   byte != '\\') {
            cnt += fprintf(out, "%c", byte);
        } else {
            cnt += fprintf(out, "\\x%02x", byte);
        }
        index++;
    }

    return cnt;
}

static inline unsigned int
data_format_attribute(FILE *out, struct attribute *attr,
                      uint8_t *data, unsigned int data_len)
{
    char ip_addr[DATA_IP_ADDR_LEN];
    const char *proto_name = NULL;
    unsigned int cnt = 0;

    switch (attr->data_type) {
        case LOG_DATA_IP4:
            memset(ip_addr, 0, DATA_IP_ADDR_LEN);
            sprintf(ip_addr, "%d.%d.%d.%d", data[0], data[1], data[2], data[3]);
            cnt += fprintf(out, "%s", ip_addr);
            break;
        case LOG_DATA_IP6:
            break;
        case LOG_DATA_PROTOID:
            proto_name = data_signame_get_byid(*((uint32_t *)data));
            if (proto_name) {
                cnt += fprintf(out, "%s", proto_name);
            } else {
                cnt += fprintf(out, "<unknown>");
            }
            break;
        default:
            cnt += data_format_stream(out, data, data_len);
    }

    return cnt;
}

#endif
