/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2019 - All rights reserved
*/
#include <sys/param.h>
#include <inttypes.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>

#include <qmdpi.h>
#include <qmdpi_bundle_api.h>

#include "data_format.h"
#include "exclusion.h"
#include "logs.h"
#include "dns.h"

#define LOGS_FILENAME_LEN 256

struct dns_ht DNS_HT;

static uint32_t flow_count = 0;

/* defines how we should split output CSV file(s) */
static logs_split_policy_t logs_split_p = LOGS_SPLIT_NO;
static uint64_t logs_split_threshold = 0;
static uint64_t logs_total_written = 0;
static time_t logs_next_rotate_ts = 0;

/* logging parameters */
static struct {
    uint8_t write_tcp_stream: 1;
    uint8_t write_udp_stream: 1;
    uint8_t with_proto_names: 1;
    uint8_t with_path: 1;
} logs_param = { 0, 0, 0, 0 };

/* static part of the CSV columns definition header line */
static char *log_columns[] = {
    "flow_id",
    "l4_proto_id",
    "top_proto_id",
    "flow_duration",
    "volume_of_pkts_cts",
    "volume_of_pkts_stc",
    "first_pkt_number",
    "first_pkt_timestamp",
    "pkt_count_cts",
    "pkt_count_stc",
};

/* transport protocol list */
static uint32_t logger_l4_list[] = {
    Q_PROTO_UDP,
    Q_PROTO_TCP,
    Q_PROTO_SCTP,
    Q_PROTO_L2TP,
    Q_PROTO_GRE,
};

void
logs_init()
{
    dns_ht_init(&DNS_HT);
    attribute_def_init();
    logs_next_rotate_ts = time(NULL) + (logs_split_threshold * 60);
}

void
logs_cleanup()
{
    dns_ht_clean(&DNS_HT);
}

void
logs_param_split_policy_set(logs_split_policy_t p, uint64_t threshold)
{
    logs_split_p = p;
    logs_split_threshold = threshold;
}

void
logs_param_tcp_stream_set(int param)
{
    logs_param.write_tcp_stream = param;
}

void
logs_param_udp_stream_set(int param)
{
    logs_param.write_udp_stream = param;
}

void
logs_param_with_proto_names(int param)
{
    logs_param.with_proto_names = param;
}

void
logs_param_with_path(int param)
{
    logs_param.with_path = param;
}

static inline int
logs_proto_is_l4(uint32_t proto_id)
{
    unsigned int proto_index;

    for (proto_index = 0;
            proto_index < (sizeof(logger_l4_list) / sizeof(uint32_t));
            proto_index++) {
        if (logger_l4_list[proto_index] == proto_id) {
            return 1;
        }
    }

    return 0;
}

static inline unsigned int
logs_dump_payload(FILE *out, struct flow_item *flow)
{
    unsigned int cnt = 0;

    if (flow->vol_pld_cts > 0) {
        cnt += fprintf(out, "%c", DATA_STR_SEP);
        cnt += data_format_stream(out, flow->payload_buffer_cts,
                                  data_umin64(flow->vol_pld_cts, L4_BUFFER_LEN));
        cnt += fprintf(out, "%c", DATA_STR_SEP);
    }
    if (flow->vol_pld_stc > 0) {
        if (flow->vol_pld_cts > 0) {
            cnt += fprintf(out, "%c", DATA_VAL_SEP);
        }
        cnt += fprintf(out, "%c", DATA_STR_SEP);
        cnt += data_format_stream(out, flow->payload_buffer_stc,
                                  data_umin64(flow->vol_pld_stc, L4_BUFFER_LEN));
        cnt += fprintf(out, "%c", DATA_STR_SEP);
    }

    return cnt;
}

static inline unsigned int
logs_dump_payload_tcp_udp(FILE *out, struct flow_item *flow)
{
    unsigned int cnt = 0;

    /* printing TCP stream */
    if (logs_param.write_tcp_stream) {
        cnt += fprintf(out, "%c", DATA_LOG_SEP);
        if (flow->top_proto_id == Q_PROTO_TCP) {
            cnt += logs_dump_payload(out, flow);
        }
    }

    /* printing UDP stream */
    if (logs_param.write_udp_stream) {
        cnt += fprintf(out, "%c", DATA_LOG_SEP);
        if (flow->top_proto_id == Q_PROTO_UDP) {
            cnt += logs_dump_payload(out, flow);
        }
    }

    return cnt;
}

/**
 * DNS Caching
 *
 * Returns: 1 if the DNS Caching add was performed,
 *          0 if attributes were missing in the flow,
 *         -1 if the DNS Caching add failed.
 */
static inline int
logs_dns_lookup(struct flow_item *flow)
{
    unsigned int      hostname_len = 0;
    char             *hostname = NULL;
    uint32_t          hostaddr = 0;
    attribute_list_t *attrs = NULL;

    attrs = flow->attributes;

    /* find needed information in this DNS flow */
    while (attrs) {
        if (attrs->item.proto_id == Q_PROTO_DNS) {
            switch (attrs->item.attr_id) {
                case Q_DNS_QUERY:
                    hostname = (char *)attrs->item.value_data;
                    hostname_len = attrs->item.value_len;
                    break;
                case Q_DNS_HOST_ADDR:
                    hostaddr = *(uint32_t *)(attrs->item.value_data);
                    break;
            }
            if (hostname && hostaddr) {
                break;
            }
        }
        attrs = attrs->next;
    }

    if (!hostname || !hostaddr) {
        return 0;
    }

    if (!dns_ht_entry_insert(&DNS_HT, hostaddr, hostname, hostname_len)) {
        return -1;
    }

    return 1;
}

int
logs_new_attribute(struct flow_item *flow_item, int proto_id, int attr_id,
                   const uint8_t *value, uint32_t value_len)
{
    /* filter attribute before adding */
    if (excls_attribute_is_excluded(proto_id, attr_id, value, value_len)) {
        return 0;
    }

    struct attribute_item new_item = {
        .attr_id = attr_id,
        .proto_id = proto_id,
        .value_data = (uint8_t *)value,
        .value_len = value_len,
    };

    /* add attribute to the related flow */
    if (flow_attr_list_item_add(flow_item, &new_item) == 0) {
        return -1;
    }

    /* dns caching */
    if (proto_id == Q_PROTO_DNS && attr_id == Q_DNS_HOST_ADDR) {
        logs_dns_lookup(flow_item);
    }

    return 1;
}

static inline void
logs_set_cnx_type(struct flow_item *flow_item,
                  uint32_t *cnx_type, uint32_t cnx_way)
{
    /* was the current packet's cnx_type provided by the ixEngine (uapp_cnx)? */
    if (*cnx_type == QMDPI_DIR_UNKNOWN) {

        /* no, we try to use saved flow direction */
        *cnx_type = flow_item->flow_dir;
        if (*cnx_type != QMDPI_DIR_UNKNOWN && cnx_way == QMDPI_DIR_1) {
            *cnx_type = (~*cnx_type)&QMDPI_DIR_UNKNOWN;
        }

        /* update/store the flow's cnx_type from cnx_type input */
    } else {
        if (cnx_way == QMDPI_DIR_1) {
            flow_item->flow_dir = (~*cnx_type)&QMDPI_DIR_UNKNOWN;
        } else if (cnx_way == QMDPI_DIR_0) {
            flow_item->flow_dir = *cnx_type;
        }
    }
}

int
logs_new_packet(struct flow_item **flow_item, uint64_t pkt_num,
                uint32_t cnx_type, uint32_t cnx_way,
                uint8_t *payload, uint32_t payload_len,
                uint32_t ip_src, uint32_t ip_dst)
{
    int ret_val = 0;

    /* allocate flow item */
    if (*flow_item == NULL) {
        *flow_item = malloc(sizeof(struct flow_item));
        if (*flow_item) {
            flow_item_init(*flow_item, flow_count++);
            ret_val = 1;
        } else {
            return -1;
        }
    }

    /* defining the direction (CTS/STC) of the packet */
    logs_set_cnx_type(*flow_item, &cnx_type, cnx_way);

    /* first packet ? */
    if ((*flow_item)->first_pkt_number == -1) {
        (*flow_item)->first_pkt_number = pkt_num;
    }

    /* IP addresses */
    if (ip_src && ip_dst) {
        if ((*flow_item)->l3_addr_clt == -1) {
            (*flow_item)->l3_addr_clt = (cnx_type == QMDPI_DIR_CTS) ? ip_src : ip_dst;
        }
        if ((*flow_item)->l3_addr_srv == -1) {
            (*flow_item)->l3_addr_srv = (cnx_type == QMDPI_DIR_CTS) ? ip_dst : ip_src;
        }
    }

    /* payload storage */
    if (!logs_param.write_tcp_stream && !logs_param.write_udp_stream) {
        return ret_val;
    }
    if (cnx_type != QMDPI_DIR_STC) {
        if ((*flow_item)->vol_pld_cts < L4_BUFFER_LEN && payload_len
                && (*flow_item)->cnt_pld_cts < FLOW_MAX_PKT_CNT) {
            memcpy((*flow_item)->payload_buffer_cts + (*flow_item)->vol_pld_cts,
                   payload,
                   data_umin64(L4_BUFFER_LEN - (*flow_item)->vol_pld_cts, payload_len));
            (*flow_item)->vol_pld_cts += payload_len;
            (*flow_item)->cnt_pld_cts ++;

        }
    } else {
        if ((*flow_item)->vol_pld_stc < L4_BUFFER_LEN && payload_len
                && (*flow_item)->cnt_pld_stc < FLOW_MAX_PKT_CNT) {
            memcpy((*flow_item)->payload_buffer_stc + (*flow_item)->vol_pld_stc,
                   payload,
                   data_umin64(L4_BUFFER_LEN - (*flow_item)->vol_pld_stc, payload_len));
            (*flow_item)->vol_pld_stc += payload_len;
            (*flow_item)->cnt_pld_stc ++;
        }
    }

    return ret_val;
}

void
logs_new_classification_path(struct flow_item *flow_item,
                             struct qmdpi_path *path)
{
    int top_proto = 0;
    unsigned int index;

    if (path->qp_len < 2) {
        return;
    }

    /* transport layer lookup */
    if (flow_item->l4_proto_id == Q_PROTO_UNKNOWN) {
        unsigned int layer_index = path->qp_len - 1;
        while (layer_index) {
            if (logs_proto_is_l4(path->qp_value[layer_index])) {
                flow_item->l4_proto_id = path->qp_value[layer_index];
                break;
            }
            layer_index--;
        }
    }

    /* if top layer is "unknown" then set the "top_proto"
     * of the current flow to the layer below. */
    if (path->qp_value[path->qp_len - 1] == Q_PROTO_UNKNOWN) {
        top_proto = path->qp_value[path->qp_len - 2];
    } else {
        top_proto = path->qp_value[path->qp_len - 1];
    }

    flow_item->top_proto_id = top_proto;

    /* full path storage */
    if (logs_param.with_path) {
        if (flow_item->path == NULL) {
            flow_item->path = malloc(PATH_LEN * sizeof(uint32_t));
        }
        for (index = 1; index < PATH_LEN; index++) {
            if (index >= path->qp_len) {
                flow_item->path[index - 1] = 0;
                break;
            }
            flow_item->path[index - 1] = path->qp_value[index];
        }
    }
}

static int
logs_dump_attributes(FILE *output, attribute_list_t *attr_list,
                     int index, int *col_counter)
{
    struct attribute_item *item = NULL;
    struct attribute *attr = attribute_def_get_by_index(index);
    unsigned int cnt = 0;

    if (attr == NULL) {
        return -1;
    }

    if (attr->activated == 0) {
        return 0;
    }

    (*col_counter)++;

    while (attr_list) {
        item = &(attr_list->item);
        attr_list = attr_list->next;

        if (item->proto_id != attr->proto_id || item->attr_id != attr->attr_id) {
            continue;
        }

        if (*col_counter != 0) {
            for (index = 0; index < *col_counter; index++) {
                fputc(DATA_LOG_SEP, output);
                cnt++;
            }
            *col_counter = 0;
            fputc(DATA_STR_SEP, output);
            cnt++;
        } else {
            fputc(DATA_VAL_SEP, output);
            cnt++;
        }

        fputc(DATA_STR_SEP2, output);
        cnt += data_format_attribute(output, attr,
                                     item->value_data,
                                     item->value_len);
        fputc(DATA_STR_SEP2, output);
        cnt += 2;
    }
    if (cnt > 0) {
        fputc(DATA_STR_SEP, output);
        cnt++;
    }
    return cnt;
}

int
logs_print_path(FILE *output,  struct flow_statistics *flow_stats)
{
    uint32_t *path = flow_stats->flow->path;
    int cnt = 0;
    int index;

    if (path == NULL) {
        return 0;
    }

    for (index = 0; index < PATH_LEN && path[index] != 0; index++) {
        if (index != 0) {
            if (logs_param.with_proto_names) {
                fputc(DATA_PATH_SEP, output);
            } else {
                fputc(DATA_VAL_SEP, output);
            }
            cnt++;
        } else if (logs_param.with_proto_names) {
            fputc(DATA_STR_SEP, output);
            cnt++;
        }

        if (logs_param.with_proto_names) {
            cnt += fprintf(output, "%s", data_signame_get_byid(path[index]));
        } else {
            cnt += fprintf(output, "%d", path[index]);
        }
    }

    if (logs_param.with_proto_names) {
        fputc(DATA_STR_SEP, output);
        cnt++;
    }

    return cnt;
}

int
logs_file_write(const char *filename, struct flow_statistics *flow_stats)
{
    const time_t ts = time(NULL);
    static int ref_tm_mday = -1;
    char write_date[9] = "YYYYMMDD";
    struct tm *curr_tm = NULL;

    static unsigned int log_id = 0;
    struct flow_item *flow = NULL;
    struct attribute *attr = NULL;
    static FILE *output = NULL;
    int cnt_attr = 0;
    int cnt = 0;
    int attr_index = 0;
    int col_counter = 0;
    unsigned int index;

    static char output_name[LOGS_FILENAME_LEN] = { 0 };
    char output_tmp[LOGS_FILENAME_LEN];

    curr_tm = localtime(&ts);
    if (ref_tm_mday == -1) {
        ref_tm_mday = curr_tm->tm_mday;
    }

    /* new LOG file */
    if (output == NULL) {
        if (filename != NULL) {
            /* new log file name */
            fprintf(stdout, "\r[MSG] Writing CSV to %s\n", filename);
            strcpy(output_name, filename);
            log_id = 0;

        } else {
            /* formatting today's date */
            sprintf(write_date, "%04u%02u%02u",
                    1900 + curr_tm->tm_year, curr_tm->tm_mon + 1, curr_tm->tm_mday);

            /* test if we have been archiving LOGs today */
            if (curr_tm->tm_mday != ref_tm_mday) {
                ref_tm_mday = curr_tm->tm_mday;
                log_id = 0;
            }

            /* making full archived LOG file name */
            strcpy(output_tmp, output_name);
            sprintf(output_tmp + strlen(output_tmp), "-%s-%03d",
                    write_date, (log_id++) % 1000);
            output_tmp[strlen(output_tmp) + 1] = '\0';

            /* LOG files rotation */
            if (rename(output_name, output_tmp) != 0) {
                fprintf(stderr, "\r[MSG] Archiving CSV to %s failed\n", output_tmp);
            } else {
                fprintf(stdout, "\r[MSG] Archived CSV to %s ...\n", output_tmp);
            }
            filename = output_tmp;
        }

        output = fopen(output_name, "w");
        if (output == NULL)  {
            fprintf(stderr, "\r[ERR] %s: can't open the log file for writing\n",
                    output_name);
            return -1;
        }
    }

    /* closing file */
    if (filename == NULL && flow_stats == NULL) {
        fclose(output);
        output = NULL;
        return 0;
    }

    /* printing CSV header line */
    if (flow_stats == NULL || filename == output_tmp) {

        /* static part */
        for (index = 0; index < sizeof(log_columns) / sizeof(char *); index++) {
            if (index != 0) {
                fputc(DATA_LOG_SEP, output);
                cnt++;
            }
            cnt += fprintf(output, "%s", log_columns[index]);
        }

        /* full path printing */
        if (logs_param.with_path) {
            cnt += fprintf(output, "%cpath", DATA_LOG_SEP);
        }

        /* attributes part */
        while ((attr = attribute_def_get_by_index(attr_index++))) {
            if (attr->proto_id == 0) {
                continue;
            }

            if (attr->activated == 0) {
                continue;
            }

            cnt += fprintf(output, "%c%s:%s", DATA_LOG_SEP,
                           data_signame_get_byid(attr->proto_id),
                           data_attrname_get_byid(attr->proto_id, attr->attr_id));
        }
        cnt += fprintf(output, "\n");
    }

    /* printing flow metrics and classification information  */
    if (flow_stats != NULL) {
        flow = flow_stats->flow;
        cnt += fprintf(output, "%d%c", flow->flow_id, DATA_LOG_SEP);
        if (logs_param.with_proto_names) {
            fprintf(output, "%c%s%c%c", DATA_STR_SEP,
                    data_signame_get_byid(flow->l4_proto_id),
                    DATA_STR_SEP, DATA_LOG_SEP);
            fprintf(output, "%c%s%c%c", DATA_STR_SEP,
                    data_signame_get_byid(flow->top_proto_id),
                    DATA_STR_SEP, DATA_LOG_SEP);
        } else {
            fprintf(output, "%u%c", flow->l4_proto_id, DATA_LOG_SEP);
            fprintf(output, "%u%c", flow->top_proto_id, DATA_LOG_SEP);
        }
        cnt += fprintf(output, "%u%c", flow_stats->duration, DATA_LOG_SEP);
        cnt += fprintf(output, "%" PRIu64 "%c", flow_stats->vol_pkt_cts, DATA_LOG_SEP);
        cnt += fprintf(output, "%" PRIu64 "%c", flow_stats->vol_pkt_stc, DATA_LOG_SEP);
        cnt += fprintf(output, "%" PRIu64 "%c", flow->first_pkt_number, DATA_LOG_SEP);
        cnt += fprintf(output, "%u%c", flow_stats->first_pkt_ts, DATA_LOG_SEP);
        cnt += fprintf(output, "%u%c", flow_stats->packet_count_cts, DATA_LOG_SEP);
        cnt += fprintf(output, "%u", flow_stats->packet_count_stc);

        /* full path printing */
        if (logs_param.with_path) {
            fputc(DATA_LOG_SEP, output);
            cnt += logs_print_path(output, flow_stats);
            cnt++;
        }

        /* printing payload bytes for unknown L4-level flows */
        cnt += logs_dump_payload_tcp_udp(output, flow);

        /* we remove the manipulation of tcp and udp payload as it has been already done in
         * logs_dump_payload_tcp_udp() */
        attr_index += 2;

        /* printing complete attribute list */
        while ((cnt_attr = logs_dump_attributes(output,
                                                flow->attributes,
                                                attr_index,
                                                &col_counter)) != -1) {
            cnt += cnt_attr;
            attr_index++;
        }

        cnt += fprintf(output, "\n");
    }

    return cnt;
}

static void
logs_dns_caching_finalize(struct flow_item *flow_dst)
{
    struct dns_ht_entry       *dns_entry = NULL;

    if (flow_dst->top_proto_id == Q_PROTO_DNS) {
        return;
    }

    dns_entry = dns_ht_entry_fetch(&DNS_HT, flow_dst->l3_addr_srv);

    if (dns_entry != NULL) {
        struct attribute_item new_item = {
            .proto_id = Q_PROTO_IP,
            .attr_id = Q_IP_RESOLV_NAME,
            .value_data = (uint8_t *)dns_entry->hostname,
            .value_len = strlen(dns_entry->hostname),
        };

        flow_attr_list_item_add(flow_dst, &new_item);
    }
}

static void logs_flow_write(struct flow_statistics *stat)
{
    logs_total_written += logs_file_write(NULL, stat);
    time_t cur_ts = 0;

    if (logs_split_p == LOGS_SPLIT_SIZE) {
        if (logs_total_written >= logs_split_threshold) {
            logs_file_write(NULL, NULL);
            logs_total_written = 0;
        }
    } else if (logs_split_p == LOGS_SPLIT_TIME) {
        cur_ts = time(NULL);
        if (cur_ts >= logs_next_rotate_ts) {
            logs_file_write(NULL, NULL);
            logs_next_rotate_ts = cur_ts + (logs_split_threshold * 60);
        }
    }
}

static inline void
logs_store_ip_addresses(struct flow_statistics *stat)
{
    struct attribute *attr = NULL;

    attr = attribute_def_get_by_ids(Q_PROTO_IP, Q_IP_ADDRESS);
    if (!attr || !attr->activated) {
        return;
    }

    struct attribute_item new_item = {
        .proto_id = Q_PROTO_IP,
        .attr_id = Q_IP_ADDRESS,
        .value_len = sizeof(uint32_t),
    };

    /* client address */
    new_item.value_data = (uint8_t *)&stat->flow->l3_addr_clt,
             flow_attr_list_item_add(stat->flow, &new_item);

    /* server address */
    new_item.value_data = (uint8_t *)&stat->flow->l3_addr_srv,
             flow_attr_list_item_add(stat->flow, &new_item);
}

void
logs_flow_finalize(struct flow_statistics *stats)
{
    /* DNS caching */
    logs_dns_caching_finalize(stats->flow);

    /* TCP established */
    if (stats->established && stats->flow->top_proto_id == Q_PROTO_TCP) {
        stats->flow->top_proto_id = Q_PROTO_ESTABLISHED;
    }

    /* Get IP addresses */
    logs_store_ip_addresses(stats);

    /* CSV write */
    logs_flow_write(stats);

    /* Cleanup */
    flow_item_clean(stats->flow);
    free(stats->flow);
}

