/*
*   This file is a part of Qosmos ixEngine.
*   Copyright  Qosmos 2000-2017 - All rights reserved
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>

#include "parse_options.h"


static void
parse_usage(char *name)
{
    fprintf(stderr, "PCAP logger\n");

    fprintf(stderr, "\nUsage:\t%s [options] [pcap_files|interface]\n", name);
    fprintf(stderr, "Options:\n"\

            "\t--live:" \
            "\t\t\tLive capture from \033[01minterface\033[00m instead of \033[01mpcap_files\033[00m reading.\n"
            \
            "\t\t\t\tBy default tries the first interface if no \033[01minterface\033[00m is given.\n"
            \
            "\t\t\t\tCapture is interrupted with CTRL+C.\n" \

            "\n\t--csv \033[01mfile\033[00m:" \
            "\t\tSet output CSV \033[01mfile\033[00m path (default: " LOG_FILENAME ").\n" \

            "\n\t--nb-flows \033[01mcount\033[00m:" \
            "\tSet a maximum flow \033[01mcount\033[00m for the ixEngine (default: 200000).\n"
            \

            "\n\t--rotate \033[01mn<unit>\033[00m:" \
            "\tSet a condition for CSV LOG file rotation, based on duration or file size.\n"
            \
            "\t\t\t\tThe \033[01munit\033[00m must be in [m|h|d|w|B|kB|MB|GB]; the \033[01mn\033[00mumber must be integer.\n"
            \

            "\n\t--config \033[01mfile\033[00m:" \
            "\t\tSet the path of the configuration file.\n" \

            "\n\t--print-names:" \
            "\t\tUse signature names instead of IDs in output CSV.\n" \

            "\n\t--print-path:" \
            "\t\tPrint additional protocol path column in output CSV.\n" \

            "\n\t--version:" \
            "\t\tPrint program version information.\n" \

            "\n");
    exit(0);
}

static void
parse_usage_rotate(void)
{
    fprintf(stderr, "PCAP logger\n");

    fprintf(stderr, "\nUsage:\t --rotate <number>[m|h|d|w|B|kB|MB|GB]\n");
    fprintf(stderr, "\nThe <number> must be an integer value.\n"\
            "\nDuration units:\n"\
            "\tm:\tminute(s)\n" \
            "\th:\thour(s)\n" \
            "\td:\tday(s)\n" \
            "\tw:\tweek(s)\n" \
            "\nFile size units:\n"\
            "\tB:\tByte(s)\n" \
            "\tkB:\tkilobyte(s)\n" \
            "\tMB:\tMegabyte(s)\n" \
            "\tGB:\tGigabyte(s)\n" \
            "\n");
    exit(0);
}

static logs_split_policy_t
parse_rotate(char *val, uint64_t *out)
{
    uint64_t param_value = 0;
    char *param_unit = NULL;

    /* parse number */
    param_value = strtol(val, &param_unit, 10);
    if (param_value == 0 || val == param_unit) {
        parse_usage_rotate();
    }

    if (!strlen(param_unit)) {
        parse_usage_rotate();
    }

    /* parse unit */
    if (!strcmp(param_unit, "B")) {
        *out = param_value;
        return LOGS_SPLIT_SIZE;
    }
    if (!strcmp(param_unit, "kB")) {
        *out = param_value * 1000;
        return LOGS_SPLIT_SIZE;
    }
    if (!strcmp(param_unit, "MB")) {
        *out = param_value * 1000000;
        return LOGS_SPLIT_SIZE;
    }
    if (!strcmp(param_unit, "GB")) {
        *out = param_value * 1000000000;
        return LOGS_SPLIT_SIZE;
    }
    if (!strcmp(param_unit, "m")) {
        *out = param_value;
        return LOGS_SPLIT_TIME;
    }
    if (!strcmp(param_unit, "h")) {
        *out = param_value * 60;
        return LOGS_SPLIT_TIME;
    }
    if (!strcmp(param_unit, "d")) {
        *out = param_value * 1440;
        return LOGS_SPLIT_TIME;
    }
    if (!strcmp(param_unit, "w")) {
        *out = param_value * 10080;
        return LOGS_SPLIT_TIME;
    }

    parse_usage_rotate();
    return LOGS_SPLIT_NO;
}

int
parse_param(int argc, char *argv[], struct opt *opt)
{
    static struct option opts[] = {
        {"nb-flows", 1, 0, 0},
        {"rotate", 1, 0, 0},
        {"link", 1, 0, 0},
        {"config", 1, 0, 0},
        {"csv", 1, 0, 0},
        {"live", 0, 0, 0},
        {"version", 0, 0, 0},
        {"print-names", 0, 0, 0},
        {"print-path", 0, 0, 0},
        {0, 0, 0, 0},
    };

    logs_split_policy_t s;
    uint64_t value = 0;

    int path_len = 0;
    int opti = 0;
    int c = 0;

    /* finding configuration file */
    char *slash = strrchr(argv[0], '/');
    if (slash != NULL) {
        path_len = (int)(slash - argv[0]) + 1;
        strncpy(OPT_CONFIG_FILENAME(opt), argv[0], path_len);
    }
    strcpy(OPT_CONFIG_FILENAME(opt) + path_len, CONFIG_FILE_NAME);

    if (argc <  2) {
        parse_usage(argv[0]);
        return 0;
    }

    /* parsing options */
    while ((c = getopt_long(argc, argv, "q", opts, &opti)) != -1) {
        switch (c) {
            case 0 :
                if (strcmp(opts[opti].name, "csv") == 0) {
                    strcpy(OPT_LOG_FILENAME(opt), optarg);
                } else if (strcmp(opts[opti].name, "nb-flows") == 0) {
                    OPT_SET_MCNX(opt, atoi(optarg));
                } else if (strcmp(opts[opti].name, "rotate") == 0) {
                    s = parse_rotate(optarg, &value);
                    logs_param_split_policy_set(s, value);
                } else if (strcmp(opts[opti].name, "live") == 0) {
                    OPT_LIVE_CAPTURE(opt) = 1;
                } else if (strcmp(opts[opti].name, "config") == 0) {
                    strcpy(OPT_CONFIG_FILENAME(opt), optarg);
                } else if (strcmp(opts[opti].name, "print-names") == 0) {
                    logs_param_with_proto_names(1);
                } else if (strcmp(opts[opti].name, "print-path") == 0) {
                    logs_param_with_path(1);
                } else if (strcmp(opts[opti].name, "version") == 0) {
                    fprintf(stdout, "PCAP logger version: %d.%d.%d-%s\n",
                            ENGINE_VERSION_MAJOR,
                            ENGINE_VERSION_MINOR,
                            ENGINE_VERSION_REVISION,
                            ENGINE_VERSION_EXTRA);
                    fprintf(stdout, "CSV revision: %s\n",
                            LOG_VERSION_CSV);
                    fprintf(stdout, "---------------------\n" \
                            "Copyright 2016 Qosmos\n");
                    exit(0);
                }
                break;
            default :
                parse_usage(argv[0]);
                break;
        }
    }

    return optind;
}

