/**********************************************************************
Each of the companies; Qualcomm, Motorola, Lucent, and Nokia (hereinafter 
referred to individually as "Source" or collectively as "Sources") do 
hereby state:

To the extent to which the Source(s) may legally and freely do so, the 
Source(s), upon submission of a Contribution, grant(s) a free, 
irrevocable, non-exclusive, license to the Third Generation Partnership 
Project 2 (3GPP2) and its Organizational Partners: ARIB, CCSA, TIA, TTA, 
and TTC, under the Source's copyright or copyright license rights in the 
Contribution, to, in whole or in part, copy, make derivative works, 
perform, display and distribute the Contribution and derivative works 
thereof consistent with 3GPP2's and each Organizational Partner's 
policies and procedures, with the right to (i) sublicense the foregoing 
rights consistent with 3GPP2's and each Organizational Partner's  policies 
and procedures and (ii) copyright and sell, if applicable) in 3GPP2's name 
or each Organizational Partner's name any 3GPP2 or transposed Publication 
even though this Publication may contain the Contribution or a derivative 
work thereof.  The Contribution shall disclose any known limitations on 
the Source's rights to license as herein provided.

When a Contribution is submitted by the Source(s) to assist the 
formulating groups of 3GPP2 or any of its Organizational Partners, it 
is proposed to the Committee as a basis for discussion and is not to 
be construed as a binding proposal on the Source(s).  The Source(s) 
specifically reserve(s) the right to amend or modify the material 
contained in the Contribution. Nothing contained in the Contribution 
shall, except as herein expressly provided, be construed as conferring 
by implication, estoppel or otherwise, any license or right under (i) 
any existing or later issuing patent, whether or not the use of 
information in the document necessarily employs an invention of any 
existing or later issued patent, (ii) any copyright, (iii) any 
trademark, or (iv) any other intellectual property right.

With respect to the Software necessary for the practice of any or 
all Normative portions of the EVRC-WB Variable Rate Speech Codec as 
it exists on the date of submittal of this form, should the EVRC-WB be 
approved as a Specification or Report by 3GPP2, or as a transposed 
Standard by any of the 3GPP2's Organizational Partners, the Source(s) 
state(s) that a worldwide license to reproduce, use and distribute the 
Software, the license rights to which are held by the Source(s), will 
be made available to applicants under terms and conditions that are 
reasonable and non-discriminatory, which may include monetary compensation, 
and only to the extent necessary for the practice of any or all of the 
Normative portions of the EVRC-WB or the field of use of practice of the 
EVRC-WB Specification, Report, or Standard.  The statement contained above 
is irrevocable and shall be binding upon the Source(s).  In the event 
the rights of the Source(s) in and to copyright or copyright license 
rights subject to such commitment are assigned or transferred, the 
Source(s) shall notify the assignee or transferee of the existence of 
such commitments.
*******************************************************************/

/*======================================================================*/
/*  4GV - Fourth Generation Vocoder Speech Service Option for             */
/*  Wideband Spread Spectrum Digital System                             */
/*  C Source Code Simulation                                            */
/*                                                                      */
/*  Copyright (C) 1999 Qualcomm Incorporated. All rights                */
/*  reserved.                                                           */
/*----------------------------------------------------------------------*/
#include "struct.h"
#include "defines.h"
#include "macro.h"
#include "proto.h"
#include "transient_consts.h"
#include "MSencode.h"
#include "GainCodebooks.h"
#include <math.h>

/**************************************************************************
 *
 *  Protection for voiced transients in case of frame erasures
 *  Inputs:
 *      short lag - pitchlag,
 *      num_syn_pk - number of peaks in the excitation
 *  Outputs:
 *       short - difference between estimated number of peaks in the excitation
 *              and (framesize/pitchlag)
 * 
 *************************************************************************/ 
short determine_fer_depfrm(short lag, short num_syn_pk)
{
    short i,estim_num_pulses;
    
    estim_num_pulses=(short)(((float)FrameSize/(float)lag)+0.5);
    i=estim_num_pulses-num_syn_pk+2;
    if(i>3) i=3;
    if(i<0) i=0;
    return(i);
}

/**************************************************************************
 *
 *  Check for mis-estimation in the number of peaks at the decoder
 *  Inputs:
 *      short lag - pitchlag,
 *      short num_syn_pk - number of peaks in the excitation
 *      short feval - reveived FER check value
 *  Outputs:
 *       short - 0 - no mis estimation or  number of actual peaks 
 * 
 *************************************************************************/ 
short check_misestim_numpks(short lag, short num_syn_pk, short feval)
{
    short i,estim_num_pulses;
    estim_num_pulses=(short)(((float)FrameSize/(float)lag)+0.5);
	if((feval==3) && (estim_num_pulses <= num_syn_pk))
    {
        return(0);
    }
    i=estim_num_pulses-(feval-2);
    if(i!=num_syn_pk) {
        
        return(i-num_syn_pk);
    }else
        return(0);

}

/**************************************************************************
 *
 *  Quantize the location of the first peak in the transient frame
 *  Inputs:
 *      int pkloc - true location of the peak
 *  Outputs:
 *      short *indx - index to indicate the location of the first peak
 * 
 *************************************************************************/ 
short quantize_loc_first_pk(int pkloc, short *indx)
{
    int j;
    *indx=0;
    for(j=0;j<8;j++) {
        if(pkloc<((j+1)*20)) {
            *indx=j;
            return((j*20)+10);
        }
    }
	return(0);
}
short dequantize_loc_first_pk(short indx)
{
    return((indx*20)+10);
}
short next_hop(float a, float b, float c)
{
    if(a>=b) {
        if(a>=c) return(0);
        else return(2);
    }else{
        if(b>=c) return(1);
        else return(2);
    }
}
float fnd_mx(float a, float b, float c, short *mxind)
{
    if(a>=b) {
        if(a>=c){ 
            *mxind=0;
            return(a);
        } else {
            *mxind=2;
            return(c);
        }
    }else{
        if(b>=c) {
        *mxind=1;
        return(b);
        } else {
            *mxind=2;
            return(c);
        }
    }
}

/**************************************************************************
 *
 *  Dynamic programming between peaks of current frame residual and synthesized excitation
 *  Inputs:
 *      int *truepks - location of peaks in the residual,
 *      short numTpks - number of peaks in the residual,
 *      short* synpks - location of peaks in the excitation,
 *      short numSpks- number of peaks in the excitation,
 *      short lag - pitchlag,
 *  Outputs:
 *       short *mapped_pks - locations of peaks in the residual for each peak in the excitation
 *
 * 
 *************************************************************************/ 
void map_peaks(int *truepks,short numTpks,short* synpks, short numSpks, short lag, short *mapped_pks)
{
    short i,j;
    short imx,jmx,mloc;
    float scoremat[10][10], scoreval,scoreval1,scoreval2,mxscore=0;
    short tp_sel[10],sp_sel[10],cnt=0, tracemat[10][10],mxind;
    for(i=0;i<10;i++) {
        for(j=0;j<10;j++){
            scoremat[i][j]=0.0;
            tracemat[i][j]=0;
        }
		mapped_pks[i]=0;
    }
    //create score matrix
    for(i=1;i<=numTpks;i++) {
        for(j=1;j<=numSpks;j++) {

            scoreval=1-(fabs((float)truepks[i-1]-(float)synpks[j-1])/(float) lag);
            if(scoreval<-1) scoreval=-1;

            scoremat[i][j]=fnd_mx(scoremat[i-1][j-1]+scoreval,scoremat[i-1][j],scoremat[i][j-1],&mxind);
            tracemat[i][j]=mxind;
            if(scoremat[i][j] > mxscore) {
                mxscore=scoremat[i][j];
                imx=i;jmx=j;
            }

        }
    }
     //traceback
    i=imx;j=jmx;cnt=0;
    while (j>0) {
        mloc=tracemat[i][j];
        switch(mloc) {
        case 0:
            tp_sel[cnt]=truepks[i-1];
            sp_sel[cnt]=synpks[j-1];
            i=i-1; 
            if(i<1) i=1;
            j=j-1;
            break;
        case 1:
            tp_sel[cnt]=truepks[i-1];
            sp_sel[cnt]=0;
            i=i-1;
            if(i<1) i=1;
            break;
        case 2:
            tp_sel[cnt]=0;
            sp_sel[cnt]=synpks[j-1];
            j=j-1;
            break;
        }
        cnt++;
    }
 
    if(cnt<0) fprintf (stderr, "warning: Mapping Error\n");
    for(i=0;i<numSpks;i++) {
        mapped_pks[i]=0;
        for(j=0;j<cnt;j++) if(sp_sel[j]==synpks[i]) break;
        if(j!=cnt) mapped_pks[i]=tp_sel[j];
    }
    for(i=1;i<numSpks;i++) {
        if(mapped_pks[i]==mapped_pks[i-1]) {
            mapped_pks[i]=0;
        }
    }

   


}

/**************************************************************************
 *
 *  Determine whether to code a frame as voiced transient or other transient
 *  Inputs:
 *      int lag -pitch lag
 *      int numpks_inframe - number of pitch spikes in current frame
 *      int prevlag - previous frame lag,
 *      float enratio - ratio of energy between previous frame and current frame,
 *      short prev_frm_mode,
 *      int last_pk -location of the last peak in present frame,
 *      int last_pk_prev -location of the last peak in previous frame,
 *      int first_pk -location of the first peak in current frame.
 *  Outputs:
 *      int - 0 - other transient
 *            1 - voiced transient
 * 
 *************************************************************************/ 
int determine_coding_mode(int lag, int numpks_inframe, int prevlag, float enratio, short prev_frm_mode, int last_pk, int last_pk_prev, int first_pk)
{
    short fcflag=0;
    short estim_numpks_inframe=0;

    estim_numpks_inframe=(short)((FrameSize/(float)lag));
    if(numpks_inframe>=estim_numpks_inframe) fcflag=1;

    if(enratio>10 || enratio <0.1) fcflag=0;
    if(last_pk > (FrameSize-((1.2*(float)lag)))&&(first_pk<1.2*(float)lag)) 
        fcflag=1;

    if(abs(lag-prevlag)>30) fcflag=0;
    if(prev_frm_mode==1 || prev_frm_mode==2) return(0);
    if(fabs(last_pk_prev+prevlag-FrameSize-first_pk)>0.6*(float)lag) return(0); //need to set this 0.6 to a lower value based on debugging.

    return(fcflag);
}
/**************************************************************************
 *
 *  Generate excitation for transient frame based on model
 * 
 *  Inputs/ Output:
 *      float *transient_residual - contains previous frame residual,
 *                  synthesiszed excitation will populate this buffer
 *  Inputs:
 *      TRANS_MODEL_PARAMS trans_mod - model parameters,
 *      short lag_prev - pitchlag of previous frame,
 *      short prev_frm_mode -last frame's mode (1- silence, 2- unvoiced,
 *                              3- voiced , 6- transient),
 *      short loc_lastpk_pf -location of the last peak in previous frame,
 *      short *end_prev_syn -location where the synthesis ended in previous frame
 *  Outputs:
 *      short *syn_pkloc -locations of peaks in synthesized excitation,
 *      short *num_syn_pk -number of peaks in synthesized excitation,
 * 
 *************************************************************************/ 

void FGV_MEM::create_syn_excitation(float *transient_residual, short *syn_pkloc, short *num_syn_pk, TRANS_MODEL_PARAMS trans_mod, short lag_prev, 
                           short prev_frm_mode, short loc_lastpk_pf, short *end_prev_syn)
{
    short trans_strt_loc, tsl_temp, int_lag;
    short cnt=0,j,i;
    short modeflag,lag, wi_len,diffloc,offset=0;
    float ssum,gf,tener,out_temp[2*FrameSize+MAXLAG_2KBPS_TRMODE],ifac, tegy;
    DTFS prev_voiced_frm, curr_trans_frm;
    float tmplsp[ORDER], lpc1[ORDER], lpc2[ORDER], ph_offset = 0.0,lagacb[3],tmpbuf[MAXLAG_2KBPS_TRMODE];
    long seedx;
    float filtiir3kN[3]={0.6518,1.2771,0.6518};
    float filtiir3kD[3]={1.0, 1.3156,0.5801};
       

    modeflag=trans_mod.indep_coding_flag;
    lag=trans_mod.pitchlag;
    diffloc=trans_mod.diffloc;

    #ifndef EXPLICIT_LEADUP_GAIN
    if(prev_frm_mode>2) {
        for(j=0, tegy=0.0;j<lag_prev; j++) {
            tegy+=(transient_residual[trans_mod.loc_prev_frm_cyc+j]*transient_residual[trans_mod.loc_prev_frm_cyc+j]);
        }
        adj_cycle_egy(trans_mod.prototype, trans_mod.prototype,tegy,trans_mod.pitchlag);
    }
    #endif


    if(modeflag==1) { // code the frame as an extension of the past
        tsl_temp=loc_lastpk_pf+lag_prev-FrameSize;
        int_lag=lag_prev;
#ifndef TRANS_INDEP_MODEL_SYN_BUG_FIX
        while(tsl_temp < FrameSize) {
#else
        while(tsl_temp-trans_mod.diffloc < FrameSize) {
#endif
            if(tsl_temp >= 0) {
                syn_pkloc[cnt]=tsl_temp;
                trans_strt_loc=tsl_temp; //tsl has the location of the last cycle.
                
                cnt++;
                num_syn_pk[0]=cnt;
                ifac=((float)tsl_temp+0.5*(float)trans_mod.pitchlag)/(float) FrameSize;
                if(ifac>1.0) ifac=1.0;
                    
                int_lag=(short)(ifac*(float)lag +(1-ifac)*(float)lag_prev); //needs adjustment
            }
            tsl_temp=tsl_temp+int_lag;
        }
        tsl_temp=trans_strt_loc-diffloc;

        if(tsl_temp<0) {
            tsl_temp=0;
            fprintf (stderr, "warning: Negative Transient Start Location\n");
        }

        for(j=0;j<lag;j++){ //put the last pitch cycle.
            transient_residual[FrameSize+tsl_temp+j]=trans_mod.prototype[j];
        }


    }else{
        tsl_temp=trans_mod.loc_first_pk-trans_mod.diffloc;
        if(tsl_temp<0) {
            offset=-tsl_temp;
            tsl_temp=0;
        }else
            offset=0;
        trans_strt_loc=tsl_temp;
        cnt=0;
        while (tsl_temp-offset < FrameSize) {
            syn_pkloc[cnt]=tsl_temp+trans_mod.diffloc-offset;
            cnt++;
            num_syn_pk[0]=cnt;
            for (j=0;j<trans_mod.pitchlag;j++, tsl_temp++)
                transient_residual[FrameSize+tsl_temp]=trans_mod.prototype[(j+offset)%trans_mod.pitchlag];
        }
        tsl_temp=trans_strt_loc;
        
    }
    if(prev_frm_mode>2) {
        Interpol (tmplsp, trans_mod.plsp, trans_mod.plsp, 2, ORDER);
        lsp2a (lpc1, tmplsp, ORDER);
    
        Interpol (tmplsp, trans_mod.lsp, trans_mod.plsp, 2, ORDER);
        lsp2a (lpc2, tmplsp, ORDER);
    
        #ifdef EXPLICIT_LEADUP_GAIN
        for(j=0;j<lag_prev;j++) tmpbuf[j]=transient_residual[trans_mod.loc_prev_frm_cyc+j];
        normalize_cycle(tmpbuf,tmpbuf,lag_prev);

        prev_voiced_frm.to_fs(tmpbuf, lag_prev);
        curr_trans_frm.to_fs(&transient_residual[FrameSize+tsl_temp],lag);

        #else

        prev_voiced_frm.to_fs(&(transient_residual[trans_mod.loc_prev_frm_cyc]), lag_prev);
        curr_trans_frm.to_fs(&transient_residual[FrameSize+tsl_temp],lag);

        #endif

        wi_len=tsl_temp-(trans_mod.loc_prev_frm_cyc+lag_prev-FrameSize)+lag;
        if(wi_len>lag) {
            WIsyn (prev_voiced_frm, &curr_trans_frm, lpc1, lpc2, &ph_offset,out_temp, wi_len);
            for(j=trans_mod.loc_prev_frm_cyc+lag_prev,i=0;i<wi_len-lag;j++,i++) transient_residual[j]=out_temp[i];
        }
    }else{
        lagacb[0]=lagacb[1]=lagacb[2]=(float)120;
        #ifdef EXND_DEP_FRM_PAST
            putacbc(&(transient_residual[FrameSize]),&(transient_residual[FrameSize]), 0,tsl_temp,0,lagacb,(float)0.9,8);
        #else
        seedx=(long)(trans_mod.lsp[2]*10000);
        for(j=0;j<tsl_temp;j++) {
            tmpbuf[j]=ran0(&seedx)-0.5;
        }
        double memtemp[2]={0,0};
        struct POLEZERO_FILTER pzfm={2,2,2,0,memtemp};
        polezero_filter(tmpbuf,transient_residual+FrameSize,tsl_temp,filtiir3kN,filtiir3kD,pzfm);
        
        #endif
        
    }
  
    if(modeflag==0 && (*end_prev_syn > (FrameSize+syn_pkloc[0]-trans_mod.diffloc))) 
            *end_prev_syn=(FrameSize+syn_pkloc[0]-trans_mod.diffloc);
        


}
/**************************************************************************
 *
 *  Variable dimensional target pitch cycle energy quantization for transient frames
 *  Inputs:
 *      float *gains - input, unquantized "dim" dimensional target pitch cycle energy vector
 *      short dim - number of gains in vector
 *      short *qindx - quanitzation indices.
 *  Output:
 *      float *q_gains - quantized "dim" dimensional target pitch cycle energy vector
 * 
 *************************************************************************/ 
void quantize_vardim_gains(float *gains, float *q_gains, short dim, short *qindx)
{

    float *GainCB, log_gains[10], qlog_gains[10], weight[10]={1,1,1,1,1,1,1,1,1,1};
    short i;

    switch(dim) {
    case 1:
        GainCB=GainCB1;
        break;
    case 2:
        GainCB=GainCB2;
        break;    
    case 3:
        GainCB=GainCB3;
        break;
    case 4:
        GainCB=GainCB4;
        break;
    case 5:
        GainCB=GainCB5;
        break;
    case 6:
        GainCB=GainCB6;
        break;
    case 7:
        GainCB=GainCB7;
        break;
    case 8:
        GainCB=GainCB8;
        break;
    case 9:
        GainCB=GainCB9;
        break;
    }
    for(i=0;i<dim;i++) log_gains[i]=10*log10(gains[i]+0.0001);
    singlevectortest(log_gains,dim,256,qindx,weight,qlog_gains,GainCB);

    for(i=0;i<dim;i++) q_gains[i]=pow(10,(qlog_gains[i])/10);
    

}

/**************************************************************************
 *
 *  Variable dimensional target pitch cycle energy codebook lookup for transient frames
 *  Inputs:
 *      short dim - number of gains in vector
 *      short qindx - received quanitzation indices.
 *  Output:
 *      float *q_gains - quantized "dim" dimensional target pitch cycle energy vector
 * 
 *************************************************************************/ 
void dequantize_vardim_gains(float *q_gains, short dim, short qindx)
{

    float *GainCB, log_gains[10], qlog_gains[10], weight[10]={1,1,1,1,1,1,1,1,1,1};
    short i;

    switch(dim) {
    case 1:
        GainCB=GainCB1;
        break;
    case 2:
        GainCB=GainCB2;
        break;    
    case 3:
        GainCB=GainCB3;
        break;
    case 4:
        GainCB=GainCB4;
        break;
    case 5:
        GainCB=GainCB5;
        break;
    case 6:
        GainCB=GainCB6;
        break;
    case 7:
        GainCB=GainCB7;
        break;
    case 8:
        GainCB=GainCB8;
        break;
    case 9:
        GainCB=GainCB9;
        break;
	default:
		GainCB=GainCB1;
		break;
    }
    SingleStageDecode(qindx,dim,qlog_gains,GainCB);

    for(i=0;i<dim;i++) q_gains[i]=pow(10,(qlog_gains[i])/10);
    

}
/**************************************************************************
 *
 *  Decode variable dimensional target pitch cycle energies for transient frames
 *  Input/ Output:
 *      TRANS_MODEL_PARAMS *trans_model - gains_PS field updated,
 *  Inputs
 *      short numsynpks -number of synthesized peaks,
 *      short adj - parameter for FER protection. Prevents bad rate.
 *************************************************************************/ 
void decode_gains_transient(TRANS_MODEL_PARAMS *trans_model, short numsynpks, short adj)
{
    float qtse[10];
    short i;

    if(trans_model->indep_coding_flag==1 || trans_model->leadupflag ==0) {
        dequantize_vardim_gains(qtse,numsynpks+adj,trans_model->gain_indx);
        if(adj>=0) {
            for(i=0;i<numsynpks;i++) trans_model->gains_PS[i]=qtse[i]*trans_model->pitchlag;
        }else{
            for(i=0;i<numsynpks+adj;i++) trans_model->gains_PS[i]=qtse[i]*trans_model->pitchlag;
            for(i=numsynpks+adj;i<numsynpks;i++) trans_model->gains_PS[i]=trans_model->gains_PS[i-1];
        }
    }else{
        dequantize_vardim_gains(qtse,numsynpks+1,trans_model->gain_indx);
        trans_model->leadEInd=qtse[0]*trans_model->pitchlag;
        for(i=0;i<numsynpks;i++) trans_model->gains_PS[i]=qtse[i+1]*trans_model->pitchlag;
    }
    
}

/**************************************************************************
 *
 *  Estimate target pitch cycle energy in speech domain
 *  Inputs:
 *      float *speech - input speech signal (after regularization)
 *      float *excitation - input and output signal
 *      short *syn_pkloc - location of peaks in the excitation signal,
 *      short *numSpk - number of peaks in the synthesized excitation
 *      short* mapped_pks - mapping between peaks in original residual and synthesized excitation
 *      TRANS_MODEL_PARAMS * trans_model - other parameters needed for synthesis,
 *      float *memory - synthesis memory,
 *      short end_prev_syn - location where last frame's gain scaling ended
 *
 *************************************************************************/   
void estimate_gains(float *speech, float *excitation, short* syn_pkloc, short *numSpks, short* mapped_pks, TRANS_MODEL_PARAMS *trans_model, 
                    short sp_diffloc, float *memory, float *residual, short end_prev_syn)
{
    short i,j,k,len, strtloc, gqindx;
    float tse[10],tse_tmp[10], qtse[10];
    float leadupE=0.0;
    short TransExplLUP=0,dcnt=0;

	for(i=0;i<10;i++) tse[i]=0;

    for(i=0;i<numSpks[0];i++) {
        tse[i]=0.0;
        if(mapped_pks[i]!=0) {
            strtloc=mapped_pks[i];
            if(i == numSpks[0]-1 || mapped_pks[i+1] == 0) {
                if(strtloc+trans_model->pitchlag < (FrameSize+LOOKAHEAD_LEN)) 
                    len=trans_model->pitchlag;
                else
                    len=(FrameSize+LOOKAHEAD_LEN)-strtloc+1;
            }else{
                len=mapped_pks[i+1]-mapped_pks[i]+1;
            }
            for(j=0;j<len;j++) tse[i]+=(speech[strtloc+j-sp_diffloc]*speech[strtloc+j-sp_diffloc]); //the index can go negative...will just look back into the buffer
            #ifdef SAMP_NORM_EN
            tse[i]=tse[i]/len;
            #endif
        }
    }
   strtloc+=trans_model->pitchlag;

    for(i=0;i<numSpks[0];i++) {
        if(tse[i]==0.0) {
            if(i==0) {
                tse[i]=0.5*tse[i+1];
            } else{
                
                short interpFlag=0;
                for(j=i;j<numSpks[0];j++) {
                    if(tse[j]>0) interpFlag=1;
                }
                if(interpFlag==1) {
                    if(i==numSpks[0]-1){ 
                        tse[i]=tse[i-1];
                    } else{
                        tse[i]=0.5*(tse[i-1]+tse[i+1]);
                    }
                }else{
                    if(strtloc+trans_model->pitchlag < FrameSize) {
                        len=trans_model->pitchlag;
                        for(j=0;j<len;j++) tse[i]+=(speech[strtloc+j-sp_diffloc]*speech[strtloc+j-sp_diffloc]); 
                        #ifdef SAMP_NORM_EN
                        tse[i]=tse[i]/len;
                        #endif
                        strtloc+=trans_model->pitchlag;
                    }
                    else
                        tse[i]=0.25*tse[i-1];
                }


            }
        }
    }
    len=end_prev_syn-FrameSize;
    
    if(syn_pkloc[0]-trans_model->diffloc-len > (trans_model->pitchlag >> 1)) trans_model->leadupflag =1;
    else trans_model->leadupflag =0;


    for(leadupE=0.0,j=len;j<mapped_pks[0]-sp_diffloc;j++) leadupE+=(speech[j]*speech[j]); //-sp_diffloc
    

   

    if(trans_model->indep_coding_flag==1 || trans_model->leadupflag ==0) { 
        dcnt=numSpks[0];
        for(i=0;i<numSpks[0];i++) {
            tse_tmp[i]=((tse[i]/trans_model->pitchlag));
        }
    }else{
        dcnt=numSpks[0]+1;
        tse_tmp[0]=(leadupE/trans_model->pitchlag);
        for(i=1;i<numSpks[0]+1;i++) {
            tse_tmp[i]=((tse[i-1]/trans_model->pitchlag));
        }
    }


     //quantize the gains
     quantize_vardim_gains(tse_tmp,qtse,dcnt,&gqindx);
     trans_model->gain_indx=gqindx;
//printf("numpk=%d gain dim=%d ",numSpks[0],dcnt);
//for(i=0; i<dcnt; ++i) printf("%d %d ",(int)tse_tmp[i],(int) qtse[i]); printf("\n");

      if(trans_model->indep_coding_flag==1 || trans_model->leadupflag ==0) {
          for(i=0;i<numSpks[0];i++) trans_model->gains_PS[i]=qtse[i]*trans_model->pitchlag;
      }else{
          trans_model->leadEInd=qtse[0]*trans_model->pitchlag;
          for(i=0;i<numSpks[0];i++) trans_model->gains_PS[i]=qtse[i+1]*trans_model->pitchlag;
      }


}
/**************************************************************************
 *
 *  Apply target pitch cycle energy to a residual pitch cycle and synthesize in speech domain
 *  Inputs:
 *      float *excitation - input and output signal
 *      short *syn_pkloc - location of peaks in the excitation signal,
 *      short num_syn_pk - number of peaks in the synthesized excitation
 *      TRANS_MODEL_PARAMS * trans_model - other parameters needed for synthesis,
 *      float *memory - synthesis memory,
 *      short end_prev_syn - location where last frame's gain scaling ended
 *
 *  Output:
 *      float *sptemp - synthesised speech,
 *
 *************************************************************************/   
void apply_gains(float *excitation,float *sptemp, short *syn_pkloc, short num_syn_pk, TRANS_MODEL_PARAMS * trans_model,
                        float *memory, short end_prev_syn, short lag_prev)  
{
    short i,j,k,len, strtloc, offset,sloc,gqindx;
    float pci[10],lspi[10],zir[FrameSize],zsr[FrameSize], tse[10],tse_tmp[10], qtse[10];
    float intfac, mem_temp[10],leadupE=0.0;
    short TransExplLUP=0,dcnt=0;
    float e1,g1,g2,pg,eng;


    len=end_prev_syn-FrameSize;
    
    lsp2a(pci,trans_model->plsp,10);
    len=end_prev_syn-FrameSize;
    if(len>0) {
        SynthesisFilter(sptemp,excitation,pci,memory,10,len);
    }

   if(trans_model->indep_coding_flag==0 && ((syn_pkloc[0]-trans_model->diffloc)>(end_prev_syn-FrameSize))) {
        
        lsp2a(pci,trans_model->plsp,10);

        strtloc=end_prev_syn-FrameSize;
        len=(syn_pkloc[0]-trans_model->diffloc)-strtloc;
        
        if(trans_model->leadupflag==1) {
            //first, scale the exictation by an approximate gain
            for(j=0,e1=0.0;j<len;j++) e1+=(excitation[strtloc+j]*excitation[strtloc+j]);
    
            zir[0]=1.0; for(j=1;j<len;j++) zir[j]=0.0;
            for(j=0;j<10;j++) mem_temp[j]=0.0;
            SynthesisFilter(zsr,zir,pci,mem_temp,10,len);
            for(j=0,pg=0.0;j<len;j++) pg+=(zsr[j]*zsr[j]);
    
            g1=sqrt((trans_model->leadEInd)/((e1*pg)+0.0001));
            for(j=0;j<len;j++) excitation[strtloc+j]=excitation[strtloc+j]*g1;
    
            for(j=0;j<10;j++) mem_temp[j]=memory[j];
            SynthesisFilter(sptemp+strtloc,excitation+strtloc,pci,mem_temp,10,len);
    
            for(j=0,e1=0.0;j<len;j++) {
                e1+=(sptemp[strtloc+j]*sptemp[strtloc+j]);
            }
            g2=sqrt((trans_model->leadEInd)/(e1+0.00001));
            
            for(j=0;j<len;j++) excitation[strtloc+j]=excitation[strtloc+j]*g2;
        }
        
        SynthesisFilter(sptemp+strtloc,excitation+strtloc,pci,memory,10,len);
        

     }

     for(i=0;i<num_syn_pk;i++) {
        // interpolate the lsps
        intfac=((float)syn_pkloc[i]+0.5*(float)trans_model->pitchlag)/(float)FrameSize;
        if(intfac>1.0) intfac=1.0;

        Interpol_fac(lspi,trans_model->plsp,trans_model->lsp,intfac,10);
        lsp2a(pci,lspi,10);
        
        strtloc=syn_pkloc[i]-trans_model->diffloc;
        if(strtloc<0) {
            offset=-strtloc;
            strtloc=0;
        }else
            offset=0;


        if(i<num_syn_pk-1) {
            len=(syn_pkloc[i+1]-syn_pkloc[i])-offset;
        } else { 
            //len=trans_model->pitchlag;
			if(trans_model->indep_coding_flag == 1)
				len=(short)(intfac*(float)trans_model->pitchlag +(1-intfac)*(float)lag_prev); //needs adjustment
			else
				len=trans_model->pitchlag;
        }
        eng=(trans_model->gains_PS[i]/trans_model->pitchlag)*len;

        //first, scale the exictation by an approximate gain
        for(j=0,e1=0.0;j<len;j++) {
            e1+=(excitation[strtloc+j]*excitation[strtloc+j]);
        }
        //calculate the impusle response of the synthesis filter

        zir[0]=1.0; for(j=1;j<len;j++) zir[j]=0.0;
        for(j=0;j<10;j++) mem_temp[j]=0.0;
        SynthesisFilter(zsr,zir,pci,mem_temp,10,len);
        for(j=0,pg=0.0;j<len;j++) pg+=(zsr[j]*zsr[j]);       

        #ifdef SAMP_NORM_EN
        g1=sqrt((trans_model->gains_PS[i]*len)/((e1*pg)+0.0001));
        #else
        g1=sqrt(eng/((e1*pg)+0.0001));
        #endif
        for(j=0;j<len;j++) excitation[strtloc+j]=excitation[strtloc+j]*g1;

        
        for(j=0;j<10;j++) mem_temp[j]=memory[j];
        //synthesize the speech with the temporary memory
        SynthesisFilter(&(sptemp[strtloc]),&(excitation[strtloc]),pci,mem_temp,10,len);

        for(j=0,e1=0.0;j<len;j++) {
            e1+=(sptemp[strtloc+j]*sptemp[strtloc+j]);
        }
        #ifdef SAMP_NORM_EN
        g2=sqrt((trans_model->gains_PS[i]*len)/(e1+0.00001));
        #else
        g2=sqrt(eng/(e1+0.00001));
        #endif
        
        for(j=0;j<len;j++) excitation[strtloc+j]=excitation[strtloc+j]*g2;

        //resynthesize for proper memory update
        if(strtloc+len<FrameSize) {
            SynthesisFilter(&(sptemp[strtloc]),&(excitation[strtloc]),pci,memory,10,len);
        }else{
            len=FrameSize-strtloc;
            SynthesisFilter(&(sptemp[strtloc]),&(excitation[strtloc]),pci,memory,10,len);
        }
     }
    
}

/**************************************************************************
 *
 *  Constructor for the transient mode class
 *  
 *************************************************************************/


TRANSMODE::TRANSMODE() 
{
    int i,j;
    for (i=0;i<10;i++){
        pitchspikes[i]=0;
        pitchspikes_inframe[i]=0;
    }
    numpeaks=0;
    numpeaks_inframe=0;
    for(i=0;i<(FrameSize+LOOKAHEAD_LEN);i++) {
        resid_signal[i]=0.0;
        inCw[i]=0.0; //holds the smoothed out residual.
    }
    for(i=0;i<FrameSize;i++) pitchproto[i]=0.0;
    last_pitchspike=(FrameSize+LOOKAHEAD_LEN);
    inSpM=0.0; //sparseness measure of the residual
}

/**************************************************************************
 *
 *  Normalize the energy of a vector of length "len" to 1
 *  Inputs:
 *      float *input - pointer to input
 *      short len    - length of input vector
 *
 *  Output:
 *      float* outp - output vector, normalized to unit energy
 *
 *************************************************************************/
void normalize_cycle(float *input, float *outp, short len)
{
    int j;
    double ssum=0.0;
    for(j=0;j<len;j++) ssum+=(input[j]*input[j]);
    if(ssum==0.0) {
        for(j=0;j<len;j++) outp[j]=input[j];
        return;
    }
    ssum=1/sqrt(ssum);
    for(j=0;j<len;j++) outp[j]=input[j]*ssum;

}

/**************************************************************************
 *
 *  Adjust the energy of a vector of length "len" to target energy
 *  Inputs:
 *      float *input - pointer to input
 *      short len    - length of input vector
 *      float tar_egy - the value to which the energy of input needs to be
 *                      adjusted
 *
 *  Output:
 *      float* outp - output vector, normalized to tar_egy energy
 *
 *************************************************************************/
void adj_cycle_egy(float *input, float *outp, float tar_egy,short len)
{
    int j;
    double ssum=0.0;
    for(j=0;j<len;j++) ssum+=(input[j]*input[j]);
    if(ssum==0.0) {
        for(j=0;j<len;j++) outp[j]=input[j];
        return;
    }
    ssum=sqrt(tar_egy/ssum);
    for(j=0;j<len;j++) outp[j]=input[j]*ssum;

}

int max_val(int i, int j)
{
    if(i>=j) return(i); else return(j);
}
int min_val(int i, int j)
{
    if(i<=j) return(i); else return(j);
}


/**************************************************************************
 *
 *  Convolve two signals and align them 
 *  Inputs:
 *      float *a    - first input vector
 *      int lena    - length of first input vector
 *      float *b    - second input vector
 *      int lenb    - length of second input vector
 *      
 *  Output:
 *      float* cresult - output vector
 *
 *************************************************************************/
void conv_and_adj(float *a, int lena, float *b, int lenb, float *cresult)
{
    int i,j;
    int sp=(short)(lenb/2);
    for(i=0;i<sp;i++) {
        cresult[i]=0.0;
        for(j=0;j<sp+i;j++) {
            cresult[i]+=(a[j]*b[sp-i+j]);
        }
    }
    for(i=sp;i<lena;i++) {
        cresult[i]=0.0;
        for(j=0;j<lenb;j++) {
            cresult[i]+=(a[i-sp+j]*b[j]);
        }
        
    }
}

/**************************************************************************
 *
 *  Calculate the normalized cross correlation factor between two vectors
 *  Inputs:
 *      float *a    - first input vector
 *      float *b    - second input vector
 *      int len    - length of input vector
 *      
 *  Output:
 *      float    - normalized cross correlation factor
 *
 *************************************************************************/
float conv_pc(float *a, float *b, int len)
{
    float *tmpa, *tmpb, cc;
    float sa, sb;
    int i,j,k;
    for(cc=0.0,sa=0.0,sb=0.0,j=0;j<len;j++) {
        cc+=(a[j]*b[j]);
        sa+=(a[j]*a[j]);
        sb+=(b[j]*b[j]);
    }
    return(cc/pow((double)sa*(double)sb, 0.5));
}

/**************************************************************************
 *
 *  Normalize vector so that largest value in vector is unit amplitude
 *  Input/Output:
 *      float *bufr    - input output vector
 *      int len    - length of input/output vector
 *      
 *
 *************************************************************************/
float TRANSMODE::Normalize_maxAmp(float *bufr, int len)
{
    float mxval=0;
    int j;
    for(j=0;j<len;j++) if(fabs(bufr[j])>mxval)   mxval=fabs(bufr[j]);
    for(j=0;j<len;j++) bufr[j]=bufr[j]/mxval;
    return(mxval);
    
}

/**************************************************************************
 *
 *  Calculate the difference between successive peaks and the interpeak
 *  correlation
 *  Class members affected:
 *      pdiff    - differences between the locations of successive peaks
 *      interpkcorr - correlation between successive peaks
 *      rat -  ratio between the mean and the variance of the pdiff
 *
 *************************************************************************/
float TRANSMODE::SpikeDiff_Rel()
{
    int i,j;
    float sm, mn, rat;
    if(numpeaks<=1) return(0);
    for(j=0,mn=0;j<numpeaks-1;j++) {
        pdiff[j]=pitchspikes[j+1]-pitchspikes[j];
        mn+=(float)pdiff[j];
        interpkcorr[j]=find_corr_pkpairs(pitchspikes[j],pitchspikes[j+1]);
    }
    mn=mn/(numpeaks-1);
    for(j=0,sm=0;j<numpeaks-1;j++) sm+=(pdiff[j]-mn)*(pdiff[j]-mn);
    sm=pow((sm/(numpeaks-1)),0.5);
    rat=sm/mn;
    return(rat);
}


/**************************************************************************
 *
 *  Calculate a measure of sparseness on the input vector
 *  Inputs:
 *      float *signl    - input vector
 *      int len    - length of input vector
 *      
 *  Output:
 *      float    - sparseness measure, close to zero means less sparse, close
 *              to 1 means very sparse.
 *
 *************************************************************************/
float TRANSMODE::GetSparsenessMeasure(float *signl, int len)
{
    float temp=0.0;
    int flag=1, i,j,jmax,cnt=0;
    float enerincl=0.0, tener=0.0,mxval=0.0;
    float *sigener;
    sigener=(float *) malloc(len*sizeof(float));
    for(tener=0.0,j=0;j<len;j++){
       sigener[j]=(signl[j]*signl[j]);
       tener+=sigener[j];
    }

    for(i=0;i<len;i++) {
        if(flag==1) {
            //locate the largest remaining amplitude
            mxval=0.0;jmax=0;
            for(j=0;j<len;j++) {
                if(sigener[j]>=mxval) {
                    mxval=sigener[j];
                    jmax=j;
                }
            }
            sigener[jmax]=0.0;
            enerincl+=mxval;
            cnt++;
            if((enerincl/tener)> SPRSNSS_TH) flag=0;
        }
    }
    temp=(float)(len-cnt)/(float)len;
    free(sigener);
    return(temp);
}

/**************************************************************************
 *
 *  Find peaks in a vector of signal contour (inCw, which is a member of the
 *  class)
 * 
 *  Inputs:
 *      int srchrange   - minimum distance between successive peaks.
 *      float betap -   pitch gain for the present frame
 *      float len - signal length
 *      short pdelay - previous frame pitch lag
 *      short delay - estimate of current frame pitch
 *      
 *  Class members affected:
 *      pitchspikes - list of peaks in the current frame and lookahead
 *      pitchspikes_inframe - list of peaks in the current frame
 *      numpeaks - number of peaks in the current frame and lookahead
 *      numpeaks_inframe - number of peaks in the current frame
 * 
 *************************************************************************/
int TRANSMODE::PeakPicking(int srchrange_in, float betap, int len, short pdelay, short delay)
{
    int i,j,jsel;
    float *delta1, *delta2,pkmx,mxpkval=0, pk_elim_th=0.25,pkstd, proxth,pkrat,proxx;
    short cont_srch_flag=1,num_attmpts=0,incflag=0, prox, keep_dec;
    int pktmp[100],pkcnttmp=0,pktmp1[100],pkcnttmp1=0,savepk=0,mxpk_loc, secpk;
    short keepflagD=0,keepflagU=0;
    delta1=(float *)malloc(len*sizeof(float));
    delta2=(float *)malloc(len*sizeof(float));
	int srchrange;
	srchrange = srchrange_in;
    
    for(i=1;i<len;i++)delta1[i]=inCw[i]-inCw[i-1]; delta1[0]=0.0;
    delta2[len-1]=0.0;
    for(i=0;i<len-1;i++)delta2[i]=delta1[i+1]-delta1[i];
    

    while(cont_srch_flag==1&& num_attmpts<2) {
        pkcnttmp=0; 
        num_attmpts++;
        mxpk_loc=0;
        mxpkval=0;
        for(i=0;i<len-1;i++){
            incflag=0;
             
            if(delta2[i]<=-0.01*nrmval){ //potential peak
                pkmx=0.0;
                jsel=-320;
                for(j=max_val(0,i-srchrange);j<min_val(len,i+srchrange);j++){
                    if(inCw[j]>=pkmx){
                        pkmx=inCw[j];
                        jsel=j;
                    }
                }
         
                if(jsel==i|| incflag==1){
                    pktmp[pkcnttmp]=i;
                    pkcnttmp++;

                    if(inCw[i]>mxpkval&& i<200){ //this used to have "&& i<FrameSize"...but eliminated because it picks up too many spurious peaks. Whether a value is a peak or not has to be seen in the context of the entire frame and the lookahead.
                        mxpkval=inCw[i];
                        mxpk_loc=i;
                    }
                }
            }
        }
        if(mxpk_loc>200) mxpkval=0.8*mxpkval;

        if(pkcnttmp>=target_num_pulses) {
            cont_srch_flag=0;
        }
        else{
            srchrange=(int)(0.8*(float)srchrange);
        }

    }
    if(mxpkval==0.0) { //no peaks were found within the frame boundary
        numpeaks=0;
        numpeaks_inframe=0;
        return(0);
    }
    float spkval=0;			//Find second largest peak
    secpk=0;
    for(i=0;i<pkcnttmp;i++) {
        if(inCw[pktmp[i]]>spkval && inCw[pktmp[i]] < mxpkval) {
            spkval=inCw[pktmp[i]];
            secpk=pktmp[i];
        }
    }

   if(delay<20) { //unreliable pitchlag estimate
       pitchlag=max_val(abs(mxpk_loc-secpk),20);
       delay=pdelay=pitchlag;
   }
 

    cont_srch_flag=1,num_attmpts=0;
    //eliminate peaks below set% of the max val with in the frame
    pkcnttmp1=0;pkmx=0.0;
    for(i=0;i<pkcnttmp;i++){
        if(betap>0.2) {
            float lagspan;
            lagspan=((float)abs(mxpk_loc-pktmp[i]))/(float)delay;
            lagspan=fabs(lagspan-(float)((short)(lagspan+0.5)));
            if(lagspan>0.8 || lagspan< 0.2) {
                if(lagspan>0.98 || lagspan<0.02) {
                    pk_elim_th=0.1;
                }else{
                    pk_elim_th=0.25;  //setting at 0.25 avoids spurious peaks (see frm 842)
                }
            }else
                pk_elim_th=0.4;  //setting it to 0.4 to avoid spurious peaks (see frame 389)
        }else
            pk_elim_th=0.25;

        
        if(pkcnttmp1>0) {								//keep_dec 0 don't keep the peak, if 1 then replace the last peak with the current one and 2 keep the current peak
            prox=pktmp[i]-pktmp1[pkcnttmp1-1];
            proxth=(float)prox/(float)delay;
            pkrat=(inCw[pktmp1[pkcnttmp1-1]]/inCw[pktmp[i]]);
           
            if(proxth<0.25) { //keep one of two peaks
                if(inCw[pktmp1[pkcnttmp1-1]]>= inCw[pktmp[i]]) { //eliminate present peak
                    keep_dec=0;
                }else
                    keep_dec=1;
            }
            if(proxth>=0.25 && proxth <0.7) {
                pkrat=(inCw[pktmp1[pkcnttmp1-1]]/inCw[pktmp[i]]) ;
                if(pkrat>2){ 
                    keep_dec=0;
                }else{
                    if(pkrat<0.5) keep_dec=1;
                    else keep_dec=2;
                }
            }
            if(proxth>0.7) keep_dec=2;
        }else
            keep_dec=2;
            

        if(inCw[pktmp[i]] > pk_elim_th*mxpkval && keep_dec!=0) {
            if(keep_dec==2) {
                pktmp1[pkcnttmp1]=pktmp[i];
                pkcnttmp1++;
            }else{
                pktmp1[pkcnttmp1-1]=pktmp[i];
            }

        }


    }

    
    
    //eliminate peaks that are very close to each other
    pkcnttmp=pkcnttmp1;
    for(i=0;i<pkcnttmp1;i++) pktmp[i]=pktmp1[i];
    
	//Alok
	//Eliminate false peak
	int loc_diff[100], good_peaks, elim_flag[100], elim_num,elimination_performed;
	elimination_performed=1;
	while(elimination_performed==1){
	for(i=0; i<pkcnttmp-1; i++) loc_diff[i]= pktmp[i+1]-pktmp[i];
	good_peaks=0;
	for(i=0; i<pkcnttmp-2; i++) if(abs(loc_diff[i]-delay) <= 2 && abs(loc_diff[i+1]-delay) <= 2) good_peaks=1;
	for(i=0; i<pkcnttmp; i++) elim_flag[i]=0; 
	elimination_performed=0;
	if(good_peaks == 1 && betap>= 0.4){
		for(i=0; i<pkcnttmp-1; i++){
			if(elimination_performed == 0){
			if(abs(loc_diff[i]-delay) >= 0.25*delay){
				if(i==pkcnttmp-2) elim_flag[i+1]=1;	
				else{
				if(abs(loc_diff[i+1]-delay) >= 0.25*delay) elim_flag[i+1]=1;
				else elim_flag[i]=1;
				}
				elimination_performed=1;
			}
		}
		}
	}
	if(elimination_performed==1){
	pkcnttmp1=0;
	for(i=0; i<pkcnttmp; i++){ 
		if(elim_flag[i] == 0){
		 pktmp1[pkcnttmp1]=pktmp[i];
		 ++pkcnttmp1;
		}
	}
    pkcnttmp=pkcnttmp1;
    for(i=0;i<pkcnttmp1;i++) pktmp[i]=pktmp1[i];
	}
	}

	
	//return peaks;
    numpeaks=pkcnttmp;
    if(numpeaks>8) {
        numpeaks=8;
    }
    numpeaks_inframe=0;
    for(i=0;i<numpeaks;i++) {
        pitchspikes[i]=pktmp[i];
        if(pktmp[i]<FrameSize) {
            pitchspikes_inframe[numpeaks_inframe]=pktmp[i];
            numpeaks_inframe++;
        }
    }
       
    free(delta1);
    free(delta2);

    return 0;
}

/**************************************************************************
 *
 *  Preprocess input signal and call function to estimate peaks
 * 
 *  Inputs:
 *      float *resid_buffer - input residual
 *      short pdelay - previous frame pitch lag
 *      short delay - estimate of current frame pitch
 *      float len - signal length
 *      short prev_frm_mode - last frame's mode (1- silence, 2- unvoiced,
 *                              3- voiced , 6- transient);
 *      float fe_rat - ratio between energy of previous frame and present frame
 *      float *beta - pitch gain for current frame
 *      
 *  Class members affected:
 *      inCw - hold the energy contour of the residuals of the current frame
 *      pitchspikes - list of peaks in the current frame and lookahead
 *      pitchspikes_inframe - list of peaks in the current frame
 *      numpeaks - number of peaks in the current frame and lookahead
 *      numpeaks_inframe - number of peaks in the current frame
 * 
 *************************************************************************/
void TRANSMODE::get_signal_peaks(float *resid_buffer,short delay, short pdelay, short len,short prev_frm_mode, float fe_rat, float *beta)
{
    int i;
    //float nrmval;
    int srchRange, spflag=0;
    float absresid[(FrameSize+LOOKAHEAD_LEN)+8];
    
    pitchlag=delay;
    for(i=0;i<len;i++) {
        resid_signal[i]=resid_buffer[i];
        absresid[i]=fabs(resid_buffer[i]);
    }
    for(i=len;i<(FrameSize+LOOKAHEAD_LEN+8);i++) absresid[i]=0.0;//additional length for convolution

    conv_and_adj(absresid,len,gausswin8,8, inCw);
    //nrmval=Normalize_maxAmp(inCw,FrameSize);
    nrmval=0;
	for(i=0;i<FrameSize;i++) if(fabs(inCw[i])>nrmval)   nrmval=fabs(inCw[i]);
	//if(nrmval != 0) for(i=0;i<len;i++) inCw[i]=inCw[i]/nrmval;
    inSpM=GetSparsenessMeasure(resid_buffer,FrameSize);

    
    int pl_temp=pitchlag;
    if(fabs(pdelay-delay)>15) 
        pl_temp=min_val(pdelay, delay);

    if(pl_temp>127) pl_temp=127;
    if(pl_temp < 120 || (pl_temp>120 && inSpM >0.7) ){ //if the pitchlag is very large, then the frame has likely unvoiced characteristics, especially if the inSpM is < 0.7
        if(inSpM>0.7)
            srchRange=(int)(0.35*(float)pl_temp);
        else{
            srchRange=(int)(0.4*(float)pl_temp);
            spflag=0;
        }
    }else{
        srchRange=20;
        spflag=1;
    }

    if(inSpM>0.9 && pitchlag==127)
        srchRange=(pitchlag >> 2);

    if(prev_frm_mode>2 && fe_rat>0.9) { // the previous frame was voiced and the present frame is almost equal in energy
        target_num_pulses=(int)((float) (FrameSize+LOOKAHEAD_LEN)/(float)delay);
    }else
        target_num_pulses=0;
    
    if(prev_frm_mode==3) { //previous was voiced, so we need to keep peaks that are related to pitchlag
       pk_srch_strt=last_pk_prev_frm-FrameSize;
    }else
        pk_srch_strt=0;

    if(inSpM>0.75 && *beta < 0.1 && prev_frm_mode>2) {
        pitchlag=pdelay;
        *beta=0.21;
    }


    if(inSpM>0.7 && *beta > 0.6) {
        srchRange=(short)(0.5*(float)srchRange);
    }

    PeakPicking(srchRange, *beta, len, pdelay, pitchlag);
//	if(nrmval != 0) for(i=0;i<len;i++) inCw[i]=inCw[i]/nrmval;
}


/**************************************************************************
 *
 *  Find correlation between pairs of signal peaks
 * 
 *  Inputs:
 *      int pk1 - location of the first peak in the signal amplitude contour buffer
 *      int pk2 - location of the second peak in the signal amplitude contour buffer
 *      
 *  Output:
 *      float - interpeak correlation coefficient
 * 
 *************************************************************************/
 
float TRANSMODE::find_corr_pkpairs(int pk1, int pk2)
{
    int i,j,k;
    int adjst,rng,rngmin1,rngmin2, rngmax1,rngmax2,tmp,exceed;
    int ptr1, ptr2,cl;
    float mfact,mxc,cc;
    adjst= (int)(0.2*(float)pitchlag);
    rngmin1=max_val(pk1-adjst,0);
    rngmin2=max_val(pk2-adjst,0);
    rngmax1=rngmin1+pitchlag; rngmax2=rngmin2+pitchlag;
    if(rngmax1<=(FrameSize+LOOKAHEAD_LEN) && rngmax2 <= (FrameSize+LOOKAHEAD_LEN)){
        mfact=1.0;
    }else{
        
            tmp=max_val(rngmax1,rngmax2);
            exceed=tmp-(FrameSize+LOOKAHEAD_LEN);
            mfact=(1-((float)exceed/pitchlag));
            
        
    }
    //rng=max_val(adjst,2);
    rng=1;
    mxc=0.0;
    for(k=-rng;k<=rng;k++){
        ptr1=max_val(pk1-rng+k,0);
        ptr2=max_val(pk2-rng,0);
        cl=pitchlag;tmp=pitchlag;
        if((ptr1+pitchlag)>(FrameSize+LOOKAHEAD_LEN))
            tmp=pitchlag-((FrameSize+LOOKAHEAD_LEN)-ptr1);
        if((ptr2+pitchlag)>(FrameSize+LOOKAHEAD_LEN))
            cl=pitchlag-((FrameSize+LOOKAHEAD_LEN)-ptr2);
        cl=min_val(tmp,cl);
        cc=conv_pc(&(resid_signal[ptr1]),&(resid_signal[ptr2]),cl);
        cc=cc*mfact;
        if(cc>mxc)
            mxc=cc;
    }
    return(mxc);
}

/**************************************************************************
 *
 *  Check to see if a transient frame can be marked as unvoiced for NELP coding
 * 
 *  Inputs:
 *      float frm_egy_evol - ratio between energy of previous frame and present frame
 *      float pitchcorr - pitch gain
 *      prev_mode -  last frame's mode (1- silence, 2- unvoiced,
 *                              3- voiced , 6- transient);
 *      
 *  Output:
 *      0 - if the frame is still to be coded as transient
 *      1 - if the frame is to be coded as unvoiced instead of transient
 * 
 *************************************************************************/
int TRANSMODE::addlcheck_if_unvoiced(float frm_egy_evol, float pitchcorr, short prev_mode)
{
    int unvoiced_q=0;
    

   
    //case :- no pitchspikes in the frame
    if(numpeaks_inframe==0 || numpeaks==0) {
        unvoiced_q=1;
        return(unvoiced_q);
    }
    

    //case :- only one pitchspike in the frame and the residual is not sparse, then call unvoiced
    if(numpeaks==1 && inSpM < 0.65) 
        unvoiced_q=1;
    
 
 

    //case :- more than one pitchspike in frame
    psrel=SpikeDiff_Rel();
    if(numpeaks > 1) {
        if(inSpM<0.60){                                        // to account for non sparse but periodic waveforms
            if(pitchcorr<0.3 )
                unvoiced_q=1;
        }else{                                                // for sparse signals 
            if(inSpM<0.7) {                                  // sparseness is between 0.6 and 0.7, then call unvoiced if the spikes are not correlated
                if(pitchcorr<0.3 && ((inSpM-0.6)*pitchcorr*100 < 1.2))       // 1.2 is good.  added 0.7 threshold on mxcorr to account for frames which have voiced characteristics by means of high correlation but is not very sparse due to poor lpc
                    unvoiced_q=1;
                if(psrel>0.5) unvoiced_q=1;
            }else
                if(psrel > 0.6) unvoiced_q=1;                                                
        }
    }
    
    if(inSpM< 0.65 && pitchlag<= 20) 
        unvoiced_q=1;
    
    if((prev_mode==1 || prev_mode==2) && frm_egy_evol<0.5)  unvoiced_q=1;//previous frame was silence or NELP
        
    if(prev_mode>2 && frm_egy_evol>10 && inSpM >0.55) unvoiced_q=0;

    
    if(psrel>0.3 && inSpM<0.6) unvoiced_q=1;

//Alok
	if((prev_mode==2) && numpeaks_inframe > 1 && frm_egy_evol>0.50  && frm_egy_evol<1.25 && (pitchcorr < 0.30 || inSpM<0.55))  unvoiced_q=1;
	if((prev_mode==1) && pitchcorr < 0.27) unvoiced_q=1;

    return(unvoiced_q);

}

/**************************************************************************
 *
 *  Find the next peak after or before a given peak based on flagud flag
 * 
 *  Inputs:
 *      int pkanc - location of the given peak
 *      int flagud - indicate if the next peak should be before or after given peak
 *              0 - find next peak after the given peak
 *              1 - find next peak before the given peak
 *      
 *  Output:
 *      pkloc - location of the next peak or 11 if there is no next peak
 * 
 *************************************************************************/
int TRANSMODE::find_closest_peak(int pkanc, int flagud)
{
    int mxc,mxl;
    int i;
    if(flagud==0) {
        mxc=-1;mxl=11;
        for(i=0;i<numpeaks;i++) {
            if(pitchspikes[i]<pkanc) {
                if(pitchspikes[i]>mxc) {
                    mxc=pitchspikes[i];
                    mxl=i;
                }
            }
        }
    }else{
        mxc=241;mxl=11;
        for(i=0;i<numpeaks;i++) {
            if(pitchspikes[i]>pkanc) {
                if(pitchspikes[i]<mxc) {
                    mxc=pitchspikes[i];
                    mxl=i;
                }
            }
        }
    }
    return(mxl);
    
}

/**************************************************************************
 *
 *  Calcuate an adjustment factor around loc where a energy minima lies
 * 
 *  Inputs:
 *      int loc - the location around which the adj factor has to be found
 *      int rng - range of the search
 *      
 *  Output:
 *      adjusted location.
 * 
 *************************************************************************/
int TRANSMODE::adj_to_minima(int loc, int rng, int len)
{
    int i,j=0,sv;
    double mxval=1e9,avsum;
    sv=max_val(loc-rng,0);

    for (i=loc;i>=sv;i--) {
        avsum=((0.2*inCw[max_val(i-2,0)]*inCw[max_val(i-2,0)])     +
               (0.5*inCw[max_val(i-1,0)]*inCw[max_val(i-1,0)])     +
               (1.0*inCw[max_val(i,0)]*inCw[max_val(i,0)])         +
               (0.5*inCw[min_val(i+1,len-1)]*inCw[min_val(i+1,len-1)]) +
               (0.2*inCw[min_val(i+2,len-1)]*inCw[min_val(i+2,len-1)]));
        if (avsum<mxval) {
            mxval=avsum;
            j=i;
        }
    }
	return(j);


}

/**************************************************************************
 *
 *  Adjust the last peak location in the current frame, if not within
 *  FrameSize-pitchlag
 * 
 *  Class members affected:
 *  last_pitchspike - location of the last peak in the current frame
 * 
 *************************************************************************/
void TRANSMODE::adj_last_pk_loc()
{
    int j,cnt=0, pk_within_pcyc[10],i;
	float mnpkval=0; 

	for (i=0;i<numpeaks_inframe; i++) {
        if (pitchspikes_inframe[i]> FrameSize-pitchlag) {
            pk_within_pcyc[cnt]=pitchspikes_inframe[i];
            cnt++;
        }
    }

    if (cnt>1) {
        for (i=0;i<cnt;i++) {
            if (inCw[pk_within_pcyc[i]]>mnpkval) {
                last_pitchspike=pk_within_pcyc[i];
                mnpkval=inCw[pk_within_pcyc[i]];
            }
        }
    }
}

/**************************************************************************
 *
 *  Call to the function to calcuate an adjustment factor around loc
 *  where a energy minima lies
 * 
 *  Inputs:
 *      int loc - the location around which the adj factor has to be found
 *      int rng - range of the search
 *      
 *  Output:
 *      adjusted location.
 * 
 *************************************************************************/
void TRANSMODE::adj_cycle_boundry(int len)
{
    int range,dloc1;

    range=pitchlag>>2;
 
    dloc1=adj_to_minima(pitchspikes[0],range,len);
    diffloc=pitchspikes[0]-dloc1;

}

/**************************************************************************
 *
 *  Calculate the median of the elements of a vector
 * 
 *  Inputs:
 *      int* vec - input vector
 *      int len - length of input vector
 *      
 *  Outputs:
 *      median of the elements of the input vector.
 * 
 *************************************************************************/
int median(int* vec, int len)
{
    int i,j,jsel;
    int tmp[10],tmpvec[10],mval,mvt=0;

    if (len==1) {
        return(vec[0]);
    } else {
        for (i=0;i<len;i++) tmpvec[i]=vec[i];
        for (i=0;i<len;i++) {
            mval=(FrameSize+LOOKAHEAD_LEN);
            for (j=0;j<len;j++) {
                if (tmpvec[j]<mval) {
                    mval=vec[j];
                    jsel=j;
                }
            }
            tmp[i]=mval;
            tmpvec[jsel]=(FrameSize+LOOKAHEAD_LEN);
            mvt=mval;
        }
    }
    if ((len % 2)==0) {
        i=len>>1;
        return((tmp[i]+tmp[i-1])>>1);
    } else {
        return(tmp[len>>1]);
    }


}

/**************************************************************************
 *
 *  Reestimate the pitchlag
 * 
 *  Inputs:
 *      short prev_lag - previous frame pitchlag
 *      short prev_frm_mode -last frame's mode (1- silence, 2- unvoiced,
 *                              3- voiced , 6- transient);
 *      float beta_n - pitch gain of current frame
 *      float beta_prev - pitch gain of previous frame
 *      float frm_egy_evol - ratio of previous frame to current frame energy
 *      short stdy_voiced_cnt - number of successive voiced frames encountered
 *      short stdy_pitch_flag - pitchlags of the successive voiced frame
 *            
 *  Class members affected:
 *      pitchlag_mod -modified pitchlag.
 * 
 *************************************************************************/
void TRANSMODE::reestimate_pitch(short prev_lag,short prev_frm_mode, float beta_n, float beta_prev, float frm_egy_evol, short stdy_voiced_cnt, short stdy_pitch_flag)
{
    int i,j;
    int tmppk[10], pksel[10];
    int rval=0;
    float plc,mean_pdiff,stdpk,fact,mnpk;
    float tmp[10], csum, mdist;
    short num_sel_can,num_lag_can;
    short flg=0, jsel,cnt, skipflag=1;
    short tmplag,estim_num_pks,estim_num_pks_inframe;
    short score,lagcand[45],mxscore,sel_lag;


    if(numpeaks == 1) { 
        if(prev_frm_mode>2 && beta_prev>0.3) 
            pitchlag_mod=prev_lag;
        else
            pitchlag_mod=pitchlag;

    }else{
        num_lag_can=numpeaks-1;
        if(beta_n>0.1) {
            pdiff[num_lag_can]=pitchlag;
            interpkcorr[num_lag_can]=beta_n;
            num_lag_can++;
            for(j=0;j<num_lag_can;j++) {
                if(interpkcorr[j]>beta_n) {
                    skipflag=0;
                }
            }

        }else
            skipflag=0;
        if(beta_prev>0.2 && prev_frm_mode>2 &&( frm_egy_evol >0.1 && frm_egy_evol<10)) {
            pdiff[num_lag_can]=prev_lag;
            interpkcorr[num_lag_can]=beta_prev;
            num_lag_can++;
            

        }
        for(i=0,mnpk=0,csum=0.0;i<num_lag_can;i++) {
            mnpk+=((float)pdiff[i]*interpkcorr[i]);
            csum+=(interpkcorr[i]);
        }
        mnpk=(short)(((float)mnpk/csum)+0.5);

        num_sel_can=num_lag_can;
        while(num_sel_can > 1 && skipflag==0) {
            float mxdist=0;
            for(j=0;j<num_sel_can;j++) {
                if(fabs((float)pdiff[j]-mnpk)>mxdist) {
                    mxdist=fabs((float)pdiff[j]-mnpk);
                    jsel=j;
                }
            }
            for(j=0,cnt=0;j<num_sel_can;j++) {
                if(j!=jsel) {
                    pksel[cnt]=pdiff[j];
                    tmp[cnt]=interpkcorr[j];
                    cnt++;
                }
            }
            num_sel_can--;
            for(j=0;j<num_sel_can;j++) {
                pdiff[j]=pksel[j];
                interpkcorr[j]=tmp[j];
            }
            for(i=0,mnpk=0,csum=0.0;i<num_sel_can;i++) {
                mnpk+=((float)pdiff[i]*interpkcorr[i]);
                csum+=(interpkcorr[i]);
            }
            mnpk=(short)(((float)mnpk/csum)+0.5);
        }
        if(skipflag==0) {
            pitchlag_mod=pdiff[0];
        }else
            pitchlag_mod=pitchlag;
    }

    if(beta_n>0.4 && abs(pitchlag-(pitchlag_mod<<1)) <=(pitchlag >>3)) {
        //potential pitch doubling
        pitchlag_mod=pitchlag;
    }


    if((beta_prev>=beta_n) && (prev_frm_mode>2) && (abs(prev_lag-(pitchlag_mod<<1)) <= (pitchlag_mod >>3) )) {
        //potential pitch doubling
        pitchlag_mod=prev_lag;
    }


    if(stdy_voiced_cnt>2 && stdy_pitch_flag==1 && psrel >0.18) { //unreliable estimate, just rely on the past
        pitchlag_mod=pitchlag=prev_lag;
    }

 //Alok
	if(stdy_voiced_cnt>2 && stdy_pitch_flag==1 && pitchlag_mod < 50 && (prev_lag-pitchlag_mod) > 45 ) { //unreliable estimate, just rely on the past
        pitchlag_mod=pitchlag=prev_lag;
    }

    if(pitchlag_mod>MAXLAG_2KBPS_TRMODE) {
        pitchlag_mod=MAXLAG_2KBPS_TRMODE;
    }
        

    if(pitchlag_mod<20) {
        if(beta_n > 0.2) {
            pitchlag_mod=pitchlag;
        }else{
            if(prev_frm_mode>2 && beta_prev>0.4) 
                pitchlag_mod=prev_lag;
            else
                pitchlag_mod=20;
        }
    }

   
}

/**************************************************************************
 *
 *  Fetch the last peak location
 * 
 *  Class members affected:
 *  last_pitchspike - location of the last peak in the current frame
 * 
 *************************************************************************/
void TRANSMODE::get_last_pk_inframe()
{
	if(numpeaks_inframe>0) {
		last_pitchspike=pitchspikes_inframe[numpeaks_inframe-1];
	}else
		last_pitchspike=0;
}

/**************************************************************************
 *
 *  Generate a flat spectrum prototype pitch waveform
 * 
 *  Inputs:
 *      short len - length of the waveform to be generated (usually = pitchlag)
 *      short voicing_cf - frequency above which noise has to be added
 *      long *seed_rndph - seed for noise additon
 *      short rl - rotate so that waveform has minima at extremities
 *            
 *  Class members affected:
 *      pitchproto -holds prototype pitch waveform.
 * 
 *************************************************************************/

void TRANSMODE::gen_proto_wf(short len, short voicing_cf, long *seed_rndph, short rl)
{
    short i,j;
    short fc_300, fc_3500, fc_rndph,mf;
    
    float mag[FrameSize], ph[FrameSize],a[FrameSize],b[FrameSize],pulse[FrameSize];
    float krnl;
    float psum;

    
    fc_300=(short)(((2.0/80.0)*(float)len)+1.0);
    fc_3500=((short)((3.0/8.0)*(float)len)+1.0);
    fc_rndph=(short)(((float)voicing_cf*(float)len/8000.0)+1.0);

    mf=len>>1;
    krnl=TWO_PI/len;
    
    
    for(j=0;j<len;j++) {
        mag[j]=1.0;
        ph[j]=0.0;
    }

    mag[0]=0.0; //dc is set to 0
    for(j=1;j<fc_300;j++) mag[j]=(float)j/(float)fc_300;                //upramp from 0-300 Hz
    if(mag[1]>0.5) mag[1]=0.5;
    if(mf!=fc_3500) {
       // for(j=fc_3500;j<=mf;j++) mag[j]=(float)(mf-j)/(float)(mf-fc_3500);  //down ramp from 3500 to 4K
         for(j=fc_3500;j<=mf;j++) mag[j]=0.1;  //down ramp from 3500 to 4K
    }else{
        mag[mf]=0.1;
    }

    //random phase past the voicing cutoff frequency
    for(j=fc_rndph; j<=mf;j++) {
        ph[j]=TWO_PI*ran0(seed_rndph);
        //mag[j]=0.7*mag[j];
    }

    if(len%2 == 0) {    //even lag
        mag[mf]=0.0;    //this will be 0 anyways
        ph[mf]=0.0;
    }
    for(j=mf+1;j<len;j++) {
        mag[j]=mag[len-j];
        ph[j]=-ph[len-j];
    }
    for(j=0;j<len;j++) {
        a[j]=mag[j]*cos(ph[j]);
        b[j]=mag[j]*sin(ph[j]);
    }
    for(psum=0.0,i=0;i<len;i++) {
        for(pulse[i]=0.0,j=0;j<len;j++) {
            pulse[i]+=((a[j]*cos(-krnl*i*j))-(b[j]*sin(-krnl*i*j)));
        }
        psum+=(pulse[i]*pulse[i]);
    }
    psum=1/sqrt(psum);

    //rotate the pulse by 1/8 of its length and normalize it to unit energy
    if(rl==0) {
        rl=(len>>3)+1; 
    }
    for(i=0;i<len;i++) {
        pitchproto[(i+rl)%len]=pulse[i]*psum;
    }
    diffloc=rl;
}


/**************************************************************************
 *
 *  Eliminate from a list of peaks any peaks that have been coded in the
 *  previous frames.
 * 
 *  Inputs:
 *      short prev_frm_last_coded_pk - last peak of the previous frame to be coded
 *      short estim_loc_first_pk - estimated location of the first peak in the present frame
 *            
 *  Class members affected:
 *      pitchspikes, pitchspikes_inframe,numpeaks, numpeaks_inframe - peaks in the current frame and lookahead
 * 
 *************************************************************************/
void TRANSMODE::eliminate_included_pk(short prev_frm_last_coded_pk, short  estim_loc_first_pk)
{
    int j;
    short elim_flag=0;


    if ((abs(prev_frm_last_coded_pk-FrameSize-pitchspikes[0])<(short)(0.25*pitchlag_mod))&& prev_frm_last_coded_pk!=(FrameSize+LOOKAHEAD_LEN))
        elim_flag=1;

    if(numpeaks>1 && estim_loc_first_pk!=(FrameSize+LOOKAHEAD_LEN)) {
        if((abs(estim_loc_first_pk-FrameSize-pitchspikes[0])>(short)(0.7*pitchlag_mod)) && 
           (abs(estim_loc_first_pk-FrameSize-pitchspikes[1]) <(short)(0.2*pitchlag_mod))) {
            elim_flag=1;
        }
    }



     if(elim_flag==1){
        numpeaks_inframe=0;
        for (j=0;j<numpeaks-1;j++) {
            pitchspikes[j]=pitchspikes[j+1];
            if (pitchspikes[j+1]<FrameSize) {
                pitchspikes_inframe[j]=pitchspikes[j+1];
                numpeaks_inframe++;
            }
        }
        numpeaks--;
    }
}
