/***************************************************************************
 *
 *  FILE NAME:  decoder.c
 *
 *         Usage : decoder  bitstream_file  synth_file
 *
 *         Format for bitstream_file:
 *           One word (2-byte) for bad frame indication (BFI) flag bit
 *               0x0000 -> good frame;  0x0001 -> bad frame
 *           244  words (2-byte) containing 244 bits.
 *               Bit 0 = 0x0000 and Bit 1 = 0x0001
 *           One word (2-byte) for ternary Silence Descriptor (SID) flag
 *               0x0000 -> inactive (no detected speech activity);
 *               0x0001 -> active
 *           One word (2-byte) for Time Alignment Flag (TAF) bit
 *               0x0000 -> inactive (no transmission of speech frames);
 *               0x0001 -> active
 *
 *         Format for synth_file:
 *           Synthesis is written to a binary file of 16 bits data.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "typedef.h"
#include "n_stack.h"
#include "basic_op.h"
#include "sig_proc.h"
#include "count.h"
#include "codec.h"
#include "cnst.h"
#include "d_homing.h"
#include "gsmefr.h"


/* These constants define the number of consecutive parameters
   that function decoder_homing_frame_test() checks */

#define WHOLE_FRAME 57
#define TO_FIRST_SUBFRAME 18


Word16 synth_buf[L_FRAME + M];

/* L_FRAME, M, PRM_SIZE, AZ_SIZE, SERIAL_SIZE: defined in "cnst.h" */

/*-----------------------------------------------------------------*
 *             Global variables                                    *
 *-----------------------------------------------------------------*/

//#if (WMOPS)
Word16 dtx_mode = 0;

//#endif

Word16 *synth;              /* Synthesis                  */
static Word16 reset_flag_old = 1;

/***************************************************************************
 *
 *      Local function prototypes
 *
 **************************************************************************/

static Word16 encoder_interface (const char* payload, Word16 serial_in_para[]);
static void random_parameters (Word16 serial_params[]);

static Word16 enc_dec_interface (const char* payload, Word16 serial_out_para[])
{

#define SPEECH      1
#define CNIFIRSTSID 2
#define CNICONT     3
#define VALIDSID    11
#define GOODSPEECH  33

    static Word16 decoding_mode = {SPEECH};
    static Word16 TAF_count = {1};
    Word16 serial_in_para[244], i, frame_type;
 
    /* Set channel status (BFI) flag to zero */
    /* --------------------------------------*/
    serial_out_para[0] = 0;     /* BFI flag */

    if(encoder_interface (payload, serial_in_para) == -1)
	{
		serial_out_para[0] = 1;     /* BFI flag */
	}

    /* Copy input parameters to output parameters */
    /* ------------------------------------------ */
    for (i = 0; i < 244; i++)
    {
        serial_out_para[i+1] = serial_in_para[i];
    }

    /* Evaluate SID flag                                  */
    /* Function sid_frame_detection() is defined in dtx.c */
    /* -------------------------------------------------- */
    serial_out_para[245] = sid_frame_detection (&serial_out_para[1]);

    /* Evaluate TAF flag */
    /* ----------------- */
    if (TAF_count == 0)
    {
        serial_out_para[246] = 1;
    }
    else
    {
        serial_out_para[246] = 0;
    }

    TAF_count = (TAF_count + 1) % 24;
   
    /* Frame classification:                                                */
    /* Since the transmission is error free, the received frames are either */
    /* valid speech or valid SID frames                                     */
    /* -------------------------------------------------------------------- */
    if (serial_out_para[245] == 2)
    {
        frame_type = VALIDSID;
    }
    else if (serial_out_para[245] == 0)
    {
        frame_type = GOODSPEECH;
    }
    else {
        return (1);
    }

    /* Update of decoder state */
    /* ----------------------- */
    if (decoding_mode == SPEECH) /* State of previous frame */
    {
        if (frame_type == VALIDSID)
        {
            decoding_mode = CNIFIRSTSID;
        }
        else if (frame_type == GOODSPEECH)
        {
            decoding_mode = SPEECH;
        }
    }
    else  /* comfort noise insertion mode */
    {
        if (frame_type == VALIDSID)
        {
            decoding_mode = CNICONT;
        }
        else if (frame_type == GOODSPEECH)
        {
            decoding_mode = SPEECH;
        }
    }

    /* Replace parameters by random data if in CNICONT-mode and TAF=0 */
    /* -------------------------------------------------------------- */
    if ((decoding_mode == CNICONT) && (serial_out_para[246] == 0))
    {
        random_parameters (&serial_out_para[1]);

        /* Set flags such that an "unusable frame" is produced */
        serial_out_para[0] = 1;       /* BFI flag */
        serial_out_para[245] = 0;     /* SID flag */
    }

    return (0);
}

static Word16 encoder_interface (const char* payload, Word16 serial_in_para[])
{
    int i,j,ret=0;
    int shift;
	int GSM_MAGIC = ((*payload & 0xF0) >> 4);
	if(GSM_MAGIC != 0xC)
	{
		ret = -1;
	}
    shift = 0x8;
    for(i=0; i<4; i++)
	{
        if(payload[0] & shift)
		{
			serial_in_para[i] = 1;
		}
		else
		{
			serial_in_para[i] = 0;
		}
		shift = shift >> 1;
	}

	for(i=1; i<31; i++)
	{
		shift = 0x80;
        for(j=0; j<8; j++)
		{
            if(payload[i] & shift)
			{
				serial_in_para[i*8+j-4] = 1;
			}
			else
			{
				serial_in_para[i*8+j-4] = 0;
			}
			shift = shift >> 1;
		}
	}

    return ret;
}

static void random_parameters (Word16 serial_params[])
{
    static Word32 L_PN_seed = 0x321CEDE2L;
    Word16 i;

    /* Set the 244 speech parameter bits to random bit values */
    /* Function pseudonoise() is defined in dtx.c             */
    /*--------------------------------------------------------*/

    for (i = 0; i < 244; i++)
    {
        serial_params[i] = pseudonoise (&L_PN_seed, 1);
    }

    return;
}

__attribute__ ((visibility ("default"))) int apiGSMEFRInit()
{
    /*-----------------------------------------------------------------*
     *           Initialization of decoder                             *
     *-----------------------------------------------------------------*/
    synth = synth_buf + M;

    reset_dec (); /* Bring the decoder and receive DTX to the initial state */
}

/*-----------------------------------------------------------------*
 *            gsm-efr decoder routine                                 *
 *-----------------------------------------------------------------*/

__attribute__ ((visibility ("default"))) int apiGSMEFRdecode(const char*payload, short *buf)
{
    Word16 parm[PRM_SIZE + 1];  /* Synthesis parameters       */
    Word16 serial[SERIAL_SIZE+2];/* Serial stream              */
    Word16 Az_dec[AZ_SIZE];     /* Decoded Az for post-filter */
                                /* in 4 subframes, length= 44 */
    Word16 i, frame, temp;

    Word16 TAF, SID_flag;

    Word16 reset_flag;

    if(enc_dec_interface (payload, serial) == 1)
	{
        memset((void*)buf, 0, 160*sizeof(short));
		return 0;
	}


    SID_flag = serial[245];         /* Receive SID flag */
    TAF = serial[246];              /* Receive TAF flag */

    Bits2prm_12k2 (serial, parm);   /* serial to parameters   */


    if (parm[0] == 0)               /* BFI == 0, perform DHF check */
    {
        if (reset_flag_old == 1)    /* Check for second and further
                                        successive DHF (to first subfr.) */
        {
            reset_flag = decoder_homing_frame_test (&parm[1],
                                                    TO_FIRST_SUBFRAME);
        }
        else
        {
            reset_flag = 0;
        }
    }
    else                          /* BFI==1, bypass DHF check (frame
                                        is taken as not being a DHF) */
    {
        reset_flag = 0;
    }

    if ((reset_flag != 0) && (reset_flag_old != 0))
    {
        /* Force the output to be the encoder homing frame pattern */

        for (i = 0; i < L_FRAME; i++)
        {
            synth[i] = EHF_MASK;
        }
    }
    else
    {
        Decoder_12k2 (parm, synth, Az_dec, TAF, SID_flag);/* Synthesis */

        Post_Filter (synth, Az_dec);                      /* Post-filter */

        for (i = 0; i < L_FRAME; i++) 
            /* Upscale the 15 bit linear PCM to 16 bits,
                then truncate to 13 bits */
        {
            temp = GsmEfr_shl (synth[i], 1);
            synth[i] = temp & 0xfff8;       logic16 (); move16 (); 
        }

    }                       /* else */

	memcpy((void*)buf,(void*)synth,L_FRAME*sizeof(short));

    /* BFI == 0, perform check for first DHF (whole frame) */
    if ((parm[0] == 0) && (reset_flag_old == 0))
    {
        reset_flag = decoder_homing_frame_test (&parm[1], WHOLE_FRAME);
    }

    if (reset_flag != 0)
    {
        /* Bring the decoder and receive DTX to the home state */
        reset_dec ();
    }
    reset_flag_old = reset_flag;


    return 0;
}
