/*************************************************************************
 *
 *  FUNCTION:  Prm2bits_12k2
 *
 *  PURPOSE:  converts the encoder parameter vector into a vector of serial
 *                      bits.
 *
 *  DESCRIPTION: The encoder parameters are:
 *
 *     LPC:
 *              1st codebook             7 bit
 *              2nd codebook             8 bit
 *              3rd codebook             8+1 bit
 *              4th codebook             8 bit
 *              5th codebook             6 bit
 *
 *     1st and 3rd subframes:
 *           pitch period                9 bit
 *           pitch gain                  4 bit
 *           codebook index              35 bit
 *           codebook gain               5 bit
 *
 *     2nd and 4th subframes:
 *           pitch period                6 bit
 *           pitch gain                  4 bit
 *           codebook index              35 bit
 *           codebook gain               5 bit
 *
 *************************************************************************/

#include "typedef.h"
#include "basic_op.h"
#include "count.h"

/* Local function */

void Int2bin (
    Word16 value,       /* input : value to be converted to binary      */
    Word16 no_of_bits,  /* input : number of bits associated with value */
    Word16 *bitstream   /* output: address where bits are written       */
);

#define BIT_0     0
#define BIT_1     1
#define MASK      0x0001
#define PRM_NO    57

void Prm2bits_12k2 (
    Word16 prm[],       /* input : analysis parameters  (57 parameters)   */
    Word16 bits[]       /* output: 244 serial bits                        */
)
{
    Word16 i;

    static const Word16 bitno[PRM_NO] =
    {
        7, 8, 9, 8, 6,                          /* LSP VQ          */
        9, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 5,  /* first subframe  */
        6, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 5,  /* second subframe */
        9, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 5,  /* third subframe  */
        6, 4, 4, 4, 4, 4, 4, 3, 3, 3, 3, 3, 5   /* fourth subframe */
    };
    for (i = 0; i < PRM_NO; i++)
    {
        Int2bin (prm[i], bitno[i], bits);
        bits += bitno[i];
    }

    return;
}

/*************************************************************************
 *
 *  FUNCTION:  Int2bin
 *
 *  PURPOSE:  convert integer to binary and write the bits to the array
 *            bitstream[]. The most significant bits are written first.
 *
 *************************************************************************/

void Int2bin (
    Word16 value,       /* input : value to be converted to binary      */
    Word16 no_of_bits,  /* input : number of bits associated with value */
    Word16 *bitstream   /* output: address where bits are written       */
)
{
    Word16 *pt_bitstream, i, bit;

    pt_bitstream = &bitstream[no_of_bits];      move16 (); 

    for (i = 0; i < no_of_bits; i++)
    {
        bit = value & MASK;                     logic16 (); 
        test (); 
        if (bit == 0)
        {
            *--pt_bitstream = BIT_0;            move16 (); 
        }
        else
        {
            *--pt_bitstream = BIT_1;            move16 (); 
        }
        value = GsmEfr_shr (value, 1);
    }
}
