/* ###########################################################################
#
# Copyright (c) 2000-2016 Telchemy, Incorporated. All Rights Reserved.
#
# Telchemy Confidential and Proprietary
#
# The following software source code ("Software") is strictly confidential
# and is proprietary to Telchemy, Incorporated ("Telchemy").  It may only
# be read, used, copied, adapted, modified or otherwise utilized by parties
# (individuals, corporations, or organizations) that have entered into a
# license agreement or confidentiality agreement with Telchemy, and are thus
# subject to the terms of that license agreement or confidentiality agreement
# and any other applicable agreement between the party and Telchemy.  If
# there is any doubt as to whether a party is entitled to access, read, use,
# copy, adapt, modify or otherwise utilize the Software, or whether a party
# is entitled to disclose the Software to any other party, you should contact
# Telchemy.  If you, as a party, have not entered into a license agreement or
# confidentiality agreement with Telchemy granting access to this Software,
# all media, copies and printed listings containing the Software should be
# forthwith returned to Telchemy.
#
# This Software is protected by Granted and Pending Patents owned by Telchemy,
# Incorporated.
#
# Telchemy reserves the right to take legal action against any party that
# violates Telchemy's rights to the Software, including without limitation a
# party's breach of the above conditions.
#
# If you have questions about any of the above conditions, or wish to report
# violations, please contact:  support@telchemy.com
#
# ##########################################################################*/

/*
 * vqmonstm.h
 *
 * VQmon Public Header File - Multimedia Stream
 *
 * Copyright (c) 2000-2016 Telchemy, Incorporated
 */

#ifndef _TELCHEMY_VQMONSTM_H_
#define _TELCHEMY_VQMONSTM_H_

#include "vqmon.h"
#include "vqmontyp.h"

#ifdef __cplusplus
extern "C" {
#endif

/* --------------------------------------------------------------------------
 *
 *      VQmon Public Multimedia Stream Constant Definitions
 *
 * --------------------------------------------------------------------------
 */

/*
 * Stream default configuration block identifiers for an interface.
 */
#define VQMON_STREAMCFGBLOCKID_RTPTSTHRESHDEF       (0x0045)
#define VQMON_STREAMCFGBLOCKID_ALERTTHRESHDEF       (0x0400)
#define VQMON_STREAMCFGBLOCKID_RTPPTMAPDEF          (0x0401)
#define VQMON_STREAMCFGBLOCKID_PESTMAPDEF           (0x0405)

/*
 * Stream configuration block identifiers.
 */
#define VQMON_STREAMCFGBLOCKID_NETEPDESC            (0x0001)
#define VQMON_STREAMCFGBLOCKID_TRANSPORTINFO        (0x0002)
#define VQMON_STREAMCFGBLOCKID_RELATIONSHIPS        (0x0003)
#define VQMON_STREAMCFGBLOCKID_USERDATA             (0x0004)
#define VQMON_STREAMCFGBLOCKID_CODECINFO            (0x0005)

#define VQMON_STREAMCFGBLOCKID_PLAYOUTBUF           (0x0010)
#define VQMON_STREAMCFGBLOCKID_ROUNDTRIPDELAY       (0x0015)

#define VQMON_STREAMCFGBLOCKID_FEC                  (0x0020)
#define VQMON_STREAMCFGBLOCKID_FECTYPE              (0x0021)
#define VQMON_STREAMCFGBLOCKID_RTPTSTHRESH          (0x0025)

#define VQMON_STREAMCFGBLOCKID_SPECTIMEOUTSET       (0x0031)
#define VQMON_STREAMCFGBLOCKID_SPECTIMEOUTRESET     (0x0032)

#define VQMON_STREAMCFGBLOCKID_ASSOCSESSION         (0x0041)

#define VQMON_STREAMCFGBLOCKID_ALERTTHRESH          (0x0100)
#define VQMON_STREAMCFGBLOCKID_RTPPTMAP             (0x0101)

/*
 * Stream metric block identifiers.
 */
#define VQMON_STREAMMETRICBLOCKID_STATUS            (0x0101)
#define VQMON_STREAMMETRICBLOCKID_PKTTRANS          (0x0102)
#define VQMON_STREAMMETRICBLOCKID_JITTER            (0x0103)
#define VQMON_STREAMMETRICBLOCKID_DELAY             (0x0104)
#define VQMON_STREAMMETRICBLOCKID_RTPPKTS           (0x0105)
#define VQMON_STREAMMETRICBLOCKID_PCRJITTER         (0x0106)
#define VQMON_STREAMMETRICBLOCKID_MMCOUNTERS        (0x0107)
#define VQMON_STREAMMETRICBLOCKID_ESDELAY           (0x0108)

#define VQMON_STREAMMETRICBLOCKID_FEC               (0x0201)
#define VQMON_STREAMMETRICBLOCKID_RELIABLEUDP       (0x0202)

#define VQMON_STREAMMETRICBLOCKID_RTCPMETRICS       (0x0301)
#define VQMON_STREAMMETRICBLOCKID_RTCPSRLAST        (0x0302)
#define VQMON_STREAMMETRICBLOCKID_RTCPSDES          (0x0303)


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Multimedia Stream Enumerations
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon stream types.
 */
#define VQMON_STREAMTYPE_UNKNOWN                    (0x0)
#define VQMON_STREAMTYPE_AUDIO                      (0x1000)
#define VQMON_STREAMTYPE_VIDEO                      (0x2000)
#define VQMON_STREAMTYPE_VOICE                      (0x4000)
#define VQMON_STREAMTYPE_MP2TS                      (0x8000)
#define VQMON_STREAMTYPE_FEC                        (0x0100)
#define VQMON_STREAMTYPE_TCP                        (0x0010)
#define VQMON_STREAMTYPE_STRMSESSION                (0x0020)
#define VQMON_STREAMTYPE_QUIC                       (0x0040)

typedef enum
{
    vqmonStreamTypeUnknown              = VQMON_STREAMTYPE_UNKNOWN,
    vqmonStreamTypeAudio                = VQMON_STREAMTYPE_AUDIO,
    vqmonStreamTypeVideo                = VQMON_STREAMTYPE_VIDEO,
    vqmonStreamTypeVoice                = VQMON_STREAMTYPE_VOICE,
    vqmonStreamTypeMP2TS                = VQMON_STREAMTYPE_MP2TS,
    vqmonStreamTypeFEC                  = VQMON_STREAMTYPE_FEC,
    vqmonStreamTypeTCP                  = VQMON_STREAMTYPE_TCP,
    vqmonStreamTypeStrmSession          = VQMON_STREAMTYPE_STRMSESSION,
    vqmonStreamTypeQUIC                 = VQMON_STREAMTYPE_QUIC

} vqmon_streamtype_t;

/*
 * VQmon stream transport protocol.
 */
#define VQMON_STREAMTRANSPROTO_UDP                  (0x01)
#define VQMON_STREAMTRANSPROTO_TCP                  (0x02)
#define VQMON_STREAMTRANSPROTO_RTP                  (0x10)
#define VQMON_STREAMTRANSPROTO_MP2TS                (0x20)
#define VQMON_STREAMTRANSPROTO_QUIC                 (0x40)
#define VQMON_STREAMTRANSPROTO_IUUP                 (0x80)

typedef enum
{
    vqmonStreamTransProtoUDP            = VQMON_STREAMTRANSPROTO_UDP,
    vqmonStreamTransProtoTCP            = VQMON_STREAMTRANSPROTO_TCP,
    vqmonStreamTransProtoRTP            = VQMON_STREAMTRANSPROTO_RTP,
    vqmonStreamTransProtoMP2TS          = VQMON_STREAMTRANSPROTO_MP2TS,
    vqmonStreamTransProtoQUIC           = VQMON_STREAMTRANSPROTO_QUIC,
    vqmonStreamTransProtoIUUP           = VQMON_STREAMTRANSPROTO_IUUP

} vqmon_streamtransproto_t;


/*
 * VQmon stream relationship types.
 */
#define VQMON_STREAMRELATION_PARENT                 (0x01)
#define VQMON_STREAMRELATION_CHILD                  (0x02)
#define VQMON_STREAMRELATION_SIBLING                (0x03)
#define VQMON_STREAMRELATION_PEER                   (0x04)
#define VQMON_STREAMRELATION_FECSOURCE              (0x05)
#define VQMON_STREAMRELATION_RUDPSOURCE             (0x06)
#define VQMON_STREAMRELATION_ASSOCMEDIA             (0x07)

typedef enum
{
    vqmonStreamRelationParent           = VQMON_STREAMRELATION_PARENT,
    vqmonStreamRelationChild            = VQMON_STREAMRELATION_CHILD,
    vqmonStreamRelationSibling          = VQMON_STREAMRELATION_SIBLING,
    vqmonStreamRelationPeer             = VQMON_STREAMRELATION_PEER,
    vqmonStreamRelationFECSource        = VQMON_STREAMRELATION_FECSOURCE,
    vqmonStreamRelationRUDPSource       = VQMON_STREAMRELATION_RUDPSOURCE,
    vqmonStreamRelationAssocMedia       = VQMON_STREAMRELATION_ASSOCMEDIA

} vqmon_streamrelation_t;


/*
 * VQmon Stream relationship set/get config options.
 */
#define VQMON_STREAMRELATIONSHIP_ADD                (0x01)
#define VQMON_STREAMRELATIONSHIP_REMOVE             (0x02)
#define VQMON_STREAMRELATIONSHIP_QUERY              (0x03)

typedef enum
{
    vqmonStreamRelationshipAdd          = VQMON_STREAMRELATIONSHIP_ADD,
    vqmonStreamRelationshipRemove       = VQMON_STREAMRELATIONSHIP_REMOVE,
    vqmonStreamRelationshipQuery        = VQMON_STREAMRELATIONSHIP_QUERY,

} vqmon_streamrelationop_t;


/*
 * VQmon stream playout buffer modes.
 */
#define VQMON_STREAMPLAYOUTBUFMODE_NORMAL           (0x00)
#define VQMON_STREAMPLAYOUTBUFMODE_NORESEQ          (0x10)
#define VQMON_STREAMPLAYOUTBUFMODE_NODISCARD        (0x20)

typedef enum
{
    vqmonStreamPlayoutBufferModeNormal         = VQMON_STREAMPLAYOUTBUFMODE_NORMAL,
    vqmonStreamPlayoutBufferModeNoResequencing = VQMON_STREAMPLAYOUTBUFMODE_NORESEQ,
    vqmonStreamPlayoutBufferModeNoDiscard      = VQMON_STREAMPLAYOUTBUFMODE_NODISCARD

} vqmon_streamplayoutbufmode_t;


/*
 * VQmon stream metric value sources.
 */
typedef enum
{
    vqmonStreamMetricSourceUnknown     = 0,
    
    vqmonStreamMetricSourceDefault,
    vqmonStreamMetricSourceUserConfig,
    vqmonStreamMetricSourceRTCPRpt,
    vqmonStreamMetricSourceRTCPXRRpt,
    vqmonStreamMetricSourceSIPRTCPRpt,

    /* For reporting only. */
    vqmonStreamMetricSourceUserConfigCorrected,
    vqmonStreamMetricSourceRTCPXRRptCorrected

} vqmon_streammetricsource_t;


/*
 * VQmon report types.
 */
typedef enum
{
    vqmonStreamReportTypeRTCPXR         = 0x4001,
    vqmonStreamReportTypeRTCPHR         = 0x4002,
    vqmonStreamReportTypeSIPVQSession   = 0x4003

} vqmon_streamreporttype_t;


/*
 * VQmon stream status
 */
#define VQMON_STREAMSTATUS_UNKNOWN                  (0x00)
#define VQMON_STREAMSTATUS_INACTIVE                 (0x01)
#define VQMON_STREAMSTATUS_ACTIVE                   (0x02)
#define VQMON_STREAMSTATUS_TERMINATED               (0x10)

typedef enum
{
    vqmonStreamStatusUnknown            = VQMON_STREAMSTATUS_UNKNOWN,
    vqmonStreamStatusInactive           = VQMON_STREAMSTATUS_INACTIVE,
    vqmonStreamStatusActive             = VQMON_STREAMSTATUS_ACTIVE,
    vqmonStreamStatusTerminated         = VQMON_STREAMSTATUS_TERMINATED

} vqmon_streamstatus_t;


/*
 * VQmon stream progress event types
 */
typedef enum
{
    vqmonStreamProgEventNone            = 0,
    vqmonStreamProgEventOnHold,
    vqmonStreamProgEventOffHold,
    vqmonStreamProgEventChannelChange,
    vqmonStreamProgEventIntervalEnd,
    vqmonStreamProgEventIntervalExtSample,
    vqmonStreamProgEventResetCounters,
    vqmonStreamProgEventTerminate,
    vqmonStreamProgEventResetTR101290Counters

} vqmon_streamprogevttype_t;


/*
 * VQmon stream management notification types
 */
typedef enum
{
    vqmonStreamMgmtNotifyNone                   = 0,
    vqmonStreamMgmtNotifyCreated                = VQMON_STREAMMGMTNOTIFY_CREATED,
    vqmonStreamMgmtNotifyActivating             = VQMON_STREAMMGMTNOTIFY_ACTIVATING,
    vqmonStreamMgmtNotifyCODECUpdate            = VQMON_STREAMMGMTNOTIFY_CODECUPDATE,
    vqmonStreamMgmtNotifyCODECRequest           = VQMON_STREAMMGMTNOTIFY_CODECREQUEST,
    vqmonStreamMgmtNotifyDelayUpdate            = VQMON_STREAMMGMTNOTIFY_DELAYUPDATE,
    vqmonStreamMgmtNotifyPlayoutBufUpdate       = VQMON_STREAMMGMTNOTIFY_PLAYOUTBUFUPDATE,
    vqmonStreamMgmtNotifyProgInfoUpdate         = VQMON_STREAMMGMTNOTIFY_PROGINFOUPDATE,
    vqmonStreamMgmtNotifyIntervalUpdate         = VQMON_STREAMMGMTNOTIFY_INTERVALUPDATE,
    vqmonStreamMgmtNotifyDeactivating           = VQMON_STREAMMGMTNOTIFY_DEACTIVATING,
    vqmonStreamMgmtNotifyTerminated             = VQMON_STREAMMGMTNOTIFY_TERMINATED,
    vqmonStreamMgmtNotifyRTPHeaderExtensionRcvd = VQMON_STREAMMGMTNOTIFY_RTPEXTHEADERRCVD,
    vqmonStreamMgmtNotifyRTCPAppRcvd            = VQMON_STREAMMGMTNOTIFY_RTCPAPPRCVD,
    vqmonStreamMgmtNotifyRTCPSRRcvd             = VQMON_STREAMMGMTNOTIFY_RTCPSRRCVD,
    vqmonStreamMgmtNotifyRTCPXRRcvd             = VQMON_STREAMMGMTNOTIFY_RTCPXRRCVD,
    vqmonStreamMgmtNotifyRTCPByePacket          = VQMON_STREAMMGMTNOTIFY_RTCPBYEPACKET,
    vqmonStreamMgmtNotifyRTCPSDESPacket         = VQMON_STREAMMGMTNOTIFY_RTCPSDESPACKET,
    vqmonStreamMgmtNotifyConfigError            = VQMON_STREAMMGMTNOTIFY_CONFIGERROR,
    vqmonStreamMgmtNotifyReset                  = VQMON_STREAMMGMTNOTIFY_RESET,
    vqmonStreamMgmtNotifyRTPSSRCChanging        = VQMON_STREAMMGMTNOTIFY_RTPSSRCCHANGING,
    vqmonStreamMgmtNotifyRTPSSRCChanged         = VQMON_STREAMMGMTNOTIFY_RTPSSRCCHANGED,
    vqmonStreamMgmtNotifyStrmVidSampleAvail     = VQMON_STREAMMGMTNOTIFY_STRMVIDSAMPLEAVAIL,
    vqmonStreamMgmtNotifyStrmVidSampleFull      = VQMON_STREAMMGMTNOTIFY_STRMVIDSAMPLEFULL

} vqmon_streammgmtnotifytype_t;


/*
 * VQmon stream CODEC Request actions
 */
typedef enum
{
    vqmonStreamCODECRequestActionContinue       = 0,
    vqmonStreamCODECRequestActionContinueNoType,
    vqmonStreamCODECRequestActionDelay,
    vqmonStreamCODECRequestActionHalt

} vqmon_streamcodecrequestaction_t;


/*
 * VQmon stream configuration error flags
 */
typedef enum
{
    vqmonStreamConfigErrorTypeMismatch          = 0x0001,
    vqmonStreamConfigErrorSampleRateUnsupported = 0x0002,
    vqmonStreamConfigErrorCODECUnknown          = 0x0004

} vqmon_streamconfigerrorflag_t;

/*
 * VQmon stream FEC types
 */
typedef enum
{
    vqmonStreamFECTypeUnspecified = 0x00,
    vqmonStreamFECTypeDisabled    = 0x01,
    vqmonStreamFECType1           = 0x02,
    vqmonStreamFECTypeLPR         = 0x03,
    vqmonStreamFECTypeReTx        = 0x04

} vqmon_stream_fectype_t;


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Multimedia Stream Simple Data Types
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon stream configuration block identifier types.
 */
typedef tcmyU16 vqmon_streamcfgblkid_t;


/*
 * VQmon stream metric block identifier types.
 */
typedef tcmyU16 vqmon_streammetricblkid_t;


/*
 * VQmon stream event handler callback function definition.
 */
struct _vqmon_streammgmtnotify_info_s;
typedef vqmon_result_t (*vqmon_func_streammgmtnotifyhdlrcb_t)(
                 const vqmon_handle_t hInterface,
                 const vqmon_handle_t hStream,
                 const vqmon_streammgmtnotifytype_t eNotification,
                 struct _vqmon_streammgmtnotify_info_s* pNotifyInfo
                 );


/* --------------------------------------------------------------------------
 *
 *      VQmon Public Multimedia Stream Complex Data Structures
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon stream endpoint identification configuration.
 */
struct _vqmon_streamcfg_netepdesc_s
{
    vqmon_ipaddr_t              tSrcAddress;
    vqmon_ipaddr_t              tDstAddress;
    tcmyU16                     nSrcPort;
    tcmyU16                     nDstPort;

    tcmyU16                     nVlanTag;
    tcmyU16                     nServiceVlanTag;
    tcmyU8                      nPacketTOS;
#ifdef VQMON_BUILDCFG_FLOWHASH
    tcmyU16                     nAdditionalTag;
    tcmyU64                     nFlowHash;
#endif
    tcmyU8                      aSrcMACAddress[6];
    tcmyU8                      aDstMACAddress[6];

    vqmon_streamtype_t          eType;
    vqmon_streamtransproto_t    eTransportProtos;
};

typedef struct _vqmon_streamcfg_netepdesc_s vqmon_streamcfg_netepdesc_t;


/*
 * VQmon stream transport protocol extended information.
 */
struct _vqmon_streamcfg_transprotoext_s
{
    vqmon_streamtransproto_t    eTransportProto;

    union _transport_proto_u
    {
        struct _transport_proto_rtp_s
        {
            tcmyU32             nSSRC;

        } rtp;

        struct _transport_proto_mp2ts_s
        {
            tcmyU16             fCtrlFlags;
            tcmyU16             nPID;

        } mp2ts;
  
        struct _transport_proto_quic_s
        {
            tcmyU64             nConnectionID;

        } quic;
    } proto;
};

typedef struct _vqmon_streamcfg_transprotoext_s \
    vqmon_streamcfg_transprotoext_t;


/*
 * VQmon stream relationship information.
 */
struct _vqmon_streamrelationship_s
{
    vqmon_streamrelation_t      eRelation;
    vqmon_handle_t              hStream;
};

typedef struct _vqmon_streamrelationship_s \
    vqmon_streamrelationship_t;

struct _vqmon_streamcfg_relationinfo_s
{
    vqmon_streamrelationop_t    eRelationOp;

    tcmyU16                     nNumRelationships;
    vqmon_streamrelationship_t  aRelationships[1];
};

typedef struct _vqmon_streamcfg_relationinfo_s vqmon_streamcfg_relationinfo_t;


/*
 * VQmon stream CODEC configuration.
 */
#define VQMON_STREAMCODECUSERCFG_TYPE       (0x01)
#define VQMON_STREAMCODECUSERCFG_VAD        (0x02)
#define VQMON_STREAMCODECUSERCFG_RATE       (0x04)
#define VQMON_STREAMCODECUSERCFG_NOAUTO     (0x08)

struct _vqmon_streamcfg_codec_s
{
    vqmon_codecid_t             idCODECType;
    tcmyBOOL                    bActivityDetectAvail;
    tcmyS32                     nCurrentRate;
    tcmyU16                     fUserConfigured;
    vqmon_percent_t             nConfidence;
};

typedef struct _vqmon_streamcfg_codec_s vqmon_streamcfg_codec_t;


/*
 * VQmon stream playout buffer configuration.
 */
struct _vqmon_streamcfg_playoutbuf_s
{
    vqmon_streamplayoutbufmode_t eMode;
    vqmon_fpq4_t                 nNominalDelaySec;
    vqmon_fpq4_t                 nMaxDelaySec;
    tcmyU32                      nMaxPackets;
    vqmon_fpq4_t                 nIncRebufDelaySec;
};

typedef struct _vqmon_streamcfg_playoutbuf_s vqmon_streamcfg_playoutbuf_t;


/*
 * VQmon stream round-trip delay configuration
 */
struct _vqmon_streamcfg_rtdelay_s
{
    vqmon_streammetricsource_t  eDelaySource;
    vqmon_fpq4_t                nDelayMs;
};

typedef struct _vqmon_streamcfg_rtdelay_s vqmon_streamcfg_rtdelay_t;


/*
 * VQmon stream Forward Error Correction (FEC) configuration.
 */
struct _vqmon_streamcfg_fec_s
{
    tcmyU8                      nFECBlockSz;
    tcmyU8                      nFECCorrPkts;
};

typedef struct _vqmon_streamcfg_fec_s vqmon_streamcfg_fec_t;

/*
 * VQmon stream Forward Error Correction (FEC) type configuration.
 */
struct _vqmon_streamcfg_fectype_s
{
    vqmon_stream_fectype_t      eFECType;
};

typedef struct _vqmon_streamcfg_fectype_s vqmon_streamcfg_fectype_t;

/*
 * VQmon stream RTP timestamp threshold configuration.
 */
struct _vqmon_streamcfg_rtptsthresh_s
{
    tcmyU32                     nThresholdMs;
};

typedef struct _vqmon_streamcfg_rtptsthresh_s vqmon_streamcfg_rtptsthresh_t;

/*
 * VQmon stream user data configuration
 */

struct _vqmon_streamcfg_userdata_s
{
    void                     *pUserData;
};

typedef struct _vqmon_streamcfg_userdata_s vqmon_streamcfg_userdata_t;


/*
 * VQmon stream RTP payload type mapping configuration
 */
struct _vqmon_streamcfg_rtpptmap_mapping_s
{
    tcmyU8                  nPT;
    vqmon_codecid_t         eCODEC;
};

typedef struct _vqmon_streamcfg_rtpptmap_mapping_s
    vqmon_streamcfg_rtpptmap_mapping_t;

struct _vqmon_streamcfg_rtpptmap_table_s
{
    tcmyU8                  nPTCount;
    vqmon_streamcfg_rtpptmap_mapping_t aRTPMap[1];
};

typedef struct _vqmon_streamcfg_rtpptmap_table_s
            vqmon_streamcfg_rtpptmap_table_t;


/*
 * VQmon stream associated (streaming) session configuration
 */

struct _vqmon_streamcfg_assocsession_s
{
    vqmon_handle_t              hAssocSession;
    tcmyBOOL                    bIsTerminating;
};

typedef struct _vqmon_streamcfg_assocsession_s vqmon_streamcfg_assocsession_t;


/*
 * VQmon stream management notification information
 */
struct _vqmon_streammgmtnotify_info_s
{
    vqmon_streamtype_t          eStreamType;

    union
    {
        struct
        {
            vqmon_time_t        tNotifyTime;
            
        } activating;

        struct
        {
            vqmon_time_t        tNotifyTime;

        } terminated;

        struct
        {
            vqmon_time_t        tNotifyTime;
            vqmon_fpq4_t        nOldRTDelayMs;
            vqmon_fpq4_t        nNewRTDelayMs;

        } rtdelayupdate;

        struct
        {
            vqmon_codecid_t     idOldCODECType;
            tcmyU16             nOldRateIndex;
            vqmon_codecid_t     idNewCODECType;
            tcmyU16             nNewRateIndex;
            vqmon_percent_t     nConfidence;

        } codecupdate;

        struct
        {
            /*
             * Requested from the user
             */
            vqmon_streamtype_t          eType;
            vqmon_codecid_t             idCODECType;
            vqmon_streamcodecrequestaction_t eAction;

            /*
             * Provided to the user
             */
            vqmon_ipaddr_t              tSrcAddress;
            vqmon_ipaddr_t              tDstAddress;
            tcmyU16                     nSrcPort;
            tcmyU16                     nDstPort;
            tcmyU32                     nSSRC;
            tcmyU8                      nRTPPayloadType;
        
        } codecrequest;

        struct
        {
            vqmon_fpq4_t        nNomDelaySecs;
            tcmyU16             nMaxNumPkts;
            vqmon_fpq4_t        nMaxDelaySecs;
            vqmon_fpq4_t        nIncRebufDelaySecs;

        } playoutbufupdate;

        struct
        {
            tcmyU16             nHeaderExtType;
            tcmyU8*             pHeaderExtBuf;
            tcmyU32             nHeaderExtLen;
        } rtpheaderext;

        struct
        {
            tcmyU8              nSubtype;
            tcmyU32             nSSRC;
            tcmyCHAR            aName[4];

            tcmyU8*             pDataBuf;
            tcmyU32             nDataBufLen;
        } rtcpapp;

        struct
        {
            tcmyU8                  nSourceCount;
            tcmyU32                 nChunkLength;
            tcmyU8*                 pChunkBuffer;
        } rtcpsdes;

        struct
        {
            tcmyU32             fConfigError;

        } config;

        struct
        {
            tcmyU32             nOldSSRC;
            tcmyU32             nNewSSRC;

        } rtpssrc;
    } state;
};

typedef struct _vqmon_streammgmtnotify_info_s vqmon_streammgmtnotify_info_t;


/*
 * VQmon stream status information.
 */
struct _vqmon_streammetrics_status_s
{
    vqmon_handle_t              hInterface;

    vqmon_time_t                tStartTime;
    vqmon_time_t                tDuration;
 
    vqmon_streamstatus_t        eStatus;
};

typedef struct _vqmon_streammetrics_status_s vqmon_streammetrics_status_t;

    
/*
 * VQmon stream Markov Model Counters.
 */
struct _vqmon_streammetrics_mmcounters_s
{
    tcmyU32             c11;
    tcmyU32             c13;
    tcmyU32             c14;
    tcmyU32             c22;
    tcmyU32             c23;
    tcmyU32             c33;
    tcmyU32             c5;
    tcmyU32             c6;

};

typedef struct _vqmon_streammetrics_mmcounters_s vqmon_streammetrics_mmcounters_t;


/*
 * VQmon stream transport packet metrics block.
 */
struct _vqmon_streammetrics_pkttrans_s
{
    tcmyU32                     nPktsRcvd;
    tcmyU32                     nPktsCorrected;
    tcmyU32                     nPktsLost;
    tcmyU32                     nPktsDiscarded;
    tcmyU32                     nPktsOutOfSeq;
    tcmyU32                     nPktsDuplicated;

    vqmon_proportion_t          nNetworkLossProp;
    vqmon_proportion_t          nCorrectedLossProp;
    vqmon_proportion_t          nDiscardProp;
    vqmon_proportion_t          nOutOfSeqProp;
    vqmon_proportion_t          nDuplicateProp;

    tcmyU16                     nBurstCount;
    vqmon_proportion_t          nBurstLossRate;
    tcmyU32                     nBurstLengthPkts;
    tcmyU16                     nGapCount;
    vqmon_proportion_t          nGapLossRate;
    tcmyU32                     nGapLengthPkts;

    tcmyU8                      nCurPacketTTL;
    tcmyU8                      nMinPacketTTL;
    tcmyU8                      nMaxPacketTTL;
};

typedef struct _vqmon_streammetrics_pkttrans_s vqmon_streammetrics_pkttrans_t;


/*
 * VQmon stream RTP transport packet metrics block for MPEG-2 transport
 *   over RTP.
 */
struct _vqmon_streammetrics_rtppkts_s
{
    tcmyU32                     nPktsRcvd;
    tcmyU32                     nPktsCorrected;
    tcmyU32                     nPktsLost;
    tcmyU32                     nPktsDiscarded;
    tcmyU32                     nPktsOutOfSeq;
    tcmyU32                     nPktsDuplicated;

    tcmyU32                     nTimestampError;
};

typedef struct _vqmon_streammetrics_rtppkts_s vqmon_streammetrics_rtppkts_t;


/*
 * VQmon stream transport packet jitter metrics block.
 */
struct _vqmon_streammetrics_jitter_s
{
    vqmon_ufpq4_t               nPPDVMs;
    vqmon_ufpq4_t               nMaxPPDVMs;
};

typedef struct _vqmon_streammetrics_jitter_s vqmon_streammetrics_jitter_t;


/*
 * VQmon MPEG-2 transport stream PCR jitter metrics block.
 */
struct _vqmon_streammetrics_pcrjitter_s
{
    vqmon_ufpq4_t               nPCRJitterMs;
    vqmon_ufpq4_t               nMaxPCRJitterMs;
};

typedef struct _vqmon_streammetrics_pcrjitter_s vqmon_streammetrics_pcrjitter_t;


/*
 * VQmon stream transport packet delay metrics block.
 */
struct _vqmon_streammetrics_delay_s
{
    vqmon_streammetricsource_t  eRTDelaySource;
    tcmyS16                     nInstRTDelayMs;
    tcmyS16                     nAvgRTDelayMs;
    tcmyS16                     nMaxRTDelayMs;

    vqmon_streammetricsource_t  eOWDelaySource;
    tcmyS16                     nInstOWDelayMs;
    tcmyS16                     nAvgOWDelayMs;
    tcmyS16                     nMaxOWDelayMs;
};

typedef struct _vqmon_streammetrics_delay_s vqmon_streammetrics_delay_t;


struct _vqmon_streammetrics_esdelay_s
{
    tcmyS16                     nInstOrigESDelayMs;
    tcmyS16                     nAvgOrigESDelayMs;
    tcmyS16                     nMaxOrigESDelayMs;
    vqmon_streammetricsource_t  eOrigESDelaySource;

    tcmyS16                     nInstTermESDelayMs;
    tcmyS16                     nAvgTermESDelayMs;
    tcmyS16                     nMaxTermESDelayMs;
    vqmon_streammetricsource_t  eTermESDelaySource;
};

typedef struct _vqmon_streammetrics_esdelay_s vqmon_streammetrics_esdelay_t;

/*
 * VQmon stream Forward Error Correction (FEC) effectiveness statistics.
 */
struct _vqmon_streammetrics_fecstats_s
{
    vqmon_percent_t             nFECEffectiveness;
    tcmyU8                      nOptimumFECBlockSz;
    tcmyU8                      nOptimumFECCorrPkts;
};

typedef struct _vqmon_streammetrics_fecstats_s vqmon_streammetrics_fecstats_t;


/*
 * VQmon stream Reliable UDP effectiveness statistics.
 */
struct _vqmon_streammetrics_reliableudp_s
{
    vqmon_proportion_t          nPropRetransmission;
    vqmon_proportion_t          nPeakToMeanBWRatio;
};

typedef struct _vqmon_streammetrics_reliableudp_s \
    vqmon_streammetrics_reliableudp_t;


/*
 * RTCP Sender Report information
 */

struct _vqmon_streammetrics_rtcprrblock_s
{
    vqmon_handle_t              hStream;
    tcmyU32                     nSSRC;
    tcmyU8                      nFractionLost;
    tcmyU32                     nPktsLost;
    tcmyU32                     nExtHighestSeqNumber;
    tcmyU32                     nInterArrivalJitter;
    tcmyU32                     nLSR;
    tcmyU32                     nDLSR;
};

typedef struct _vqmon_streammetrics_rtcprrblock_s \
            vqmon_streammetrics_rtcprrblock_t;

struct _vqmon_streammetrics_rtcpsr_s
{
    tcmyU32                             nSenderSSRC;
    tcmyU32                             nSenderInfoNTPTimeMSW;
    tcmyU32                             nSenderInfoNTPTimeLSW;
    tcmyU32                             nSenderInfoRTPTime;
    tcmyU32                             nSenderInfoPkts;
    tcmyU32                             nSenderInfoOctets;

    tcmyU8                              nRRCount;
    vqmon_streammetrics_rtcprrblock_t   aReceptionReports[1];
};

typedef struct _vqmon_streammetrics_rtcpsr_s vqmon_streammetrics_rtcpsr_t;


struct _vqmon_streammetrics_rtcpmetrics_s
{
    tcmyU32                             nPktsRcvd;

    tcmyU16                             nSDESPDUsRcvd;
    tcmyU16                             nSRPDUsRcvd;
    tcmyU16                             nRRPDUsRcvd;
    tcmyU16                             nRRBlocksRcvd;
    tcmyU16                             nXRVoIPBlocksRcvd;
    tcmyU16                             nOtherPDUsRcvd;
};

typedef struct _vqmon_streammetrics_rtcpmetrics_s \
            vqmon_streammetrics_rtcpmetrics_t;

struct _vqmon_streammetrics_rtcpsdes_s
{
    tcmyCHAR                            CNAME[VQMON_MAXSTRLENGTH];
    tcmyCHAR                            NAME[VQMON_MAXSTRLENGTH];
    tcmyCHAR                            PHONE[VQMON_MAXSTRLENGTH];    
    tcmyCHAR                            TOOL[VQMON_MAXSTRLENGTH];
};

typedef struct _vqmon_streammetrics_rtcpsdes_s vqmon_streammetrics_rtcpsdes_t;

/*
 * Inclusion of media type-specific constants, enumerations and data types.
 */
#ifdef VQMON_BUILDCFG_HAVEAUDIO
#include "vqmonaud.h"
#endif
#ifdef VQMON_BUILDCFG_HAVEVIDEO
#include "vqmonvid.h"
#endif
#ifdef VQMON_BUILDCFG_HAVEVOICE
#include "vqmonvce.h"
#endif
#ifdef VQMON_BUILDCFG_HAVETCPSTRM
#include "vqmontcp.h"
#include "vqmonsvd.h"
#endif
#ifdef VQMON_BUILDCFG_HAVEDATA
#include "vqmondat.h"
#endif

/* --------------------------------------------------------------------------
 *
 *      VQmon Public Multimedia Stream Functions
 *
 * --------------------------------------------------------------------------
 */

/*
 * VQmon stream methods.
 */
VQMON_PUBLIC(vqmon_result_t)
VQmonStreamCreate(
    vqmon_handle_t* const phStream,
    const vqmon_handle_t hInterface,
    const vqmon_streamtype_t eStreamType
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamDestroy(
    vqmon_handle_t hStream
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamConfigGet(
    const vqmon_handle_t hStream,
    const vqmon_streamcfgblkid_t idConfigBlock,
    tcmyU32* const pnConfigBufSz,
    void* const pConfigBuf
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamConfigSet(
    vqmon_handle_t const hStream,
    const vqmon_streamcfgblkid_t idConfigBlock,
    const tcmyU32 nConfigBufSz,
    const void* pConfigBuf
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamMetricsGet(
    const vqmon_handle_t hStream,
    const vqmon_streammetricblkid_t idMetricsBlock,
    tcmyU32* const pnMetricsBufSz,
    void* const pMetricsBuf
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamIndicatePacket(
    vqmon_handle_t const hStream,
    const vqmon_time_t tRcvTimestamp,
    vqmon_pktdesc_t* pPktDesc
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamIndicateEvent(
    vqmon_handle_t const hStream,
    const vqmon_streamprogevttype_t eEventType,
    const vqmon_time_t tEventTimestamp,
    const tcmyU32 nEventDetailSz,
    const void* pEventDetail
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamAlertsClear(
    vqmon_handle_t const hStream,
    const vqmon_alerttype_t eAlertType
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamReportGenerate(
    vqmon_handle_t const hStream,
    const vqmon_streamreporttype_t eReportType,
    const vqmon_time_t tReportTimestamp,
    const void* pReportInfo,
    tcmyU32* const pnReportBufSz,
    void* const pReportBuf
    );

VQMON_PUBLIC(vqmon_result_t)
VQmonStreamIndicateAssocFECPacket(
    vqmon_handle_t const hStream,
    const vqmon_time_t tRcvTimestamp,
    vqmon_pktdesc_t* pPktDesc
    );


#ifdef __cplusplus
}
#endif

#endif  /* _TELCHEMY_VQMONSTM_H_ */
